// package proxy is a wrapper around proxyguard that integrates it with eduvpn-common settings
// - leaves out some options not applicable to the common integration, e.g. fwmark
// - integrates with eduvpn-common's logger
// - integrates eduvpn-common's user agent
package proxy

import (
	"context"

	"codeberg.org/eduVPN/proxyguard"

	"codeberg.org/eduVPN/eduvpn-common/i18nerr"
	httpw "codeberg.org/eduVPN/eduvpn-common/internal/http"
	"codeberg.org/eduVPN/eduvpn-common/internal/log"
)

// Logger is defined here such that we can update the proxyguard logger
type Logger struct{}

// Logf logs a message with parameters
func (l *Logger) Logf(msg string, params ...interface{}) {
	log.Logger.Infof("[Proxyguard] "+msg, params...)
}

// Log logs a message
func (l *Logger) Log(msg string) {
	log.Logger.Infof("[Proxyguard] %s", msg)
}

type Proxy struct {
	proxyguard.Client
	resChan chan struct{}
}

// NewProxyguard sets up proxyguard for proxied WireGuard connections
func NewProxyguard(ctx context.Context, lp int, tcpsp int, peer string, setupSocket func(fd int)) (*Proxy, error) {
	proxyguard.UpdateLogger(&Logger{})
	proxy := Proxy{
		Client: proxyguard.Client{
			Peer:          peer,
			ListenPort:    lp,
			TCPSourcePort: tcpsp,
			SetupSocket:   setupSocket,
			UserAgent:     httpw.UserAgent,
		},
		resChan: make(chan struct{}),
	}
	_, err := proxy.Client.Setup(ctx)
	if err != nil {
		return nil, i18nerr.WrapInternal(err, "The ProxyGuard DNS could not be resolved")
	}

	return &proxy, nil
}

func (p *Proxy) Tunnel(ctx context.Context, wglisten int) error {
	log.Logger.Infof("callying tunnel")
	errChan := make(chan error, 1)
	gctx, cancel := context.WithCancel(ctx)
	go func() {
		err := p.Client.Tunnel(gctx, wglisten)
		if err != nil {
			err = i18nerr.WrapInternal(err, "The VPN proxy exited")
		}
		errChan <- err
	}()

	select {
	case err := <-errChan:
		cancel()
		return err
	case <-p.resChan:
		cancel()
		<-errChan
		return p.Tunnel(ctx, wglisten)
	}
}

func (p *Proxy) Restart() {
	p.resChan <- struct{}{}
}
