pub const ext = @import("ext.zig");
const glib = @This();

const std = @import("std");
const compat = @import("compat");
/// Integer representing a day of the month; between 1 and 31.
///
/// The `G_DATE_BAD_DAY` value represents an invalid day of the month.
pub const DateDay = u8;

/// Integer type representing a year.
///
/// The `G_DATE_BAD_YEAR` value is the invalid value. The year
/// must be 1 or higher; negative ([BCE](https://en.wikipedia.org/wiki/Common_Era))
/// years are not allowed.
///
/// The year is represented with four digits.
pub const DateYear = u16;

/// Opaque type. See `g_main_context_pusher_new` for details.
pub const MainContextPusher = void;

/// Opaque type. See `g_mutex_locker_new` for details.
pub const MutexLocker = void;

/// A type which is used to hold a process identification.
///
/// On UNIX, processes are identified by a process id (an integer),
/// while Windows uses process handles (which are pointers).
///
/// GPid is used in GLib only for descendant processes spawned with
/// the g_spawn functions.
pub const Pid = c_int;

/// A GQuark is a non-zero integer which uniquely identifies a
/// particular string.
///
/// A GQuark value of zero is associated to `NULL`.
///
/// Given either the string or the `GQuark` identifier it is possible to
/// retrieve the other.
///
/// Quarks are used for both
/// [datasets and keyed data lists](datalist-and-dataset.html).
///
/// To create a new quark from a string, use `glib.quarkFromString`
/// or `glib.quarkFromStaticString`.
///
/// To find the string corresponding to a given `GQuark`, use
/// `glib.quarkToString`.
///
/// To find the `GQuark` corresponding to a given string, use
/// `glib.quarkTryString`.
///
/// Another use for the string pool maintained for the quark functions
/// is string interning, using `glib.internString` or
/// `glib.internStaticString`. An interned string is a canonical
/// representation for a string. One important advantage of interned
/// strings is that they can be compared for equality by a simple
/// pointer comparison, rather than using ``strcmp``.
pub const Quark = u32;

/// Opaque type. See `g_rw_lock_reader_locker_new` for details.
pub const RWLockReaderLocker = void;

/// Opaque type. See `g_rw_lock_writer_locker_new` for details.
pub const RWLockWriterLocker = void;

/// Opaque type. See `g_rec_mutex_locker_new` for details.
pub const RecMutexLocker = void;

/// A typedef for a reference-counted string. A pointer to a `glib.RefString` can be
/// treated like a standard `char*` array by all code, but can additionally have
/// `g_ref_string_*()` methods called on it. `g_ref_string_*()` methods cannot be
/// called on `char*` arrays not allocated using `glib.refStringNew`.
///
/// If using `glib.RefString` with autocleanups, `g_autoptr` must be used rather than
/// `g_autofree`, so that the reference counting metadata is also freed.
pub const RefString = u8;

/// A typedef alias for gchar**. This is mostly useful when used together with
/// ``g_auto``.
pub const Strv = *[*:0]u8;

/// Simply a replacement for `time_t`. It has been deprecated
/// since it is not equivalent to `time_t` on 64-bit platforms
/// with a 64-bit `time_t`.
///
/// Unrelated to `glib.Timer`.
///
/// Note that `glib.Time` is defined to always be a 32-bit integer,
/// unlike `time_t` which may be 64-bit on some systems. Therefore,
/// `glib.Time` will overflow in the year 2038, and you cannot use the
/// address of a `glib.Time` variable as argument to the UNIX `time`
/// function.
///
/// Instead, do the following:
///
/// ```
/// time_t ttime;
/// GTime gtime;
///
/// time (&ttime);
/// gtime = (GTime)ttime;
/// ```
pub const Time = i32;

/// A value representing an interval of time, in microseconds.
pub const TimeSpan = i64;

pub const Allocator = opaque {
    extern fn g_allocator_new(p_name: [*:0]const u8, p_n_preallocs: c_uint) *glib.Allocator;
    pub const new = g_allocator_new;

    extern fn g_allocator_free(p_allocator: *Allocator) void;
    pub const free = g_allocator_free;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Contains the public fields of a GArray.
pub const Array = extern struct {
    /// a pointer to the element data. The data may be moved as
    ///     elements are added to the `glib.Array`.
    f_data: ?[*:0]u8,
    /// the number of elements in the `glib.Array` not including the
    ///     possible terminating zero element.
    f_len: c_uint,

    /// Adds `len` elements onto the end of the array.
    extern fn g_array_append_vals(p_array: *glib.Array, p_data: *const anyopaque, p_len: c_uint) *glib.Array;
    pub const appendVals = g_array_append_vals;

    /// Checks whether `target` exists in `array` by performing a binary
    /// search based on the given comparison function `compare_func` which
    /// get pointers to items as arguments. If the element is found, `TRUE`
    /// is returned and the element’s index is returned in `out_match_index`
    /// (if non-`NULL`). Otherwise, `FALSE` is returned and `out_match_index`
    /// is undefined. If `target` exists multiple times in `array`, the index
    /// of the first instance is returned. This search is using a binary
    /// search, so the `array` must absolutely be sorted to return a correct
    /// result (if not, the function may produce false-negative).
    ///
    /// This example defines a comparison function and search an element in a `glib.Array`:
    /// ```
    /// static gint
    /// cmpint (gconstpointer a, gconstpointer b)
    /// {
    ///   const gint *_a = a;
    ///   const gint *_b = b;
    ///
    ///   return *_a - *_b;
    /// }
    /// ...
    /// gint i = 424242;
    /// guint matched_index;
    /// gboolean result = g_array_binary_search (garray, &i, cmpint, &matched_index);
    /// ...
    /// ```
    extern fn g_array_binary_search(p_array: *glib.Array, p_target: ?*const anyopaque, p_compare_func: glib.CompareFunc, p_out_match_index: ?*c_uint) c_int;
    pub const binarySearch = g_array_binary_search;

    /// Create a shallow copy of a `glib.Array`. If the array elements consist of
    /// pointers to data, the pointers are copied but the actual data is not.
    extern fn g_array_copy(p_array: *glib.Array) *glib.Array;
    pub const copy = g_array_copy;

    /// Frees the memory allocated for the `glib.Array`. If `free_segment` is
    /// `TRUE` it frees the memory block holding the elements as well. Pass
    /// `FALSE` if you want to free the `glib.Array` wrapper but preserve the
    /// underlying array for use elsewhere. If the reference count of
    /// `array` is greater than one, the `glib.Array` wrapper is preserved but
    /// the size of  `array` will be set to zero.
    ///
    /// If array contents point to dynamically-allocated memory, they should
    /// be freed separately if `free_segment` is `TRUE` and no `clear_func`
    /// function has been set for `array`.
    ///
    /// This function is not thread-safe. If using a `glib.Array` from multiple
    /// threads, use only the atomic `glib.Array.ref` and `glib.Array.unref`
    /// functions.
    extern fn g_array_free(p_array: *glib.Array, p_free_segment: c_int) [*:0]u8;
    pub const free = g_array_free;

    /// Gets the size of the elements in `array`.
    extern fn g_array_get_element_size(p_array: *glib.Array) c_uint;
    pub const getElementSize = g_array_get_element_size;

    /// Inserts `len` elements into a `glib.Array` at the given index.
    ///
    /// If `index_` is greater than the array’s current length, the array is expanded.
    /// The elements between the old end of the array and the newly inserted elements
    /// will be initialised to zero if the array was configured to clear elements;
    /// otherwise their values will be undefined.
    ///
    /// If `index_` is less than the array’s current length, new entries will be
    /// inserted into the array, and the existing entries above `index_` will be moved
    /// upwards.
    ///
    /// `data` may be `NULL` if (and only if) `len` is zero. If `len` is zero, this
    /// function is a no-op.
    extern fn g_array_insert_vals(p_array: *glib.Array, p_index_: c_uint, p_data: ?*const anyopaque, p_len: c_uint) *glib.Array;
    pub const insertVals = g_array_insert_vals;

    /// Creates a new `glib.Array` with a reference count of 1.
    extern fn g_array_new(p_zero_terminated: c_int, p_clear_: c_int, p_element_size: c_uint) *glib.Array;
    pub const new = g_array_new;

    /// Creates a new `glib.Array` with `data` as array data, `len` as length and a
    /// reference count of 1.
    ///
    /// This avoids having to copy the data manually, when it can just be
    /// inherited.
    /// After this call, `data` belongs to the `glib.Array` and may no longer be
    /// modified by the caller. The memory of `data` has to be dynamically
    /// allocated and will eventually be freed with `glib.free`.
    ///
    /// In case the elements need to be cleared when the array is freed, use
    /// `glib.Array.setClearFunc` to set a `glib.DestroyNotify` function to perform
    /// such task.
    ///
    /// Do not use it if `len` or `element_size` are greater than `G_MAXUINT`.
    /// `glib.Array` stores the length of its data in `guint`, which may be shorter
    /// than `gsize`.
    extern fn g_array_new_take(p_data: ?[*]*anyopaque, p_len: usize, p_clear: c_int, p_element_size: usize) *glib.Array;
    pub const newTake = g_array_new_take;

    /// Creates a new `glib.Array` with `data` as array data, computing the length of it
    /// and setting the reference count to 1.
    ///
    /// This avoids having to copy the data manually, when it can just be
    /// inherited.
    /// After this call, `data` belongs to the `glib.Array` and may no longer be
    /// modified by the caller. The memory of `data` has to be dynamically
    /// allocated and will eventually be freed with `glib.free`.
    ///
    /// The length is calculated by iterating through `data` until the first `NULL`
    /// element is found.
    ///
    /// In case the elements need to be cleared when the array is freed, use
    /// `glib.Array.setClearFunc` to set a `glib.DestroyNotify` function to perform
    /// such task.
    ///
    /// Do not use it if `data` length or `element_size` are greater than `G_MAXUINT`.
    /// `glib.Array` stores the length of its data in `guint`, which may be shorter
    /// than `gsize`.
    extern fn g_array_new_take_zero_terminated(p_data: [*]*anyopaque, p_clear: c_int, p_element_size: usize) *glib.Array;
    pub const newTakeZeroTerminated = g_array_new_take_zero_terminated;

    /// Adds `len` elements onto the start of the array.
    ///
    /// `data` may be `NULL` if (and only if) `len` is zero. If `len` is zero, this
    /// function is a no-op.
    ///
    /// This operation is slower than `glib.Array.appendVals` since the
    /// existing elements in the array have to be moved to make space for
    /// the new elements.
    extern fn g_array_prepend_vals(p_array: *glib.Array, p_data: ?*const anyopaque, p_len: c_uint) *glib.Array;
    pub const prependVals = g_array_prepend_vals;

    /// Atomically increments the reference count of `array` by one.
    /// This function is thread-safe and may be called from any thread.
    extern fn g_array_ref(p_array: *glib.Array) *glib.Array;
    pub const ref = g_array_ref;

    /// Removes the element at the given index from a `glib.Array`. The following
    /// elements are moved down one place.
    extern fn g_array_remove_index(p_array: *glib.Array, p_index_: c_uint) *glib.Array;
    pub const removeIndex = g_array_remove_index;

    /// Removes the element at the given index from a `glib.Array`. The last
    /// element in the array is used to fill in the space, so this function
    /// does not preserve the order of the `glib.Array`. But it is faster than
    /// `glib.Array.removeIndex`.
    extern fn g_array_remove_index_fast(p_array: *glib.Array, p_index_: c_uint) *glib.Array;
    pub const removeIndexFast = g_array_remove_index_fast;

    /// Removes the given number of elements starting at the given index
    /// from a `glib.Array`.  The following elements are moved to close the gap.
    extern fn g_array_remove_range(p_array: *glib.Array, p_index_: c_uint, p_length: c_uint) *glib.Array;
    pub const removeRange = g_array_remove_range;

    /// Sets a function to clear an element of `array`.
    ///
    /// The `clear_func` will be called when an element in the array
    /// data segment is removed and when the array is freed and data
    /// segment is deallocated as well. `clear_func` will be passed a
    /// pointer to the element to clear, rather than the element itself.
    ///
    /// Note that in contrast with other uses of `glib.DestroyNotify`
    /// functions, `clear_func` is expected to clear the contents of
    /// the array element it is given, but not free the element itself.
    ///
    /// ```
    /// typedef struct
    /// {
    ///   gchar *str;
    ///   GObject *obj;
    /// } ArrayElement;
    ///
    /// static void
    /// array_element_clear (ArrayElement *element)
    /// {
    ///   g_clear_pointer (&element->str, g_free);
    ///   g_clear_object (&element->obj);
    /// }
    ///
    /// // main code
    /// GArray *garray = g_array_new (FALSE, FALSE, sizeof (ArrayElement));
    /// g_array_set_clear_func (garray, (GDestroyNotify) array_element_clear);
    /// // assign data to the structure
    /// g_array_free (garray, TRUE);
    /// ```
    extern fn g_array_set_clear_func(p_array: *glib.Array, p_clear_func: glib.DestroyNotify) void;
    pub const setClearFunc = g_array_set_clear_func;

    /// Sets the size of the array, expanding it if necessary. If the array
    /// was created with `clear_` set to `TRUE`, the new elements are set to 0.
    extern fn g_array_set_size(p_array: *glib.Array, p_length: c_uint) *glib.Array;
    pub const setSize = g_array_set_size;

    /// Creates a new `glib.Array` with `reserved_size` elements preallocated and
    /// a reference count of 1. This avoids frequent reallocation, if you
    /// are going to add many elements to the array. Note however that the
    /// size of the array is still 0.
    extern fn g_array_sized_new(p_zero_terminated: c_int, p_clear_: c_int, p_element_size: c_uint, p_reserved_size: c_uint) *glib.Array;
    pub const sizedNew = g_array_sized_new;

    /// Sorts a `glib.Array` using `compare_func` which should be a `qsort`-style
    /// comparison function (returns less than zero for first arg is less
    /// than second arg, zero for equal, greater zero if first arg is
    /// greater than second arg).
    ///
    /// This is guaranteed to be a stable sort since version 2.32.
    extern fn g_array_sort(p_array: *glib.Array, p_compare_func: glib.CompareFunc) void;
    pub const sort = g_array_sort;

    /// Like `glib.Array.sort`, but the comparison function receives an extra
    /// user data argument.
    ///
    /// This is guaranteed to be a stable sort since version 2.32.
    ///
    /// There used to be a comment here about making the sort stable by
    /// using the addresses of the elements in the comparison function.
    /// This did not actually work, so any such code should be removed.
    extern fn g_array_sort_with_data(p_array: *glib.Array, p_compare_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const sortWithData = g_array_sort_with_data;

    /// Frees the data in the array and resets the size to zero, while
    /// the underlying array is preserved for use elsewhere and returned
    /// to the caller.
    ///
    /// If the array was created with the `zero_terminate` property
    /// set to `TRUE`, the returned data is zero terminated too.
    ///
    /// If array elements contain dynamically-allocated memory,
    /// the array elements should also be freed by the caller.
    ///
    /// A short example of use:
    /// ```
    /// ...
    /// gpointer data;
    /// gsize data_len;
    /// data = g_array_steal (some_array, &data_len);
    /// ...
    /// ```
    extern fn g_array_steal(p_array: *glib.Array, p_len: ?*usize) ?*anyopaque;
    pub const steal = g_array_steal;

    /// Atomically decrements the reference count of `array` by one. If the
    /// reference count drops to 0, the effect is the same as calling
    /// `glib.Array.free` with `free_segment` set to `TRUE`. This function is
    /// thread-safe and may be called from any thread.
    extern fn g_array_unref(p_array: *glib.Array) void;
    pub const unref = g_array_unref;

    extern fn g_array_get_type() usize;
    pub const getGObjectType = g_array_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// An opaque data structure which represents an asynchronous queue.
///
/// It should only be accessed through the `g_async_queue_*` functions.
pub const AsyncQueue = opaque {
    /// Creates a new asynchronous queue.
    extern fn g_async_queue_new() *glib.AsyncQueue;
    pub const new = g_async_queue_new;

    /// Creates a new asynchronous queue and sets up a destroy notify
    /// function that is used to free any remaining queue items when
    /// the queue is destroyed after the final unref.
    extern fn g_async_queue_new_full(p_item_free_func: ?glib.DestroyNotify) *glib.AsyncQueue;
    pub const newFull = g_async_queue_new_full;

    /// Returns the length of the queue.
    ///
    /// Actually this function returns the number of data items in
    /// the queue minus the number of waiting threads, so a negative
    /// value means waiting threads, and a positive value means available
    /// entries in the `queue`. A return value of 0 could mean n entries
    /// in the queue and n threads waiting. This can happen due to locking
    /// of the queue or due to scheduling.
    extern fn g_async_queue_length(p_queue: *AsyncQueue) c_int;
    pub const length = g_async_queue_length;

    /// Returns the length of the queue.
    ///
    /// Actually this function returns the number of data items in
    /// the queue minus the number of waiting threads, so a negative
    /// value means waiting threads, and a positive value means available
    /// entries in the `queue`. A return value of 0 could mean n entries
    /// in the queue and n threads waiting. This can happen due to locking
    /// of the queue or due to scheduling.
    ///
    /// This function must be called while holding the `queue`'s lock.
    extern fn g_async_queue_length_unlocked(p_queue: *AsyncQueue) c_int;
    pub const lengthUnlocked = g_async_queue_length_unlocked;

    /// Acquires the `queue`'s lock. If another thread is already
    /// holding the lock, this call will block until the lock
    /// becomes available.
    ///
    /// Call `glib.AsyncQueue.unlock` to drop the lock again.
    ///
    /// While holding the lock, you can only call the
    /// g_async_queue_*`_unlocked` functions on `queue`. Otherwise,
    /// deadlock may occur.
    extern fn g_async_queue_lock(p_queue: *AsyncQueue) void;
    pub const lock = g_async_queue_lock;

    /// Pops data from the `queue`. If `queue` is empty, this function
    /// blocks until data becomes available.
    extern fn g_async_queue_pop(p_queue: *AsyncQueue) ?*anyopaque;
    pub const pop = g_async_queue_pop;

    /// Pops data from the `queue`. If `queue` is empty, this function
    /// blocks until data becomes available.
    ///
    /// This function must be called while holding the `queue`'s lock.
    extern fn g_async_queue_pop_unlocked(p_queue: *AsyncQueue) ?*anyopaque;
    pub const popUnlocked = g_async_queue_pop_unlocked;

    /// Pushes the `data` into the `queue`.
    ///
    /// The `data` parameter must not be `NULL`.
    extern fn g_async_queue_push(p_queue: *AsyncQueue, p_data: *anyopaque) void;
    pub const push = g_async_queue_push;

    /// Pushes the `item` into the `queue`. `item` must not be `NULL`.
    /// In contrast to `glib.AsyncQueue.push`, this function
    /// pushes the new item ahead of the items already in the queue,
    /// so that it will be the next one to be popped off the queue.
    extern fn g_async_queue_push_front(p_queue: *AsyncQueue, p_item: *anyopaque) void;
    pub const pushFront = g_async_queue_push_front;

    /// Pushes the `item` into the `queue`. `item` must not be `NULL`.
    /// In contrast to `glib.AsyncQueue.pushUnlocked`, this function
    /// pushes the new item ahead of the items already in the queue,
    /// so that it will be the next one to be popped off the queue.
    ///
    /// This function must be called while holding the `queue`'s lock.
    extern fn g_async_queue_push_front_unlocked(p_queue: *AsyncQueue, p_item: *anyopaque) void;
    pub const pushFrontUnlocked = g_async_queue_push_front_unlocked;

    /// Inserts `data` into `queue` using `func` to determine the new
    /// position.
    ///
    /// This function requires that the `queue` is sorted before pushing on
    /// new elements, see `glib.AsyncQueue.sort`.
    ///
    /// This function will lock `queue` before it sorts the queue and unlock
    /// it when it is finished.
    ///
    /// For an example of `func` see `glib.AsyncQueue.sort`.
    extern fn g_async_queue_push_sorted(p_queue: *AsyncQueue, p_data: *anyopaque, p_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const pushSorted = g_async_queue_push_sorted;

    /// Inserts `data` into `queue` using `func` to determine the new
    /// position.
    ///
    /// The sort function `func` is passed two elements of the `queue`.
    /// It should return 0 if they are equal, a negative value if the
    /// first element should be higher in the `queue` or a positive value
    /// if the first element should be lower in the `queue` than the second
    /// element.
    ///
    /// This function requires that the `queue` is sorted before pushing on
    /// new elements, see `glib.AsyncQueue.sort`.
    ///
    /// This function must be called while holding the `queue`'s lock.
    ///
    /// For an example of `func` see `glib.AsyncQueue.sort`.
    extern fn g_async_queue_push_sorted_unlocked(p_queue: *AsyncQueue, p_data: ?*anyopaque, p_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const pushSortedUnlocked = g_async_queue_push_sorted_unlocked;

    /// Pushes the `data` into the `queue`.
    ///
    /// The `data` parameter must not be `NULL`.
    ///
    /// This function must be called while holding the `queue`'s lock.
    extern fn g_async_queue_push_unlocked(p_queue: *AsyncQueue, p_data: *anyopaque) void;
    pub const pushUnlocked = g_async_queue_push_unlocked;

    /// Increases the reference count of the asynchronous `queue` by 1.
    /// You do not need to hold the lock to call this function.
    extern fn g_async_queue_ref(p_queue: *AsyncQueue) *glib.AsyncQueue;
    pub const ref = g_async_queue_ref;

    /// Increases the reference count of the asynchronous `queue` by 1.
    extern fn g_async_queue_ref_unlocked(p_queue: *AsyncQueue) void;
    pub const refUnlocked = g_async_queue_ref_unlocked;

    /// Remove an item from the queue.
    extern fn g_async_queue_remove(p_queue: *AsyncQueue, p_item: *anyopaque) c_int;
    pub const remove = g_async_queue_remove;

    /// Remove an item from the queue.
    ///
    /// This function must be called while holding the `queue`'s lock.
    extern fn g_async_queue_remove_unlocked(p_queue: *AsyncQueue, p_item: ?*anyopaque) c_int;
    pub const removeUnlocked = g_async_queue_remove_unlocked;

    /// Sorts `queue` using `func`.
    ///
    /// The sort function `func` is passed two elements of the `queue`.
    /// It should return 0 if they are equal, a negative value if the
    /// first element should be higher in the `queue` or a positive value
    /// if the first element should be lower in the `queue` than the second
    /// element.
    ///
    /// This function will lock `queue` before it sorts the queue and unlock
    /// it when it is finished.
    ///
    /// If you were sorting a list of priority numbers to make sure the
    /// lowest priority would be at the top of the queue, you could use:
    /// ```
    ///  gint32 id1;
    ///  gint32 id2;
    ///
    ///  id1 = GPOINTER_TO_INT (element1);
    ///  id2 = GPOINTER_TO_INT (element2);
    ///
    ///  return (id1 > id2 ? +1 : id1 == id2 ? 0 : -1);
    /// ```
    extern fn g_async_queue_sort(p_queue: *AsyncQueue, p_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const sort = g_async_queue_sort;

    /// Sorts `queue` using `func`.
    ///
    /// The sort function `func` is passed two elements of the `queue`.
    /// It should return 0 if they are equal, a negative value if the
    /// first element should be higher in the `queue` or a positive value
    /// if the first element should be lower in the `queue` than the second
    /// element.
    ///
    /// This function must be called while holding the `queue`'s lock.
    extern fn g_async_queue_sort_unlocked(p_queue: *AsyncQueue, p_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const sortUnlocked = g_async_queue_sort_unlocked;

    /// Pops data from the `queue`. If the queue is empty, blocks until
    /// `end_time` or until data becomes available.
    ///
    /// If no data is received before `end_time`, `NULL` is returned.
    ///
    /// To easily calculate `end_time`, a combination of `glib.getRealTime`
    /// and `glib.TimeVal.add` can be used.
    extern fn g_async_queue_timed_pop(p_queue: *AsyncQueue, p_end_time: *glib.TimeVal) ?*anyopaque;
    pub const timedPop = g_async_queue_timed_pop;

    /// Pops data from the `queue`. If the queue is empty, blocks until
    /// `end_time` or until data becomes available.
    ///
    /// If no data is received before `end_time`, `NULL` is returned.
    ///
    /// To easily calculate `end_time`, a combination of `glib.getRealTime`
    /// and `glib.TimeVal.add` can be used.
    ///
    /// This function must be called while holding the `queue`'s lock.
    extern fn g_async_queue_timed_pop_unlocked(p_queue: *AsyncQueue, p_end_time: *glib.TimeVal) ?*anyopaque;
    pub const timedPopUnlocked = g_async_queue_timed_pop_unlocked;

    /// Pops data from the `queue`. If the queue is empty, blocks for
    /// `timeout` microseconds, or until data becomes available.
    ///
    /// If no data is received before the timeout, `NULL` is returned.
    extern fn g_async_queue_timeout_pop(p_queue: *AsyncQueue, p_timeout: u64) ?*anyopaque;
    pub const timeoutPop = g_async_queue_timeout_pop;

    /// Pops data from the `queue`. If the queue is empty, blocks for
    /// `timeout` microseconds, or until data becomes available.
    ///
    /// If no data is received before the timeout, `NULL` is returned.
    ///
    /// This function must be called while holding the `queue`'s lock.
    extern fn g_async_queue_timeout_pop_unlocked(p_queue: *AsyncQueue, p_timeout: u64) ?*anyopaque;
    pub const timeoutPopUnlocked = g_async_queue_timeout_pop_unlocked;

    /// Tries to pop data from the `queue`. If no data is available,
    /// `NULL` is returned.
    extern fn g_async_queue_try_pop(p_queue: *AsyncQueue) ?*anyopaque;
    pub const tryPop = g_async_queue_try_pop;

    /// Tries to pop data from the `queue`. If no data is available,
    /// `NULL` is returned.
    ///
    /// This function must be called while holding the `queue`'s lock.
    extern fn g_async_queue_try_pop_unlocked(p_queue: *AsyncQueue) ?*anyopaque;
    pub const tryPopUnlocked = g_async_queue_try_pop_unlocked;

    /// Releases the queue's lock.
    ///
    /// Calling this function when you have not acquired
    /// the with `glib.AsyncQueue.lock` leads to undefined
    /// behaviour.
    extern fn g_async_queue_unlock(p_queue: *AsyncQueue) void;
    pub const unlock = g_async_queue_unlock;

    /// Decreases the reference count of the asynchronous `queue` by 1.
    ///
    /// If the reference count went to 0, the `queue` will be destroyed
    /// and the memory allocated will be freed. So you are not allowed
    /// to use the `queue` afterwards, as it might have disappeared.
    /// You do not need to hold the lock to call this function.
    extern fn g_async_queue_unref(p_queue: *AsyncQueue) void;
    pub const unref = g_async_queue_unref;

    /// Decreases the reference count of the asynchronous `queue` by 1
    /// and releases the lock. This function must be called while holding
    /// the `queue`'s lock. If the reference count went to 0, the `queue`
    /// will be destroyed and the memory allocated will be freed.
    extern fn g_async_queue_unref_and_unlock(p_queue: *AsyncQueue) void;
    pub const unrefAndUnlock = g_async_queue_unref_and_unlock;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GBookmarkFile` lets you parse, edit or create files containing bookmarks.
///
/// Bookmarks refer to a URI, along with some meta-data about the resource
/// pointed by the URI like its MIME type, the application that is registering
/// the bookmark and the icon that should be used to represent the bookmark.
/// The data is stored using the
/// [Desktop Bookmark Specification](https://www.freedesktop.org/wiki/Specifications/desktop-bookmark-spec/).
///
/// The syntax of the bookmark files is described in detail inside the
/// Desktop Bookmark Specification, here is a quick summary: bookmark
/// files use a sub-class of the XML Bookmark Exchange Language
/// specification, consisting of valid UTF-8 encoded XML, under the
/// `<xbel>` root element; each bookmark is stored inside a
/// `<bookmark>` element, using its URI: no relative paths can
/// be used inside a bookmark file. The bookmark may have a user defined
/// title and description, to be used instead of the URI. Under the
/// `<metadata>` element, with its owner attribute set to
/// `http://freedesktop.org`, is stored the meta-data about a resource
/// pointed by its URI. The meta-data consists of the resource's MIME
/// type; the applications that have registered a bookmark; the groups
/// to which a bookmark belongs to; a visibility flag, used to set the
/// bookmark as "private" to the applications and groups that has it
/// registered; the URI and MIME type of an icon, to be used when
/// displaying the bookmark inside a GUI.
///
/// Here is an example of a bookmark file:
/// [bookmarks.xbel](https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/glib/tests/bookmarks.xbel)
///
/// A bookmark file might contain more than one bookmark; each bookmark
/// is accessed through its URI.
///
/// The important caveat of bookmark files is that when you add a new
/// bookmark you must also add the application that is registering it, using
/// `glib.BookmarkFile.addApplication` or `glib.BookmarkFile.setApplicationInfo`.
/// If a bookmark has no applications then it won't be dumped when creating
/// the on disk representation, using `glib.BookmarkFile.toData` or
/// `glib.BookmarkFile.toFile`.
pub const BookmarkFile = opaque {
    extern fn g_bookmark_file_error_quark() glib.Quark;
    pub const errorQuark = g_bookmark_file_error_quark;

    /// Creates a new empty `glib.BookmarkFile` object.
    ///
    /// Use `glib.BookmarkFile.loadFromFile`, `glib.BookmarkFile.loadFromData`
    /// or `glib.BookmarkFile.loadFromDataDirs` to read an existing bookmark
    /// file.
    extern fn g_bookmark_file_new() *glib.BookmarkFile;
    pub const new = g_bookmark_file_new;

    /// Adds the application with `name` and `exec` to the list of
    /// applications that have registered a bookmark for `uri` into
    /// `bookmark`.
    ///
    /// Every bookmark inside a `glib.BookmarkFile` must have at least an
    /// application registered.  Each application must provide a name, a
    /// command line useful for launching the bookmark, the number of times
    /// the bookmark has been registered by the application and the last
    /// time the application registered this bookmark.
    ///
    /// If `name` is `NULL`, the name of the application will be the
    /// same returned by `glib.getApplicationName`; if `exec` is `NULL`, the
    /// command line will be a composition of the program name as
    /// returned by `glib.getPrgname` and the "\%u" modifier, which will be
    /// expanded to the bookmark's URI.
    ///
    /// This function will automatically take care of updating the
    /// registrations count and timestamping in case an application
    /// with the same `name` had already registered a bookmark for
    /// `uri` inside `bookmark`.
    ///
    /// If no bookmark for `uri` is found, one is created.
    extern fn g_bookmark_file_add_application(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_name: ?[*:0]const u8, p_exec: ?[*:0]const u8) void;
    pub const addApplication = g_bookmark_file_add_application;

    /// Adds `group` to the list of groups to which the bookmark for `uri`
    /// belongs to.
    ///
    /// If no bookmark for `uri` is found then it is created.
    extern fn g_bookmark_file_add_group(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_group: [*:0]const u8) void;
    pub const addGroup = g_bookmark_file_add_group;

    /// Deeply copies a `bookmark` `glib.BookmarkFile` object to a new one.
    extern fn g_bookmark_file_copy(p_bookmark: *BookmarkFile) *glib.BookmarkFile;
    pub const copy = g_bookmark_file_copy;

    /// Frees a `glib.BookmarkFile`.
    extern fn g_bookmark_file_free(p_bookmark: *BookmarkFile) void;
    pub const free = g_bookmark_file_free;

    /// Gets the time the bookmark for `uri` was added to `bookmark`
    ///
    /// In the event the URI cannot be found, -1 is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_added(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) std.posix.time_t;
    pub const getAdded = g_bookmark_file_get_added;

    /// Gets the time the bookmark for `uri` was added to `bookmark`
    ///
    /// In the event the URI cannot be found, `NULL` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_added_date_time(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) ?*glib.DateTime;
    pub const getAddedDateTime = g_bookmark_file_get_added_date_time;

    /// Gets the registration information of `app_name` for the bookmark for
    /// `uri`.  See `glib.BookmarkFile.setApplicationInfo` for more information about
    /// the returned data.
    ///
    /// The string returned in `app_exec` must be freed.
    ///
    /// In the event the URI cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.  In the
    /// event that no application with name `app_name` has registered a bookmark
    /// for `uri`,  `FALSE` is returned and error is set to
    /// `G_BOOKMARK_FILE_ERROR_APP_NOT_REGISTERED`. In the event that unquoting
    /// the command line fails, an error of the `G_SHELL_ERROR` domain is
    /// set and `FALSE` is returned.
    extern fn g_bookmark_file_get_app_info(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_name: [*:0]const u8, p_exec: ?*[*:0]u8, p_count: ?*c_uint, p_stamp: ?*std.posix.time_t, p_error: ?*?*glib.Error) c_int;
    pub const getAppInfo = g_bookmark_file_get_app_info;

    /// Gets the registration information of `app_name` for the bookmark for
    /// `uri`.  See `glib.BookmarkFile.setApplicationInfo` for more information about
    /// the returned data.
    ///
    /// The string returned in `app_exec` must be freed.
    ///
    /// In the event the URI cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.  In the
    /// event that no application with name `app_name` has registered a bookmark
    /// for `uri`,  `FALSE` is returned and error is set to
    /// `G_BOOKMARK_FILE_ERROR_APP_NOT_REGISTERED`. In the event that unquoting
    /// the command line fails, an error of the `G_SHELL_ERROR` domain is
    /// set and `FALSE` is returned.
    extern fn g_bookmark_file_get_application_info(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_name: [*:0]const u8, p_exec: ?*[*:0]u8, p_count: ?*c_uint, p_stamp: ?**glib.DateTime, p_error: ?*?*glib.Error) c_int;
    pub const getApplicationInfo = g_bookmark_file_get_application_info;

    /// Retrieves the names of the applications that have registered the
    /// bookmark for `uri`.
    ///
    /// In the event the URI cannot be found, `NULL` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_applications(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*][*:0]u8;
    pub const getApplications = g_bookmark_file_get_applications;

    /// Retrieves the description of the bookmark for `uri`.
    ///
    /// In the event the URI cannot be found, `NULL` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_description(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const getDescription = g_bookmark_file_get_description;

    /// Retrieves the list of group names of the bookmark for `uri`.
    ///
    /// In the event the URI cannot be found, `NULL` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    ///
    /// The returned array is `NULL` terminated, so `length` may optionally
    /// be `NULL`.
    extern fn g_bookmark_file_get_groups(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*][*:0]u8;
    pub const getGroups = g_bookmark_file_get_groups;

    /// Gets the icon of the bookmark for `uri`.
    ///
    /// In the event the URI cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_icon(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_href: ?*[*:0]u8, p_mime_type: ?*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const getIcon = g_bookmark_file_get_icon;

    /// Gets whether the private flag of the bookmark for `uri` is set.
    ///
    /// In the event the URI cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.  In the
    /// event that the private flag cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_INVALID_VALUE`.
    extern fn g_bookmark_file_get_is_private(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const getIsPrivate = g_bookmark_file_get_is_private;

    /// Retrieves the MIME type of the resource pointed by `uri`.
    ///
    /// In the event the URI cannot be found, `NULL` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.  In the
    /// event that the MIME type cannot be found, `NULL` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_INVALID_VALUE`.
    extern fn g_bookmark_file_get_mime_type(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const getMimeType = g_bookmark_file_get_mime_type;

    /// Gets the time when the bookmark for `uri` was last modified.
    ///
    /// In the event the URI cannot be found, -1 is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_modified(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) std.posix.time_t;
    pub const getModified = g_bookmark_file_get_modified;

    /// Gets the time when the bookmark for `uri` was last modified.
    ///
    /// In the event the URI cannot be found, `NULL` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_modified_date_time(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) ?*glib.DateTime;
    pub const getModifiedDateTime = g_bookmark_file_get_modified_date_time;

    /// Gets the number of bookmarks inside `bookmark`.
    extern fn g_bookmark_file_get_size(p_bookmark: *BookmarkFile) c_int;
    pub const getSize = g_bookmark_file_get_size;

    /// Returns the title of the bookmark for `uri`.
    ///
    /// If `uri` is `NULL`, the title of `bookmark` is returned.
    ///
    /// In the event the URI cannot be found, `NULL` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_title(p_bookmark: *BookmarkFile, p_uri: ?[*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const getTitle = g_bookmark_file_get_title;

    /// Returns all URIs of the bookmarks in the bookmark file `bookmark`.
    /// The array of returned URIs will be `NULL`-terminated, so `length` may
    /// optionally be `NULL`.
    extern fn g_bookmark_file_get_uris(p_bookmark: *BookmarkFile, p_length: ?*usize) [*][*:0]u8;
    pub const getUris = g_bookmark_file_get_uris;

    /// Gets the time the bookmark for `uri` was last visited.
    ///
    /// In the event the URI cannot be found, -1 is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_visited(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) std.posix.time_t;
    pub const getVisited = g_bookmark_file_get_visited;

    /// Gets the time the bookmark for `uri` was last visited.
    ///
    /// In the event the URI cannot be found, `NULL` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_get_visited_date_time(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) ?*glib.DateTime;
    pub const getVisitedDateTime = g_bookmark_file_get_visited_date_time;

    /// Checks whether the bookmark for `uri` inside `bookmark` has been
    /// registered by application `name`.
    ///
    /// In the event the URI cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_has_application(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const hasApplication = g_bookmark_file_has_application;

    /// Checks whether `group` appears in the list of groups to which
    /// the bookmark for `uri` belongs to.
    ///
    /// In the event the URI cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_has_group(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_group: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const hasGroup = g_bookmark_file_has_group;

    /// Looks whether the desktop bookmark has an item with its URI set to `uri`.
    extern fn g_bookmark_file_has_item(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8) c_int;
    pub const hasItem = g_bookmark_file_has_item;

    /// Loads a bookmark file from memory into an empty `glib.BookmarkFile`
    /// structure.  If the object cannot be created then `error` is set to a
    /// `glib.BookmarkFileError`.
    extern fn g_bookmark_file_load_from_data(p_bookmark: *BookmarkFile, p_data: [*]const u8, p_length: usize, p_error: ?*?*glib.Error) c_int;
    pub const loadFromData = g_bookmark_file_load_from_data;

    /// This function looks for a desktop bookmark file named `file` in the
    /// paths returned from `glib.getUserDataDir` and `glib.getSystemDataDirs`,
    /// loads the file into `bookmark` and returns the file's full path in
    /// `full_path`.  If the file could not be loaded then `error` is
    /// set to either a `glib.FileError` or `glib.BookmarkFileError`.
    extern fn g_bookmark_file_load_from_data_dirs(p_bookmark: *BookmarkFile, p_file: [*:0]const u8, p_full_path: ?*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const loadFromDataDirs = g_bookmark_file_load_from_data_dirs;

    /// Loads a desktop bookmark file into an empty `glib.BookmarkFile` structure.
    /// If the file could not be loaded then `error` is set to either a `glib.FileError`
    /// or `glib.BookmarkFileError`.
    extern fn g_bookmark_file_load_from_file(p_bookmark: *BookmarkFile, p_filename: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const loadFromFile = g_bookmark_file_load_from_file;

    /// Changes the URI of a bookmark item from `old_uri` to `new_uri`.  Any
    /// existing bookmark for `new_uri` will be overwritten.  If `new_uri` is
    /// `NULL`, then the bookmark is removed.
    ///
    /// In the event the URI cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    extern fn g_bookmark_file_move_item(p_bookmark: *BookmarkFile, p_old_uri: [*:0]const u8, p_new_uri: ?[*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const moveItem = g_bookmark_file_move_item;

    /// Removes application registered with `name` from the list of applications
    /// that have registered a bookmark for `uri` inside `bookmark`.
    ///
    /// In the event the URI cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    /// In the event that no application with name `app_name` has registered
    /// a bookmark for `uri`,  `FALSE` is returned and error is set to
    /// `G_BOOKMARK_FILE_ERROR_APP_NOT_REGISTERED`.
    extern fn g_bookmark_file_remove_application(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const removeApplication = g_bookmark_file_remove_application;

    /// Removes `group` from the list of groups to which the bookmark
    /// for `uri` belongs to.
    ///
    /// In the event the URI cannot be found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`.
    /// In the event no group was defined, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_INVALID_VALUE`.
    extern fn g_bookmark_file_remove_group(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_group: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const removeGroup = g_bookmark_file_remove_group;

    /// Removes the bookmark for `uri` from the bookmark file `bookmark`.
    extern fn g_bookmark_file_remove_item(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const removeItem = g_bookmark_file_remove_item;

    /// Sets the time the bookmark for `uri` was added into `bookmark`.
    ///
    /// If no bookmark for `uri` is found then it is created.
    extern fn g_bookmark_file_set_added(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_added: std.posix.time_t) void;
    pub const setAdded = g_bookmark_file_set_added;

    /// Sets the time the bookmark for `uri` was added into `bookmark`.
    ///
    /// If no bookmark for `uri` is found then it is created.
    extern fn g_bookmark_file_set_added_date_time(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_added: *glib.DateTime) void;
    pub const setAddedDateTime = g_bookmark_file_set_added_date_time;

    /// Sets the meta-data of application `name` inside the list of
    /// applications that have registered a bookmark for `uri` inside
    /// `bookmark`.
    ///
    /// You should rarely use this function; use `glib.BookmarkFile.addApplication`
    /// and `glib.BookmarkFile.removeApplication` instead.
    ///
    /// `name` can be any UTF-8 encoded string used to identify an
    /// application.
    /// `exec` can have one of these two modifiers: "\%f", which will
    /// be expanded as the local file name retrieved from the bookmark's
    /// URI; "\%u", which will be expanded as the bookmark's URI.
    /// The expansion is done automatically when retrieving the stored
    /// command line using the `glib.BookmarkFile.getApplicationInfo` function.
    /// `count` is the number of times the application has registered the
    /// bookmark; if is < 0, the current registration count will be increased
    /// by one, if is 0, the application with `name` will be removed from
    /// the list of registered applications.
    /// `stamp` is the Unix time of the last registration; if it is -1, the
    /// current time will be used.
    ///
    /// If you try to remove an application by setting its registration count to
    /// zero, and no bookmark for `uri` is found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`; similarly,
    /// in the event that no application `name` has registered a bookmark
    /// for `uri`,  `FALSE` is returned and error is set to
    /// `G_BOOKMARK_FILE_ERROR_APP_NOT_REGISTERED`.  Otherwise, if no bookmark
    /// for `uri` is found, one is created.
    extern fn g_bookmark_file_set_app_info(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_name: [*:0]const u8, p_exec: [*:0]const u8, p_count: c_int, p_stamp: std.posix.time_t, p_error: ?*?*glib.Error) c_int;
    pub const setAppInfo = g_bookmark_file_set_app_info;

    /// Sets the meta-data of application `name` inside the list of
    /// applications that have registered a bookmark for `uri` inside
    /// `bookmark`.
    ///
    /// You should rarely use this function; use `glib.BookmarkFile.addApplication`
    /// and `glib.BookmarkFile.removeApplication` instead.
    ///
    /// `name` can be any UTF-8 encoded string used to identify an
    /// application.
    /// `exec` can have one of these two modifiers: "\%f", which will
    /// be expanded as the local file name retrieved from the bookmark's
    /// URI; "\%u", which will be expanded as the bookmark's URI.
    /// The expansion is done automatically when retrieving the stored
    /// command line using the `glib.BookmarkFile.getApplicationInfo` function.
    /// `count` is the number of times the application has registered the
    /// bookmark; if is < 0, the current registration count will be increased
    /// by one, if is 0, the application with `name` will be removed from
    /// the list of registered applications.
    /// `stamp` is the Unix time of the last registration.
    ///
    /// If you try to remove an application by setting its registration count to
    /// zero, and no bookmark for `uri` is found, `FALSE` is returned and
    /// `error` is set to `G_BOOKMARK_FILE_ERROR_URI_NOT_FOUND`; similarly,
    /// in the event that no application `name` has registered a bookmark
    /// for `uri`,  `FALSE` is returned and error is set to
    /// `G_BOOKMARK_FILE_ERROR_APP_NOT_REGISTERED`.  Otherwise, if no bookmark
    /// for `uri` is found, one is created.
    extern fn g_bookmark_file_set_application_info(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_name: [*:0]const u8, p_exec: [*:0]const u8, p_count: c_int, p_stamp: ?*glib.DateTime, p_error: ?*?*glib.Error) c_int;
    pub const setApplicationInfo = g_bookmark_file_set_application_info;

    /// Sets `description` as the description of the bookmark for `uri`.
    ///
    /// If `uri` is `NULL`, the description of `bookmark` is set.
    ///
    /// If a bookmark for `uri` cannot be found then it is created.
    extern fn g_bookmark_file_set_description(p_bookmark: *BookmarkFile, p_uri: ?[*:0]const u8, p_description: [*:0]const u8) void;
    pub const setDescription = g_bookmark_file_set_description;

    /// Sets a list of group names for the item with URI `uri`.  Each previously
    /// set group name list is removed.
    ///
    /// If `uri` cannot be found then an item for it is created.
    extern fn g_bookmark_file_set_groups(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_groups: ?[*][*:0]const u8, p_length: usize) void;
    pub const setGroups = g_bookmark_file_set_groups;

    /// Sets the icon for the bookmark for `uri`. If `href` is `NULL`, unsets
    /// the currently set icon. `href` can either be a full URL for the icon
    /// file or the icon name following the Icon Naming specification.
    ///
    /// If no bookmark for `uri` is found one is created.
    extern fn g_bookmark_file_set_icon(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_href: ?[*:0]const u8, p_mime_type: [*:0]const u8) void;
    pub const setIcon = g_bookmark_file_set_icon;

    /// Sets the private flag of the bookmark for `uri`.
    ///
    /// If a bookmark for `uri` cannot be found then it is created.
    extern fn g_bookmark_file_set_is_private(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_is_private: c_int) void;
    pub const setIsPrivate = g_bookmark_file_set_is_private;

    /// Sets `mime_type` as the MIME type of the bookmark for `uri`.
    ///
    /// If a bookmark for `uri` cannot be found then it is created.
    extern fn g_bookmark_file_set_mime_type(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_mime_type: [*:0]const u8) void;
    pub const setMimeType = g_bookmark_file_set_mime_type;

    /// Sets the last time the bookmark for `uri` was last modified.
    ///
    /// If no bookmark for `uri` is found then it is created.
    ///
    /// The "modified" time should only be set when the bookmark's meta-data
    /// was actually changed.  Every function of `glib.BookmarkFile` that
    /// modifies a bookmark also changes the modification time, except for
    /// `glib.BookmarkFile.setVisitedDateTime`.
    extern fn g_bookmark_file_set_modified(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_modified: std.posix.time_t) void;
    pub const setModified = g_bookmark_file_set_modified;

    /// Sets the last time the bookmark for `uri` was last modified.
    ///
    /// If no bookmark for `uri` is found then it is created.
    ///
    /// The "modified" time should only be set when the bookmark's meta-data
    /// was actually changed.  Every function of `glib.BookmarkFile` that
    /// modifies a bookmark also changes the modification time, except for
    /// `glib.BookmarkFile.setVisitedDateTime`.
    extern fn g_bookmark_file_set_modified_date_time(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_modified: *glib.DateTime) void;
    pub const setModifiedDateTime = g_bookmark_file_set_modified_date_time;

    /// Sets `title` as the title of the bookmark for `uri` inside the
    /// bookmark file `bookmark`.
    ///
    /// If `uri` is `NULL`, the title of `bookmark` is set.
    ///
    /// If a bookmark for `uri` cannot be found then it is created.
    extern fn g_bookmark_file_set_title(p_bookmark: *BookmarkFile, p_uri: ?[*:0]const u8, p_title: [*:0]const u8) void;
    pub const setTitle = g_bookmark_file_set_title;

    /// Sets the time the bookmark for `uri` was last visited.
    ///
    /// If no bookmark for `uri` is found then it is created.
    ///
    /// The "visited" time should only be set if the bookmark was launched,
    /// either using the command line retrieved by `glib.BookmarkFile.getApplicationInfo`
    /// or by the default application for the bookmark's MIME type, retrieved
    /// using `glib.BookmarkFile.getMimeType`.  Changing the "visited" time
    /// does not affect the "modified" time.
    extern fn g_bookmark_file_set_visited(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_visited: std.posix.time_t) void;
    pub const setVisited = g_bookmark_file_set_visited;

    /// Sets the time the bookmark for `uri` was last visited.
    ///
    /// If no bookmark for `uri` is found then it is created.
    ///
    /// The "visited" time should only be set if the bookmark was launched,
    /// either using the command line retrieved by `glib.BookmarkFile.getApplicationInfo`
    /// or by the default application for the bookmark's MIME type, retrieved
    /// using `glib.BookmarkFile.getMimeType`.  Changing the "visited" time
    /// does not affect the "modified" time.
    extern fn g_bookmark_file_set_visited_date_time(p_bookmark: *BookmarkFile, p_uri: [*:0]const u8, p_visited: *glib.DateTime) void;
    pub const setVisitedDateTime = g_bookmark_file_set_visited_date_time;

    /// This function outputs `bookmark` as a string.
    extern fn g_bookmark_file_to_data(p_bookmark: *BookmarkFile, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*]u8;
    pub const toData = g_bookmark_file_to_data;

    /// This function outputs `bookmark` into a file.  The write process is
    /// guaranteed to be atomic by using `glib.fileSetContents` internally.
    extern fn g_bookmark_file_to_file(p_bookmark: *BookmarkFile, p_filename: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const toFile = g_bookmark_file_to_file;

    extern fn g_bookmark_file_get_type() usize;
    pub const getGObjectType = g_bookmark_file_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Contains the public fields of a GByteArray.
pub const ByteArray = extern struct {
    /// a pointer to the element data. The data may be moved as
    ///     elements are added to the `glib.ByteArray`
    f_data: ?*u8,
    /// the number of elements in the `glib.ByteArray`
    f_len: c_uint,

    /// Adds the given bytes to the end of the `glib.ByteArray`.
    /// The array will grow in size automatically if necessary.
    extern fn g_byte_array_append(p_array: *glib.ByteArray, p_data: *const u8, p_len: c_uint) *glib.ByteArray;
    pub const append = g_byte_array_append;

    /// Frees the memory allocated by the `glib.ByteArray`. If `free_segment` is
    /// `TRUE` it frees the actual byte data. If the reference count of
    /// `array` is greater than one, the `glib.ByteArray` wrapper is preserved but
    /// the size of `array` will be set to zero.
    extern fn g_byte_array_free(p_array: *glib.ByteArray, p_free_segment: c_int) *u8;
    pub const free = g_byte_array_free;

    /// Transfers the data from the `glib.ByteArray` into a new immutable `glib.Bytes`.
    ///
    /// The `glib.ByteArray` is freed unless the reference count of `array` is greater
    /// than one, the `glib.ByteArray` wrapper is preserved but the size of `array`
    /// will be set to zero.
    ///
    /// This is identical to using `glib.Bytes.newTake` and `glib.byteArrayFree`
    /// together.
    extern fn g_byte_array_free_to_bytes(p_array: *glib.ByteArray) *glib.Bytes;
    pub const freeToBytes = g_byte_array_free_to_bytes;

    /// Creates a new `glib.ByteArray` with a reference count of 1.
    extern fn g_byte_array_new() *glib.ByteArray;
    pub const new = g_byte_array_new;

    /// Creates a byte array containing the `data`.
    /// After this call, `data` belongs to the `glib.ByteArray` and may no longer be
    /// modified by the caller. The memory of `data` has to be dynamically
    /// allocated and will eventually be freed with `glib.free`.
    ///
    /// Do not use it if `len` is greater than `G_MAXUINT`. `glib.ByteArray`
    /// stores the length of its data in `guint`, which may be shorter than
    /// `gsize`.
    extern fn g_byte_array_new_take(p_data: [*]u8, p_len: usize) *glib.ByteArray;
    pub const newTake = g_byte_array_new_take;

    /// Adds the given data to the start of the `glib.ByteArray`.
    /// The array will grow in size automatically if necessary.
    extern fn g_byte_array_prepend(p_array: *glib.ByteArray, p_data: *const u8, p_len: c_uint) *glib.ByteArray;
    pub const prepend = g_byte_array_prepend;

    /// Atomically increments the reference count of `array` by one.
    /// This function is thread-safe and may be called from any thread.
    extern fn g_byte_array_ref(p_array: *glib.ByteArray) *glib.ByteArray;
    pub const ref = g_byte_array_ref;

    /// Removes the byte at the given index from a `glib.ByteArray`.
    /// The following bytes are moved down one place.
    extern fn g_byte_array_remove_index(p_array: *glib.ByteArray, p_index_: c_uint) *glib.ByteArray;
    pub const removeIndex = g_byte_array_remove_index;

    /// Removes the byte at the given index from a `glib.ByteArray`. The last
    /// element in the array is used to fill in the space, so this function
    /// does not preserve the order of the `glib.ByteArray`. But it is faster
    /// than `glib.byteArrayRemoveIndex`.
    extern fn g_byte_array_remove_index_fast(p_array: *glib.ByteArray, p_index_: c_uint) *glib.ByteArray;
    pub const removeIndexFast = g_byte_array_remove_index_fast;

    /// Removes the given number of bytes starting at the given index from a
    /// `glib.ByteArray`.  The following elements are moved to close the gap.
    extern fn g_byte_array_remove_range(p_array: *glib.ByteArray, p_index_: c_uint, p_length: c_uint) *glib.ByteArray;
    pub const removeRange = g_byte_array_remove_range;

    /// Sets the size of the `glib.ByteArray`, expanding it if necessary.
    extern fn g_byte_array_set_size(p_array: *glib.ByteArray, p_length: c_uint) *glib.ByteArray;
    pub const setSize = g_byte_array_set_size;

    /// Creates a new `glib.ByteArray` with `reserved_size` bytes preallocated.
    /// This avoids frequent reallocation, if you are going to add many
    /// bytes to the array. Note however that the size of the array is still
    /// 0.
    extern fn g_byte_array_sized_new(p_reserved_size: c_uint) *glib.ByteArray;
    pub const sizedNew = g_byte_array_sized_new;

    /// Sorts a byte array, using `compare_func` which should be a
    /// `qsort`-style comparison function (returns less than zero for first
    /// arg is less than second arg, zero for equal, greater than zero if
    /// first arg is greater than second arg).
    ///
    /// If two array elements compare equal, their order in the sorted array
    /// is undefined. If you want equal elements to keep their order (i.e.
    /// you want a stable sort) you can write a comparison function that,
    /// if two elements would otherwise compare equal, compares them by
    /// their addresses.
    extern fn g_byte_array_sort(p_array: *glib.ByteArray, p_compare_func: glib.CompareFunc) void;
    pub const sort = g_byte_array_sort;

    /// Like `glib.byteArraySort`, but the comparison function takes an extra
    /// user data argument.
    extern fn g_byte_array_sort_with_data(p_array: *glib.ByteArray, p_compare_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const sortWithData = g_byte_array_sort_with_data;

    /// Frees the data in the array and resets the size to zero, while
    /// the underlying array is preserved for use elsewhere and returned
    /// to the caller.
    extern fn g_byte_array_steal(p_array: *glib.ByteArray, p_len: ?*usize) *u8;
    pub const steal = g_byte_array_steal;

    /// Atomically decrements the reference count of `array` by one. If the
    /// reference count drops to 0, all memory allocated by the array is
    /// released. This function is thread-safe and may be called from any
    /// thread.
    extern fn g_byte_array_unref(p_array: *glib.ByteArray) void;
    pub const unref = g_byte_array_unref;

    extern fn g_byte_array_get_type() usize;
    pub const getGObjectType = g_byte_array_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A simple reference counted data type representing an immutable sequence of
/// zero or more bytes from an unspecified origin.
///
/// The purpose of a `GBytes` is to keep the memory region that it holds
/// alive for as long as anyone holds a reference to the bytes.  When
/// the last reference count is dropped, the memory is released. Multiple
/// unrelated callers can use byte data in the `GBytes` without coordinating
/// their activities, resting assured that the byte data will not change or
/// move while they hold a reference.
///
/// A `GBytes` can come from many different origins that may have
/// different procedures for freeing the memory region.  Examples are
/// memory from `glib.malloc`, from memory slices, from a
/// `glib.MappedFile` or memory from other allocators.
///
/// `GBytes` work well as keys in `glib.HashTable`. Use
/// `glib.Bytes.equal` and `glib.Bytes.hash` as parameters to
/// `glib.HashTable.new` or `glib.HashTable.newFull`.
/// `GBytes` can also be used as keys in a `glib.Tree` by passing the
/// `glib.Bytes.compare` function to `glib.Tree.new`.
///
/// The data pointed to by this bytes must not be modified. For a mutable
/// array of bytes see `glib.ByteArray`. Use
/// `glib.Bytes.unrefToArray` to create a mutable array for a `GBytes`
/// sequence. To create an immutable `GBytes` from a mutable
/// `glib.ByteArray`, use the `glib.ByteArray.freeToBytes`
/// function.
pub const Bytes = opaque {
    /// Creates a new `glib.Bytes` from `data`.
    ///
    /// `data` is copied. If `size` is 0, `data` may be `NULL`.
    ///
    /// As an optimization, `glib.Bytes.new` may avoid an extra allocation by
    /// copying the data within the resulting bytes structure if sufficiently small
    /// (since GLib 2.84).
    extern fn g_bytes_new(p_data: ?[*]const u8, p_size: usize) *glib.Bytes;
    pub const new = g_bytes_new;

    /// Creates a new `glib.Bytes` from static data.
    ///
    /// `data` must be static (ie: never modified or freed). It may be `NULL` if `size`
    /// is 0.
    extern fn g_bytes_new_static(p_data: ?[*]const u8, p_size: usize) *glib.Bytes;
    pub const newStatic = g_bytes_new_static;

    /// Creates a new `glib.Bytes` from `data`.
    ///
    /// After this call, `data` belongs to the `GBytes` and may no longer be
    /// modified by the caller. The memory of `data` has to be dynamically
    /// allocated and will eventually be freed with `glib.free`.
    ///
    /// For creating `GBytes` with memory from other allocators, see
    /// `glib.Bytes.newWithFreeFunc`.
    ///
    /// `data` may be `NULL` if `size` is 0.
    extern fn g_bytes_new_take(p_data: ?[*]u8, p_size: usize) *glib.Bytes;
    pub const newTake = g_bytes_new_take;

    /// Creates a `glib.Bytes` from `data`.
    ///
    /// When the last reference is dropped, `free_func` will be called with the
    /// `user_data` argument.
    ///
    /// `data` must not be modified after this call is made until `free_func` has
    /// been called to indicate that the bytes is no longer in use.
    ///
    /// `data` may be `NULL` if `size` is 0.
    extern fn g_bytes_new_with_free_func(p_data: ?[*]const u8, p_size: usize, p_free_func: glib.DestroyNotify, p_user_data: ?*anyopaque) *glib.Bytes;
    pub const newWithFreeFunc = g_bytes_new_with_free_func;

    /// Compares the two `glib.Bytes` values.
    ///
    /// This function can be used to sort `GBytes` instances in lexicographical
    /// order.
    ///
    /// If `bytes1` and `bytes2` have different length but the shorter one is a
    /// prefix of the longer one then the shorter one is considered to be less than
    /// the longer one. Otherwise the first byte where both differ is used for
    /// comparison. If `bytes1` has a smaller value at that position it is
    /// considered less, otherwise greater than `bytes2`.
    extern fn g_bytes_compare(p_bytes1: *const Bytes, p_bytes2: *const glib.Bytes) c_int;
    pub const compare = g_bytes_compare;

    /// Compares the two `glib.Bytes` values being pointed to and returns
    /// `TRUE` if they are equal.
    ///
    /// This function can be passed to `glib.HashTable.new` as the
    /// `key_equal_func` parameter, when using non-`NULL` `GBytes` pointers as keys in
    /// a `glib.HashTable`.
    extern fn g_bytes_equal(p_bytes1: *const Bytes, p_bytes2: *const glib.Bytes) c_int;
    pub const equal = g_bytes_equal;

    /// Get the byte data in the `glib.Bytes`.
    ///
    /// This data should not be modified.
    ///
    /// This function will always return the same pointer for a given `GBytes`.
    ///
    /// `NULL` may be returned if `size` is 0. This is not guaranteed, as the `GBytes`
    /// may represent an empty string with `data` non-`NULL` and `size` as 0. `NULL`
    /// will not be returned if `size` is non-zero.
    extern fn g_bytes_get_data(p_bytes: *Bytes, p_size: ?*usize) ?[*]const u8;
    pub const getData = g_bytes_get_data;

    /// Gets a pointer to a region in `bytes`.
    ///
    /// The region starts at `offset` many bytes from the start of the data
    /// and contains `n_elements` many elements of `element_size` size.
    ///
    /// `n_elements` may be zero, but `element_size` must always be non-zero.
    /// Ideally, `element_size` is a static constant (eg: `sizeof` a struct).
    ///
    /// This function does careful bounds checking (including checking for
    /// arithmetic overflows) and returns a non-`NULL` pointer if the
    /// specified region lies entirely within the `bytes`. If the region is
    /// in some way out of range, or if an overflow has occurred, then `NULL`
    /// is returned.
    ///
    /// Note: it is possible to have a valid zero-size region. In this case,
    /// the returned pointer will be equal to the base pointer of the data of
    /// `bytes`, plus `offset`.  This will be non-`NULL` except for the case
    /// where `bytes` itself was a zero-sized region.  Since it is unlikely
    /// that you will be using this function to check for a zero-sized region
    /// in a zero-sized `bytes`, `NULL` effectively always means ‘error’.
    extern fn g_bytes_get_region(p_bytes: *Bytes, p_element_size: usize, p_offset: usize, p_n_elements: usize) ?*const anyopaque;
    pub const getRegion = g_bytes_get_region;

    /// Get the size of the byte data in the `glib.Bytes`.
    ///
    /// This function will always return the same value for a given `GBytes`.
    extern fn g_bytes_get_size(p_bytes: *Bytes) usize;
    pub const getSize = g_bytes_get_size;

    /// Creates an integer hash code for the byte data in the `glib.Bytes`.
    ///
    /// This function can be passed to `glib.HashTable.new` as the
    /// `key_hash_func` parameter, when using non-`NULL` `GBytes` pointers as keys in
    /// a `glib.HashTable`.
    extern fn g_bytes_hash(p_bytes: *const Bytes) c_uint;
    pub const hash = g_bytes_hash;

    /// Creates a `glib.Bytes` which is a subsection of another `GBytes`.
    ///
    /// The `offset` + `length` may not be longer than the size of `bytes`.
    ///
    /// A reference to `bytes` will be held by the newly created `GBytes` until
    /// the byte data is no longer needed.
    ///
    /// Since 2.56, if `offset` is 0 and `length` matches the size of `bytes`, then
    /// `bytes` will be returned with the reference count incremented by 1. If `bytes`
    /// is a slice of another `GBytes`, then the resulting `GBytes` will reference
    /// the same `GBytes` instead of `bytes`. This allows consumers to simplify the
    /// usage of `GBytes` when asynchronously writing to streams.
    extern fn g_bytes_new_from_bytes(p_bytes: *Bytes, p_offset: usize, p_length: usize) *glib.Bytes;
    pub const newFromBytes = g_bytes_new_from_bytes;

    /// Increase the reference count on `bytes`.
    extern fn g_bytes_ref(p_bytes: *Bytes) *glib.Bytes;
    pub const ref = g_bytes_ref;

    /// Releases a reference on `bytes`.
    ///
    /// This may result in the bytes being freed. If `bytes` is `NULL`, it will
    /// return immediately.
    extern fn g_bytes_unref(p_bytes: ?*Bytes) void;
    pub const unref = g_bytes_unref;

    /// Unreferences the bytes, and returns a new mutable `glib.ByteArray`
    /// containing the same byte data.
    ///
    /// As an optimization, the byte data is transferred to the array without copying
    /// if this was the last reference to `bytes` and `bytes` was created with
    /// `glib.Bytes.new`, `glib.Bytes.newTake` or
    /// `glib.ByteArray.freeToBytes` and the buffer was larger than the size
    /// `glib.Bytes` may internalize within its allocation. In all other cases
    /// the data is copied.
    ///
    /// Do not use it if `bytes` contains more than `G_MAXUINT`
    /// bytes. `glib.ByteArray` stores the length of its data in `guint`,
    /// which may be shorter than `gsize`, that `bytes` is using.
    extern fn g_bytes_unref_to_array(p_bytes: *Bytes) *glib.ByteArray;
    pub const unrefToArray = g_bytes_unref_to_array;

    /// Unreferences the bytes, and returns a pointer the same byte data
    /// contents.
    ///
    /// As an optimization, the byte data is returned without copying if this was
    /// the last reference to `bytes` and `bytes` was created with
    /// `glib.Bytes.new`, `glib.Bytes.newTake` or
    /// `glib.ByteArray.freeToBytes` and the buffer was larger than the size
    /// `glib.Bytes` may internalize within its allocation. In all other cases
    /// the data is copied.
    extern fn g_bytes_unref_to_data(p_bytes: *Bytes, p_size: *usize) [*]u8;
    pub const unrefToData = g_bytes_unref_to_data;

    extern fn g_bytes_get_type() usize;
    pub const getGObjectType = g_bytes_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `GCache` allows sharing of complex data structures, in order to
/// save system resources.
///
/// `GCache` uses keys and values. A `GCache` key describes the properties
/// of a particular resource. A `GCache` value is the actual resource.
///
/// `GCache` has been marked as deprecated, since this API is rarely
/// used and not very actively maintained.
pub const Cache = opaque {
    /// Creates a new `glib.Cache`.
    extern fn g_cache_new(p_value_new_func: glib.CacheNewFunc, p_value_destroy_func: glib.CacheDestroyFunc, p_key_dup_func: glib.CacheDupFunc, p_key_destroy_func: glib.CacheDestroyFunc, p_hash_key_func: glib.HashFunc, p_hash_value_func: glib.HashFunc, p_key_equal_func: glib.EqualFunc) *glib.Cache;
    pub const new = g_cache_new;

    /// Frees the memory allocated for the `glib.Cache`.
    ///
    /// Note that it does not destroy the keys and values which were
    /// contained in the `glib.Cache`.
    extern fn g_cache_destroy(p_cache: *Cache) void;
    pub const destroy = g_cache_destroy;

    /// Gets the value corresponding to the given key, creating it if
    /// necessary. It first checks if the value already exists in the
    /// `glib.Cache`, by using the `key_equal_func` function passed to
    /// `glib.Cache.new`. If it does already exist it is returned, and its
    /// reference count is increased by one. If the value does not currently
    /// exist, if is created by calling the `value_new_func`. The key is
    /// duplicated by calling `key_dup_func` and the duplicated key and value
    /// are inserted into the `glib.Cache`.
    extern fn g_cache_insert(p_cache: *Cache, p_key: ?*anyopaque) ?*anyopaque;
    pub const insert = g_cache_insert;

    /// Calls the given function for each of the keys in the `glib.Cache`.
    ///
    /// NOTE `func` is passed three parameters, the value and key of a cache
    /// entry and the `user_data`. The order of value and key is different
    /// from the order in which `glib.hashTableForeach` passes key-value
    /// pairs to its callback function !
    extern fn g_cache_key_foreach(p_cache: *Cache, p_func: glib.HFunc, p_user_data: ?*anyopaque) void;
    pub const keyForeach = g_cache_key_foreach;

    /// Decreases the reference count of the given value. If it drops to 0
    /// then the value and its corresponding key are destroyed, using the
    /// `value_destroy_func` and `key_destroy_func` passed to `glib.Cache.new`.
    extern fn g_cache_remove(p_cache: *Cache, p_value: ?*const anyopaque) void;
    pub const remove = g_cache_remove;

    /// Calls the given function for each of the values in the `glib.Cache`.
    extern fn g_cache_value_foreach(p_cache: *Cache, p_func: glib.HFunc, p_user_data: ?*anyopaque) void;
    pub const valueForeach = g_cache_value_foreach;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// GLib provides a generic API for computing checksums (or ‘digests’)
/// for a sequence of arbitrary bytes, using various hashing algorithms
/// like MD5, SHA-1 and SHA-256. Checksums are commonly used in various
/// environments and specifications.
///
/// To create a new `GChecksum`, use `glib.Checksum.new`. To free
/// a `GChecksum`, use `glib.Checksum.free`.
///
/// GLib supports incremental checksums using the `GChecksum` data
/// structure, by calling `glib.Checksum.update` as long as there’s data
/// available and then using `glib.Checksum.getString` or
/// `glib.Checksum.getDigest` to compute the checksum and return it
/// either as a string in hexadecimal form, or as a raw sequence of bytes. To
/// compute the checksum for binary blobs and nul-terminated strings in
/// one go, use the convenience functions `glib.computeChecksumForData`
/// and `glib.computeChecksumForString`, respectively.
pub const Checksum = opaque {
    /// Gets the length in bytes of digests of type `checksum_type`
    extern fn g_checksum_type_get_length(p_checksum_type: glib.ChecksumType) isize;
    pub const typeGetLength = g_checksum_type_get_length;

    /// Creates a new `glib.Checksum`, using the checksum algorithm `checksum_type`.
    /// If the `checksum_type` is not known, `NULL` is returned.
    /// A `glib.Checksum` can be used to compute the checksum, or digest, of an
    /// arbitrary binary blob, using different hashing algorithms.
    ///
    /// A `glib.Checksum` works by feeding a binary blob through `glib.Checksum.update`
    /// until there is data to be checked; the digest can then be extracted
    /// using `glib.Checksum.getString`, which will return the checksum as a
    /// hexadecimal string; or `glib.Checksum.getDigest`, which will return a
    /// vector of raw bytes. Once either `glib.Checksum.getString` or
    /// `glib.Checksum.getDigest` have been called on a `glib.Checksum`, the checksum
    /// will be closed and it won't be possible to call `glib.Checksum.update`
    /// on it anymore.
    extern fn g_checksum_new(p_checksum_type: glib.ChecksumType) ?*glib.Checksum;
    pub const new = g_checksum_new;

    /// Copies a `glib.Checksum`. If `checksum` has been closed, by calling
    /// `glib.Checksum.getString` or `glib.Checksum.getDigest`, the copied
    /// checksum will be closed as well.
    extern fn g_checksum_copy(p_checksum: *const Checksum) *glib.Checksum;
    pub const copy = g_checksum_copy;

    /// Frees the memory allocated for `checksum`.
    extern fn g_checksum_free(p_checksum: *Checksum) void;
    pub const free = g_checksum_free;

    /// Gets the digest from `checksum` as a raw binary vector and places it
    /// into `buffer`. The size of the digest depends on the type of checksum.
    ///
    /// Once this function has been called, the `glib.Checksum` is closed and can
    /// no longer be updated with `glib.Checksum.update`.
    extern fn g_checksum_get_digest(p_checksum: *Checksum, p_buffer: [*]u8, p_digest_len: *usize) void;
    pub const getDigest = g_checksum_get_digest;

    /// Gets the digest as a hexadecimal string.
    ///
    /// Once this function has been called the `glib.Checksum` can no longer be
    /// updated with `glib.Checksum.update`.
    ///
    /// The hexadecimal characters will be lower case.
    extern fn g_checksum_get_string(p_checksum: *Checksum) [*:0]const u8;
    pub const getString = g_checksum_get_string;

    /// Resets the state of the `checksum` back to its initial state.
    extern fn g_checksum_reset(p_checksum: *Checksum) void;
    pub const reset = g_checksum_reset;

    /// Feeds `data` into an existing `glib.Checksum`. The checksum must still be
    /// open, that is `glib.Checksum.getString` or `glib.Checksum.getDigest` must
    /// not have been called on `checksum`.
    extern fn g_checksum_update(p_checksum: *Checksum, p_data: [*]const u8, p_length: isize) void;
    pub const update = g_checksum_update;

    extern fn g_checksum_get_type() usize;
    pub const getGObjectType = g_checksum_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GCompletion` provides support for automatic completion of a string
/// using any group of target strings. It is typically used for file
/// name completion as is common in many UNIX shells.
///
/// A `GCompletion` is created using `glib.Completion.new`. Target items are
/// added and removed with `glib.Completion.addItems`,
/// `glib.Completion.removeItems` and
/// `glib.Completion.clearItems`. A completion attempt is requested with
/// `glib.Completion.complete` or `glib.Completion.completeUtf8`.
/// When no longer needed, the `GCompletion` is freed with
/// `glib.Completion.free`.
///
/// Items in the completion can be simple strings (e.g. filenames), or
/// pointers to arbitrary data structures. If data structures are used
/// you must provide a `glib.CompletionFunc` in `glib.Completion.new`,
/// which retrieves the item’s string from the data structure. You can change
/// the way in which strings are compared by setting a different
/// `glib.CompletionStrncmpFunc` in `glib.Completion.setCompare`.
///
/// `GCompletion` has been marked as deprecated, since this API is rarely
/// used and not very actively maintained.
pub const Completion = extern struct {
    /// list of target items (strings or data structures).
    f_items: ?*glib.List,
    /// function which is called to get the string associated with a
    ///        target item. It is `NULL` if the target items are strings.
    f_func: ?glib.CompletionFunc,
    /// the last prefix passed to `glib.Completion.complete` or
    ///          `glib.Completion.completeUtf8`.
    f_prefix: ?[*:0]u8,
    /// the list of items which begin with `prefix`.
    f_cache: ?*glib.List,
    /// The function to use when comparing strings.  Use
    ///                `glib.Completion.setCompare` to modify this function.
    f_strncmp_func: ?glib.CompletionStrncmpFunc,

    /// Creates a new `glib.Completion`.
    extern fn g_completion_new(p_func: glib.CompletionFunc) *glib.Completion;
    pub const new = g_completion_new;

    /// Adds items to the `glib.Completion`.
    extern fn g_completion_add_items(p_cmp: *Completion, p_items: *glib.List) void;
    pub const addItems = g_completion_add_items;

    /// Removes all items from the `glib.Completion`. The items are not freed, so if the
    /// memory was dynamically allocated, it should be freed after calling this
    /// function.
    extern fn g_completion_clear_items(p_cmp: *Completion) void;
    pub const clearItems = g_completion_clear_items;

    /// Attempts to complete the string `prefix` using the `glib.Completion`
    /// target items.
    extern fn g_completion_complete(p_cmp: *Completion, p_prefix: [*:0]const u8, p_new_prefix: *[*:0]u8) *glib.List;
    pub const complete = g_completion_complete;

    /// Attempts to complete the string `prefix` using the `glib.Completion` target items.
    /// In contrast to `glib.Completion.complete`, this function returns the largest common
    /// prefix that is a valid UTF-8 string, omitting a possible common partial
    /// character.
    ///
    /// You should use this function instead of `glib.Completion.complete` if your
    /// items are UTF-8 strings.
    extern fn g_completion_complete_utf8(p_cmp: *Completion, p_prefix: [*:0]const u8, p_new_prefix: *[*:0]u8) *glib.List;
    pub const completeUtf8 = g_completion_complete_utf8;

    /// Frees all memory used by the `glib.Completion`. The items are not freed, so if
    /// the memory was dynamically allocated, it should be freed after calling this
    /// function.
    extern fn g_completion_free(p_cmp: *Completion) void;
    pub const free = g_completion_free;

    /// Removes items from a `glib.Completion`. The items are not freed, so if the memory
    /// was dynamically allocated, free `items` with `glib.List.freeFull` after calling
    /// this function.
    extern fn g_completion_remove_items(p_cmp: *Completion, p_items: *glib.List) void;
    pub const removeItems = g_completion_remove_items;

    /// Sets the function to use for string comparisons. The default string
    /// comparison function is `strncmp`.
    extern fn g_completion_set_compare(p_cmp: *Completion, p_strncmp_func: glib.CompletionStrncmpFunc) void;
    pub const setCompare = g_completion_set_compare;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.Cond` struct is an opaque data structure that represents a
/// condition. Threads can block on a `glib.Cond` if they find a certain
/// condition to be false. If other threads change the state of this
/// condition they signal the `glib.Cond`, and that causes the waiting
/// threads to be woken up.
///
/// Consider the following example of a shared variable.  One or more
/// threads can wait for data to be published to the variable and when
/// another thread publishes the data, it can signal one of the waiting
/// threads to wake up to collect the data.
///
/// Here is an example for using GCond to block a thread until a condition
/// is satisfied:
/// ```
///   gpointer current_data = NULL;
///   GMutex data_mutex;
///   GCond data_cond;
///
///   void
///   push_data (gpointer data)
///   {
///     g_mutex_lock (&data_mutex);
///     current_data = data;
///     g_cond_signal (&data_cond);
///     g_mutex_unlock (&data_mutex);
///   }
///
///   gpointer
///   pop_data (void)
///   {
///     gpointer data;
///
///     g_mutex_lock (&data_mutex);
///     while (!current_data)
///       g_cond_wait (&data_cond, &data_mutex);
///     data = current_data;
///     current_data = NULL;
///     g_mutex_unlock (&data_mutex);
///
///     return data;
///   }
/// ```
/// Whenever a thread calls `pop_data` now, it will wait until
/// current_data is non-`NULL`, i.e. until some other thread
/// has called `push_data`.
///
/// The example shows that use of a condition variable must always be
/// paired with a mutex.  Without the use of a mutex, there would be a
/// race between the check of `current_data` by the while loop in
/// `pop_data` and waiting. Specifically, another thread could set
/// `current_data` after the check, and signal the cond (with nobody
/// waiting on it) before the first thread goes to sleep. `glib.Cond` is
/// specifically useful for its ability to release the mutex and go
/// to sleep atomically.
///
/// It is also important to use the `glib.Cond.wait` and `glib.Cond.waitUntil`
/// functions only inside a loop which checks for the condition to be
/// true.  See `glib.Cond.wait` for an explanation of why the condition may
/// not be true even after it returns.
///
/// If a `glib.Cond` is allocated in static storage then it can be used
/// without initialisation.  Otherwise, you should call `glib.Cond.init`
/// on it and `glib.Cond.clear` when done.
///
/// A `glib.Cond` should only be accessed via the g_cond_ functions.
pub const Cond = extern struct {
    f_p: ?*anyopaque,
    f_i: [2]c_uint,

    /// Allocates and initializes a new `glib.Cond`.
    extern fn g_cond_new() *glib.Cond;
    pub const new = g_cond_new;

    /// If threads are waiting for `cond`, all of them are unblocked.
    /// If no threads are waiting for `cond`, this function has no effect.
    /// It is good practice to lock the same mutex as the waiting threads
    /// while calling this function, though not required.
    extern fn g_cond_broadcast(p_cond: *Cond) void;
    pub const broadcast = g_cond_broadcast;

    /// Frees the resources allocated to a `glib.Cond` with `glib.Cond.init`.
    ///
    /// This function should not be used with a `glib.Cond` that has been
    /// statically allocated.
    ///
    /// Calling `glib.Cond.clear` for a `glib.Cond` on which threads are
    /// blocking leads to undefined behaviour.
    extern fn g_cond_clear(p_cond: *Cond) void;
    pub const clear = g_cond_clear;

    /// Destroys a `glib.Cond` that has been created with `glib.condNew`.
    ///
    /// Calling `glib.Cond.free` for a `glib.Cond` on which threads are
    /// blocking leads to undefined behaviour.
    extern fn g_cond_free(p_cond: *Cond) void;
    pub const free = g_cond_free;

    /// Initialises a `glib.Cond` so that it can be used.
    ///
    /// This function is useful to initialise a `glib.Cond` that has been
    /// allocated as part of a larger structure.  It is not necessary to
    /// initialise a `glib.Cond` that has been statically allocated.
    ///
    /// To undo the effect of `glib.Cond.init` when a `glib.Cond` is no longer
    /// needed, use `glib.Cond.clear`.
    ///
    /// Calling `glib.Cond.init` on an already-initialised `glib.Cond` leads
    /// to undefined behaviour.
    extern fn g_cond_init(p_cond: *Cond) void;
    pub const init = g_cond_init;

    /// If threads are waiting for `cond`, at least one of them is unblocked.
    /// If no threads are waiting for `cond`, this function has no effect.
    /// It is good practice to hold the same lock as the waiting thread
    /// while calling this function, though not required.
    extern fn g_cond_signal(p_cond: *Cond) void;
    pub const signal = g_cond_signal;

    /// Waits until this thread is woken up on `cond`, but not longer than
    /// until the time specified by `abs_time`. The `mutex` is unlocked before
    /// falling asleep and locked again before resuming.
    ///
    /// If `abs_time` is `NULL`, `glib.Cond.timedWait` acts like `glib.Cond.wait`.
    ///
    /// This function can be used even if `glib.threadInit` has not yet been
    /// called, and, in that case, will immediately return `TRUE`.
    ///
    /// To easily calculate `abs_time` a combination of `glib.getRealTime`
    /// and `glib.TimeVal.add` can be used.
    extern fn g_cond_timed_wait(p_cond: *Cond, p_mutex: *glib.Mutex, p_abs_time: *glib.TimeVal) c_int;
    pub const timedWait = g_cond_timed_wait;

    /// Atomically releases `mutex` and waits until `cond` is signalled.
    /// When this function returns, `mutex` is locked again and owned by the
    /// calling thread.
    ///
    /// When using condition variables, it is possible that a spurious wakeup
    /// may occur (ie: `glib.Cond.wait` returns even though `glib.Cond.signal` was
    /// not called).  It's also possible that a stolen wakeup may occur.
    /// This is when `glib.Cond.signal` is called, but another thread acquires
    /// `mutex` before this thread and modifies the state of the program in
    /// such a way that when `glib.Cond.wait` is able to return, the expected
    /// condition is no longer met.
    ///
    /// For this reason, `glib.Cond.wait` must always be used in a loop.  See
    /// the documentation for `glib.Cond` for a complete example.
    extern fn g_cond_wait(p_cond: *Cond, p_mutex: *glib.Mutex) void;
    pub const wait = g_cond_wait;

    /// Waits until either `cond` is signalled or `end_time` has passed.
    ///
    /// As with `glib.Cond.wait` it is possible that a spurious or stolen wakeup
    /// could occur.  For that reason, waiting on a condition variable should
    /// always be in a loop, based on an explicitly-checked predicate.
    ///
    /// `TRUE` is returned if the condition variable was signalled (or in the
    /// case of a spurious wakeup).  `FALSE` is returned if `end_time` has
    /// passed.
    ///
    /// The following code shows how to correctly perform a timed wait on a
    /// condition variable (extending the example presented in the
    /// documentation for `glib.Cond`):
    ///
    /// ```
    /// gpointer
    /// pop_data_timed (void)
    /// {
    ///   gint64 end_time;
    ///   gpointer data;
    ///
    ///   g_mutex_lock (&data_mutex);
    ///
    ///   end_time = g_get_monotonic_time () + 5 * G_TIME_SPAN_SECOND;
    ///   while (!current_data)
    ///     if (!g_cond_wait_until (&data_cond, &data_mutex, end_time))
    ///       {
    ///         // timeout has passed.
    ///         g_mutex_unlock (&data_mutex);
    ///         return NULL;
    ///       }
    ///
    ///   // there is data for us
    ///   data = current_data;
    ///   current_data = NULL;
    ///
    ///   g_mutex_unlock (&data_mutex);
    ///
    ///   return data;
    /// }
    /// ```
    ///
    /// Notice that the end time is calculated once, before entering the
    /// loop and reused.  This is the motivation behind the use of absolute
    /// time on this API -- if a relative time of 5 seconds were passed
    /// directly to the call and a spurious wakeup occurred, the program would
    /// have to start over waiting again (which would lead to a total wait
    /// time of more than 5 seconds).
    extern fn g_cond_wait_until(p_cond: *Cond, p_mutex: *glib.Mutex, p_end_time: i64) c_int;
    pub const waitUntil = g_cond_wait_until;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// An opaque data structure that represents a keyed data list.
///
/// See also: [Keyed data lists](datalist-and-dataset.html).
pub const Data = opaque {
    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GDate` is a struct for calendrical calculations.
///
/// The `GDate` data structure represents a day between January 1, Year 1,
/// and sometime a few thousand years in the future (right now it will go
/// to the year 65535 or so, but `glib.Date.setParse` only parses up to the
/// year 8000 or so - just count on "a few thousand"). `GDate` is meant to
/// represent everyday dates, not astronomical dates or historical dates
/// or ISO timestamps or the like. It extrapolates the current Gregorian
/// calendar forward and backward in time; there is no attempt to change
/// the calendar to match time periods or locations. `GDate` does not store
/// time information; it represents a day.
///
/// The `GDate` implementation has several nice features; it is only a
/// 64-bit struct, so storing large numbers of dates is very efficient. It
/// can keep both a Julian and day-month-year representation of the date,
/// since some calculations are much easier with one representation or the
/// other. A Julian representation is simply a count of days since some
/// fixed day in the past; for `glib.Date` the fixed day is January 1, 1 AD.
/// ("Julian" dates in the `glib.Date` API aren't really Julian dates in the
/// technical sense; technically, Julian dates count from the start of the
/// Julian period, Jan 1, 4713 BC).
///
/// `GDate` is simple to use. First you need a "blank" date; you can get a
/// dynamically allocated date from `glib.Date.new`, or you can declare an
/// automatic variable or array and initialize it by calling `glib.Date.clear`.
/// A cleared date is safe; it's safe to call `glib.Date.setDmy` and the other
/// mutator functions to initialize the value of a cleared date. However, a cleared date
/// is initially invalid, meaning that it doesn't represent a day that exists.
/// It is undefined to call any of the date calculation routines on an invalid date.
/// If you obtain a date from a user or other unpredictable source, you should check
/// its validity with the `glib.Date.valid` predicate. `glib.Date.valid`
/// is also used to check for errors with `glib.Date.setParse` and other functions
/// that can fail. Dates can be invalidated by calling `glib.Date.clear` again.
///
/// It is very important to use the API to access the `GDate` struct. Often only the
/// day-month-year or only the Julian representation is valid. Sometimes neither is valid.
/// Use the API.
///
/// GLib also features `GDateTime` which represents a precise time.
pub const Date = extern struct {
    bitfields0: packed struct(c_uint) {
        /// the Julian representation of the date
        f_julian_days: u32,
    },
    bitfields1: packed struct(c_uint) {
        /// this bit is set if `julian_days` is valid
        f_julian: u1,
        /// this is set if `day`, `month` and `year` are valid
        f_dmy: u1,
        /// the day of the day-month-year representation of the date,
        ///   as a number between 1 and 31
        f_day: u6,
        /// the month of the day-month-year representation of the date,
        ///   as a number between 1 and 12
        f_month: u4,
        /// the year of the day-month-year representation of the date
        f_year: u16,
        _: u4,
    },

    /// Returns the number of days in a month, taking leap
    /// years into account.
    extern fn g_date_get_days_in_month(p_month: glib.DateMonth, p_year: glib.DateYear) u8;
    pub const getDaysInMonth = g_date_get_days_in_month;

    /// Returns the number of weeks in the year, where weeks
    /// are taken to start on Monday. Will be 52 or 53. The
    /// date must be valid. (Years always have 52 7-day periods,
    /// plus 1 or 2 extra days depending on whether it's a leap
    /// year. This function is basically telling you how many
    /// Mondays are in the year, i.e. there are 53 Mondays if
    /// one of the extra days happens to be a Monday.)
    extern fn g_date_get_monday_weeks_in_year(p_year: glib.DateYear) u8;
    pub const getMondayWeeksInYear = g_date_get_monday_weeks_in_year;

    /// Returns the number of weeks in the year, where weeks
    /// are taken to start on Sunday. Will be 52 or 53. The
    /// date must be valid. (Years always have 52 7-day periods,
    /// plus 1 or 2 extra days depending on whether it's a leap
    /// year. This function is basically telling you how many
    /// Sundays are in the year, i.e. there are 53 Sundays if
    /// one of the extra days happens to be a Sunday.)
    extern fn g_date_get_sunday_weeks_in_year(p_year: glib.DateYear) u8;
    pub const getSundayWeeksInYear = g_date_get_sunday_weeks_in_year;

    /// Returns `TRUE` if the year is a leap year.
    ///
    /// For the purposes of this function, leap year is every year
    /// divisible by 4 unless that year is divisible by 100. If it
    /// is divisible by 100 it would be a leap year only if that year
    /// is also divisible by 400.
    extern fn g_date_is_leap_year(p_year: glib.DateYear) c_int;
    pub const isLeapYear = g_date_is_leap_year;

    /// Generates a printed representation of the date, in a
    /// [locale](running.html`locale`)-specific way.
    /// Works just like the platform's C library `strftime` function,
    /// but only accepts date-related formats; time-related formats
    /// give undefined results. Date must be valid. Unlike `strftime`
    /// (which uses the locale encoding), works on a UTF-8 format
    /// string and stores a UTF-8 result.
    ///
    /// This function does not provide any conversion specifiers in
    /// addition to those implemented by the platform's C library.
    /// For example, don't expect that using `glib.dateStrftime` would
    /// make the \%F provided by the C99 `strftime` work on Windows
    /// where the C library only complies to C89.
    extern fn g_date_strftime(p_s: [*:0]u8, p_slen: usize, p_format: [*:0]const u8, p_date: *const glib.Date) usize;
    pub const strftime = g_date_strftime;

    /// Returns `TRUE` if the day of the month is valid (a day is valid if it's
    /// between 1 and 31 inclusive).
    extern fn g_date_valid_day(p_day: glib.DateDay) c_int;
    pub const validDay = g_date_valid_day;

    /// Returns `TRUE` if the day-month-year triplet forms a valid, existing day
    /// in the range of days `glib.Date` understands (Year 1 or later, no more than
    /// a few thousand years in the future).
    extern fn g_date_valid_dmy(p_day: glib.DateDay, p_month: glib.DateMonth, p_year: glib.DateYear) c_int;
    pub const validDmy = g_date_valid_dmy;

    /// Returns `TRUE` if the Julian day is valid. Anything greater than zero
    /// is basically a valid Julian, though there is a 32-bit limit.
    extern fn g_date_valid_julian(p_julian_date: u32) c_int;
    pub const validJulian = g_date_valid_julian;

    /// Returns `TRUE` if the month value is valid. The 12 `glib.DateMonth`
    /// enumeration values are the only valid months.
    extern fn g_date_valid_month(p_month: glib.DateMonth) c_int;
    pub const validMonth = g_date_valid_month;

    /// Returns `TRUE` if the weekday is valid. The seven `glib.DateWeekday` enumeration
    /// values are the only valid weekdays.
    extern fn g_date_valid_weekday(p_weekday: glib.DateWeekday) c_int;
    pub const validWeekday = g_date_valid_weekday;

    /// Returns `TRUE` if the year is valid. Any year greater than 0 is valid,
    /// though there is a 16-bit limit to what `glib.Date` will understand.
    extern fn g_date_valid_year(p_year: glib.DateYear) c_int;
    pub const validYear = g_date_valid_year;

    /// Allocates a `glib.Date` and initializes
    /// it to a safe state. The new date will
    /// be cleared (as if you'd called `glib.Date.clear`) but invalid (it won't
    /// represent an existing day). Free the return value with `glib.Date.free`.
    extern fn g_date_new() *glib.Date;
    pub const new = g_date_new;

    /// Create a new `glib.Date` representing the given day-month-year triplet.
    ///
    /// The triplet you pass in must represent a valid date. Use `glib.dateValidDmy`
    /// if needed to validate it. The returned `glib.Date` is guaranteed to be non-`NULL`
    /// and valid.
    extern fn g_date_new_dmy(p_day: glib.DateDay, p_month: glib.DateMonth, p_year: glib.DateYear) *glib.Date;
    pub const newDmy = g_date_new_dmy;

    /// Create a new `glib.Date` representing the given Julian date.
    ///
    /// The `julian_day` you pass in must be valid. Use `glib.dateValidJulian` if
    /// needed to validate it. The returned `glib.Date` is guaranteed to be non-`NULL` and
    /// valid.
    extern fn g_date_new_julian(p_julian_day: u32) *glib.Date;
    pub const newJulian = g_date_new_julian;

    /// Increments a date some number of days.
    /// To move forward by weeks, add weeks*7 days.
    /// The date must be valid.
    extern fn g_date_add_days(p_date: *Date, p_n_days: c_uint) void;
    pub const addDays = g_date_add_days;

    /// Increments a date by some number of months.
    /// If the day of the month is greater than 28,
    /// this routine may change the day of the month
    /// (because the destination month may not have
    /// the current day in it). The date must be valid.
    extern fn g_date_add_months(p_date: *Date, p_n_months: c_uint) void;
    pub const addMonths = g_date_add_months;

    /// Increments a date by some number of years.
    /// If the date is February 29, and the destination
    /// year is not a leap year, the date will be changed
    /// to February 28. The date must be valid.
    extern fn g_date_add_years(p_date: *Date, p_n_years: c_uint) void;
    pub const addYears = g_date_add_years;

    /// If `date` is prior to `min_date`, sets `date` equal to `min_date`.
    /// If `date` falls after `max_date`, sets `date` equal to `max_date`.
    /// Otherwise, `date` is unchanged.
    /// Either of `min_date` and `max_date` may be `NULL`.
    /// All non-`NULL` dates must be valid.
    extern fn g_date_clamp(p_date: *Date, p_min_date: *const glib.Date, p_max_date: *const glib.Date) void;
    pub const clamp = g_date_clamp;

    /// Initializes one or more `glib.Date` structs to a safe but invalid
    /// state. The cleared dates will not represent an existing date, but will
    /// not contain garbage. Useful to init a date declared on the stack.
    /// Validity can be tested with `glib.Date.valid`.
    extern fn g_date_clear(p_date: *Date, p_n_dates: c_uint) void;
    pub const clear = g_date_clear;

    /// `qsort`-style comparison function for dates.
    /// Both dates must be valid.
    extern fn g_date_compare(p_lhs: *const Date, p_rhs: *const glib.Date) c_int;
    pub const compare = g_date_compare;

    /// Copies a GDate to a newly-allocated GDate. If the input was invalid
    /// (as determined by `glib.Date.valid`), the invalid state will be copied
    /// as is into the new object.
    extern fn g_date_copy(p_date: *const Date) *glib.Date;
    pub const copy = g_date_copy;

    /// Computes the number of days between two dates.
    /// If `date2` is prior to `date1`, the returned value is negative.
    /// Both dates must be valid.
    extern fn g_date_days_between(p_date1: *const Date, p_date2: *const glib.Date) c_int;
    pub const daysBetween = g_date_days_between;

    /// Frees a `glib.Date` returned from `glib.Date.new`.
    extern fn g_date_free(p_date: *Date) void;
    pub const free = g_date_free;

    /// Returns the day of the month. The date must be valid.
    extern fn g_date_get_day(p_date: *const Date) glib.DateDay;
    pub const getDay = g_date_get_day;

    /// Returns the day of the year, where Jan 1 is the first day of the
    /// year. The date must be valid.
    extern fn g_date_get_day_of_year(p_date: *const Date) c_uint;
    pub const getDayOfYear = g_date_get_day_of_year;

    /// Returns the week of the year, where weeks are interpreted according
    /// to ISO 8601.
    extern fn g_date_get_iso8601_week_of_year(p_date: *const Date) c_uint;
    pub const getIso8601WeekOfYear = g_date_get_iso8601_week_of_year;

    /// Returns the Julian day or "serial number" of the `glib.Date`. The
    /// Julian day is simply the number of days since January 1, Year 1; i.e.,
    /// January 1, Year 1 is Julian day 1; January 2, Year 1 is Julian day 2,
    /// etc. The date must be valid.
    extern fn g_date_get_julian(p_date: *const Date) u32;
    pub const getJulian = g_date_get_julian;

    /// Returns the week of the year, where weeks are understood to start on
    /// Monday. If the date is before the first Monday of the year, return 0.
    /// The date must be valid.
    extern fn g_date_get_monday_week_of_year(p_date: *const Date) c_uint;
    pub const getMondayWeekOfYear = g_date_get_monday_week_of_year;

    /// Returns the month of the year. The date must be valid.
    extern fn g_date_get_month(p_date: *const Date) glib.DateMonth;
    pub const getMonth = g_date_get_month;

    /// Returns the week of the year during which this date falls, if
    /// weeks are understood to begin on Sunday. The date must be valid.
    /// Can return 0 if the day is before the first Sunday of the year.
    extern fn g_date_get_sunday_week_of_year(p_date: *const Date) c_uint;
    pub const getSundayWeekOfYear = g_date_get_sunday_week_of_year;

    /// Returns the day of the week for a `glib.Date`. The date must be valid.
    extern fn g_date_get_weekday(p_date: *const Date) glib.DateWeekday;
    pub const getWeekday = g_date_get_weekday;

    /// Returns the year of a `glib.Date`. The date must be valid.
    extern fn g_date_get_year(p_date: *const Date) glib.DateYear;
    pub const getYear = g_date_get_year;

    /// Returns `TRUE` if the date is on the first of a month.
    /// The date must be valid.
    extern fn g_date_is_first_of_month(p_date: *const Date) c_int;
    pub const isFirstOfMonth = g_date_is_first_of_month;

    /// Returns `TRUE` if the date is the last day of the month.
    /// The date must be valid.
    extern fn g_date_is_last_of_month(p_date: *const Date) c_int;
    pub const isLastOfMonth = g_date_is_last_of_month;

    /// Checks if `date1` is less than or equal to `date2`,
    /// and swap the values if this is not the case.
    extern fn g_date_order(p_date1: *Date, p_date2: *glib.Date) void;
    pub const order = g_date_order;

    /// Sets the day of the month for a `glib.Date`. If the resulting
    /// day-month-year triplet is invalid, the date will be invalid.
    extern fn g_date_set_day(p_date: *Date, p_day: glib.DateDay) void;
    pub const setDay = g_date_set_day;

    /// Sets the value of a `glib.Date` from a day, month, and year.
    /// The day-month-year triplet must be valid; if you aren't
    /// sure it is, call `glib.dateValidDmy` to check before you
    /// set it.
    extern fn g_date_set_dmy(p_date: *Date, p_day: glib.DateDay, p_month: glib.DateMonth, p_y: glib.DateYear) void;
    pub const setDmy = g_date_set_dmy;

    /// Sets the value of a `glib.Date` from a Julian day number.
    extern fn g_date_set_julian(p_date: *Date, p_julian_date: u32) void;
    pub const setJulian = g_date_set_julian;

    /// Sets the month of the year for a `glib.Date`.  If the resulting
    /// day-month-year triplet is invalid, the date will be invalid.
    extern fn g_date_set_month(p_date: *Date, p_month: glib.DateMonth) void;
    pub const setMonth = g_date_set_month;

    /// Parses a user-inputted string `str`, and try to figure out what date it
    /// represents, taking the [current locale](running.html`locale`)
    /// into account. If the string is successfully parsed, the date will be
    /// valid after the call. Otherwise, it will be invalid. You should check
    /// using `glib.Date.valid` to see whether the parsing succeeded.
    ///
    /// This function is not appropriate for file formats and the like; it
    /// isn't very precise, and its exact behavior varies with the locale.
    /// It's intended to be a heuristic routine that guesses what the user
    /// means by a given string (and it does work pretty well in that
    /// capacity).
    extern fn g_date_set_parse(p_date: *Date, p_str: [*:0]const u8) void;
    pub const setParse = g_date_set_parse;

    /// Sets the value of a date from a `glib.Time` value.
    /// The time to date conversion is done using the user's current timezone.
    extern fn g_date_set_time(p_date: *Date, p_time_: glib.Time) void;
    pub const setTime = g_date_set_time;

    /// Sets the value of a date to the date corresponding to a time
    /// specified as a time_t. The time to date conversion is done using
    /// the user's current timezone.
    ///
    /// To set the value of a date to the current day, you could write:
    /// ```
    ///  time_t now = time (NULL);
    ///  if (now == (time_t) -1)
    ///    // handle the error
    ///  g_date_set_time_t (date, now);
    /// ```
    extern fn g_date_set_time_t(p_date: *Date, p_timet: std.posix.time_t) void;
    pub const setTimeT = g_date_set_time_t;

    /// Sets the value of a date from a `glib.TimeVal` value.  Note that the
    /// `tv_usec` member is ignored, because `glib.Date` can't make use of the
    /// additional precision.
    ///
    /// The time to date conversion is done using the user's current timezone.
    extern fn g_date_set_time_val(p_date: *Date, p_timeval: *glib.TimeVal) void;
    pub const setTimeVal = g_date_set_time_val;

    /// Sets the year for a `glib.Date`. If the resulting day-month-year
    /// triplet is invalid, the date will be invalid.
    extern fn g_date_set_year(p_date: *Date, p_year: glib.DateYear) void;
    pub const setYear = g_date_set_year;

    /// Moves a date some number of days into the past.
    /// To move by weeks, just move by weeks*7 days.
    /// The date must be valid.
    extern fn g_date_subtract_days(p_date: *Date, p_n_days: c_uint) void;
    pub const subtractDays = g_date_subtract_days;

    /// Moves a date some number of months into the past.
    /// If the current day of the month doesn't exist in
    /// the destination month, the day of the month
    /// may change. The date must be valid.
    extern fn g_date_subtract_months(p_date: *Date, p_n_months: c_uint) void;
    pub const subtractMonths = g_date_subtract_months;

    /// Moves a date some number of years into the past.
    /// If the current day doesn't exist in the destination
    /// year (i.e. it's February 29 and you move to a non-leap-year)
    /// then the day is changed to February 29. The date
    /// must be valid.
    extern fn g_date_subtract_years(p_date: *Date, p_n_years: c_uint) void;
    pub const subtractYears = g_date_subtract_years;

    /// Fills in the date-related bits of a struct tm using the `date` value.
    /// Initializes the non-date parts with something safe but meaningless.
    extern fn g_date_to_struct_tm(p_date: *const Date, p_tm: *anyopaque) void;
    pub const toStructTm = g_date_to_struct_tm;

    /// Returns `TRUE` if the `glib.Date` represents an existing day. The date must not
    /// contain garbage; it should have been initialized with `glib.Date.clear`
    /// if it wasn't allocated by one of the `glib.Date.new` variants.
    extern fn g_date_valid(p_date: *const Date) c_int;
    pub const valid = g_date_valid;

    extern fn g_date_get_type() usize;
    pub const getGObjectType = g_date_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GDateTime` is a structure that combines a Gregorian date and time
/// into a single structure.
///
/// `GDateTime` provides many conversion and methods to manipulate dates and times.
/// Time precision is provided down to microseconds and the time can range
/// (proleptically) from 0001-01-01 00:00:00 to 9999-12-31 23:59:59.999999.
/// `GDateTime` follows POSIX time in the sense that it is oblivious to leap
/// seconds.
///
/// `GDateTime` is an immutable object; once it has been created it cannot
/// be modified further. All modifiers will create a new `GDateTime`.
/// Nearly all such functions can fail due to the date or time going out
/// of range, in which case `NULL` will be returned.
///
/// `GDateTime` is reference counted: the reference count is increased by calling
/// `glib.DateTime.ref` and decreased by calling `glib.DateTime.unref`.
/// When the reference count drops to 0, the resources allocated by the `GDateTime`
/// structure are released.
///
/// Many parts of the API may produce non-obvious results. As an
/// example, adding two months to January 31st will yield March 31st
/// whereas adding one month and then one month again will yield either
/// March 28th or March 29th.  Also note that adding 24 hours is not
/// always the same as adding one day (since days containing daylight
/// savings time transitions are either 23 or 25 hours in length).
pub const DateTime = opaque {
    /// Creates a new `glib.DateTime` corresponding to the given date and time in
    /// the time zone `tz`.
    ///
    /// The `year` must be between 1 and 9999, `month` between 1 and 12 and `day`
    /// between 1 and 28, 29, 30 or 31 depending on the month and the year.
    ///
    /// `hour` must be between 0 and 23 and `minute` must be between 0 and 59.
    ///
    /// `seconds` must be at least 0.0 and must be strictly less than 60.0.
    /// It will be rounded down to the nearest microsecond.
    ///
    /// If the given time is not representable in the given time zone (for
    /// example, 02:30 on March 14th 2010 in Toronto, due to daylight savings
    /// time) then the time will be rounded up to the nearest existing time
    /// (in this case, 03:00).  If this matters to you then you should verify
    /// the return value for containing the same as the numbers you gave.
    ///
    /// In the case that the given time is ambiguous in the given time zone
    /// (for example, 01:30 on November 7th 2010 in Toronto, due to daylight
    /// savings time) then the time falling within standard (ie:
    /// non-daylight) time is taken.
    ///
    /// It not considered a programmer error for the values to this function
    /// to be out of range, but in the case that they are, the function will
    /// return `NULL`.
    ///
    /// You should release the return value by calling `glib.DateTime.unref`
    /// when you are done with it.
    extern fn g_date_time_new(p_tz: *glib.TimeZone, p_year: c_int, p_month: c_int, p_day: c_int, p_hour: c_int, p_minute: c_int, p_seconds: f64) ?*glib.DateTime;
    pub const new = g_date_time_new;

    /// Creates a `glib.DateTime` corresponding to the given
    /// [ISO 8601 formatted string](https://en.wikipedia.org/wiki/ISO_8601)
    /// `text`. ISO 8601 strings of the form `<date><sep><time><tz>` are supported, with
    /// some extensions from [RFC 3339](https://tools.ietf.org/html/rfc3339) as
    /// mentioned below.
    ///
    /// Note that as `glib.DateTime` "is oblivious to leap seconds", leap seconds information
    /// in an ISO-8601 string will be ignored, so a `23:59:60` time would be parsed as
    /// `23:59:59`.
    ///
    /// `<sep>` is the separator and can be either 'T', 't' or ' '. The latter two
    /// separators are an extension from
    /// [RFC 3339](https://tools.ietf.org/html/rfc3339`section`-5.6).
    ///
    /// `<date>` is in the form:
    ///
    /// - `YYYY-MM-DD` - Year/month/day, e.g. 2016-08-24.
    /// - `YYYYMMDD` - Same as above without dividers.
    /// - `YYYY-DDD` - Ordinal day where DDD is from 001 to 366, e.g. 2016-237.
    /// - `YYYYDDD` - Same as above without dividers.
    /// - `YYYY-Www-D` - Week day where ww is from 01 to 52 and D from 1-7,
    ///   e.g. 2016-W34-3.
    /// - `YYYYWwwD` - Same as above without dividers.
    ///
    /// `<time>` is in the form:
    ///
    /// - `hh:mm:ss(.sss)` - Hours, minutes, seconds (subseconds), e.g. 22:10:42.123.
    /// - `hhmmss(.sss)` - Same as above without dividers.
    ///
    /// `<tz>` is an optional timezone suffix of the form:
    ///
    /// - `Z` - UTC.
    /// - `+hh:mm` or `-hh:mm` - Offset from UTC in hours and minutes, e.g. +12:00.
    /// - `+hh` or `-hh` - Offset from UTC in hours, e.g. +12.
    ///
    /// If the timezone is not provided in `text` it must be provided in `default_tz`
    /// (this field is otherwise ignored).
    ///
    /// This call can fail (returning `NULL`) if `text` is not a valid ISO 8601
    /// formatted string.
    ///
    /// You should release the return value by calling `glib.DateTime.unref`
    /// when you are done with it.
    extern fn g_date_time_new_from_iso8601(p_text: [*:0]const u8, p_default_tz: ?*glib.TimeZone) ?*glib.DateTime;
    pub const newFromIso8601 = g_date_time_new_from_iso8601;

    /// Creates a `glib.DateTime` corresponding to the given `glib.TimeVal` `tv` in the
    /// local time zone.
    ///
    /// The time contained in a `glib.TimeVal` is always stored in the form of
    /// seconds elapsed since 1970-01-01 00:00:00 UTC, regardless of the
    /// local time offset.
    ///
    /// This call can fail (returning `NULL`) if `tv` represents a time outside
    /// of the supported range of `glib.DateTime`.
    ///
    /// You should release the return value by calling `glib.DateTime.unref`
    /// when you are done with it.
    extern fn g_date_time_new_from_timeval_local(p_tv: *const glib.TimeVal) ?*glib.DateTime;
    pub const newFromTimevalLocal = g_date_time_new_from_timeval_local;

    /// Creates a `glib.DateTime` corresponding to the given `glib.TimeVal` `tv` in UTC.
    ///
    /// The time contained in a `glib.TimeVal` is always stored in the form of
    /// seconds elapsed since 1970-01-01 00:00:00 UTC.
    ///
    /// This call can fail (returning `NULL`) if `tv` represents a time outside
    /// of the supported range of `glib.DateTime`.
    ///
    /// You should release the return value by calling `glib.DateTime.unref`
    /// when you are done with it.
    extern fn g_date_time_new_from_timeval_utc(p_tv: *const glib.TimeVal) ?*glib.DateTime;
    pub const newFromTimevalUtc = g_date_time_new_from_timeval_utc;

    /// Creates a `glib.DateTime` corresponding to the given Unix time `t` in the
    /// local time zone.
    ///
    /// Unix time is the number of seconds that have elapsed since 1970-01-01
    /// 00:00:00 UTC, regardless of the local time offset.
    ///
    /// This call can fail (returning `NULL`) if `t` represents a time outside
    /// of the supported range of `glib.DateTime`.
    ///
    /// You should release the return value by calling `glib.DateTime.unref`
    /// when you are done with it.
    extern fn g_date_time_new_from_unix_local(p_t: i64) ?*glib.DateTime;
    pub const newFromUnixLocal = g_date_time_new_from_unix_local;

    /// Creates a `glib.DateTime` corresponding to the given Unix time `t` in the
    /// local time zone.
    ///
    /// Unix time is the number of microseconds that have elapsed since 1970-01-01
    /// 00:00:00 UTC, regardless of the local time offset.
    ///
    /// This call can fail (returning `NULL`) if `t` represents a time outside
    /// of the supported range of `glib.DateTime`.
    ///
    /// You should release the return value by calling `glib.DateTime.unref`
    /// when you are done with it.
    extern fn g_date_time_new_from_unix_local_usec(p_usecs: i64) ?*glib.DateTime;
    pub const newFromUnixLocalUsec = g_date_time_new_from_unix_local_usec;

    /// Creates a `glib.DateTime` corresponding to the given Unix time `t` in UTC.
    ///
    /// Unix time is the number of seconds that have elapsed since 1970-01-01
    /// 00:00:00 UTC.
    ///
    /// This call can fail (returning `NULL`) if `t` represents a time outside
    /// of the supported range of `glib.DateTime`.
    ///
    /// You should release the return value by calling `glib.DateTime.unref`
    /// when you are done with it.
    extern fn g_date_time_new_from_unix_utc(p_t: i64) ?*glib.DateTime;
    pub const newFromUnixUtc = g_date_time_new_from_unix_utc;

    /// Creates a `glib.DateTime` corresponding to the given Unix time `t` in UTC.
    ///
    /// Unix time is the number of microseconds that have elapsed since 1970-01-01
    /// 00:00:00 UTC.
    ///
    /// This call can fail (returning `NULL`) if `t` represents a time outside
    /// of the supported range of `glib.DateTime`.
    ///
    /// You should release the return value by calling `glib.DateTime.unref`
    /// when you are done with it.
    extern fn g_date_time_new_from_unix_utc_usec(p_usecs: i64) ?*glib.DateTime;
    pub const newFromUnixUtcUsec = g_date_time_new_from_unix_utc_usec;

    /// Creates a new `glib.DateTime` corresponding to the given date and time in
    /// the local time zone.
    ///
    /// This call is equivalent to calling `glib.DateTime.new` with the time
    /// zone returned by `glib.TimeZone.newLocal`.
    extern fn g_date_time_new_local(p_year: c_int, p_month: c_int, p_day: c_int, p_hour: c_int, p_minute: c_int, p_seconds: f64) ?*glib.DateTime;
    pub const newLocal = g_date_time_new_local;

    /// Creates a `glib.DateTime` corresponding to this exact instant in the given
    /// time zone `tz`.  The time is as accurate as the system allows, to a
    /// maximum accuracy of 1 microsecond.
    ///
    /// This function will always succeed unless GLib is still being used after the
    /// year 9999.
    ///
    /// You should release the return value by calling `glib.DateTime.unref`
    /// when you are done with it.
    extern fn g_date_time_new_now(p_tz: *glib.TimeZone) ?*glib.DateTime;
    pub const newNow = g_date_time_new_now;

    /// Creates a `glib.DateTime` corresponding to this exact instant in the local
    /// time zone.
    ///
    /// This is equivalent to calling `glib.DateTime.newNow` with the time
    /// zone returned by `glib.TimeZone.newLocal`.
    extern fn g_date_time_new_now_local() ?*glib.DateTime;
    pub const newNowLocal = g_date_time_new_now_local;

    /// Creates a `glib.DateTime` corresponding to this exact instant in UTC.
    ///
    /// This is equivalent to calling `glib.DateTime.newNow` with the time
    /// zone returned by `glib.TimeZone.newUtc`.
    extern fn g_date_time_new_now_utc() ?*glib.DateTime;
    pub const newNowUtc = g_date_time_new_now_utc;

    /// Creates a new `glib.DateTime` corresponding to the given date and time in
    /// UTC.
    ///
    /// This call is equivalent to calling `glib.DateTime.new` with the time
    /// zone returned by `glib.TimeZone.newUtc`.
    extern fn g_date_time_new_utc(p_year: c_int, p_month: c_int, p_day: c_int, p_hour: c_int, p_minute: c_int, p_seconds: f64) ?*glib.DateTime;
    pub const newUtc = g_date_time_new_utc;

    /// Creates a copy of `datetime` and adds the specified timespan to the copy.
    extern fn g_date_time_add(p_datetime: *DateTime, p_timespan: glib.TimeSpan) ?*glib.DateTime;
    pub const add = g_date_time_add;

    /// Creates a copy of `datetime` and adds the specified number of days to the
    /// copy. Add negative values to subtract days.
    extern fn g_date_time_add_days(p_datetime: *DateTime, p_days: c_int) ?*glib.DateTime;
    pub const addDays = g_date_time_add_days;

    /// Creates a new `glib.DateTime` adding the specified values to the current date and
    /// time in `datetime`. Add negative values to subtract.
    extern fn g_date_time_add_full(p_datetime: *DateTime, p_years: c_int, p_months: c_int, p_days: c_int, p_hours: c_int, p_minutes: c_int, p_seconds: f64) ?*glib.DateTime;
    pub const addFull = g_date_time_add_full;

    /// Creates a copy of `datetime` and adds the specified number of hours.
    /// Add negative values to subtract hours.
    extern fn g_date_time_add_hours(p_datetime: *DateTime, p_hours: c_int) ?*glib.DateTime;
    pub const addHours = g_date_time_add_hours;

    /// Creates a copy of `datetime` adding the specified number of minutes.
    /// Add negative values to subtract minutes.
    extern fn g_date_time_add_minutes(p_datetime: *DateTime, p_minutes: c_int) ?*glib.DateTime;
    pub const addMinutes = g_date_time_add_minutes;

    /// Creates a copy of `datetime` and adds the specified number of months to the
    /// copy. Add negative values to subtract months.
    ///
    /// The day of the month of the resulting `glib.DateTime` is clamped to the number
    /// of days in the updated calendar month. For example, if adding 1 month to
    /// 31st January 2018, the result would be 28th February 2018. In 2020 (a leap
    /// year), the result would be 29th February.
    extern fn g_date_time_add_months(p_datetime: *DateTime, p_months: c_int) ?*glib.DateTime;
    pub const addMonths = g_date_time_add_months;

    /// Creates a copy of `datetime` and adds the specified number of seconds.
    /// Add negative values to subtract seconds.
    extern fn g_date_time_add_seconds(p_datetime: *DateTime, p_seconds: f64) ?*glib.DateTime;
    pub const addSeconds = g_date_time_add_seconds;

    /// Creates a copy of `datetime` and adds the specified number of weeks to the
    /// copy. Add negative values to subtract weeks.
    extern fn g_date_time_add_weeks(p_datetime: *DateTime, p_weeks: c_int) ?*glib.DateTime;
    pub const addWeeks = g_date_time_add_weeks;

    /// Creates a copy of `datetime` and adds the specified number of years to the
    /// copy. Add negative values to subtract years.
    ///
    /// As with `glib.DateTime.addMonths`, if the resulting date would be 29th
    /// February on a non-leap year, the day will be clamped to 28th February.
    extern fn g_date_time_add_years(p_datetime: *DateTime, p_years: c_int) ?*glib.DateTime;
    pub const addYears = g_date_time_add_years;

    /// A comparison function for `GDateTimes` that is suitable
    /// as a `glib.CompareFunc`. Both `GDateTimes` must be non-`NULL`.
    extern fn g_date_time_compare(p_dt1: *const DateTime, p_dt2: *const glib.DateTime) c_int;
    pub const compare = g_date_time_compare;

    /// Calculates the difference in time between `end` and `begin`.  The
    /// `glib.TimeSpan` that is returned is effectively `end` - `begin` (ie:
    /// positive if the first parameter is larger).
    extern fn g_date_time_difference(p_end: *DateTime, p_begin: *glib.DateTime) glib.TimeSpan;
    pub const difference = g_date_time_difference;

    /// Checks to see if `dt1` and `dt2` are equal.
    ///
    /// Equal here means that they represent the same moment after converting
    /// them to the same time zone.
    extern fn g_date_time_equal(p_dt1: *const DateTime, p_dt2: *const glib.DateTime) c_int;
    pub const equal = g_date_time_equal;

    /// Creates a newly allocated string representing the requested `format`.
    ///
    /// The format strings understood by this function are a subset of the
    /// ``strftime`` format language as specified by C99.  The ``D``, ``U`` and ``W``
    /// conversions are not supported, nor is the `E` modifier.  The GNU
    /// extensions ``k``, ``l``, ``s`` and ``P`` are supported, however, as are the
    /// `0`, `_` and `-` modifiers. The Python extension ``f`` is also supported.
    ///
    /// In contrast to ``strftime``, this function always produces a UTF-8
    /// string, regardless of the current locale.  Note that the rendering of
    /// many formats is locale-dependent and may not match the ``strftime``
    /// output exactly.
    ///
    /// The following format specifiers are supported:
    ///
    /// - ``a``: the abbreviated weekday name according to the current locale
    /// - ``A``: the full weekday name according to the current locale
    /// - ``b``: the abbreviated month name according to the current locale
    /// - ``B``: the full month name according to the current locale
    /// - ``c``: the preferred date and time representation for the current locale
    /// - ``C``: the century number (year/100) as a 2-digit integer (00-99)
    /// - ``d``: the day of the month as a decimal number (range 01 to 31)
    /// - ``e``: the day of the month as a decimal number (range 1 to 31);
    ///   single digits are preceded by a figure space (U+2007)
    /// - ``F``: equivalent to ``Y`-`m`-`d`` (the ISO 8601 date format)
    /// - ``g``: the last two digits of the ISO 8601 week-based year as a
    ///   decimal number (00-99). This works well with ``V`` and ``u``.
    /// - ``G``: the ISO 8601 week-based year as a decimal number. This works
    ///   well with ``V`` and ``u``.
    /// - ``h``: equivalent to ``b``
    /// - ``H``: the hour as a decimal number using a 24-hour clock (range 00 to 23)
    /// - ``I``: the hour as a decimal number using a 12-hour clock (range 01 to 12)
    /// - ``j``: the day of the year as a decimal number (range 001 to 366)
    /// - ``k``: the hour (24-hour clock) as a decimal number (range 0 to 23);
    ///   single digits are preceded by a figure space (U+2007)
    /// - ``l``: the hour (12-hour clock) as a decimal number (range 1 to 12);
    ///   single digits are preceded by a figure space (U+2007)
    /// - ``m``: the month as a decimal number (range 01 to 12)
    /// - ``M``: the minute as a decimal number (range 00 to 59)
    /// - ``f``: the microsecond as a decimal number (range 000000 to 999999)
    /// - ``p``: either ‘AM’ or ‘PM’ according to the given time value, or the
    ///   corresponding  strings for the current locale.  Noon is treated as
    ///   ‘PM’ and midnight as ‘AM’. Use of this format specifier is discouraged, as
    ///   many locales have no concept of AM/PM formatting. Use ``c`` or ``X`` instead.
    /// - ``P``: like ``p`` but lowercase: ‘am’ or ‘pm’ or a corresponding string for
    ///   the current locale. Use of this format specifier is discouraged, as
    ///   many locales have no concept of AM/PM formatting. Use ``c`` or ``X`` instead.
    /// - ``r``: the time in a.m. or p.m. notation. Use of this format specifier is
    ///   discouraged, as many locales have no concept of AM/PM formatting. Use ``c``
    ///   or ``X`` instead.
    /// - ``R``: the time in 24-hour notation (``H`:`M``)
    /// - ``s``: the number of seconds since the Epoch, that is, since 1970-01-01
    ///   00:00:00 UTC
    /// - ``S``: the second as a decimal number (range 00 to 60)
    /// - ``t``: a tab character
    /// - ``T``: the time in 24-hour notation with seconds (``H`:`M`:`S``)
    /// - ``u``: the ISO 8601 standard day of the week as a decimal, range 1 to 7,
    ///    Monday being 1. This works well with ``G`` and ``V``.
    /// - ``V``: the ISO 8601 standard week number of the current year as a decimal
    ///   number, range 01 to 53, where week 1 is the first week that has at
    ///   least 4 days in the new year. See `glib.DateTime.getWeekOfYear`.
    ///   This works well with ``G`` and ``u``.
    /// - ``w``: the day of the week as a decimal, range 0 to 6, Sunday being 0.
    ///   This is not the ISO 8601 standard format — use ``u`` instead.
    /// - ``x``: the preferred date representation for the current locale without
    ///   the time
    /// - ``X``: the preferred time representation for the current locale without
    ///   the date
    /// - ``y``: the year as a decimal number without the century
    /// - ``Y``: the year as a decimal number including the century
    /// - ``z``: the time zone as an offset from UTC (`+hhmm`)
    /// - `%:z`: the time zone as an offset from UTC (`+hh:mm`).
    ///   This is a gnulib ``strftime`` extension. Since: 2.38
    /// - `%::z`: the time zone as an offset from UTC (`+hh:mm:ss`). This is a
    ///   gnulib ``strftime`` extension. Since: 2.38
    /// - `%:::z`: the time zone as an offset from UTC, with `:` to necessary
    ///   precision (e.g., `-04`, `+05:30`). This is a gnulib ``strftime`` extension. Since: 2.38
    /// - ``Z``: the time zone or name or abbreviation
    /// - `%%`: a literal `%` character
    ///
    /// Some conversion specifications can be modified by preceding the
    /// conversion specifier by one or more modifier characters.
    ///
    /// The following modifiers are supported for many of the numeric
    /// conversions:
    ///
    /// - `O`: Use alternative numeric symbols, if the current locale supports those.
    /// - `_`: Pad a numeric result with spaces. This overrides the default padding
    ///   for the specifier.
    /// - `-`: Do not pad a numeric result. This overrides the default padding
    ///   for the specifier.
    /// - `0`: Pad a numeric result with zeros. This overrides the default padding
    ///   for the specifier.
    ///
    /// The following modifiers are supported for many of the alphabetic conversions:
    ///
    /// - `^`: Use upper case if possible. This is a gnulib ``strftime`` extension.
    ///   Since: 2.80
    /// - `#`: Use opposite case if possible. This is a gnulib ``strftime``
    ///   extension. Since: 2.80
    ///
    /// Additionally, when `O` is used with `B`, `b`, or `h`, it produces the alternative
    /// form of a month name. The alternative form should be used when the month
    /// name is used without a day number (e.g., standalone). It is required in
    /// some languages (Baltic, Slavic, Greek, and more) due to their grammatical
    /// rules. For other languages there is no difference. ``OB`` is a GNU and BSD
    /// ``strftime`` extension expected to be added to the future POSIX specification,
    /// ``Ob`` and ``Oh`` are GNU ``strftime`` extensions. Since: 2.56
    ///
    /// Since GLib 2.80, when `E` is used with ``c``, ``C``, ``x``, ``X``, ``y`` or ``Y``,
    /// the date is formatted using an alternate era representation specific to the
    /// locale. This is typically used for the Thai solar calendar or Japanese era
    /// names, for example.
    ///
    /// - ``Ec``: the preferred date and time representation for the current locale,
    ///   using the alternate era representation
    /// - ``EC``: the name of the era
    /// - ``Ex``: the preferred date representation for the current locale without
    ///   the time, using the alternate era representation
    /// - ``EX``: the preferred time representation for the current locale without
    ///   the date, using the alternate era representation
    /// - ``Ey``: the year since the beginning of the era denoted by the ``EC``
    ///   specifier
    /// - ``EY``: the full alternative year representation
    extern fn g_date_time_format(p_datetime: *DateTime, p_format: [*:0]const u8) ?[*:0]u8;
    pub const format = g_date_time_format;

    /// Format `datetime` in [ISO 8601 format](https://en.wikipedia.org/wiki/ISO_8601),
    /// including the date, time and time zone, and return that as a UTF-8 encoded
    /// string.
    ///
    /// Since GLib 2.66, this will output to sub-second precision if needed.
    extern fn g_date_time_format_iso8601(p_datetime: *DateTime) ?[*:0]u8;
    pub const formatIso8601 = g_date_time_format_iso8601;

    /// Retrieves the day of the month represented by `datetime` in the gregorian
    /// calendar.
    extern fn g_date_time_get_day_of_month(p_datetime: *DateTime) c_int;
    pub const getDayOfMonth = g_date_time_get_day_of_month;

    /// Retrieves the ISO 8601 day of the week on which `datetime` falls (1 is
    /// Monday, 2 is Tuesday... 7 is Sunday).
    extern fn g_date_time_get_day_of_week(p_datetime: *DateTime) c_int;
    pub const getDayOfWeek = g_date_time_get_day_of_week;

    /// Retrieves the day of the year represented by `datetime` in the Gregorian
    /// calendar.
    extern fn g_date_time_get_day_of_year(p_datetime: *DateTime) c_int;
    pub const getDayOfYear = g_date_time_get_day_of_year;

    /// Retrieves the hour of the day represented by `datetime`
    extern fn g_date_time_get_hour(p_datetime: *DateTime) c_int;
    pub const getHour = g_date_time_get_hour;

    /// Retrieves the microsecond of the date represented by `datetime`
    extern fn g_date_time_get_microsecond(p_datetime: *DateTime) c_int;
    pub const getMicrosecond = g_date_time_get_microsecond;

    /// Retrieves the minute of the hour represented by `datetime`
    extern fn g_date_time_get_minute(p_datetime: *DateTime) c_int;
    pub const getMinute = g_date_time_get_minute;

    /// Retrieves the month of the year represented by `datetime` in the Gregorian
    /// calendar.
    extern fn g_date_time_get_month(p_datetime: *DateTime) c_int;
    pub const getMonth = g_date_time_get_month;

    /// Retrieves the second of the minute represented by `datetime`
    extern fn g_date_time_get_second(p_datetime: *DateTime) c_int;
    pub const getSecond = g_date_time_get_second;

    /// Retrieves the number of seconds since the start of the last minute,
    /// including the fractional part.
    extern fn g_date_time_get_seconds(p_datetime: *DateTime) f64;
    pub const getSeconds = g_date_time_get_seconds;

    /// Get the time zone for this `datetime`.
    extern fn g_date_time_get_timezone(p_datetime: *DateTime) *glib.TimeZone;
    pub const getTimezone = g_date_time_get_timezone;

    /// Determines the time zone abbreviation to be used at the time and in
    /// the time zone of `datetime`.
    ///
    /// For example, in Toronto this is currently "EST" during the winter
    /// months and "EDT" during the summer months when daylight savings
    /// time is in effect.
    extern fn g_date_time_get_timezone_abbreviation(p_datetime: *DateTime) [*:0]const u8;
    pub const getTimezoneAbbreviation = g_date_time_get_timezone_abbreviation;

    /// Determines the offset to UTC in effect at the time and in the time
    /// zone of `datetime`.
    ///
    /// The offset is the number of microseconds that you add to UTC time to
    /// arrive at local time for the time zone (ie: negative numbers for time
    /// zones west of GMT, positive numbers for east).
    ///
    /// If `datetime` represents UTC time, then the offset is always zero.
    extern fn g_date_time_get_utc_offset(p_datetime: *DateTime) glib.TimeSpan;
    pub const getUtcOffset = g_date_time_get_utc_offset;

    /// Returns the ISO 8601 week-numbering year in which the week containing
    /// `datetime` falls.
    ///
    /// This function, taken together with `glib.DateTime.getWeekOfYear` and
    /// `glib.DateTime.getDayOfWeek` can be used to determine the full ISO
    /// week date on which `datetime` falls.
    ///
    /// This is usually equal to the normal Gregorian year (as returned by
    /// `glib.DateTime.getYear`), except as detailed below:
    ///
    /// For Thursday, the week-numbering year is always equal to the usual
    /// calendar year.  For other days, the number is such that every day
    /// within a complete week (Monday to Sunday) is contained within the
    /// same week-numbering year.
    ///
    /// For Monday, Tuesday and Wednesday occurring near the end of the year,
    /// this may mean that the week-numbering year is one greater than the
    /// calendar year (so that these days have the same week-numbering year
    /// as the Thursday occurring early in the next year).
    ///
    /// For Friday, Saturday and Sunday occurring near the start of the year,
    /// this may mean that the week-numbering year is one less than the
    /// calendar year (so that these days have the same week-numbering year
    /// as the Thursday occurring late in the previous year).
    ///
    /// An equivalent description is that the week-numbering year is equal to
    /// the calendar year containing the majority of the days in the current
    /// week (Monday to Sunday).
    ///
    /// Note that January 1 0001 in the proleptic Gregorian calendar is a
    /// Monday, so this function never returns 0.
    extern fn g_date_time_get_week_numbering_year(p_datetime: *DateTime) c_int;
    pub const getWeekNumberingYear = g_date_time_get_week_numbering_year;

    /// Returns the ISO 8601 week number for the week containing `datetime`.
    /// The ISO 8601 week number is the same for every day of the week (from
    /// Moday through Sunday).  That can produce some unusual results
    /// (described below).
    ///
    /// The first week of the year is week 1.  This is the week that contains
    /// the first Thursday of the year.  Equivalently, this is the first week
    /// that has more than 4 of its days falling within the calendar year.
    ///
    /// The value 0 is never returned by this function.  Days contained
    /// within a year but occurring before the first ISO 8601 week of that
    /// year are considered as being contained in the last week of the
    /// previous year.  Similarly, the final days of a calendar year may be
    /// considered as being part of the first ISO 8601 week of the next year
    /// if 4 or more days of that week are contained within the new year.
    extern fn g_date_time_get_week_of_year(p_datetime: *DateTime) c_int;
    pub const getWeekOfYear = g_date_time_get_week_of_year;

    /// Retrieves the year represented by `datetime` in the Gregorian calendar.
    extern fn g_date_time_get_year(p_datetime: *DateTime) c_int;
    pub const getYear = g_date_time_get_year;

    /// Retrieves the Gregorian day, month, and year of a given `glib.DateTime`.
    extern fn g_date_time_get_ymd(p_datetime: *DateTime, p_year: ?*c_int, p_month: ?*c_int, p_day: ?*c_int) void;
    pub const getYmd = g_date_time_get_ymd;

    /// Hashes `datetime` into a `guint`, suitable for use within `glib.HashTable`.
    extern fn g_date_time_hash(p_datetime: *const DateTime) c_uint;
    pub const hash = g_date_time_hash;

    /// Determines if daylight savings time is in effect at the time and in
    /// the time zone of `datetime`.
    extern fn g_date_time_is_daylight_savings(p_datetime: *DateTime) c_int;
    pub const isDaylightSavings = g_date_time_is_daylight_savings;

    /// Atomically increments the reference count of `datetime` by one.
    extern fn g_date_time_ref(p_datetime: *DateTime) *glib.DateTime;
    pub const ref = g_date_time_ref;

    /// Creates a new `glib.DateTime` corresponding to the same instant in time as
    /// `datetime`, but in the local time zone.
    ///
    /// This call is equivalent to calling `glib.DateTime.toTimezone` with the
    /// time zone returned by `glib.TimeZone.newLocal`.
    extern fn g_date_time_to_local(p_datetime: *DateTime) ?*glib.DateTime;
    pub const toLocal = g_date_time_to_local;

    /// Stores the instant in time that `datetime` represents into `tv`.
    ///
    /// The time contained in a `glib.TimeVal` is always stored in the form of
    /// seconds elapsed since 1970-01-01 00:00:00 UTC, regardless of the time
    /// zone associated with `datetime`.
    ///
    /// On systems where 'long' is 32bit (ie: all 32bit systems and all
    /// Windows systems), a `glib.TimeVal` is incapable of storing the entire
    /// range of values that `glib.DateTime` is capable of expressing.  On those
    /// systems, this function returns `FALSE` to indicate that the time is
    /// out of range.
    ///
    /// On systems where 'long' is 64bit, this function never fails.
    extern fn g_date_time_to_timeval(p_datetime: *DateTime, p_tv: *glib.TimeVal) c_int;
    pub const toTimeval = g_date_time_to_timeval;

    /// Create a new `glib.DateTime` corresponding to the same instant in time as
    /// `datetime`, but in the time zone `tz`.
    ///
    /// This call can fail in the case that the time goes out of bounds.  For
    /// example, converting 0001-01-01 00:00:00 UTC to a time zone west of
    /// Greenwich will fail (due to the year 0 being out of range).
    extern fn g_date_time_to_timezone(p_datetime: *DateTime, p_tz: *glib.TimeZone) ?*glib.DateTime;
    pub const toTimezone = g_date_time_to_timezone;

    /// Gives the Unix time corresponding to `datetime`, rounding down to the
    /// nearest second.
    ///
    /// Unix time is the number of seconds that have elapsed since 1970-01-01
    /// 00:00:00 UTC, regardless of the time zone associated with `datetime`.
    extern fn g_date_time_to_unix(p_datetime: *DateTime) i64;
    pub const toUnix = g_date_time_to_unix;

    /// Gives the Unix time corresponding to `datetime`, in microseconds.
    ///
    /// Unix time is the number of microseconds that have elapsed since 1970-01-01
    /// 00:00:00 UTC, regardless of the time zone associated with `datetime`.
    extern fn g_date_time_to_unix_usec(p_datetime: *DateTime) i64;
    pub const toUnixUsec = g_date_time_to_unix_usec;

    /// Creates a new `glib.DateTime` corresponding to the same instant in time as
    /// `datetime`, but in UTC.
    ///
    /// This call is equivalent to calling `glib.DateTime.toTimezone` with the
    /// time zone returned by `glib.TimeZone.newUtc`.
    extern fn g_date_time_to_utc(p_datetime: *DateTime) ?*glib.DateTime;
    pub const toUtc = g_date_time_to_utc;

    /// Atomically decrements the reference count of `datetime` by one.
    ///
    /// When the reference count reaches zero, the resources allocated by
    /// `datetime` are freed
    extern fn g_date_time_unref(p_datetime: *DateTime) void;
    pub const unref = g_date_time_unref;

    extern fn g_date_time_get_type() usize;
    pub const getGObjectType = g_date_time_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Associates a string with a bit flag.
/// Used in `glib.parseDebugString`.
pub const DebugKey = extern struct {
    /// the string
    f_key: ?[*:0]const u8,
    /// the flag
    f_value: c_uint,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// An opaque structure representing an opened directory.
pub const Dir = opaque {
    /// Creates a subdirectory in the preferred directory for temporary
    /// files (as returned by `glib.getTmpDir`).
    ///
    /// `tmpl` should be a string in the GLib file name encoding containing
    /// a sequence of six 'X' characters, as the parameter to `glib.mkstemp`.
    /// However, unlike these functions, the template should only be a
    /// basename, no directory components are allowed. If template is
    /// `NULL`, a default template is used.
    ///
    /// Note that in contrast to `glib.mkdtemp` (and `mkdtemp`) `tmpl` is not
    /// modified, and might thus be a read-only literal string.
    extern fn g_dir_make_tmp(p_tmpl: ?[*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const makeTmp = g_dir_make_tmp;

    /// Opens a directory for reading. The names of the files in the
    /// directory can then be retrieved using `glib.Dir.readName`.  Note
    /// that the ordering is not defined.
    extern fn g_dir_open(p_path: [*:0]const u8, p_flags: c_uint, p_error: ?*?*glib.Error) ?*glib.Dir;
    pub const open = g_dir_open;

    /// Closes the directory immediately and decrements the reference count.
    ///
    /// Once the reference count reaches zero, the `GDir` structure itself will be
    /// freed. Prior to GLib 2.80, `GDir` was not reference counted.
    ///
    /// It is an error to call any of the `GDir` methods other than
    /// `glib.Dir.ref` and `glib.Dir.unref` on a `GDir` after calling
    /// `glib.Dir.close` on it.
    extern fn g_dir_close(p_dir: *Dir) void;
    pub const close = g_dir_close;

    /// Retrieves the name of another entry in the directory, or `NULL`.
    /// The order of entries returned from this function is not defined,
    /// and may vary by file system or other operating-system dependent
    /// factors.
    ///
    /// `NULL` may also be returned in case of errors. On Unix, you can
    /// check `errno` to find out if `NULL` was returned because of an error.
    ///
    /// On Unix, the '.' and '..' entries are omitted, and the returned
    /// name is in the on-disk encoding.
    ///
    /// On Windows, as is true of all GLib functions which operate on
    /// filenames, the returned name is in UTF-8.
    extern fn g_dir_read_name(p_dir: *Dir) [*:0]const u8;
    pub const readName = g_dir_read_name;

    /// Increment the reference count of `dir`.
    extern fn g_dir_ref(p_dir: *Dir) *glib.Dir;
    pub const ref = g_dir_ref;

    /// Resets the given directory. The next call to `glib.Dir.readName`
    /// will return the first entry again.
    extern fn g_dir_rewind(p_dir: *Dir) void;
    pub const rewind = g_dir_rewind;

    /// Decrements the reference count of `dir`.
    ///
    /// Once the reference count reaches zero, the directory will be closed and all
    /// resources associated with it will be freed. If `glib.Dir.close` is
    /// called when the reference count is greater than zero, the directory is closed
    /// but the `GDir` structure will not be freed until its reference count reaches
    /// zero.
    ///
    /// It is an error to call any of the `GDir` methods other than
    /// `glib.Dir.ref` and `glib.Dir.unref` on a `GDir` after calling
    /// `glib.Dir.close` on it.
    extern fn g_dir_unref(p_dir: *Dir) void;
    pub const unref = g_dir_unref;

    extern fn g_dir_get_type() usize;
    pub const getGObjectType = g_dir_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `GError` structure contains information about
/// an error that has occurred.
pub const Error = extern struct {
    /// error domain, e.g. `G_FILE_ERROR`
    f_domain: glib.Quark,
    /// error code, e.g. `G_FILE_ERROR_NOENT`
    f_code: c_int,
    /// human-readable informative error message
    f_message: ?[*:0]u8,

    /// This function registers an extended `glib.Error` domain.
    /// `error_type_name` will be duplicated. Otherwise does the same as
    /// `glib.errorDomainRegisterStatic`.
    extern fn g_error_domain_register(p_error_type_name: [*:0]const u8, p_error_type_private_size: usize, p_error_type_init: glib.ErrorInitFunc, p_error_type_copy: glib.ErrorCopyFunc, p_error_type_clear: glib.ErrorClearFunc) glib.Quark;
    pub const domainRegister = g_error_domain_register;

    /// This function registers an extended `glib.Error` domain.
    ///
    /// `error_type_name` should not be freed. `error_type_private_size` must
    /// be greater than 0.
    ///
    /// `error_type_init` receives an initialized `glib.Error` and should then initialize
    /// the private data.
    ///
    /// `error_type_copy` is a function that receives both original and a copy
    /// `glib.Error` and should copy the fields of the private error data. The standard
    /// `glib.Error` fields are already handled.
    ///
    /// `error_type_clear` receives the pointer to the error, and it should free the
    /// fields of the private error data. It should not free the struct itself though.
    ///
    /// Normally, it is better to use `G_DEFINE_EXTENDED_ERROR`, as it
    /// already takes care of passing valid information to this function.
    extern fn g_error_domain_register_static(p_error_type_name: [*:0]const u8, p_error_type_private_size: usize, p_error_type_init: glib.ErrorInitFunc, p_error_type_copy: glib.ErrorCopyFunc, p_error_type_clear: glib.ErrorClearFunc) glib.Quark;
    pub const domainRegisterStatic = g_error_domain_register_static;

    /// Creates a new `glib.Error` with the given `domain` and `code`,
    /// and a message formatted with `format`.
    extern fn g_error_new(p_domain: glib.Quark, p_code: c_int, p_format: [*:0]const u8, ...) *glib.Error;
    pub const new = g_error_new;

    /// Creates a new `glib.Error`; unlike `glib.Error.new`, `message` is
    /// not a `printf`-style format string. Use this function if
    /// `message` contains text you don't have control over,
    /// that could include `printf` escape sequences.
    extern fn g_error_new_literal(p_domain: glib.Quark, p_code: c_int, p_message: [*:0]const u8) *glib.Error;
    pub const newLiteral = g_error_new_literal;

    /// Creates a new `glib.Error` with the given `domain` and `code`,
    /// and a message formatted with `format`.
    extern fn g_error_new_valist(p_domain: glib.Quark, p_code: c_int, p_format: [*:0]const u8, p_args: std.builtin.VaList) *glib.Error;
    pub const newValist = g_error_new_valist;

    /// Makes a copy of `error`.
    extern fn g_error_copy(p_error: *const Error) *glib.Error;
    pub const copy = g_error_copy;

    /// Frees a `glib.Error` and associated resources.
    extern fn g_error_free(p_error: *Error) void;
    pub const free = g_error_free;

    /// Returns `TRUE` if `error` matches `domain` and `code`, `FALSE`
    /// otherwise. In particular, when `error` is `NULL`, `FALSE` will
    /// be returned.
    ///
    /// If `domain` contains a `FAILED` (or otherwise generic) error code,
    /// you should generally not check for it explicitly, but should
    /// instead treat any not-explicitly-recognized error code as being
    /// equivalent to the `FAILED` code. This way, if the domain is
    /// extended in the future to provide a more specific error code for
    /// a certain case, your code will still work.
    extern fn g_error_matches(p_error: ?*const Error, p_domain: glib.Quark, p_code: c_int) c_int;
    pub const matches = g_error_matches;

    extern fn g_error_get_type() usize;
    pub const getGObjectType = g_error_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.HashTable` struct is an opaque data structure to represent a
/// [Hash Table](data-structures.html`hash`-tables). It should only be accessed via the
/// following functions.
pub const HashTable = opaque {
    /// This is a convenience function for using a `glib.HashTable` as a set.  It
    /// is equivalent to calling `glib.hashTableReplace` with `key` as both the
    /// key and the value.
    ///
    /// In particular, this means that if `key` already exists in the hash table, then
    /// the old copy of `key` in the hash table is freed and `key` replaces it in the
    /// table.
    ///
    /// When a hash table only ever contains keys that have themselves as the
    /// corresponding value it is able to be stored more efficiently.  See
    /// the discussion in the section description.
    ///
    /// Starting from GLib 2.40, this function returns a boolean value to
    /// indicate whether the newly added value was already in the hash table
    /// or not.
    extern fn g_hash_table_add(p_hash_table: *glib.HashTable, p_key: ?*anyopaque) c_int;
    pub const add = g_hash_table_add;

    /// Checks if `key` is in `hash_table`.
    extern fn g_hash_table_contains(p_hash_table: *glib.HashTable, p_key: ?*const anyopaque) c_int;
    pub const contains = g_hash_table_contains;

    /// Destroys all keys and values in the `glib.HashTable` and decrements its
    /// reference count by 1. If keys and/or values are dynamically allocated,
    /// you should either free them first or create the `glib.HashTable` with destroy
    /// notifiers using `glib.HashTable.newFull`. In the latter case the destroy
    /// functions you supplied will be called on all keys and values during the
    /// destruction phase.
    extern fn g_hash_table_destroy(p_hash_table: *glib.HashTable) void;
    pub const destroy = g_hash_table_destroy;

    /// Calls the given function for key/value pairs in the `glib.HashTable`
    /// until `predicate` returns `TRUE`. The function is passed the key
    /// and value of each pair, and the given `user_data` parameter. The
    /// hash table may not be modified while iterating over it (you can't
    /// add/remove items).
    ///
    /// Note, that hash tables are really only optimized for forward
    /// lookups, i.e. `glib.hashTableLookup`. So code that frequently issues
    /// `glib.hashTableFind` or `glib.hashTableForeach` (e.g. in the order of
    /// once per every entry in a hash table) should probably be reworked
    /// to use additional or different data structures for reverse lookups
    /// (keep in mind that an O(n) find/foreach operation issued for all n
    /// values in a hash table ends up needing O(n*n) operations).
    extern fn g_hash_table_find(p_hash_table: *glib.HashTable, p_predicate: glib.HRFunc, p_user_data: ?*anyopaque) ?*anyopaque;
    pub const find = g_hash_table_find;

    /// Calls the given function for each of the key/value pairs in the
    /// `glib.HashTable`.  The function is passed the key and value of each
    /// pair, and the given `user_data` parameter.  The hash table may not
    /// be modified while iterating over it (you can't add/remove
    /// items). To remove all items matching a predicate, use
    /// `glib.hashTableForeachRemove`.
    ///
    /// The order in which `glib.hashTableForeach` iterates over the keys/values in
    /// the hash table is not defined.
    ///
    /// See `glib.hashTableFind` for performance caveats for linear
    /// order searches in contrast to `glib.hashTableLookup`.
    extern fn g_hash_table_foreach(p_hash_table: *glib.HashTable, p_func: glib.HFunc, p_user_data: ?*anyopaque) void;
    pub const foreach = g_hash_table_foreach;

    /// Calls the given function for each key/value pair in the
    /// `glib.HashTable`. If the function returns `TRUE`, then the key/value
    /// pair is removed from the `glib.HashTable`. If you supplied key or
    /// value destroy functions when creating the `glib.HashTable`, they are
    /// used to free the memory allocated for the removed keys and values.
    ///
    /// See `glib.HashTableIter` for an alternative way to loop over the
    /// key/value pairs in the hash table.
    extern fn g_hash_table_foreach_remove(p_hash_table: *glib.HashTable, p_func: glib.HRFunc, p_user_data: ?*anyopaque) c_uint;
    pub const foreachRemove = g_hash_table_foreach_remove;

    /// Calls the given function for each key/value pair in the
    /// `glib.HashTable`. If the function returns `TRUE`, then the key/value
    /// pair is removed from the `glib.HashTable`, but no key or value
    /// destroy functions are called.
    ///
    /// See `glib.HashTableIter` for an alternative way to loop over the
    /// key/value pairs in the hash table.
    extern fn g_hash_table_foreach_steal(p_hash_table: *glib.HashTable, p_func: glib.HRFunc, p_user_data: ?*anyopaque) c_uint;
    pub const foreachSteal = g_hash_table_foreach_steal;

    /// Retrieves every key inside `hash_table`. The returned data is valid
    /// until changes to the hash release those keys.
    ///
    /// This iterates over every entry in the hash table to build its return value.
    /// To iterate over the entries in a `glib.HashTable` more efficiently, use a
    /// `glib.HashTableIter`.
    extern fn g_hash_table_get_keys(p_hash_table: *glib.HashTable) *glib.List;
    pub const getKeys = g_hash_table_get_keys;

    /// Retrieves every key inside `hash_table`, as an array.
    ///
    /// The returned array is `NULL`-terminated but may contain `NULL` as a
    /// key.  Use `length` to determine the true length if it's possible that
    /// `NULL` was used as the value for a key.
    ///
    /// Note: in the common case of a string-keyed `glib.HashTable`, the return
    /// value of this function can be conveniently cast to (const gchar **).
    ///
    /// This iterates over every entry in the hash table to build its return value.
    /// To iterate over the entries in a `glib.HashTable` more efficiently, use a
    /// `glib.HashTableIter`.
    ///
    /// You should always free the return result with `glib.free`.  In the
    /// above-mentioned case of a string-keyed hash table, it may be
    /// appropriate to use `glib.strfreev` if you call `glib.hashTableStealAll`
    /// first to transfer ownership of the keys.
    extern fn g_hash_table_get_keys_as_array(p_hash_table: *glib.HashTable, p_length: ?*c_uint) [*]*anyopaque;
    pub const getKeysAsArray = g_hash_table_get_keys_as_array;

    /// Retrieves every key inside `hash_table`, as a `glib.PtrArray`.
    /// The returned data is valid until changes to the hash release those keys.
    ///
    /// This iterates over every entry in the hash table to build its return value.
    /// To iterate over the entries in a `glib.HashTable` more efficiently, use a
    /// `glib.HashTableIter`.
    ///
    /// You should always unref the returned array with `glib.PtrArray.unref`.
    extern fn g_hash_table_get_keys_as_ptr_array(p_hash_table: *glib.HashTable) *glib.PtrArray;
    pub const getKeysAsPtrArray = g_hash_table_get_keys_as_ptr_array;

    /// Retrieves every value inside `hash_table`. The returned data
    /// is valid until `hash_table` is modified.
    ///
    /// This iterates over every entry in the hash table to build its return value.
    /// To iterate over the entries in a `glib.HashTable` more efficiently, use a
    /// `glib.HashTableIter`.
    extern fn g_hash_table_get_values(p_hash_table: *glib.HashTable) *glib.List;
    pub const getValues = g_hash_table_get_values;

    /// Retrieves every value inside `hash_table`, as a `glib.PtrArray`.
    /// The returned data is valid until changes to the hash release those values.
    ///
    /// This iterates over every entry in the hash table to build its return value.
    /// To iterate over the entries in a `glib.HashTable` more efficiently, use a
    /// `glib.HashTableIter`.
    ///
    /// You should always unref the returned array with `glib.PtrArray.unref`.
    extern fn g_hash_table_get_values_as_ptr_array(p_hash_table: *glib.HashTable) *glib.PtrArray;
    pub const getValuesAsPtrArray = g_hash_table_get_values_as_ptr_array;

    /// Inserts a new key and value into a `glib.HashTable`.
    ///
    /// If the key already exists in the `glib.HashTable` its current
    /// value is replaced with the new value. If you supplied a
    /// `value_destroy_func` when creating the `glib.HashTable`, the old
    /// value is freed using that function. If you supplied a
    /// `key_destroy_func` when creating the `glib.HashTable`, the passed
    /// key is freed using that function.
    ///
    /// Starting from GLib 2.40, this function returns a boolean value to
    /// indicate whether the newly added value was already in the hash table
    /// or not.
    extern fn g_hash_table_insert(p_hash_table: *glib.HashTable, p_key: ?*anyopaque, p_value: ?*anyopaque) c_int;
    pub const insert = g_hash_table_insert;

    /// Looks up a key in a `glib.HashTable`. Note that this function cannot
    /// distinguish between a key that is not present and one which is present
    /// and has the value `NULL`. If you need this distinction, use
    /// `glib.hashTableLookupExtended`.
    extern fn g_hash_table_lookup(p_hash_table: *glib.HashTable, p_key: ?*const anyopaque) ?*anyopaque;
    pub const lookup = g_hash_table_lookup;

    /// Looks up a key in the `glib.HashTable`, returning the original key and the
    /// associated value and a `gboolean` which is `TRUE` if the key was found. This
    /// is useful if you need to free the memory allocated for the original key,
    /// for example before calling `glib.hashTableRemove`.
    ///
    /// You can actually pass `NULL` for `lookup_key` to test
    /// whether the `NULL` key exists, provided the hash and equal functions
    /// of `hash_table` are `NULL`-safe.
    extern fn g_hash_table_lookup_extended(p_hash_table: *glib.HashTable, p_lookup_key: ?*const anyopaque, p_orig_key: ?*anyopaque, p_value: ?*anyopaque) c_int;
    pub const lookupExtended = g_hash_table_lookup_extended;

    /// Creates a new `glib.HashTable` with a reference count of 1.
    ///
    /// Hash values returned by `hash_func` are used to determine where keys
    /// are stored within the `glib.HashTable` data structure. The `glib.directHash`,
    /// `glib.intHash`, `glib.int64Hash`, `glib.doubleHash` and `glib.strHash`
    /// functions are provided for some common types of keys.
    /// If `hash_func` is `NULL`, `glib.directHash` is used.
    ///
    /// `key_equal_func` is used when looking up keys in the `glib.HashTable`.
    /// The `glib.directEqual`, `glib.intEqual`, `glib.int64Equal`, `glib.doubleEqual`
    /// and `glib.strEqual` functions are provided for the most common types
    /// of keys. If `key_equal_func` is `NULL`, keys are compared directly in
    /// a similar fashion to `glib.directEqual`, but without the overhead of
    /// a function call. `key_equal_func` is called with the key from the hash table
    /// as its first parameter, and the user-provided key to check against as
    /// its second.
    extern fn g_hash_table_new(p_hash_func: glib.HashFunc, p_key_equal_func: glib.EqualFunc) *glib.HashTable;
    pub const new = g_hash_table_new;

    /// Creates a new `glib.HashTable` like `glib.HashTable.new` with a reference
    /// count of 1 and allows to specify functions to free the memory
    /// allocated for the key and value that get called when removing the
    /// entry from the `glib.HashTable`.
    ///
    /// Since version 2.42 it is permissible for destroy notify functions to
    /// recursively remove further items from the hash table. This is only
    /// permissible if the application still holds a reference to the hash table.
    /// This means that you may need to ensure that the hash table is empty by
    /// calling `glib.hashTableRemoveAll` before releasing the last reference using
    /// `glib.hashTableUnref`.
    extern fn g_hash_table_new_full(p_hash_func: glib.HashFunc, p_key_equal_func: glib.EqualFunc, p_key_destroy_func: ?glib.DestroyNotify, p_value_destroy_func: ?glib.DestroyNotify) *glib.HashTable;
    pub const newFull = g_hash_table_new_full;

    /// Creates a new `glib.HashTable` like `glib.HashTable.newFull` with a reference
    /// count of 1.
    ///
    /// It inherits the hash function, the key equal function, the key destroy function,
    /// as well as the value destroy function, from `other_hash_table`.
    ///
    /// The returned hash table will be empty; it will not contain the keys
    /// or values from `other_hash_table`.
    extern fn g_hash_table_new_similar(p_other_hash_table: *glib.HashTable) *glib.HashTable;
    pub const newSimilar = g_hash_table_new_similar;

    /// Atomically increments the reference count of `hash_table` by one.
    /// This function is MT-safe and may be called from any thread.
    extern fn g_hash_table_ref(p_hash_table: *glib.HashTable) *glib.HashTable;
    pub const ref = g_hash_table_ref;

    /// Removes a key and its associated value from a `glib.HashTable`.
    ///
    /// If the `glib.HashTable` was created using `glib.HashTable.newFull`, the
    /// key and value are freed using the supplied destroy functions, otherwise
    /// you have to make sure that any dynamically allocated values are freed
    /// yourself.
    extern fn g_hash_table_remove(p_hash_table: *glib.HashTable, p_key: ?*const anyopaque) c_int;
    pub const remove = g_hash_table_remove;

    /// Removes all keys and their associated values from a `glib.HashTable`.
    ///
    /// If the `glib.HashTable` was created using `glib.HashTable.newFull`,
    /// the keys and values are freed using the supplied destroy functions,
    /// otherwise you have to make sure that any dynamically allocated
    /// values are freed yourself.
    extern fn g_hash_table_remove_all(p_hash_table: *glib.HashTable) void;
    pub const removeAll = g_hash_table_remove_all;

    /// Inserts a new key and value into a `glib.HashTable` similar to
    /// `glib.hashTableInsert`. The difference is that if the key
    /// already exists in the `glib.HashTable`, it gets replaced by the
    /// new key. If you supplied a `value_destroy_func` when creating
    /// the `glib.HashTable`, the old value is freed using that function.
    /// If you supplied a `key_destroy_func` when creating the
    /// `glib.HashTable`, the old key is freed using that function.
    ///
    /// Starting from GLib 2.40, this function returns a boolean value to
    /// indicate whether the newly added value was already in the hash table
    /// or not.
    extern fn g_hash_table_replace(p_hash_table: *glib.HashTable, p_key: ?*anyopaque, p_value: ?*anyopaque) c_int;
    pub const replace = g_hash_table_replace;

    /// Returns the number of elements contained in the `glib.HashTable`.
    extern fn g_hash_table_size(p_hash_table: *glib.HashTable) c_uint;
    pub const size = g_hash_table_size;

    /// Removes a key and its associated value from a `glib.HashTable` without
    /// calling the key and value destroy functions.
    extern fn g_hash_table_steal(p_hash_table: *glib.HashTable, p_key: ?*const anyopaque) c_int;
    pub const steal = g_hash_table_steal;

    /// Removes all keys and their associated values from a `glib.HashTable`
    /// without calling the key and value destroy functions.
    extern fn g_hash_table_steal_all(p_hash_table: *glib.HashTable) void;
    pub const stealAll = g_hash_table_steal_all;

    /// Removes all keys and their associated values from a `glib.HashTable`
    /// without calling the key destroy functions, returning the keys
    /// as a `glib.PtrArray` with the free func set to the `hash_table` key
    /// destroy function.
    extern fn g_hash_table_steal_all_keys(p_hash_table: *glib.HashTable) *glib.PtrArray;
    pub const stealAllKeys = g_hash_table_steal_all_keys;

    /// Removes all keys and their associated values from a `glib.HashTable`
    /// without calling the value destroy functions, returning the values
    /// as a `glib.PtrArray` with the free func set to the `hash_table` value
    /// destroy function.
    extern fn g_hash_table_steal_all_values(p_hash_table: *glib.HashTable) *glib.PtrArray;
    pub const stealAllValues = g_hash_table_steal_all_values;

    /// Looks up a key in the `glib.HashTable`, stealing the original key and the
    /// associated value and returning `TRUE` if the key was found. If the key was
    /// not found, `FALSE` is returned.
    ///
    /// If found, the stolen key and value are removed from the hash table without
    /// calling the key and value destroy functions, and ownership is transferred to
    /// the caller of this method, as with `glib.hashTableSteal`. That is the case
    /// regardless whether `stolen_key` or `stolen_value` output parameters are
    /// requested.
    ///
    /// You can pass `NULL` for `lookup_key`, provided the hash and equal functions
    /// of `hash_table` are `NULL`-safe.
    ///
    /// The dictionary implementation optimizes for having all values identical to
    /// their keys, for example by using `glib.hashTableAdd`. Before 2.82, when
    /// stealing both the key and the value from such a dictionary, the value was
    /// `NULL`. Since 2.82, the returned value and key will be the same.
    extern fn g_hash_table_steal_extended(p_hash_table: *glib.HashTable, p_lookup_key: ?*const anyopaque, p_stolen_key: ?*anyopaque, p_stolen_value: ?*anyopaque) c_int;
    pub const stealExtended = g_hash_table_steal_extended;

    /// Atomically decrements the reference count of `hash_table` by one.
    /// If the reference count drops to 0, all keys and values will be
    /// destroyed, and all memory allocated by the hash table is released.
    /// This function is MT-safe and may be called from any thread.
    extern fn g_hash_table_unref(p_hash_table: *glib.HashTable) void;
    pub const unref = g_hash_table_unref;

    extern fn g_hash_table_get_type() usize;
    pub const getGObjectType = g_hash_table_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A GHashTableIter structure represents an iterator that can be used
/// to iterate over the elements of a `glib.HashTable`. GHashTableIter
/// structures are typically allocated on the stack and then initialized
/// with `glib.HashTableIter.init`.
///
/// The iteration order of a `glib.HashTableIter` over the keys/values in a hash
/// table is not defined.
pub const HashTableIter = extern struct {
    f_dummy1: ?*anyopaque,
    f_dummy2: ?*anyopaque,
    f_dummy3: ?*anyopaque,
    f_dummy4: c_int,
    f_dummy5: c_int,
    f_dummy6: ?*anyopaque,

    /// Returns the `glib.HashTable` associated with `iter`.
    extern fn g_hash_table_iter_get_hash_table(p_iter: *HashTableIter) *glib.HashTable;
    pub const getHashTable = g_hash_table_iter_get_hash_table;

    /// Initializes a key/value pair iterator and associates it with
    /// `hash_table`. Modifying the hash table after calling this function
    /// invalidates the returned iterator.
    ///
    /// The iteration order of a `glib.HashTableIter` over the keys/values in a hash
    /// table is not defined.
    ///
    /// ```
    /// GHashTableIter iter;
    /// gpointer key, value;
    ///
    /// g_hash_table_iter_init (&iter, hash_table);
    /// while (g_hash_table_iter_next (&iter, &key, &value))
    ///   {
    ///     // do something with key and value
    ///   }
    /// ```
    extern fn g_hash_table_iter_init(p_iter: *HashTableIter, p_hash_table: *glib.HashTable) void;
    pub const init = g_hash_table_iter_init;

    /// Advances `iter` and retrieves the key and/or value that are now
    /// pointed to as a result of this advancement. If `FALSE` is returned,
    /// `key` and `value` are not set, and the iterator becomes invalid.
    extern fn g_hash_table_iter_next(p_iter: *HashTableIter, p_key: ?*anyopaque, p_value: ?*anyopaque) c_int;
    pub const next = g_hash_table_iter_next;

    /// Removes the key/value pair currently pointed to by the iterator
    /// from its associated `glib.HashTable`. Can only be called after
    /// `glib.HashTableIter.next` returned `TRUE`, and cannot be called
    /// more than once for the same key/value pair.
    ///
    /// If the `glib.HashTable` was created using `glib.HashTable.newFull`,
    /// the key and value are freed using the supplied destroy functions,
    /// otherwise you have to make sure that any dynamically allocated
    /// values are freed yourself.
    ///
    /// It is safe to continue iterating the `glib.HashTable` afterward:
    /// ```
    /// while (g_hash_table_iter_next (&iter, &key, &value))
    ///   {
    ///     if (condition)
    ///       g_hash_table_iter_remove (&iter);
    ///   }
    /// ```
    extern fn g_hash_table_iter_remove(p_iter: *HashTableIter) void;
    pub const remove = g_hash_table_iter_remove;

    /// Replaces the value currently pointed to by the iterator
    /// from its associated `glib.HashTable`. Can only be called after
    /// `glib.HashTableIter.next` returned `TRUE`.
    ///
    /// If you supplied a `value_destroy_func` when creating the
    /// `glib.HashTable`, the old value is freed using that function.
    extern fn g_hash_table_iter_replace(p_iter: *HashTableIter, p_value: ?*anyopaque) void;
    pub const replace = g_hash_table_iter_replace;

    /// Removes the key/value pair currently pointed to by the
    /// iterator from its associated `glib.HashTable`, without calling
    /// the key and value destroy functions. Can only be called
    /// after `glib.HashTableIter.next` returned `TRUE`, and cannot
    /// be called more than once for the same key/value pair.
    extern fn g_hash_table_iter_steal(p_iter: *HashTableIter) void;
    pub const steal = g_hash_table_iter_steal;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// HMACs should be used when producing a cookie or hash based on data
/// and a key. Simple mechanisms for using SHA1 and other algorithms to
/// digest a key and data together are vulnerable to various security
/// issues.
/// [HMAC](http://en.wikipedia.org/wiki/HMAC)
/// uses algorithms like SHA1 in a secure way to produce a digest of a
/// key and data.
///
/// Both the key and data are arbitrary byte arrays of bytes or characters.
///
/// Support for HMAC Digests has been added in GLib 2.30, and support for SHA-512
/// in GLib 2.42. Support for SHA-384 was added in GLib 2.52.
///
/// To create a new `GHmac`, use `glib.Hmac.new`. To free a `GHmac`, use
/// `glib.Hmac.unref`.
pub const Hmac = opaque {
    /// Creates a new `glib.Hmac`, using the digest algorithm `digest_type`.
    /// If the `digest_type` is not known, `NULL` is returned.
    /// A `glib.Hmac` can be used to compute the HMAC of a key and an
    /// arbitrary binary blob, using different hashing algorithms.
    ///
    /// A `glib.Hmac` works by feeding a binary blob through `glib.Hmac.update`
    /// until the data is complete; the digest can then be extracted
    /// using `glib.Hmac.getString`, which will return the checksum as a
    /// hexadecimal string; or `glib.Hmac.getDigest`, which will return a
    /// array of raw bytes. Once either `glib.Hmac.getString` or
    /// `glib.Hmac.getDigest` have been called on a `glib.Hmac`, the HMAC
    /// will be closed and it won't be possible to call `glib.Hmac.update`
    /// on it anymore.
    ///
    /// Support for digests of type `G_CHECKSUM_SHA512` has been added in GLib 2.42.
    /// Support for `G_CHECKSUM_SHA384` was added in GLib 2.52.
    extern fn g_hmac_new(p_digest_type: glib.ChecksumType, p_key: [*]const u8, p_key_len: usize) ?*glib.Hmac;
    pub const new = g_hmac_new;

    /// Copies a `glib.Hmac`. If `hmac` has been closed, by calling
    /// `glib.Hmac.getString` or `glib.Hmac.getDigest`, the copied
    /// HMAC will be closed as well.
    extern fn g_hmac_copy(p_hmac: *const Hmac) *glib.Hmac;
    pub const copy = g_hmac_copy;

    /// Gets the digest from `checksum` as a raw binary array and places it
    /// into `buffer`. The size of the digest depends on the type of checksum.
    ///
    /// Once this function has been called, the `glib.Hmac` is closed and can
    /// no longer be updated with `glib.Checksum.update`.
    extern fn g_hmac_get_digest(p_hmac: *Hmac, p_buffer: [*]u8, p_digest_len: *usize) void;
    pub const getDigest = g_hmac_get_digest;

    /// Gets the HMAC as a hexadecimal string.
    ///
    /// Once this function has been called the `glib.Hmac` can no longer be
    /// updated with `glib.Hmac.update`.
    ///
    /// The hexadecimal characters will be lower case.
    extern fn g_hmac_get_string(p_hmac: *Hmac) [*:0]const u8;
    pub const getString = g_hmac_get_string;

    /// Atomically increments the reference count of `hmac` by one.
    ///
    /// This function is MT-safe and may be called from any thread.
    extern fn g_hmac_ref(p_hmac: *Hmac) *glib.Hmac;
    pub const ref = g_hmac_ref;

    /// Atomically decrements the reference count of `hmac` by one.
    ///
    /// If the reference count drops to 0, all keys and values will be
    /// destroyed, and all memory allocated by the hash table is released.
    /// This function is MT-safe and may be called from any thread.
    /// Frees the memory allocated for `hmac`.
    extern fn g_hmac_unref(p_hmac: *Hmac) void;
    pub const unref = g_hmac_unref;

    /// Feeds `data` into an existing `glib.Hmac`.
    ///
    /// The HMAC must still be open, that is `glib.Hmac.getString` or
    /// `glib.Hmac.getDigest` must not have been called on `hmac`.
    extern fn g_hmac_update(p_hmac: *Hmac, p_data: [*]const u8, p_length: isize) void;
    pub const update = g_hmac_update;

    extern fn g_hmac_get_type() usize;
    pub const getGObjectType = g_hmac_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.Hook` struct represents a single hook function in a `glib.HookList`.
pub const Hook = extern struct {
    /// data which is passed to func when this hook is invoked
    f_data: ?*anyopaque,
    /// pointer to the next hook in the list
    f_next: ?*glib.Hook,
    /// pointer to the previous hook in the list
    f_prev: ?*glib.Hook,
    /// the reference count of this hook
    f_ref_count: c_uint,
    /// the id of this hook, which is unique within its list
    f_hook_id: c_ulong,
    /// flags which are set for this hook. See `glib.HookFlagMask` for
    ///     predefined flags
    f_flags: c_uint,
    /// the function to call when this hook is invoked. The possible
    ///     signatures for this function are `glib.HookFunc` and `glib.HookCheckFunc`
    f_func: ?*anyopaque,
    /// the default `finalize_hook` function of a `glib.HookList` calls
    ///     this member of the hook that is being finalized
    f_destroy: ?glib.DestroyNotify,

    /// Allocates space for a `glib.Hook` and initializes it.
    extern fn g_hook_alloc(p_hook_list: *glib.HookList) *glib.Hook;
    pub const alloc = g_hook_alloc;

    /// Destroys a `glib.Hook`, given its ID.
    extern fn g_hook_destroy(p_hook_list: *glib.HookList, p_hook_id: c_ulong) c_int;
    pub const destroy = g_hook_destroy;

    /// Removes one `glib.Hook` from a `glib.HookList`, marking it
    /// inactive and calling `glib.hookUnref` on it.
    extern fn g_hook_destroy_link(p_hook_list: *glib.HookList, p_hook: *glib.Hook) void;
    pub const destroyLink = g_hook_destroy_link;

    /// Finds a `glib.Hook` in a `glib.HookList` using the given function to
    /// test for a match.
    extern fn g_hook_find(p_hook_list: *glib.HookList, p_need_valids: c_int, p_func: glib.HookFindFunc, p_data: ?*anyopaque) *glib.Hook;
    pub const find = g_hook_find;

    /// Finds a `glib.Hook` in a `glib.HookList` with the given data.
    extern fn g_hook_find_data(p_hook_list: *glib.HookList, p_need_valids: c_int, p_data: ?*anyopaque) *glib.Hook;
    pub const findData = g_hook_find_data;

    /// Finds a `glib.Hook` in a `glib.HookList` with the given function.
    extern fn g_hook_find_func(p_hook_list: *glib.HookList, p_need_valids: c_int, p_func: ?*anyopaque) *glib.Hook;
    pub const findFunc = g_hook_find_func;

    /// Finds a `glib.Hook` in a `glib.HookList` with the given function and data.
    extern fn g_hook_find_func_data(p_hook_list: *glib.HookList, p_need_valids: c_int, p_func: *anyopaque, p_data: ?*anyopaque) *glib.Hook;
    pub const findFuncData = g_hook_find_func_data;

    /// Returns the first `glib.Hook` in a `glib.HookList` which has not been destroyed.
    /// The reference count for the `glib.Hook` is incremented, so you must call
    /// `glib.hookUnref` to restore it when no longer needed. (Or call
    /// `glib.Hook.nextValid` if you are stepping through the `glib.HookList`.)
    extern fn g_hook_first_valid(p_hook_list: *glib.HookList, p_may_be_in_call: c_int) *glib.Hook;
    pub const firstValid = g_hook_first_valid;

    /// Calls the `glib.HookList` `finalize_hook` function if it exists,
    /// and frees the memory allocated for the `glib.Hook`.
    extern fn g_hook_free(p_hook_list: *glib.HookList, p_hook: *glib.Hook) void;
    pub const free = g_hook_free;

    /// Returns the `glib.Hook` with the given id, or `NULL` if it is not found.
    extern fn g_hook_get(p_hook_list: *glib.HookList, p_hook_id: c_ulong) *glib.Hook;
    pub const get = g_hook_get;

    /// Inserts a `glib.Hook` into a `glib.HookList`, before a given `glib.Hook`.
    extern fn g_hook_insert_before(p_hook_list: *glib.HookList, p_sibling: ?*glib.Hook, p_hook: *glib.Hook) void;
    pub const insertBefore = g_hook_insert_before;

    /// Inserts a `glib.Hook` into a `glib.HookList`, sorted by the given function.
    extern fn g_hook_insert_sorted(p_hook_list: *glib.HookList, p_hook: *glib.Hook, p_func: glib.HookCompareFunc) void;
    pub const insertSorted = g_hook_insert_sorted;

    /// Returns the next `glib.Hook` in a `glib.HookList` which has not been destroyed.
    /// The reference count for the `glib.Hook` is incremented, so you must call
    /// `glib.hookUnref` to restore it when no longer needed. (Or continue to call
    /// `glib.Hook.nextValid` until `NULL` is returned.)
    extern fn g_hook_next_valid(p_hook_list: *glib.HookList, p_hook: *glib.Hook, p_may_be_in_call: c_int) *glib.Hook;
    pub const nextValid = g_hook_next_valid;

    /// Prepends a `glib.Hook` on the start of a `glib.HookList`.
    extern fn g_hook_prepend(p_hook_list: *glib.HookList, p_hook: *glib.Hook) void;
    pub const prepend = g_hook_prepend;

    /// Increments the reference count for a `glib.Hook`.
    extern fn g_hook_ref(p_hook_list: *glib.HookList, p_hook: *glib.Hook) *glib.Hook;
    pub const ref = g_hook_ref;

    /// Decrements the reference count of a `glib.Hook`.
    /// If the reference count falls to 0, the `glib.Hook` is removed
    /// from the `glib.HookList` and `glib.hookFree` is called to free it.
    extern fn g_hook_unref(p_hook_list: *glib.HookList, p_hook: *glib.Hook) void;
    pub const unref = g_hook_unref;

    /// Compares the ids of two `glib.Hook` elements, returning a negative value
    /// if the second id is greater than the first.
    extern fn g_hook_compare_ids(p_new_hook: *Hook, p_sibling: *glib.Hook) c_int;
    pub const compareIds = g_hook_compare_ids;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.HookList` struct represents a list of hook functions.
pub const HookList = extern struct {
    /// the next free `glib.Hook` id
    f_seq_id: c_ulong,
    bitfields0: packed struct(c_uint) {
        /// the size of the `glib.HookList` elements, in bytes
        f_hook_size: u16,
        /// 1 if the `glib.HookList` has been initialized
        f_is_setup: u1,
        _: u15,
    },
    /// the first `glib.Hook` element in the list
    f_hooks: ?*glib.Hook,
    /// unused
    f_dummy3: ?*anyopaque,
    /// the function to call to finalize a `glib.Hook` element.
    ///     The default behaviour is to call the hooks `destroy` function
    f_finalize_hook: ?glib.HookFinalizeFunc,
    /// unused
    f_dummy: [2]*anyopaque,

    /// Removes all the `glib.Hook` elements from a `glib.HookList`.
    extern fn g_hook_list_clear(p_hook_list: *HookList) void;
    pub const clear = g_hook_list_clear;

    /// Initializes a `glib.HookList`.
    /// This must be called before the `glib.HookList` is used.
    extern fn g_hook_list_init(p_hook_list: *HookList, p_hook_size: c_uint) void;
    pub const init = g_hook_list_init;

    /// Calls all of the `glib.Hook` functions in a `glib.HookList`.
    extern fn g_hook_list_invoke(p_hook_list: *HookList, p_may_recurse: c_int) void;
    pub const invoke = g_hook_list_invoke;

    /// Calls all of the `glib.Hook` functions in a `glib.HookList`.
    /// Any function which returns `FALSE` is removed from the `glib.HookList`.
    extern fn g_hook_list_invoke_check(p_hook_list: *HookList, p_may_recurse: c_int) void;
    pub const invokeCheck = g_hook_list_invoke_check;

    /// Calls a function on each valid `glib.Hook`.
    extern fn g_hook_list_marshal(p_hook_list: *HookList, p_may_recurse: c_int, p_marshaller: glib.HookMarshaller, p_marshal_data: ?*anyopaque) void;
    pub const marshal = g_hook_list_marshal;

    /// Calls a function on each valid `glib.Hook` and destroys it if the
    /// function returns `FALSE`.
    extern fn g_hook_list_marshal_check(p_hook_list: *HookList, p_may_recurse: c_int, p_marshaller: glib.HookCheckMarshaller, p_marshal_data: ?*anyopaque) void;
    pub const marshalCheck = g_hook_list_marshal_check;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The GIConv struct wraps an `iconv` conversion descriptor. It contains
/// private data and should only be accessed using the following functions.
pub const IConv = *opaque {
    /// Same as the standard UNIX routine `iconv_open`, but
    /// may be implemented via libiconv on UNIX flavors that lack
    /// a native implementation.
    ///
    /// GLib provides `glib.convert` and `glib.localeToUtf8` which are likely
    /// more convenient than the raw iconv wrappers.
    extern fn g_iconv_open(p_to_codeset: [*:0]const u8, p_from_codeset: [*:0]const u8) glib.IConv;
    pub const open = g_iconv_open;

    /// Same as the standard UNIX routine `iconv_close`, but
    /// may be implemented via libiconv on UNIX flavors that lack
    /// a native implementation. Should be called to clean up
    /// the conversion descriptor from `glib.iconvOpen` when
    /// you are done converting things.
    ///
    /// GLib provides `glib.convert` and `glib.localeToUtf8` which are likely
    /// more convenient than the raw iconv wrappers.
    extern fn g_iconv_close(p_converter: IConv) c_int;
    pub const close = g_iconv_close;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `GIOChannel` data type aims to provide a portable method for
/// using file descriptors, pipes, and sockets, and integrating them
/// into the main event loop (see `glib.MainContext`). Currently,
/// full support is available on UNIX platforms; support for Windows
/// is only partially complete.
///
/// To create a new `GIOChannel` on UNIX systems use
/// `glib.IOChannel.unixNew`. This works for plain file descriptors,
/// pipes and sockets. Alternatively, a channel can be created for a
/// file in a system independent manner using `glib.IOChannel.newFile`.
///
/// Once a `GIOChannel` has been created, it can be used in a generic
/// manner with the functions `glib.IOChannel.readChars`,
/// `glib.IOChannel.writeChars`, `glib.IOChannel.seekPosition`,
/// and `glib.IOChannel.shutdown`.
///
/// To add a `GIOChannel` to the main event loop, use `glib.ioAddWatch` or
/// `glib.ioAddWatchFull`. Here you specify which events you are
/// interested in on the `GIOChannel`, and provide a function to be called
/// whenever these events occur.
///
/// `GIOChannel` instances are created with an initial reference count of 1.
/// `glib.IOChannel.ref` and `glib.IOChannel.unref` can be used to
/// increment or decrement the reference count respectively. When the
/// reference count falls to 0, the `GIOChannel` is freed. (Though it
/// isn’t closed automatically, unless it was created using
/// `glib.IOChannel.newFile`.) Using `glib.ioAddWatch` or
/// `glib.ioAddWatchFull` increments a channel’s reference count.
///
/// The new functions `glib.IOChannel.readChars`,
/// `glib.IOChannel.readLine`, `glib.IOChannel.readLineString`,
/// `glib.IOChannel.readToEnd`, `glib.IOChannel.writeChars`,
/// `glib.IOChannel.seekPosition`, and `glib.IOChannel.flush`
/// should not be mixed with the deprecated functions
/// `glib.IOChannel.read`, `glib.IOChannel.write`, and
/// `glib.IOChannel.seek` on the same channel.
pub const IOChannel = extern struct {
    f_ref_count: c_int,
    f_funcs: ?*glib.IOFuncs,
    f_encoding: ?[*:0]u8,
    f_read_cd: ?glib.IConv,
    f_write_cd: ?glib.IConv,
    f_line_term: ?[*:0]u8,
    f_line_term_len: c_uint,
    f_buf_size: usize,
    f_read_buf: ?*glib.String,
    f_encoded_read_buf: ?*glib.String,
    f_write_buf: ?*glib.String,
    f_partial_write_buf: [6]u8,
    bitfields0: packed struct(c_uint) {
        f_use_buffer: u1,
        f_do_encode: u1,
        f_close_on_unref: u1,
        f_is_readable: u1,
        f_is_writeable: u1,
        f_is_seekable: u1,
        _: u26,
    },
    f_reserved1: ?*anyopaque,
    f_reserved2: ?*anyopaque,

    /// Converts an `errno` error number to a `glib.IOChannelError`.
    extern fn g_io_channel_error_from_errno(p_en: c_int) glib.IOChannelError;
    pub const errorFromErrno = g_io_channel_error_from_errno;

    extern fn g_io_channel_error_quark() glib.Quark;
    pub const errorQuark = g_io_channel_error_quark;

    /// Open a file `filename` as a `glib.IOChannel` using mode `mode`. This
    /// channel will be closed when the last reference to it is dropped,
    /// so there is no need to call `glib.IOChannel.close` (though doing
    /// so will not cause problems, as long as no attempt is made to
    /// access the channel after it is closed).
    extern fn g_io_channel_new_file(p_filename: [*:0]const u8, p_mode: [*:0]const u8, p_error: ?*?*glib.Error) ?*glib.IOChannel;
    pub const newFile = g_io_channel_new_file;

    /// Creates a new `glib.IOChannel` given a file descriptor. On UNIX systems
    /// this works for plain files, pipes, and sockets.
    ///
    /// The returned `glib.IOChannel` has a reference count of 1.
    ///
    /// The default encoding for `glib.IOChannel` is UTF-8. If your application
    /// is reading output from a command using via pipe, you may need to set
    /// the encoding to the encoding of the current locale (see
    /// `glib.getCharset`) with the `glib.IOChannel.setEncoding` function.
    /// By default, the fd passed will not be closed when the final reference
    /// to the `glib.IOChannel` data structure is dropped.
    ///
    /// If you want to read raw binary data without interpretation, then
    /// call the `glib.IOChannel.setEncoding` function with `NULL` for the
    /// encoding argument.
    ///
    /// This function is available in GLib on Windows, too, but you should
    /// avoid using it on Windows. The domain of file descriptors and
    /// sockets overlap. There is no way for GLib to know which one you mean
    /// in case the argument you pass to this function happens to be both a
    /// valid file descriptor and socket. If that happens a warning is
    /// issued, and GLib assumes that it is the file descriptor you mean.
    extern fn g_io_channel_unix_new(p_fd: c_int) *glib.IOChannel;
    pub const unixNew = g_io_channel_unix_new;

    /// Close an IO channel. Any pending data to be written will be
    /// flushed, ignoring errors. The channel will not be freed until the
    /// last reference is dropped using `glib.IOChannel.unref`.
    extern fn g_io_channel_close(p_channel: *IOChannel) void;
    pub const close = g_io_channel_close;

    /// Flushes the write buffer for the GIOChannel.
    extern fn g_io_channel_flush(p_channel: *IOChannel, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const flush = g_io_channel_flush;

    /// This function returns a `glib.IOCondition` depending on whether there
    /// is data to be read/space to write data in the internal buffers in
    /// the `glib.IOChannel`. Only the flags `G_IO_IN` and `G_IO_OUT` may be set.
    extern fn g_io_channel_get_buffer_condition(p_channel: *IOChannel) glib.IOCondition;
    pub const getBufferCondition = g_io_channel_get_buffer_condition;

    /// Gets the buffer size.
    extern fn g_io_channel_get_buffer_size(p_channel: *IOChannel) usize;
    pub const getBufferSize = g_io_channel_get_buffer_size;

    /// Returns whether `channel` is buffered.
    extern fn g_io_channel_get_buffered(p_channel: *IOChannel) c_int;
    pub const getBuffered = g_io_channel_get_buffered;

    /// Returns whether the file/socket/whatever associated with `channel`
    /// will be closed when `channel` receives its final unref and is
    /// destroyed. The default value of this is `TRUE` for channels created
    /// by g_io_channel_new_file (), and `FALSE` for all other channels.
    extern fn g_io_channel_get_close_on_unref(p_channel: *IOChannel) c_int;
    pub const getCloseOnUnref = g_io_channel_get_close_on_unref;

    /// Gets the encoding for the input/output of the channel.
    /// The internal encoding is always UTF-8. The encoding `NULL`
    /// makes the channel safe for binary data.
    extern fn g_io_channel_get_encoding(p_channel: *IOChannel) [*:0]const u8;
    pub const getEncoding = g_io_channel_get_encoding;

    /// Gets the current flags for a `glib.IOChannel`, including read-only
    /// flags such as `G_IO_FLAG_IS_READABLE`.
    ///
    /// The values of the flags `G_IO_FLAG_IS_READABLE` and `G_IO_FLAG_IS_WRITABLE`
    /// are cached for internal use by the channel when it is created.
    /// If they should change at some later point (e.g. partial shutdown
    /// of a socket with the UNIX `shutdown` function), the user
    /// should immediately call `glib.IOChannel.getFlags` to update
    /// the internal values of these flags.
    extern fn g_io_channel_get_flags(p_channel: *IOChannel) glib.IOFlags;
    pub const getFlags = g_io_channel_get_flags;

    /// This returns the string that `glib.IOChannel` uses to determine
    /// where in the file a line break occurs. A value of `NULL`
    /// indicates autodetection. Since 2.84, the return value is always
    /// nul-terminated.
    extern fn g_io_channel_get_line_term(p_channel: *IOChannel, p_length: ?*c_int) [*:0]const u8;
    pub const getLineTerm = g_io_channel_get_line_term;

    /// Initializes a `glib.IOChannel` struct.
    ///
    /// This is called by each of the above functions when creating a
    /// `glib.IOChannel`, and so is not often needed by the application
    /// programmer (unless you are creating a new type of `glib.IOChannel`).
    extern fn g_io_channel_init(p_channel: *IOChannel) void;
    pub const init = g_io_channel_init;

    /// Reads data from a `glib.IOChannel`.
    extern fn g_io_channel_read(p_channel: *IOChannel, p_buf: [*:0]u8, p_count: usize, p_bytes_read: *usize) glib.IOError;
    pub const read = g_io_channel_read;

    /// Replacement for `glib.IOChannel.read` with the new API.
    extern fn g_io_channel_read_chars(p_channel: *IOChannel, p_buf: *[*]u8, p_count: usize, p_bytes_read: ?*usize, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const readChars = g_io_channel_read_chars;

    /// Reads a line, including the terminating character(s),
    /// from a `glib.IOChannel` into a newly-allocated string.
    /// `str_return` will contain allocated memory if the return
    /// is `G_IO_STATUS_NORMAL`.
    extern fn g_io_channel_read_line(p_channel: *IOChannel, p_str_return: *[*:0]u8, p_length: ?*usize, p_terminator_pos: ?*usize, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const readLine = g_io_channel_read_line;

    /// Reads a line from a `glib.IOChannel`, using a `glib.String` as a buffer.
    extern fn g_io_channel_read_line_string(p_channel: *IOChannel, p_buffer: *glib.String, p_terminator_pos: ?*usize, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const readLineString = g_io_channel_read_line_string;

    /// Reads all the remaining data from the file.
    extern fn g_io_channel_read_to_end(p_channel: *IOChannel, p_str_return: *[*]u8, p_length: *usize, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const readToEnd = g_io_channel_read_to_end;

    /// Reads a Unicode character from `channel`.
    /// This function cannot be called on a channel with `NULL` encoding.
    extern fn g_io_channel_read_unichar(p_channel: *IOChannel, p_thechar: *u32, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const readUnichar = g_io_channel_read_unichar;

    /// Increments the reference count of a `glib.IOChannel`.
    extern fn g_io_channel_ref(p_channel: *IOChannel) *glib.IOChannel;
    pub const ref = g_io_channel_ref;

    /// Sets the current position in the `glib.IOChannel`, similar to the standard
    /// library function `fseek`.
    extern fn g_io_channel_seek(p_channel: *IOChannel, p_offset: i64, p_type: glib.SeekType) glib.IOError;
    pub const seek = g_io_channel_seek;

    /// Replacement for `glib.IOChannel.seek` with the new API.
    extern fn g_io_channel_seek_position(p_channel: *IOChannel, p_offset: i64, p_type: glib.SeekType, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const seekPosition = g_io_channel_seek_position;

    /// Sets the buffer size.
    extern fn g_io_channel_set_buffer_size(p_channel: *IOChannel, p_size: usize) void;
    pub const setBufferSize = g_io_channel_set_buffer_size;

    /// The buffering state can only be set if the channel's encoding
    /// is `NULL`. For any other encoding, the channel must be buffered.
    ///
    /// A buffered channel can only be set unbuffered if the channel's
    /// internal buffers have been flushed. Newly created channels or
    /// channels which have returned `G_IO_STATUS_EOF`
    /// not require such a flush. For write-only channels, a call to
    /// g_io_channel_flush () is sufficient. For all other channels,
    /// the buffers may be flushed by a call to g_io_channel_seek_position ().
    /// This includes the possibility of seeking with seek type `G_SEEK_CUR`
    /// and an offset of zero. Note that this means that socket-based
    /// channels cannot be set unbuffered once they have had data
    /// read from them.
    ///
    /// On unbuffered channels, it is safe to mix read and write
    /// calls from the new and old APIs, if this is necessary for
    /// maintaining old code.
    ///
    /// The default state of the channel is buffered.
    extern fn g_io_channel_set_buffered(p_channel: *IOChannel, p_buffered: c_int) void;
    pub const setBuffered = g_io_channel_set_buffered;

    /// Whether to close the channel on the final unref of the `glib.IOChannel`
    /// data structure. The default value of this is `TRUE` for channels
    /// created by g_io_channel_new_file (), and `FALSE` for all other channels.
    ///
    /// Setting this flag to `TRUE` for a channel you have already closed
    /// can cause problems when the final reference to the `glib.IOChannel` is dropped.
    extern fn g_io_channel_set_close_on_unref(p_channel: *IOChannel, p_do_close: c_int) void;
    pub const setCloseOnUnref = g_io_channel_set_close_on_unref;

    /// Sets the encoding for the input/output of the channel.
    /// The internal encoding is always UTF-8. The default encoding
    /// for the external file is UTF-8.
    ///
    /// The encoding `NULL` is safe to use with binary data.
    ///
    /// The encoding can only be set if one of the following conditions
    /// is true:
    ///
    /// - The channel was just created, and has not been written to or read from yet.
    ///
    /// - The channel is write-only.
    ///
    /// - The channel is a file, and the file pointer was just repositioned
    ///   by a call to `glib.IOChannel.seekPosition`. (This flushes all the
    ///   internal buffers.)
    ///
    /// - The current encoding is `NULL` or UTF-8.
    ///
    /// - One of the (new API) read functions has just returned `G_IO_STATUS_EOF`
    ///   (or, in the case of `glib.IOChannel.readToEnd`, `G_IO_STATUS_NORMAL`).
    ///
    /// -  One of the functions `glib.IOChannel.readChars` or
    ///    `glib.IOChannel.readUnichar` has returned `G_IO_STATUS_AGAIN` or
    ///    `G_IO_STATUS_ERROR`. This may be useful in the case of
    ///    `G_CONVERT_ERROR_ILLEGAL_SEQUENCE`.
    ///    Returning one of these statuses from `glib.IOChannel.readLine`,
    ///    `glib.IOChannel.readLineString`, or `glib.IOChannel.readToEnd`
    ///    does not guarantee that the encoding can be changed.
    ///
    /// Channels which do not meet one of the above conditions cannot call
    /// `glib.IOChannel.seekPosition` with an offset of `G_SEEK_CUR`, and, if
    /// they are "seekable", cannot call `glib.IOChannel.writeChars` after
    /// calling one of the API "read" functions.
    extern fn g_io_channel_set_encoding(p_channel: *IOChannel, p_encoding: ?[*:0]const u8, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const setEncoding = g_io_channel_set_encoding;

    /// Sets the (writeable) flags in `channel` to (`flags` & `G_IO_FLAG_SET_MASK`).
    extern fn g_io_channel_set_flags(p_channel: *IOChannel, p_flags: glib.IOFlags, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const setFlags = g_io_channel_set_flags;

    /// This sets the string that `glib.IOChannel` uses to determine
    /// where in the file a line break occurs.
    extern fn g_io_channel_set_line_term(p_channel: *IOChannel, p_line_term: ?[*:0]const u8, p_length: c_int) void;
    pub const setLineTerm = g_io_channel_set_line_term;

    /// Close an IO channel. Any pending data to be written will be
    /// flushed if `flush` is `TRUE`. The channel will not be freed until the
    /// last reference is dropped using `glib.IOChannel.unref`.
    extern fn g_io_channel_shutdown(p_channel: *IOChannel, p_flush: c_int, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const shutdown = g_io_channel_shutdown;

    /// Returns the file descriptor of the `glib.IOChannel`.
    ///
    /// On Windows this function returns the file descriptor or socket of
    /// the `glib.IOChannel`.
    extern fn g_io_channel_unix_get_fd(p_channel: *IOChannel) c_int;
    pub const unixGetFd = g_io_channel_unix_get_fd;

    /// Decrements the reference count of a `glib.IOChannel`.
    extern fn g_io_channel_unref(p_channel: *IOChannel) void;
    pub const unref = g_io_channel_unref;

    /// Writes data to a `glib.IOChannel`.
    extern fn g_io_channel_write(p_channel: *IOChannel, p_buf: [*:0]const u8, p_count: usize, p_bytes_written: *usize) glib.IOError;
    pub const write = g_io_channel_write;

    /// Replacement for `glib.IOChannel.write` with the new API.
    ///
    /// On seekable channels with encodings other than `NULL` or UTF-8, generic
    /// mixing of reading and writing is not allowed. A call to g_io_channel_write_chars ()
    /// may only be made on a channel from which data has been read in the
    /// cases described in the documentation for g_io_channel_set_encoding ().
    extern fn g_io_channel_write_chars(p_channel: *IOChannel, p_buf: [*]const u8, p_count: isize, p_bytes_written: *usize, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const writeChars = g_io_channel_write_chars;

    /// Writes a Unicode character to `channel`.
    /// This function cannot be called on a channel with `NULL` encoding.
    extern fn g_io_channel_write_unichar(p_channel: *IOChannel, p_thechar: u32, p_error: ?*?*glib.Error) glib.IOStatus;
    pub const writeUnichar = g_io_channel_write_unichar;

    extern fn g_io_channel_get_type() usize;
    pub const getGObjectType = g_io_channel_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A table of functions used to handle different types of `glib.IOChannel`
/// in a generic way.
pub const IOFuncs = extern struct {
    /// reads raw bytes from the channel.  This is called from
    ///           various functions such as `glib.IOChannel.readChars` to
    ///           read raw bytes from the channel.  Encoding and buffering
    ///           issues are dealt with at a higher level.
    f_io_read: ?*const fn (p_channel: *glib.IOChannel, p_buf: [*:0]u8, p_count: usize, p_bytes_read: *usize, p_error: ?*?*glib.Error) callconv(.c) glib.IOStatus,
    /// writes raw bytes to the channel.  This is called from
    ///            various functions such as `glib.IOChannel.writeChars` to
    ///            write raw bytes to the channel.  Encoding and buffering
    ///            issues are dealt with at a higher level.
    f_io_write: ?*const fn (p_channel: *glib.IOChannel, p_buf: [*:0]const u8, p_count: usize, p_bytes_written: *usize, p_error: ?*?*glib.Error) callconv(.c) glib.IOStatus,
    /// seeks the channel.  This is called from
    ///           `glib.IOChannel.seek` on channels that support it.
    f_io_seek: ?*const fn (p_channel: *glib.IOChannel, p_offset: i64, p_type: glib.SeekType, p_error: ?*?*glib.Error) callconv(.c) glib.IOStatus,
    /// closes the channel.  This is called from
    ///            `glib.IOChannel.close` after flushing the buffers.
    f_io_close: ?*const fn (p_channel: *glib.IOChannel, p_error: ?*?*glib.Error) callconv(.c) glib.IOStatus,
    /// creates a watch on the channel.  This call
    ///                   corresponds directly to `glib.ioCreateWatch`.
    f_io_create_watch: ?*const fn (p_channel: *glib.IOChannel, p_condition: glib.IOCondition) callconv(.c) *glib.Source,
    /// called from `glib.IOChannel.unref` when the channel needs to
    ///           be freed.  This function must free the memory associated
    ///           with the channel, including freeing the `glib.IOChannel`
    ///           structure itself.  The channel buffers have been flushed
    ///           and possibly `io_close` has been called by the time this
    ///           function is called.
    f_io_free: ?*const fn (p_channel: *glib.IOChannel) callconv(.c) void,
    /// sets the `glib.IOFlags` on the channel.  This is called
    ///                from `glib.IOChannel.setFlags` with all flags except
    ///                for `G_IO_FLAG_APPEND` and `G_IO_FLAG_NONBLOCK` masked
    ///                out.
    f_io_set_flags: ?*const fn (p_channel: *glib.IOChannel, p_flags: glib.IOFlags, p_error: ?*?*glib.Error) callconv(.c) glib.IOStatus,
    /// gets the `glib.IOFlags` for the channel.  This function
    ///                need only return the `G_IO_FLAG_APPEND` and
    ///                `G_IO_FLAG_NONBLOCK` flags; `glib.IOChannel.getFlags`
    ///                automatically adds the others as appropriate.
    f_io_get_flags: ?*const fn (p_channel: *glib.IOChannel) callconv(.c) glib.IOFlags,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GKeyFile` parses .ini-like config files.
///
/// `GKeyFile` lets you parse, edit or create files containing groups of
/// key-value pairs, which we call ‘key files’ for lack of a better name.
/// Several freedesktop.org specifications use key files. For example, the
/// [Desktop Entry Specification](https://specifications.freedesktop.org/desktop-entry-spec/latest/)
/// and the [Icon Theme Specification](https://specifications.freedesktop.org/icon-theme-spec/latest/).
///
/// The syntax of key files is described in detail in the
/// [Desktop Entry Specification](https://specifications.freedesktop.org/desktop-entry-spec/latest/),
/// here is a quick summary: Key files consists of groups of key-value pairs, interspersed
/// with comments.
///
/// ```txt
/// # this is just an example
/// # there can be comments before the first group
///
/// [First Group]
///
/// Name=Key File Example\tthis value shows\nescaping
///
/// # localized strings are stored in multiple key-value pairs
/// Welcome=Hello
/// Welcome[de]=Hallo
/// Welcome[fr_FR]=Bonjour
/// Welcome[it]=Ciao
///
/// [Another Group]
///
/// Numbers=2;20;-200;0
///
/// Booleans=true;false;true;true
/// ```
///
/// Lines beginning with a `#` and blank lines are considered comments.
///
/// Groups are started by a header line containing the group name enclosed
/// in `[` and `]`, and ended implicitly by the start of the next group or
/// the end of the file. Each key-value pair must be contained in a group.
///
/// Key-value pairs generally have the form `key=value`, with the exception
/// of localized strings, which have the form `key[locale]=value`, with a
/// locale identifier of the form `lang_COUNTRY`MODIFIER`` where `COUNTRY`
/// and `MODIFIER` are optional. As a special case, the locale `C` is associated
/// with the untranslated pair `key=value` (since GLib 2.84). Space before and
/// after the `=` character is ignored. Newline, tab, carriage return and
/// backslash characters in value are escaped as `\n`, `\t`, `\r`, and `\\\\`,
/// respectively. To preserve leading spaces in values, these can also be escaped
/// as `\s`.
///
/// Key files can store strings (possibly with localized variants), integers,
/// booleans and lists of these. Lists are separated by a separator character,
/// typically `;` or `,`. To use the list separator character in a value in
/// a list, it has to be escaped by prefixing it with a backslash.
///
/// This syntax is obviously inspired by the .ini files commonly met
/// on Windows, but there are some important differences:
///
/// - .ini files use the `;` character to begin comments,
///   key files use the `#` character.
///
/// - Key files do not allow for ungrouped keys meaning only
///   comments can precede the first group.
///
/// - Key files are always encoded in UTF-8.
///
/// - Key and Group names are case-sensitive. For example, a group called
///   `[GROUP]` is a different from `[group]`.
///
/// - .ini files don’t have a strongly typed boolean entry type,
///    they only have ``GetProfileInt``. In key files, only
///    `true` and `false` (in lower case) are allowed.
///
/// Note that in contrast to the
/// [Desktop Entry Specification](https://specifications.freedesktop.org/desktop-entry-spec/latest/),
/// groups in key files may contain the same key multiple times; the last entry wins.
/// Key files may also contain multiple groups with the same name; they are merged
/// together. Another difference is that keys and group names in key files are not
/// restricted to ASCII characters.
///
/// Here is an example of loading a key file and reading a value:
///
/// ```c
/// g_autoptr(GError) error = NULL;
/// g_autoptr(GKeyFile) key_file = g_key_file_new ();
///
/// if (!g_key_file_load_from_file (key_file, "key-file.ini", flags, &error))
///   {
///     if (!g_error_matches (error, G_FILE_ERROR, G_FILE_ERROR_NOENT))
///       g_warning ("Error loading key file: `s`", error->message);
///     return;
///   }
///
/// g_autofree gchar *val = g_key_file_get_string (key_file, "Group Name", "SomeKey", &error);
/// if (val == NULL &&
///     !g_error_matches (error, G_KEY_FILE_ERROR, G_KEY_FILE_ERROR_KEY_NOT_FOUND))
///   {
///     g_warning ("Error finding key in key file: `s`", error->message);
///     return;
///   }
/// else if (val == NULL)
///   {
///     // Fall back to a default value.
///     val = g_strdup ("default-value");
///   }
/// ```
///
/// Here is an example of creating and saving a key file:
///
/// ```c
/// g_autoptr(GKeyFile) key_file = g_key_file_new ();
/// const gchar *val = …;
/// g_autoptr(GError) error = NULL;
///
/// g_key_file_set_string (key_file, "Group Name", "SomeKey", val);
///
/// // Save as a file.
/// if (!g_key_file_save_to_file (key_file, "key-file.ini", &error))
///   {
///     g_warning ("Error saving key file: `s`", error->message);
///     return;
///   }
///
/// // Or store to a GBytes for use elsewhere.
/// gsize data_len;
/// g_autofree guint8 *data = (guint8 *) g_key_file_to_data (key_file, &data_len, &error);
/// if (data == NULL)
///   {
///     g_warning ("Error saving key file: `s`", error->message);
///     return;
///   }
/// g_autoptr(GBytes) bytes = g_bytes_new_take (g_steal_pointer (&data), data_len);
/// ```
pub const KeyFile = opaque {
    extern fn g_key_file_error_quark() glib.Quark;
    pub const errorQuark = g_key_file_error_quark;

    /// Creates a new empty `glib.KeyFile` object.
    ///
    /// Use `glib.KeyFile.loadFromFile`,
    /// `glib.KeyFile.loadFromData`, `glib.KeyFile.loadFromDirs` or
    /// `glib.KeyFile.loadFromDataDirs` to
    /// read an existing key file.
    extern fn g_key_file_new() *glib.KeyFile;
    pub const new = g_key_file_new;

    /// Clears all keys and groups from `key_file`, and decreases the
    /// reference count by 1.
    ///
    /// If the reference count reaches zero, frees the key file and all its allocated
    /// memory.
    extern fn g_key_file_free(p_key_file: *KeyFile) void;
    pub const free = g_key_file_free;

    /// Returns the value associated with `key` under `group_name` as a
    /// boolean.
    ///
    /// If `key` cannot be found then `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. Likewise, if the value associated with `key` cannot be interpreted
    /// as a boolean then `glib.@"KeyFileError.INVALID_VALUE"` is returned.
    extern fn g_key_file_get_boolean(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const getBoolean = g_key_file_get_boolean;

    /// Returns the values associated with `key` under `group_name` as
    /// booleans.
    ///
    /// If `key` cannot be found then `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. Likewise, if the values associated with `key` cannot be interpreted
    /// as booleans then `glib.@"KeyFileError.INVALID_VALUE"` is returned.
    extern fn g_key_file_get_boolean_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_length: *usize, p_error: ?*?*glib.Error) ?[*]c_int;
    pub const getBooleanList = g_key_file_get_boolean_list;

    /// Retrieves a comment above `key` from `group_name`.
    ///
    /// If `key` is `NULL` then `comment` will be read from above
    /// `group_name`. If both `key` and `group_name` are `NULL`, then
    /// `comment` will be read from above the first group in the file.
    ///
    /// Note that the returned string does not include the `#` comment markers,
    /// but does include any whitespace after them (on each line). It includes
    /// the line breaks between lines, but does not include the final line break.
    extern fn g_key_file_get_comment(p_key_file: *KeyFile, p_group_name: ?[*:0]const u8, p_key: ?[*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const getComment = g_key_file_get_comment;

    /// Returns the value associated with `key` under `group_name` as a double.
    ///
    /// If `key` cannot be found then `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. Likewise, if the value associated with `key` cannot be interpreted
    /// as a double then `glib.@"KeyFileError.INVALID_VALUE"` is returned.
    extern fn g_key_file_get_double(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_error: ?*?*glib.Error) f64;
    pub const getDouble = g_key_file_get_double;

    /// Returns the values associated with `key` under `group_name` as
    /// doubles.
    ///
    /// If `key` cannot be found then `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. Likewise, if the values associated with `key` cannot be interpreted
    /// as doubles then `glib.@"KeyFileError.INVALID_VALUE"` is returned.
    extern fn g_key_file_get_double_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_length: *usize, p_error: ?*?*glib.Error) ?[*]f64;
    pub const getDoubleList = g_key_file_get_double_list;

    /// Returns all groups in the key file loaded with `key_file`.
    ///
    /// The array of returned groups will be `NULL`-terminated, so
    /// `length` may optionally be `NULL`.
    extern fn g_key_file_get_groups(p_key_file: *KeyFile, p_length: ?*usize) [*:null]?[*:0]u8;
    pub const getGroups = g_key_file_get_groups;

    /// Returns the value associated with `key` under `group_name` as a signed
    /// 64-bit integer.
    ///
    /// This is similar to `glib.KeyFile.getInteger` but can return
    /// 64-bit results without truncation.
    extern fn g_key_file_get_int64(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_error: ?*?*glib.Error) i64;
    pub const getInt64 = g_key_file_get_int64;

    /// Returns the value associated with `key` under `group_name` as an
    /// integer.
    ///
    /// If `key` cannot be found then `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. Likewise, if the value associated with `key` cannot be interpreted
    /// as an integer, or is out of range for a `gint`, then
    /// `glib.@"KeyFileError.INVALID_VALUE"` is returned.
    extern fn g_key_file_get_integer(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const getInteger = g_key_file_get_integer;

    /// Returns the values associated with `key` under `group_name` as
    /// integers.
    ///
    /// If `key` cannot be found then `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. Likewise, if the values associated with `key` cannot be interpreted
    /// as integers, or are out of range for `gint`, then
    /// `glib.@"KeyFileError.INVALID_VALUE"` is returned.
    extern fn g_key_file_get_integer_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_length: *usize, p_error: ?*?*glib.Error) ?[*]c_int;
    pub const getIntegerList = g_key_file_get_integer_list;

    /// Returns all keys for the group name `group_name`.
    ///
    /// The array of returned keys will be `NULL`-terminated, so `length` may
    /// optionally be `NULL`. If the `group_name` cannot be found,
    /// `glib.@"KeyFileError.GROUP_NOT_FOUND"` is returned.
    extern fn g_key_file_get_keys(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*:null]?[*:0]u8;
    pub const getKeys = g_key_file_get_keys;

    /// Returns the actual locale which the result of
    /// `glib.KeyFile.getLocaleString` or
    /// `glib.KeyFile.getLocaleStringList` came from.
    ///
    /// If calling `glib.KeyFile.getLocaleString` or
    /// `glib.KeyFile.getLocaleStringList` with exactly the same `key_file`,
    /// `group_name`, `key` and `locale`, the result of those functions will
    /// have originally been tagged with the locale that is the result of
    /// this function.
    extern fn g_key_file_get_locale_for_key(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_locale: ?[*:0]const u8) ?[*:0]u8;
    pub const getLocaleForKey = g_key_file_get_locale_for_key;

    /// Returns the value associated with `key` under `group_name`
    /// translated in the given `locale` if available.
    ///
    /// If `locale` is `C` then the untranslated value is returned (since GLib 2.84).
    ///
    /// If `locale` is `NULL` then the current locale is assumed.
    ///
    /// If `locale` is to be non-`NULL`, or if the current locale will change over
    /// the lifetime of the `glib.KeyFile`, it must be loaded with
    /// `glib.@"KeyFileFlags.KEEP_TRANSLATIONS"` in order to load strings for all
    /// locales.
    ///
    /// If `key` cannot be found then `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. If the value associated
    /// with `key` cannot be interpreted or no suitable translation can
    /// be found then the untranslated value is returned.
    extern fn g_key_file_get_locale_string(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_locale: ?[*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const getLocaleString = g_key_file_get_locale_string;

    /// Returns the values associated with `key` under `group_name`
    /// translated in the given `locale` if available.
    ///
    /// If `locale` is `C` then the untranslated value is returned (since GLib 2.84).
    ///
    /// If `locale` is `NULL` then the current locale is assumed.
    ///
    /// If `locale` is to be non-`NULL`, or if the current locale will change over
    /// the lifetime of the `glib.KeyFile`, it must be loaded with
    /// `glib.@"KeyFileFlags.KEEP_TRANSLATIONS"` in order to load strings for all
    /// locales.
    ///
    /// If `key` cannot be found then `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. If the values associated
    /// with `key` cannot be interpreted or no suitable translations
    /// can be found then the untranslated values are returned. The
    /// returned array is `NULL`-terminated, so `length` may optionally
    /// be `NULL`.
    extern fn g_key_file_get_locale_string_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_locale: ?[*:0]const u8, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*:null]?[*:0]u8;
    pub const getLocaleStringList = g_key_file_get_locale_string_list;

    /// Returns the name of the start group of the file.
    extern fn g_key_file_get_start_group(p_key_file: *KeyFile) ?[*:0]u8;
    pub const getStartGroup = g_key_file_get_start_group;

    /// Returns the string value associated with `key` under `group_name`.
    ///
    /// Unlike `glib.KeyFile.getValue`, this function handles escape
    /// sequences like `\s`.
    ///
    /// If the key cannot be found, `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. If the `group_name` cannot be found,
    /// `glib.@"KeyFileError.GROUP_NOT_FOUND"` is returned.
    extern fn g_key_file_get_string(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const getString = g_key_file_get_string;

    /// Returns the values associated with `key` under `group_name`.
    ///
    /// If the key cannot be found, `glib.@"KeyFileError.KEY_NOT_FOUND"` is
    /// returned. If the `group_name` cannot be found,
    /// `glib.@"KeyFileError.GROUP_NOT_FOUND"` is returned.
    extern fn g_key_file_get_string_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*:null]?[*:0]u8;
    pub const getStringList = g_key_file_get_string_list;

    /// Returns the value associated with `key` under `group_name` as an unsigned
    /// 64-bit integer.
    ///
    /// This is similar to `glib.KeyFile.getInteger` but can return
    /// large positive results without truncation.
    extern fn g_key_file_get_uint64(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_error: ?*?*glib.Error) u64;
    pub const getUint64 = g_key_file_get_uint64;

    /// Returns the raw value associated with `key` under `group_name`.
    ///
    /// Use `glib.KeyFile.getString` to retrieve an unescaped UTF-8 string.
    ///
    /// If the key cannot be found, `glib.@"KeyFileError.KEY_NOT_FOUND"`
    /// is returned.  If the `group_name` cannot be found,
    /// `glib.@"KeyFileError.GROUP_NOT_FOUND"` is returned.
    extern fn g_key_file_get_value(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const getValue = g_key_file_get_value;

    /// Looks whether the key file has the group `group_name`.
    extern fn g_key_file_has_group(p_key_file: *KeyFile, p_group_name: [*:0]const u8) c_int;
    pub const hasGroup = g_key_file_has_group;

    /// Looks whether the key file has the key `key` in the group
    /// `group_name`.
    ///
    /// Note that this function does not follow the rules for `glib.Error`
    /// strictly;
    /// the return value both carries meaning and signals an error.  To use
    /// this function, you must pass a `glib.Error` pointer in `error`, and
    /// check whether it is not `NULL` to see if an error occurred.
    ///
    /// Language bindings should use `glib.KeyFile.getValue` to test whether
    /// a key exists.
    extern fn g_key_file_has_key(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const hasKey = g_key_file_has_key;

    /// Loads a key file from the data in `bytes` into an empty `glib.KeyFile`
    /// structure.
    ///
    /// If the object cannot be created then a `glib.KeyFileError` is returned.
    extern fn g_key_file_load_from_bytes(p_key_file: *KeyFile, p_bytes: *glib.Bytes, p_flags: glib.KeyFileFlags, p_error: ?*?*glib.Error) c_int;
    pub const loadFromBytes = g_key_file_load_from_bytes;

    /// Loads a key file from memory into an empty `glib.KeyFile` structure.
    ///
    /// If the object cannot be created then a [error`GLib`.KeyFileError is returned.
    extern fn g_key_file_load_from_data(p_key_file: *KeyFile, p_data: [*:0]const u8, p_length: usize, p_flags: glib.KeyFileFlags, p_error: ?*?*glib.Error) c_int;
    pub const loadFromData = g_key_file_load_from_data;

    /// Looks for a key file named `file` in the paths returned from
    /// `glib.getUserDataDir` and `glib.getSystemDataDirs`.
    ///
    /// The search algorithm from `glib.KeyFile.loadFromDirs` is used. If
    /// `file` is found, it’s loaded into `key_file` and its full path is returned in
    /// `full_path`.
    ///
    /// If the file could not be loaded then either a `glib.FileError` or
    /// `glib.KeyFileError` is returned.
    extern fn g_key_file_load_from_data_dirs(p_key_file: *KeyFile, p_file: [*:0]const u8, p_full_path: ?*[*:0]u8, p_flags: glib.KeyFileFlags, p_error: ?*?*glib.Error) c_int;
    pub const loadFromDataDirs = g_key_file_load_from_data_dirs;

    /// Looks for a key file named `file` in the paths specified in `search_dirs`,
    /// loads the file into `key_file` and returns the file’s full path in `full_path`.
    ///
    /// `search_dirs` are checked in the order listed in the array, with the highest
    /// priority directory listed first. Within each directory, `file` is looked for.
    /// If it’s not found, `-` characters in `file` are progressively replaced with
    /// directory separators to search subdirectories of the search directory. If the
    /// file has not been found after all `-` characters have been replaced, the next
    /// search directory in `search_dirs` is checked.
    ///
    /// If the file could not be found in any of the `search_dirs`,
    /// `glib.@"KeyFileError.NOT_FOUND"` is returned. If
    /// the file is found but the OS returns an error when opening or reading the
    /// file, a `glib.FileError` is returned. If there is a problem parsing the
    /// file, a `glib.KeyFileError` is returned.
    extern fn g_key_file_load_from_dirs(p_key_file: *KeyFile, p_file: [*:0]const u8, p_search_dirs: [*][*:0]const u8, p_full_path: ?*[*:0]u8, p_flags: glib.KeyFileFlags, p_error: ?*?*glib.Error) c_int;
    pub const loadFromDirs = g_key_file_load_from_dirs;

    /// Loads a key file into an empty `glib.KeyFile` structure.
    ///
    /// If the OS returns an error when opening or reading the file, a
    /// `glib.FileError` is returned. If there is a problem parsing the file,
    /// a `glib.KeyFileError` is returned.
    ///
    /// This function will never return a `glib.@"KeyFileError.NOT_FOUND"`
    /// error. If the `file` is not found, `glib.@"FileError.NOENT"` is returned.
    extern fn g_key_file_load_from_file(p_key_file: *KeyFile, p_file: [*:0]const u8, p_flags: glib.KeyFileFlags, p_error: ?*?*glib.Error) c_int;
    pub const loadFromFile = g_key_file_load_from_file;

    /// Increases the reference count of `key_file`.
    extern fn g_key_file_ref(p_key_file: *KeyFile) *glib.KeyFile;
    pub const ref = g_key_file_ref;

    /// Removes a comment above `key` from `group_name`.
    ///
    /// If `key` is `NULL` then `comment` will be removed above `group_name`.
    /// If both `key` and `group_name` are `NULL`, then `comment` will
    /// be removed above the first group in the file.
    extern fn g_key_file_remove_comment(p_key_file: *KeyFile, p_group_name: ?[*:0]const u8, p_key: ?[*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const removeComment = g_key_file_remove_comment;

    /// Removes the specified group, `group_name`,
    /// from the key file.
    extern fn g_key_file_remove_group(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const removeGroup = g_key_file_remove_group;

    /// Removes `key` in `group_name` from the key file.
    extern fn g_key_file_remove_key(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const removeKey = g_key_file_remove_key;

    /// Writes the contents of `key_file` to `filename` using
    /// `glib.fileSetContents`.
    ///
    /// If you need stricter guarantees about durability of
    /// the written file than are provided by `glib.fileSetContents`, use
    /// `glib.fileSetContentsFull` with the return value of
    /// `glib.KeyFile.toData`.
    ///
    /// This function can fail for any of the reasons that
    /// `glib.fileSetContents` may fail.
    extern fn g_key_file_save_to_file(p_key_file: *KeyFile, p_filename: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const saveToFile = g_key_file_save_to_file;

    /// Associates a new boolean value with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    extern fn g_key_file_set_boolean(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_value: c_int) void;
    pub const setBoolean = g_key_file_set_boolean;

    /// Associates a list of boolean values with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    extern fn g_key_file_set_boolean_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_list: [*]c_int, p_length: usize) void;
    pub const setBooleanList = g_key_file_set_boolean_list;

    /// Places a comment above `key` from `group_name`.
    ///
    /// If `key` is `NULL` then `comment` will be written above `group_name`.
    /// If both `key` and `group_name` are `NULL`, then `comment` will be
    /// written above the first group in the file.
    ///
    /// Note that this function prepends a `#` comment marker to
    /// each line of `comment`.
    extern fn g_key_file_set_comment(p_key_file: *KeyFile, p_group_name: ?[*:0]const u8, p_key: ?[*:0]const u8, p_comment: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const setComment = g_key_file_set_comment;

    /// Associates a new double value with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    extern fn g_key_file_set_double(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_value: f64) void;
    pub const setDouble = g_key_file_set_double;

    /// Associates a list of double values with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    extern fn g_key_file_set_double_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_list: [*]f64, p_length: usize) void;
    pub const setDoubleList = g_key_file_set_double_list;

    /// Associates a new integer value with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    extern fn g_key_file_set_int64(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_value: i64) void;
    pub const setInt64 = g_key_file_set_int64;

    /// Associates a new integer value with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    extern fn g_key_file_set_integer(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_value: c_int) void;
    pub const setInteger = g_key_file_set_integer;

    /// Associates a list of integer values with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    extern fn g_key_file_set_integer_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_list: [*]c_int, p_length: usize) void;
    pub const setIntegerList = g_key_file_set_integer_list;

    /// Sets the character which is used to separate values in lists.
    ///
    /// Typically `;` or `,` are used as separators. The default list separator
    /// is `;`.
    extern fn g_key_file_set_list_separator(p_key_file: *KeyFile, p_separator: u8) void;
    pub const setListSeparator = g_key_file_set_list_separator;

    /// Associates a string value for `key` and `locale` under `group_name`.
    ///
    /// If the translation for `key` cannot be found then it is created.
    ///
    /// If `locale` is `C` then the untranslated value is set (since GLib 2.84).
    extern fn g_key_file_set_locale_string(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_locale: [*:0]const u8, p_string: [*:0]const u8) void;
    pub const setLocaleString = g_key_file_set_locale_string;

    /// Associates a list of string values for `key` and `locale` under
    /// `group_name`.
    ///
    /// If `locale` is `C` then the untranslated value is set (since GLib 2.84).
    ///
    /// If the translation for `key` cannot be found then it is created.
    extern fn g_key_file_set_locale_string_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_locale: [*:0]const u8, p_list: [*:null]const ?[*:0]const u8, p_length: usize) void;
    pub const setLocaleStringList = g_key_file_set_locale_string_list;

    /// Associates a new string value with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    /// If `group_name` cannot be found then it is created.
    /// Unlike `glib.KeyFile.setValue`, this function handles characters
    /// that need escaping, such as newlines.
    extern fn g_key_file_set_string(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_string: [*:0]const u8) void;
    pub const setString = g_key_file_set_string;

    /// Associates a list of string values for `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    /// If `group_name` cannot be found then it is created.
    extern fn g_key_file_set_string_list(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_list: [*:null]const ?[*:0]const u8, p_length: usize) void;
    pub const setStringList = g_key_file_set_string_list;

    /// Associates a new integer value with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created.
    extern fn g_key_file_set_uint64(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_value: u64) void;
    pub const setUint64 = g_key_file_set_uint64;

    /// Associates a new value with `key` under `group_name`.
    ///
    /// If `key` cannot be found then it is created. If `group_name` cannot
    /// be found then it is created. To set an UTF-8 string which may contain
    /// characters that need escaping (such as newlines or spaces), use
    /// `glib.KeyFile.setString`.
    extern fn g_key_file_set_value(p_key_file: *KeyFile, p_group_name: [*:0]const u8, p_key: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const setValue = g_key_file_set_value;

    /// Outputs `key_file` as a string.
    ///
    /// Note that this function never reports an error.
    extern fn g_key_file_to_data(p_key_file: *KeyFile, p_length: ?*usize, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const toData = g_key_file_to_data;

    /// Decreases the reference count of `key_file` by 1.
    ///
    /// If the reference count reaches zero, frees the key file and all its allocated
    /// memory.
    extern fn g_key_file_unref(p_key_file: *KeyFile) void;
    pub const unref = g_key_file_unref;

    extern fn g_key_file_get_type() usize;
    pub const getGObjectType = g_key_file_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.List` struct is used for each element in a doubly-linked list.
pub const List = extern struct {
    /// holds the element's data, which can be a pointer to any kind
    ///        of data, or any integer value using the
    ///        [Type Conversion Macros](conversion-macros.html`conversion`-macros)
    f_data: ?*anyopaque,
    /// contains the link to the next element in the list
    f_next: ?*glib.List,
    /// contains the link to the previous element in the list
    f_prev: ?*glib.List,

    /// Allocates space for one `glib.List` element. It is called by
    /// `glib.List.append`, `glib.List.prepend`, `glib.List.insert` and
    /// `glib.List.insertSorted` and so is rarely used on its own.
    extern fn g_list_alloc() *glib.List;
    pub const alloc = g_list_alloc;

    /// Adds a new element on to the end of the list.
    ///
    /// Note that the return value is the new start of the list,
    /// if `list` was empty; make sure you store the new value.
    ///
    /// `glib.List.append` has to traverse the entire list to find the end,
    /// which is inefficient when adding multiple elements. A common idiom
    /// to avoid the inefficiency is to use `glib.List.prepend` and reverse
    /// the list with `glib.List.reverse` when all elements have been added.
    ///
    /// ```
    /// // Notice that these are initialized to the empty list.
    /// GList *string_list = NULL, *number_list = NULL;
    ///
    /// // This is a list of strings.
    /// string_list = g_list_append (string_list, "first");
    /// string_list = g_list_append (string_list, "second");
    ///
    /// // This is a list of integers.
    /// number_list = g_list_append (number_list, GINT_TO_POINTER (27));
    /// number_list = g_list_append (number_list, GINT_TO_POINTER (14));
    /// ```
    extern fn g_list_append(p_list: *glib.List, p_data: ?*anyopaque) *glib.List;
    pub const append = g_list_append;

    /// Adds the second `glib.List` onto the end of the first `glib.List`.
    /// Note that the elements of the second `glib.List` are not copied.
    /// They are used directly.
    ///
    /// This function is for example used to move an element in the list.
    /// The following example moves an element to the top of the list:
    /// ```
    /// list = g_list_remove_link (list, llink);
    /// list = g_list_concat (llink, list);
    /// ```
    extern fn g_list_concat(p_list1: *glib.List, p_list2: *glib.List) *glib.List;
    pub const concat = g_list_concat;

    /// Copies a `glib.List`.
    ///
    /// Note that this is a "shallow" copy. If the list elements
    /// consist of pointers to data, the pointers are copied but
    /// the actual data is not. See `glib.List.copyDeep` if you need
    /// to copy the data as well.
    extern fn g_list_copy(p_list: *glib.List) *glib.List;
    pub const copy = g_list_copy;

    /// Makes a full (deep) copy of a `glib.List`.
    ///
    /// In contrast with `glib.List.copy`, this function uses `func` to make
    /// a copy of each list element, in addition to copying the list
    /// container itself.
    ///
    /// `func`, as a `glib.CopyFunc`, takes two arguments, the data to be copied
    /// and a `user_data` pointer. On common processor architectures, it's safe to
    /// pass `NULL` as `user_data` if the copy function takes only one argument. You
    /// may get compiler warnings from this though if compiling with GCC’s
    /// `-Wcast-function-type` warning.
    ///
    /// For instance, if `list` holds a list of GObjects, you can do:
    /// ```
    /// another_list = g_list_copy_deep (list, (GCopyFunc) g_object_ref, NULL);
    /// ```
    ///
    /// And, to entirely free the new list, you could do:
    /// ```
    /// g_list_free_full (another_list, g_object_unref);
    /// ```
    extern fn g_list_copy_deep(p_list: *glib.List, p_func: glib.CopyFunc, p_user_data: ?*anyopaque) *glib.List;
    pub const copyDeep = g_list_copy_deep;

    /// Removes the node link_ from the list and frees it.
    /// Compare this to `glib.List.removeLink` which removes the node
    /// without freeing it.
    extern fn g_list_delete_link(p_list: *glib.List, p_link_: *glib.List) *glib.List;
    pub const deleteLink = g_list_delete_link;

    /// Finds the element in a `glib.List` which contains the given data.
    extern fn g_list_find(p_list: *glib.List, p_data: ?*const anyopaque) *glib.List;
    pub const find = g_list_find;

    /// Finds an element in a `glib.List`, using a supplied function to
    /// find the desired element. It iterates over the list, calling
    /// the given function which should return 0 when the desired
    /// element is found. The function takes two `gconstpointer` arguments,
    /// the `glib.List` element's data as the first argument and the
    /// given user data.
    extern fn g_list_find_custom(p_list: *glib.List, p_data: ?*const anyopaque, p_func: glib.CompareFunc) *glib.List;
    pub const findCustom = g_list_find_custom;

    /// Gets the first element in a `glib.List`.
    extern fn g_list_first(p_list: *glib.List) *glib.List;
    pub const first = g_list_first;

    /// Calls a function for each element of a `glib.List`.
    ///
    /// It is safe for `func` to remove the element from `list`, but it must
    /// not modify any part of the list after that element.
    extern fn g_list_foreach(p_list: *glib.List, p_func: glib.Func, p_user_data: ?*anyopaque) void;
    pub const foreach = g_list_foreach;

    /// Frees all of the memory used by a `glib.List`.
    /// The freed elements are returned to the slice allocator.
    ///
    /// If list elements contain dynamically-allocated memory, you should
    /// either use `glib.List.freeFull` or free them manually first.
    ///
    /// It can be combined with `g_steal_pointer` to ensure the list head pointer
    /// is not left dangling:
    /// ```
    /// GList *list_of_borrowed_things = …;  /<!-- -->* (transfer container) *<!-- -->/
    /// g_list_free (g_steal_pointer (&list_of_borrowed_things));
    /// ```
    extern fn g_list_free(p_list: *glib.List) void;
    pub const free = g_list_free;

    /// Frees one `glib.List` element, but does not update links from the next and
    /// previous elements in the list, so you should not call this function on an
    /// element that is currently part of a list.
    ///
    /// It is usually used after `glib.List.removeLink`.
    extern fn g_list_free_1(p_list: *glib.List) void;
    pub const free1 = g_list_free_1;

    /// Convenience method, which frees all the memory used by a `glib.List`,
    /// and calls `free_func` on every element's data.
    ///
    /// `free_func` must not modify the list (eg, by removing the freed
    /// element from it).
    ///
    /// It can be combined with `g_steal_pointer` to ensure the list head pointer
    /// is not left dangling ­— this also has the nice property that the head pointer
    /// is cleared before any of the list elements are freed, to prevent double frees
    /// from `free_func`:
    /// ```
    /// GList *list_of_owned_things = …;  /<!-- -->* (transfer full) (element-type GObject) *<!-- -->/
    /// g_list_free_full (g_steal_pointer (&list_of_owned_things), g_object_unref);
    /// ```
    extern fn g_list_free_full(p_list: *glib.List, p_free_func: glib.DestroyNotify) void;
    pub const freeFull = g_list_free_full;

    /// Gets the position of the element containing
    /// the given data (starting from 0).
    extern fn g_list_index(p_list: *glib.List, p_data: ?*const anyopaque) c_int;
    pub const index = g_list_index;

    /// Inserts a new element into the list at the given position.
    extern fn g_list_insert(p_list: *glib.List, p_data: ?*anyopaque, p_position: c_int) *glib.List;
    pub const insert = g_list_insert;

    /// Inserts a new element into the list before the given position.
    extern fn g_list_insert_before(p_list: *glib.List, p_sibling: *glib.List, p_data: ?*anyopaque) *glib.List;
    pub const insertBefore = g_list_insert_before;

    /// Inserts `link_` into the list before the given position.
    extern fn g_list_insert_before_link(p_list: *glib.List, p_sibling: ?*glib.List, p_link_: *glib.List) *glib.List;
    pub const insertBeforeLink = g_list_insert_before_link;

    /// Inserts a new element into the list, using the given comparison
    /// function to determine its position.
    ///
    /// If you are adding many new elements to a list, and the number of
    /// new elements is much larger than the length of the list, use
    /// `glib.List.prepend` to add the new items and sort the list afterwards
    /// with `glib.List.sort`.
    extern fn g_list_insert_sorted(p_list: *glib.List, p_data: ?*anyopaque, p_func: glib.CompareFunc) *glib.List;
    pub const insertSorted = g_list_insert_sorted;

    /// Inserts a new element into the list, using the given comparison
    /// function to determine its position.
    ///
    /// If you are adding many new elements to a list, and the number of
    /// new elements is much larger than the length of the list, use
    /// `glib.List.prepend` to add the new items and sort the list afterwards
    /// with `glib.List.sort`.
    extern fn g_list_insert_sorted_with_data(p_list: *glib.List, p_data: ?*anyopaque, p_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) *glib.List;
    pub const insertSortedWithData = g_list_insert_sorted_with_data;

    /// Gets the last element in a `glib.List`.
    extern fn g_list_last(p_list: *glib.List) *glib.List;
    pub const last = g_list_last;

    /// Gets the number of elements in a `glib.List`.
    ///
    /// This function iterates over the whole list to count its elements.
    /// Use a `glib.Queue` instead of a GList if you regularly need the number
    /// of items. To check whether the list is non-empty, it is faster to check
    /// `list` against `NULL`.
    extern fn g_list_length(p_list: *glib.List) c_uint;
    pub const length = g_list_length;

    /// Gets the element at the given position in a `glib.List`.
    ///
    /// This iterates over the list until it reaches the `n`-th position. If you
    /// intend to iterate over every element, it is better to use a for-loop as
    /// described in the `glib.List` introduction.
    extern fn g_list_nth(p_list: *glib.List, p_n: c_uint) *glib.List;
    pub const nth = g_list_nth;

    /// Gets the data of the element at the given position.
    ///
    /// This iterates over the list until it reaches the `n`-th position. If you
    /// intend to iterate over every element, it is better to use a for-loop as
    /// described in the `glib.List` introduction.
    extern fn g_list_nth_data(p_list: *glib.List, p_n: c_uint) ?*anyopaque;
    pub const nthData = g_list_nth_data;

    /// Gets the element `n` places before `list`.
    extern fn g_list_nth_prev(p_list: *glib.List, p_n: c_uint) *glib.List;
    pub const nthPrev = g_list_nth_prev;

    extern fn g_list_pop_allocator() void;
    pub const popAllocator = g_list_pop_allocator;

    /// Gets the position of the given element
    /// in the `glib.List` (starting from 0).
    extern fn g_list_position(p_list: *glib.List, p_llink: *glib.List) c_int;
    pub const position = g_list_position;

    /// Prepends a new element on to the start of the list.
    ///
    /// Note that the return value is the new start of the list,
    /// which will have changed, so make sure you store the new value.
    ///
    /// ```
    /// // Notice that it is initialized to the empty list.
    /// GList *list = NULL;
    ///
    /// list = g_list_prepend (list, "last");
    /// list = g_list_prepend (list, "first");
    /// ```
    ///
    /// Do not use this function to prepend a new element to a different
    /// element than the start of the list. Use `glib.List.insertBefore` instead.
    extern fn g_list_prepend(p_list: *glib.List, p_data: ?*anyopaque) *glib.List;
    pub const prepend = g_list_prepend;

    extern fn g_list_push_allocator(p_allocator: *glib.Allocator) void;
    pub const pushAllocator = g_list_push_allocator;

    /// Removes an element from a `glib.List`.
    /// If two elements contain the same data, only the first is removed.
    /// If none of the elements contain the data, the `glib.List` is unchanged.
    extern fn g_list_remove(p_list: *glib.List, p_data: ?*const anyopaque) *glib.List;
    pub const remove = g_list_remove;

    /// Removes all list nodes with data equal to `data`.
    /// Returns the new head of the list. Contrast with
    /// `glib.List.remove` which removes only the first node
    /// matching the given data.
    extern fn g_list_remove_all(p_list: *glib.List, p_data: ?*const anyopaque) *glib.List;
    pub const removeAll = g_list_remove_all;

    /// Removes an element from a `glib.List`, without freeing the element.
    /// The removed element's prev and next links are set to `NULL`, so
    /// that it becomes a self-contained list with one element.
    ///
    /// This function is for example used to move an element in the list
    /// (see the example for `glib.List.concat`) or to remove an element in
    /// the list before freeing its data:
    /// ```
    /// list = g_list_remove_link (list, llink);
    /// free_some_data_that_may_access_the_list_again (llink->data);
    /// g_list_free (llink);
    /// ```
    extern fn g_list_remove_link(p_list: *glib.List, p_llink: *glib.List) *glib.List;
    pub const removeLink = g_list_remove_link;

    /// Reverses a `glib.List`.
    /// It simply switches the next and prev pointers of each element.
    extern fn g_list_reverse(p_list: *glib.List) *glib.List;
    pub const reverse = g_list_reverse;

    /// Sorts a `glib.List` using the given comparison function. The algorithm
    /// used is a stable sort.
    extern fn g_list_sort(p_list: *glib.List, p_compare_func: glib.CompareFunc) *glib.List;
    pub const sort = g_list_sort;

    /// Like `glib.List.sort`, but the comparison function accepts
    /// a user data argument.
    extern fn g_list_sort_with_data(p_list: *glib.List, p_compare_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) *glib.List;
    pub const sortWithData = g_list_sort_with_data;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Structure representing a single field in a structured log entry. See
/// `glib.logStructured` for details.
///
/// Log fields may contain arbitrary values, including binary with embedded nul
/// bytes. If the field contains a string, the string must be UTF-8 encoded and
/// have a trailing nul byte. Otherwise, `length` must be set to a non-negative
/// value.
pub const LogField = extern struct {
    /// field name (UTF-8 string)
    f_key: ?[*:0]const u8,
    /// field value (arbitrary bytes)
    f_value: ?*const anyopaque,
    /// length of `value`, in bytes, or -1 if it is nul-terminated
    f_length: isize,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `GMainContext` struct is an opaque data
/// type representing a set of sources to be handled in a main loop.
pub const MainContext = opaque {
    /// Returns the global-default main context. This is the main context
    /// used for main loop functions when a main loop is not explicitly
    /// specified, and corresponds to the "main" main loop. See also
    /// `glib.MainContext.getThreadDefault`.
    extern fn g_main_context_default() *glib.MainContext;
    pub const default = g_main_context_default;

    /// Gets the thread-default `glib.MainContext` for this thread. Asynchronous
    /// operations that want to be able to be run in contexts other than
    /// the default one should call this method or
    /// `glib.MainContext.refThreadDefault` to get a
    /// `glib.MainContext` to add their `glib.Source`s to. (Note that
    /// even in single-threaded programs applications may sometimes want to
    /// temporarily push a non-default context, so it is not safe to assume that
    /// this will always return `NULL` if you are running in the default thread.)
    ///
    /// If you need to hold a reference on the context, use
    /// `glib.MainContext.refThreadDefault` instead.
    extern fn g_main_context_get_thread_default() ?*glib.MainContext;
    pub const getThreadDefault = g_main_context_get_thread_default;

    /// Gets the thread-default `glib.MainContext` for this thread, as with
    /// `glib.MainContext.getThreadDefault`, but also adds a reference to
    /// it with `glib.MainContext.ref`. In addition, unlike
    /// `glib.MainContext.getThreadDefault`, if the thread-default context
    /// is the global-default context, this will return that
    /// `glib.MainContext` (with a ref added to it) rather than returning
    /// `NULL`.
    extern fn g_main_context_ref_thread_default() *glib.MainContext;
    pub const refThreadDefault = g_main_context_ref_thread_default;

    /// Creates a new `glib.MainContext` structure.
    extern fn g_main_context_new() *glib.MainContext;
    pub const new = g_main_context_new;

    /// Creates a new `glib.MainContext` structure.
    extern fn g_main_context_new_with_flags(p_flags: glib.MainContextFlags) *glib.MainContext;
    pub const newWithFlags = g_main_context_new_with_flags;

    /// Tries to become the owner of the specified context.
    /// If some other thread is the owner of the context,
    /// returns `FALSE` immediately. Ownership is properly
    /// recursive: the owner can require ownership again
    /// and will release ownership when `glib.MainContext.release`
    /// is called as many times as `glib.MainContext.acquire`.
    ///
    /// You must be the owner of a context before you
    /// can call `glib.MainContext.prepare`, `glib.MainContext.query`,
    /// `glib.MainContext.check`, `glib.MainContext.dispatch`,
    /// `glib.MainContext.release`.
    ///
    /// Since 2.76 `context` can be `NULL` to use the global-default
    /// main context.
    extern fn g_main_context_acquire(p_context: ?*MainContext) c_int;
    pub const acquire = g_main_context_acquire;

    /// Adds a file descriptor to the set of file descriptors polled for
    /// this context. This will very seldom be used directly. Instead
    /// a typical event source will use `g_source_add_unix_fd` instead.
    extern fn g_main_context_add_poll(p_context: ?*MainContext, p_fd: *glib.PollFD, p_priority: c_int) void;
    pub const addPoll = g_main_context_add_poll;

    /// Passes the results of polling back to the main loop. You should be
    /// careful to pass `fds` and its length `n_fds` as received from
    /// `glib.MainContext.query`, as this functions relies on assumptions
    /// on how `fds` is filled.
    ///
    /// You must have successfully acquired the context with
    /// `glib.MainContext.acquire` before you may call this function.
    ///
    /// Since 2.76 `context` can be `NULL` to use the global-default
    /// main context.
    extern fn g_main_context_check(p_context: ?*MainContext, p_max_priority: c_int, p_fds: [*]glib.PollFD, p_n_fds: c_int) c_int;
    pub const check = g_main_context_check;

    /// Dispatches all pending sources.
    ///
    /// You must have successfully acquired the context with
    /// `glib.MainContext.acquire` before you may call this function.
    ///
    /// Since 2.76 `context` can be `NULL` to use the global-default
    /// main context.
    extern fn g_main_context_dispatch(p_context: ?*MainContext) void;
    pub const dispatch = g_main_context_dispatch;

    /// Finds a source with the given source functions and user data.  If
    /// multiple sources exist with the same source function and user data,
    /// the first one found will be returned.
    extern fn g_main_context_find_source_by_funcs_user_data(p_context: ?*MainContext, p_funcs: *glib.SourceFuncs, p_user_data: ?*anyopaque) *glib.Source;
    pub const findSourceByFuncsUserData = g_main_context_find_source_by_funcs_user_data;

    /// Finds a `glib.Source` given a pair of context and ID.
    ///
    /// It is a programmer error to attempt to look up a non-existent source.
    ///
    /// More specifically: source IDs can be reissued after a source has been
    /// destroyed and therefore it is never valid to use this function with a
    /// source ID which may have already been removed.  An example is when
    /// scheduling an idle to run in another thread with `glib.idleAdd`: the
    /// idle may already have run and been removed by the time this function
    /// is called on its (now invalid) source ID.  This source ID may have
    /// been reissued, leading to the operation being performed against the
    /// wrong source.
    extern fn g_main_context_find_source_by_id(p_context: ?*MainContext, p_source_id: c_uint) *glib.Source;
    pub const findSourceById = g_main_context_find_source_by_id;

    /// Finds a source with the given user data for the callback.  If
    /// multiple sources exist with the same user data, the first
    /// one found will be returned.
    extern fn g_main_context_find_source_by_user_data(p_context: ?*MainContext, p_user_data: ?*anyopaque) *glib.Source;
    pub const findSourceByUserData = g_main_context_find_source_by_user_data;

    /// Gets the poll function set by `glib.MainContext.setPollFunc`.
    extern fn g_main_context_get_poll_func(p_context: ?*MainContext) glib.PollFunc;
    pub const getPollFunc = g_main_context_get_poll_func;

    /// Invokes a function in such a way that `context` is owned during the
    /// invocation of `function`.
    ///
    /// If `context` is `NULL` then the global-default main context — as
    /// returned by `glib.MainContext.default` — is used.
    ///
    /// If `context` is owned by the current thread, `function` is called
    /// directly.  Otherwise, if `context` is the thread-default main context
    /// of the current thread and `glib.MainContext.acquire` succeeds, then
    /// `function` is called and `glib.MainContext.release` is called
    /// afterwards.
    ///
    /// In any other case, an idle source is created to call `function` and
    /// that source is attached to `context` (presumably to be run in another
    /// thread).  The idle source is attached with `glib.PRIORITY_DEFAULT`
    /// priority.  If you want a different priority, use
    /// `glib.MainContext.invokeFull`.
    ///
    /// Note that, as with normal idle functions, `function` should probably
    /// return `FALSE`.  If it returns `TRUE`, it will be continuously run in a
    /// loop (and may prevent this call from returning).
    extern fn g_main_context_invoke(p_context: ?*MainContext, p_function: glib.SourceFunc, p_data: ?*anyopaque) void;
    pub const invoke = g_main_context_invoke;

    /// Invokes a function in such a way that `context` is owned during the
    /// invocation of `function`.
    ///
    /// This function is the same as `glib.MainContext.invoke` except that it
    /// lets you specify the priority in case `function` ends up being
    /// scheduled as an idle and also lets you give a `glib.DestroyNotify` for `data`.
    ///
    /// `notify` should not assume that it is called from any particular
    /// thread or with any particular context acquired.
    extern fn g_main_context_invoke_full(p_context: ?*MainContext, p_priority: c_int, p_function: glib.SourceFunc, p_data: ?*anyopaque, p_notify: ?glib.DestroyNotify) void;
    pub const invokeFull = g_main_context_invoke_full;

    /// Determines whether this thread holds the (recursive)
    /// ownership of this `glib.MainContext`. This is useful to
    /// know before waiting on another thread that may be
    /// blocking to get ownership of `context`.
    extern fn g_main_context_is_owner(p_context: ?*MainContext) c_int;
    pub const isOwner = g_main_context_is_owner;

    /// Runs a single iteration for the given main loop. This involves
    /// checking to see if any event sources are ready to be processed,
    /// then if no events sources are ready and `may_block` is `TRUE`, waiting
    /// for a source to become ready, then dispatching the highest priority
    /// events sources that are ready. Otherwise, if `may_block` is `FALSE`
    /// sources are not waited to become ready, only those highest priority
    /// events sources will be dispatched (if any), that are ready at this
    /// given moment without further waiting.
    ///
    /// Note that even when `may_block` is `TRUE`, it is still possible for
    /// `glib.MainContext.iteration` to return `FALSE`, since the wait may
    /// be interrupted for other reasons than an event source becoming ready.
    extern fn g_main_context_iteration(p_context: ?*MainContext, p_may_block: c_int) c_int;
    pub const iteration = g_main_context_iteration;

    /// Checks if any sources have pending events for the given context.
    extern fn g_main_context_pending(p_context: ?*MainContext) c_int;
    pub const pending = g_main_context_pending;

    /// Pops `context` off the thread-default context stack (verifying that
    /// it was on the top of the stack).
    extern fn g_main_context_pop_thread_default(p_context: ?*MainContext) void;
    pub const popThreadDefault = g_main_context_pop_thread_default;

    /// Prepares to poll sources within a main loop. The resulting information
    /// for polling is determined by calling `glib.MainContext.query`.
    ///
    /// You must have successfully acquired the context with
    /// `glib.MainContext.acquire` before you may call this function.
    extern fn g_main_context_prepare(p_context: ?*MainContext, p_priority: ?*c_int) c_int;
    pub const prepare = g_main_context_prepare;

    /// Acquires `context` and sets it as the thread-default context for the
    /// current thread. This will cause certain asynchronous operations
    /// (such as most [Gio](../gio/index.html)-based I/O) which are
    /// started in this thread to run under `context` and deliver their
    /// results to its main loop, rather than running under the global
    /// default main context in the main thread. Note that calling this function
    /// changes the context returned by `glib.MainContext.getThreadDefault`,
    /// not the one returned by `glib.MainContext.default`, so it does not
    /// affect the context used by functions like `glib.idleAdd`.
    ///
    /// Normally you would call this function shortly after creating a new
    /// thread, passing it a `glib.MainContext` which will be run by a
    /// `glib.MainLoop` in that thread, to set a new default context for all
    /// async operations in that thread. In this case you may not need to
    /// ever call `glib.MainContext.popThreadDefault`, assuming you want
    /// the new `glib.MainContext` to be the default for the whole lifecycle
    /// of the thread.
    ///
    /// If you don't have control over how the new thread was created (e.g.
    /// in the new thread isn't newly created, or if the thread life
    /// cycle is managed by a `glib.ThreadPool`), it is always suggested to wrap
    /// the logic that needs to use the new `glib.MainContext` inside a
    /// `glib.MainContext.pushThreadDefault` /
    /// `glib.MainContext.popThreadDefault` pair, otherwise threads that
    /// are re-used will end up never explicitly releasing the
    /// `glib.MainContext` reference they hold.
    ///
    /// In some cases you may want to schedule a single operation in a
    /// non-default context, or temporarily use a non-default context in
    /// the main thread. In that case, you can wrap the call to the
    /// asynchronous operation inside a
    /// `glib.MainContext.pushThreadDefault` /
    /// `glib.MainContext.popThreadDefault` pair, but it is up to you to
    /// ensure that no other asynchronous operations accidentally get
    /// started while the non-default context is active.
    ///
    /// Beware that libraries that predate this function may not correctly
    /// handle being used from a thread with a thread-default context. Eg,
    /// see `g_file_supports_thread_contexts`.
    extern fn g_main_context_push_thread_default(p_context: ?*MainContext) void;
    pub const pushThreadDefault = g_main_context_push_thread_default;

    /// Determines information necessary to poll this main loop. You should
    /// be careful to pass the resulting `fds` array and its length `n_fds`
    /// as is when calling `glib.MainContext.check`, as this function relies
    /// on assumptions made when the array is filled.
    ///
    /// You must have successfully acquired the context with
    /// `glib.MainContext.acquire` before you may call this function.
    extern fn g_main_context_query(p_context: ?*MainContext, p_max_priority: c_int, p_timeout_: *c_int, p_fds: *[*]glib.PollFD, p_n_fds: c_int) c_int;
    pub const query = g_main_context_query;

    /// Increases the reference count on a `glib.MainContext` object by one.
    extern fn g_main_context_ref(p_context: *MainContext) *glib.MainContext;
    pub const ref = g_main_context_ref;

    /// Releases ownership of a context previously acquired by this thread
    /// with `glib.MainContext.acquire`. If the context was acquired multiple
    /// times, the ownership will be released only when `glib.MainContext.release`
    /// is called as many times as it was acquired.
    ///
    /// You must have successfully acquired the context with
    /// `glib.MainContext.acquire` before you may call this function.
    extern fn g_main_context_release(p_context: ?*MainContext) void;
    pub const release = g_main_context_release;

    /// Removes file descriptor from the set of file descriptors to be
    /// polled for a particular context.
    extern fn g_main_context_remove_poll(p_context: ?*MainContext, p_fd: *glib.PollFD) void;
    pub const removePoll = g_main_context_remove_poll;

    /// Sets the function to use to handle polling of file descriptors. It
    /// will be used instead of the `poll` system call
    /// (or GLib's replacement function, which is used where
    /// `poll` isn't available).
    ///
    /// This function could possibly be used to integrate the GLib event
    /// loop with an external event loop.
    extern fn g_main_context_set_poll_func(p_context: ?*MainContext, p_func: glib.PollFunc) void;
    pub const setPollFunc = g_main_context_set_poll_func;

    /// Decreases the reference count on a `glib.MainContext` object by one.
    /// If
    /// the result is zero, free the context and free all associated memory.
    extern fn g_main_context_unref(p_context: *MainContext) void;
    pub const unref = g_main_context_unref;

    /// Tries to become the owner of the specified context,
    /// as with `glib.MainContext.acquire`. But if another thread
    /// is the owner, atomically drop `mutex` and wait on `cond` until
    /// that owner releases ownership or until `cond` is signaled, then
    /// try again (once) to become the owner.
    extern fn g_main_context_wait(p_context: ?*MainContext, p_cond: *glib.Cond, p_mutex: *glib.Mutex) c_int;
    pub const wait = g_main_context_wait;

    /// If `context` is currently blocking in `glib.MainContext.iteration`
    /// waiting for a source to become ready, cause it to stop blocking
    /// and return.  Otherwise, cause the next invocation of
    /// `glib.MainContext.iteration` to return without blocking.
    ///
    /// This API is useful for low-level control over `glib.MainContext`; for
    /// example, integrating it with main loop implementations such as
    /// `glib.MainLoop`.
    ///
    /// Another related use for this function is when implementing a main
    /// loop with a termination condition, computed from multiple threads:
    ///
    /// ```
    ///   `define` NUM_TASKS 10
    ///   static gint tasks_remaining = NUM_TASKS;  // (atomic)
    ///   ...
    ///
    ///   while (g_atomic_int_get (&tasks_remaining) != 0)
    ///     g_main_context_iteration (NULL, TRUE);
    /// ```
    ///
    /// Then in a thread:
    /// ```
    ///   `perform_work`;
    ///
    ///   if (g_atomic_int_dec_and_test (&tasks_remaining))
    ///     g_main_context_wakeup (NULL);
    /// ```
    extern fn g_main_context_wakeup(p_context: ?*MainContext) void;
    pub const wakeup = g_main_context_wakeup;

    extern fn g_main_context_get_type() usize;
    pub const getGObjectType = g_main_context_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `GMainLoop` struct is an opaque data type
/// representing the main event loop of a GLib or GTK application.
pub const MainLoop = opaque {
    /// Creates a new `glib.MainLoop` structure.
    extern fn g_main_loop_new(p_context: ?*glib.MainContext, p_is_running: c_int) *glib.MainLoop;
    pub const new = g_main_loop_new;

    /// Returns the `glib.MainContext` of `loop`.
    extern fn g_main_loop_get_context(p_loop: *MainLoop) *glib.MainContext;
    pub const getContext = g_main_loop_get_context;

    /// Checks to see if the main loop is currently being run via
    /// `glib.MainLoop.run`.
    extern fn g_main_loop_is_running(p_loop: *MainLoop) c_int;
    pub const isRunning = g_main_loop_is_running;

    /// Stops a `glib.MainLoop` from running. Any calls to
    /// `glib.MainLoop.run` for the loop will return.
    ///
    /// Note that sources that have already been dispatched when
    /// `glib.MainLoop.quit` is called will still be executed.
    extern fn g_main_loop_quit(p_loop: *MainLoop) void;
    pub const quit = g_main_loop_quit;

    /// Increases the reference count on a `glib.MainLoop` object by one.
    extern fn g_main_loop_ref(p_loop: *MainLoop) *glib.MainLoop;
    pub const ref = g_main_loop_ref;

    /// Runs a main loop until `glib.MainLoop.quit` is called on the loop.
    /// If this is called for the thread of the loop's `glib.MainContext`,
    /// it will process events from the loop, otherwise it will
    /// simply wait.
    extern fn g_main_loop_run(p_loop: *MainLoop) void;
    pub const run = g_main_loop_run;

    /// Decreases the reference count on a `glib.MainLoop` object by one. If
    /// the result is zero, free the loop and free all associated memory.
    extern fn g_main_loop_unref(p_loop: *MainLoop) void;
    pub const unref = g_main_loop_unref;

    extern fn g_main_loop_get_type() usize;
    pub const getGObjectType = g_main_loop_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.MappedFile` represents a file mapping created with
/// `glib.MappedFile.new`. It has only private members and should
/// not be accessed directly.
pub const MappedFile = opaque {
    /// Maps a file into memory. On UNIX, this is using the `mmap` function.
    ///
    /// If `writable` is `TRUE`, the mapped buffer may be modified, otherwise
    /// it is an error to modify the mapped buffer. Modifications to the buffer
    /// are not visible to other processes mapping the same file, and are not
    /// written back to the file.
    ///
    /// Note that modifications of the underlying file might affect the contents
    /// of the `glib.MappedFile`. Therefore, mapping should only be used if the file
    /// will not be modified, or if all modifications of the file are done
    /// atomically (e.g. using `glib.fileSetContents`).
    ///
    /// If `filename` is the name of an empty, regular file, the function
    /// will successfully return an empty `glib.MappedFile`. In other cases of
    /// size 0 (e.g. device files such as /dev/null), `error` will be set
    /// to the `glib.FileError` value `G_FILE_ERROR_INVAL`.
    extern fn g_mapped_file_new(p_filename: [*:0]const u8, p_writable: c_int, p_error: ?*?*glib.Error) ?*glib.MappedFile;
    pub const new = g_mapped_file_new;

    /// Maps a file into memory. On UNIX, this is using the `mmap` function.
    ///
    /// If `writable` is `TRUE`, the mapped buffer may be modified, otherwise
    /// it is an error to modify the mapped buffer. Modifications to the buffer
    /// are not visible to other processes mapping the same file, and are not
    /// written back to the file.
    ///
    /// Note that modifications of the underlying file might affect the contents
    /// of the `glib.MappedFile`. Therefore, mapping should only be used if the file
    /// will not be modified, or if all modifications of the file are done
    /// atomically (e.g. using `glib.fileSetContents`).
    extern fn g_mapped_file_new_from_fd(p_fd: c_int, p_writable: c_int, p_error: ?*?*glib.Error) ?*glib.MappedFile;
    pub const newFromFd = g_mapped_file_new_from_fd;

    /// This call existed before `glib.MappedFile` had refcounting and is currently
    /// exactly the same as `glib.MappedFile.unref`.
    extern fn g_mapped_file_free(p_file: *MappedFile) void;
    pub const free = g_mapped_file_free;

    /// Creates a new `glib.Bytes` which references the data mapped from `file`.
    /// The mapped contents of the file must not be modified after creating this
    /// bytes object, because a `glib.Bytes` should be immutable.
    extern fn g_mapped_file_get_bytes(p_file: *MappedFile) *glib.Bytes;
    pub const getBytes = g_mapped_file_get_bytes;

    /// Returns the contents of a `glib.MappedFile`.
    ///
    /// Note that the contents may not be zero-terminated,
    /// even if the `glib.MappedFile` is backed by a text file.
    ///
    /// If the file is empty then `NULL` is returned.
    extern fn g_mapped_file_get_contents(p_file: *MappedFile) ?[*:0]u8;
    pub const getContents = g_mapped_file_get_contents;

    /// Returns the length of the contents of a `glib.MappedFile`.
    extern fn g_mapped_file_get_length(p_file: *MappedFile) usize;
    pub const getLength = g_mapped_file_get_length;

    /// Increments the reference count of `file` by one.  It is safe to call
    /// this function from any thread.
    extern fn g_mapped_file_ref(p_file: *MappedFile) *glib.MappedFile;
    pub const ref = g_mapped_file_ref;

    /// Decrements the reference count of `file` by one.  If the reference count
    /// drops to 0, unmaps the buffer of `file` and frees it.
    ///
    /// It is safe to call this function from any thread.
    ///
    /// Since 2.22
    extern fn g_mapped_file_unref(p_file: *MappedFile) void;
    pub const unref = g_mapped_file_unref;

    extern fn g_mapped_file_get_type() usize;
    pub const getGObjectType = g_mapped_file_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A parse context is used to parse a stream of bytes that
/// you expect to contain marked-up text.
///
/// See `glib.MarkupParseContext.new`, `glib.MarkupParser`, and so
/// on for more details.
pub const MarkupParseContext = opaque {
    /// Creates a new parse context. A parse context is used to parse
    /// marked-up documents. You can feed any number of documents into
    /// a context, as long as no errors occur; once an error occurs,
    /// the parse context can't continue to parse text (you have to
    /// free it and create a new parse context).
    extern fn g_markup_parse_context_new(p_parser: *const glib.MarkupParser, p_flags: glib.MarkupParseFlags, p_user_data: ?*anyopaque, p_user_data_dnotify: glib.DestroyNotify) *glib.MarkupParseContext;
    pub const new = g_markup_parse_context_new;

    /// Signals to the `glib.MarkupParseContext` that all data has been
    /// fed into the parse context with `glib.MarkupParseContext.parse`.
    ///
    /// This function reports an error if the document isn't complete,
    /// for example if elements are still open.
    extern fn g_markup_parse_context_end_parse(p_context: *MarkupParseContext, p_error: ?*?*glib.Error) c_int;
    pub const endParse = g_markup_parse_context_end_parse;

    /// Frees a `glib.MarkupParseContext`.
    ///
    /// This function can't be called from inside one of the
    /// `glib.MarkupParser` functions or while a subparser is pushed.
    extern fn g_markup_parse_context_free(p_context: *MarkupParseContext) void;
    pub const free = g_markup_parse_context_free;

    /// Retrieves the name of the currently open element.
    ///
    /// If called from the start_element or end_element handlers this will
    /// give the element_name as passed to those functions. For the parent
    /// elements, see `glib.MarkupParseContext.getElementStack`.
    extern fn g_markup_parse_context_get_element(p_context: *MarkupParseContext) [*:0]const u8;
    pub const getElement = g_markup_parse_context_get_element;

    /// Retrieves the element stack from the internal state of the parser.
    ///
    /// The returned `glib.SList` is a list of strings where the first item is
    /// the currently open tag (as would be returned by
    /// `glib.MarkupParseContext.getElement`) and the next item is its
    /// immediate parent.
    ///
    /// This function is intended to be used in the start_element and
    /// end_element handlers where `glib.MarkupParseContext.getElement`
    /// would merely return the name of the element that is being
    /// processed.
    extern fn g_markup_parse_context_get_element_stack(p_context: *MarkupParseContext) *const glib.SList;
    pub const getElementStack = g_markup_parse_context_get_element_stack;

    /// Retrieves the current line number and the number of the character on
    /// that line. Intended for use in error messages; there are no strict
    /// semantics for what constitutes the "current" line number other than
    /// "the best number we could come up with for error messages."
    extern fn g_markup_parse_context_get_position(p_context: *MarkupParseContext, p_line_number: ?*c_int, p_char_number: ?*c_int) void;
    pub const getPosition = g_markup_parse_context_get_position;

    /// Returns the user_data associated with `context`.
    ///
    /// This will either be the user_data that was provided to
    /// `glib.MarkupParseContext.new` or to the most recent call
    /// of `glib.MarkupParseContext.push`.
    extern fn g_markup_parse_context_get_user_data(p_context: *MarkupParseContext) ?*anyopaque;
    pub const getUserData = g_markup_parse_context_get_user_data;

    /// Feed some data to the `glib.MarkupParseContext`.
    ///
    /// The data need not be valid UTF-8; an error will be signaled if
    /// it's invalid. The data need not be an entire document; you can
    /// feed a document into the parser incrementally, via multiple calls
    /// to this function. Typically, as you receive data from a network
    /// connection or file, you feed each received chunk of data into this
    /// function, aborting the process if an error occurs. Once an error
    /// is reported, no further data may be fed to the `glib.MarkupParseContext`;
    /// all errors are fatal.
    extern fn g_markup_parse_context_parse(p_context: *MarkupParseContext, p_text: [*:0]const u8, p_text_len: isize, p_error: ?*?*glib.Error) c_int;
    pub const parse = g_markup_parse_context_parse;

    /// Completes the process of a temporary sub-parser redirection.
    ///
    /// This function exists to collect the user_data allocated by a
    /// matching call to `glib.MarkupParseContext.push`. It must be called
    /// in the end_element handler corresponding to the start_element
    /// handler during which `glib.MarkupParseContext.push` was called.
    /// You must not call this function from the error callback -- the
    /// `user_data` is provided directly to the callback in that case.
    ///
    /// This function is not intended to be directly called by users
    /// interested in invoking subparsers. Instead, it is intended to
    /// be used by the subparsers themselves to implement a higher-level
    /// interface.
    extern fn g_markup_parse_context_pop(p_context: *MarkupParseContext) ?*anyopaque;
    pub const pop = g_markup_parse_context_pop;

    /// Temporarily redirects markup data to a sub-parser.
    ///
    /// This function may only be called from the start_element handler of
    /// a `glib.MarkupParser`. It must be matched with a corresponding call to
    /// `glib.MarkupParseContext.pop` in the matching end_element handler
    /// (except in the case that the parser aborts due to an error).
    ///
    /// All tags, text and other data between the matching tags is
    /// redirected to the subparser given by `parser`. `user_data` is used
    /// as the user_data for that parser. `user_data` is also passed to the
    /// error callback in the event that an error occurs. This includes
    /// errors that occur in subparsers of the subparser.
    ///
    /// The end tag matching the start tag for which this call was made is
    /// handled by the previous parser (which is given its own user_data)
    /// which is why `glib.MarkupParseContext.pop` is provided to allow "one
    /// last access" to the `user_data` provided to this function. In the
    /// case of error, the `user_data` provided here is passed directly to
    /// the error callback of the subparser and `glib.MarkupParseContext.pop`
    /// should not be called. In either case, if `user_data` was allocated
    /// then it ought to be freed from both of these locations.
    ///
    /// This function is not intended to be directly called by users
    /// interested in invoking subparsers. Instead, it is intended to be
    /// used by the subparsers themselves to implement a higher-level
    /// interface.
    ///
    /// As an example, see the following implementation of a simple
    /// parser that counts the number of tags encountered.
    ///
    /// ```
    /// typedef struct
    /// {
    ///   gint tag_count;
    /// } CounterData;
    ///
    /// static void
    /// counter_start_element (GMarkupParseContext  *context,
    ///                        const gchar          *element_name,
    ///                        const gchar         **attribute_names,
    ///                        const gchar         **attribute_values,
    ///                        gpointer              user_data,
    ///                        GError              **error)
    /// {
    ///   CounterData *data = user_data;
    ///
    ///   data->tag_count++;
    /// }
    ///
    /// static void
    /// counter_error (GMarkupParseContext *context,
    ///                GError              *error,
    ///                gpointer             user_data)
    /// {
    ///   CounterData *data = user_data;
    ///
    ///   g_slice_free (CounterData, data);
    /// }
    ///
    /// static GMarkupParser counter_subparser =
    /// {
    ///   counter_start_element,
    ///   NULL,
    ///   NULL,
    ///   NULL,
    ///   counter_error
    /// };
    /// ```
    ///
    /// In order to allow this parser to be easily used as a subparser, the
    /// following interface is provided:
    ///
    /// ```
    /// void
    /// start_counting (GMarkupParseContext *context)
    /// {
    ///   CounterData *data = g_slice_new (CounterData);
    ///
    ///   data->tag_count = 0;
    ///   g_markup_parse_context_push (context, &counter_subparser, data);
    /// }
    ///
    /// gint
    /// end_counting (GMarkupParseContext *context)
    /// {
    ///   CounterData *data = g_markup_parse_context_pop (context);
    ///   int result;
    ///
    ///   result = data->tag_count;
    ///   g_slice_free (CounterData, data);
    ///
    ///   return result;
    /// }
    /// ```
    ///
    /// The subparser would then be used as follows:
    ///
    /// ```
    /// static void start_element (context, element_name, ...)
    /// {
    ///   if (strcmp (element_name, "count-these") == 0)
    ///     start_counting (context);
    ///
    ///   // else, handle other tags...
    /// }
    ///
    /// static void end_element (context, element_name, ...)
    /// {
    ///   if (strcmp (element_name, "count-these") == 0)
    ///     g_print ("Counted `d` tags\n", end_counting (context));
    ///
    ///   // else, handle other tags...
    /// }
    /// ```
    extern fn g_markup_parse_context_push(p_context: *MarkupParseContext, p_parser: *const glib.MarkupParser, p_user_data: ?*anyopaque) void;
    pub const push = g_markup_parse_context_push;

    /// Increases the reference count of `context`.
    extern fn g_markup_parse_context_ref(p_context: *MarkupParseContext) *glib.MarkupParseContext;
    pub const ref = g_markup_parse_context_ref;

    /// Decreases the reference count of `context`.  When its reference count
    /// drops to 0, it is freed.
    extern fn g_markup_parse_context_unref(p_context: *MarkupParseContext) void;
    pub const unref = g_markup_parse_context_unref;

    extern fn g_markup_parse_context_get_type() usize;
    pub const getGObjectType = g_markup_parse_context_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Any of the fields in `glib.MarkupParser` can be `NULL`, in which case they
/// will be ignored. Except for the `error` function, any of these callbacks
/// can set an error; in particular the `G_MARKUP_ERROR_UNKNOWN_ELEMENT`,
/// `G_MARKUP_ERROR_UNKNOWN_ATTRIBUTE`, and `G_MARKUP_ERROR_INVALID_CONTENT`
/// errors are intended to be set from these callbacks. If you set an error
/// from a callback, `glib.MarkupParseContext.parse` will report that error
/// back to its caller.
pub const MarkupParser = extern struct {
    /// Callback to invoke when the opening tag of an element
    ///     is seen. The callback's `attribute_names` and `attribute_values` parameters
    ///     are `NULL`-terminated.
    f_start_element: ?*const fn (p_context: *glib.MarkupParseContext, p_element_name: [*:0]const u8, p_attribute_names: *[*:0]const u8, p_attribute_values: *[*:0]const u8, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) callconv(.c) void,
    /// Callback to invoke when the closing tag of an element
    ///     is seen. Note that this is also called for empty tags like
    ///     `<empty/>`.
    f_end_element: ?*const fn (p_context: *glib.MarkupParseContext, p_element_name: [*:0]const u8, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) callconv(.c) void,
    /// Callback to invoke when some text is seen (text is always
    ///     inside an element). Note that the text of an element may be spread
    ///     over multiple calls of this function. If the
    ///     `G_MARKUP_TREAT_CDATA_AS_TEXT` flag is set, this function is also
    ///     called for the content of CDATA marked sections.
    f_text: ?*const fn (p_context: *glib.MarkupParseContext, p_text: [*:0]const u8, p_text_len: usize, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) callconv(.c) void,
    /// Callback to invoke for comments, processing instructions
    ///     and doctype declarations; if you're re-writing the parsed document,
    ///     write the passthrough text back out in the same position. If the
    ///     `G_MARKUP_TREAT_CDATA_AS_TEXT` flag is not set, this function is also
    ///     called for CDATA marked sections.
    f_passthrough: ?*const fn (p_context: *glib.MarkupParseContext, p_passthrough_text: [*:0]const u8, p_text_len: usize, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) callconv(.c) void,
    /// Callback to invoke when an error occurs.
    f_error: ?*const fn (p_context: *glib.MarkupParseContext, p_error: *glib.Error, p_user_data: ?*anyopaque) callconv(.c) void,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A GMatchInfo is an opaque struct used to return information about
/// matches.
pub const MatchInfo = opaque {
    /// Returns a new string containing the text in `string_to_expand` with
    /// references and escape sequences expanded. References refer to the last
    /// match done with `string` against `regex` and have the same syntax used by
    /// `glib.Regex.replace`.
    ///
    /// The `string_to_expand` must be UTF-8 encoded even if `G_REGEX_RAW` was
    /// passed to `glib.Regex.new`.
    ///
    /// The backreferences are extracted from the string passed to the match
    /// function, so you cannot call this function after freeing the string.
    ///
    /// `match_info` may be `NULL` in which case `string_to_expand` must not
    /// contain references. For instance "foo\n" does not refer to an actual
    /// pattern and '\n' merely will be replaced with \n character,
    /// while to expand "\0" (whole match) one needs the result of a match.
    /// Use `glib.regexCheckReplacement` to find out whether `string_to_expand`
    /// contains references.
    extern fn g_match_info_expand_references(p_match_info: ?*const MatchInfo, p_string_to_expand: [*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const expandReferences = g_match_info_expand_references;

    /// Retrieves the text matching the `match_num`'th capturing
    /// parentheses. 0 is the full text of the match, 1 is the first paren
    /// set, 2 the second, and so on.
    ///
    /// If `match_num` is a valid sub pattern but it didn't match anything
    /// (e.g. sub pattern 1, matching "b" against "(a)?b") then an empty
    /// string is returned.
    ///
    /// If the match was obtained using the DFA algorithm, that is using
    /// `glib.Regex.matchAll` or `glib.Regex.matchAllFull`, the retrieved
    /// string is not that of a set of parentheses but that of a matched
    /// substring. Substrings are matched in reverse order of length, so
    /// 0 is the longest match.
    ///
    /// The string is fetched from the string passed to the match function,
    /// so you cannot call this function after freeing the string.
    extern fn g_match_info_fetch(p_match_info: *const MatchInfo, p_match_num: c_int) ?[*:0]u8;
    pub const fetch = g_match_info_fetch;

    /// Bundles up pointers to each of the matching substrings from a match
    /// and stores them in an array of gchar pointers. The first element in
    /// the returned array is the match number 0, i.e. the entire matched
    /// text.
    ///
    /// If a sub pattern didn't match anything (e.g. sub pattern 1, matching
    /// "b" against "(a)?b") then an empty string is inserted.
    ///
    /// If the last match was obtained using the DFA algorithm, that is using
    /// `glib.Regex.matchAll` or `glib.Regex.matchAllFull`, the retrieved
    /// strings are not that matched by sets of parentheses but that of the
    /// matched substring. Substrings are matched in reverse order of length,
    /// so the first one is the longest match.
    ///
    /// The strings are fetched from the string passed to the match function,
    /// so you cannot call this function after freeing the string.
    extern fn g_match_info_fetch_all(p_match_info: *const MatchInfo) [*:null]?[*:0]u8;
    pub const fetchAll = g_match_info_fetch_all;

    /// Retrieves the text matching the capturing parentheses named `name`.
    ///
    /// If `name` is a valid sub pattern name but it didn't match anything
    /// (e.g. sub pattern `"X"`, matching `"b"` against `"(?P<X>a)?b"`)
    /// then an empty string is returned.
    ///
    /// The string is fetched from the string passed to the match function,
    /// so you cannot call this function after freeing the string.
    extern fn g_match_info_fetch_named(p_match_info: *const MatchInfo, p_name: [*:0]const u8) ?[*:0]u8;
    pub const fetchNamed = g_match_info_fetch_named;

    /// Retrieves the position in bytes of the capturing parentheses named `name`.
    ///
    /// If `name` is a valid sub pattern name but it didn't match anything
    /// (e.g. sub pattern `"X"`, matching `"b"` against `"(?P<X>a)?b"`)
    /// then `start_pos` and `end_pos` are set to -1 and `TRUE` is returned.
    extern fn g_match_info_fetch_named_pos(p_match_info: *const MatchInfo, p_name: [*:0]const u8, p_start_pos: ?*c_int, p_end_pos: ?*c_int) c_int;
    pub const fetchNamedPos = g_match_info_fetch_named_pos;

    /// Retrieves the position in bytes of the `match_num`'th capturing
    /// parentheses. 0 is the full text of the match, 1 is the first
    /// paren set, 2 the second, and so on.
    ///
    /// If `match_num` is a valid sub pattern but it didn't match anything
    /// (e.g. sub pattern 1, matching "b" against "(a)?b") then `start_pos`
    /// and `end_pos` are set to -1 and `TRUE` is returned.
    ///
    /// If the match was obtained using the DFA algorithm, that is using
    /// `glib.Regex.matchAll` or `glib.Regex.matchAllFull`, the retrieved
    /// position is not that of a set of parentheses but that of a matched
    /// substring. Substrings are matched in reverse order of length, so
    /// 0 is the longest match.
    extern fn g_match_info_fetch_pos(p_match_info: *const MatchInfo, p_match_num: c_int, p_start_pos: ?*c_int, p_end_pos: ?*c_int) c_int;
    pub const fetchPos = g_match_info_fetch_pos;

    /// If `match_info` is not `NULL`, calls `glib.MatchInfo.unref`; otherwise does
    /// nothing.
    extern fn g_match_info_free(p_match_info: ?*MatchInfo) void;
    pub const free = g_match_info_free;

    /// Retrieves the number of matched substrings (including substring 0,
    /// that is the whole matched text), so 1 is returned if the pattern
    /// has no substrings in it and 0 is returned if the match failed.
    ///
    /// If the last match was obtained using the DFA algorithm, that is
    /// using `glib.Regex.matchAll` or `glib.Regex.matchAllFull`, the retrieved
    /// count is not that of the number of capturing parentheses but that of
    /// the number of matched substrings.
    extern fn g_match_info_get_match_count(p_match_info: *const MatchInfo) c_int;
    pub const getMatchCount = g_match_info_get_match_count;

    /// Returns `glib.Regex` object used in `match_info`. It belongs to Glib
    /// and must not be freed. Use `glib.Regex.ref` if you need to keep it
    /// after you free `match_info` object.
    extern fn g_match_info_get_regex(p_match_info: *const MatchInfo) *glib.Regex;
    pub const getRegex = g_match_info_get_regex;

    /// Returns the string searched with `match_info`. This is the
    /// string passed to `glib.Regex.match` or `glib.Regex.replace` so
    /// you may not free it before calling this function.
    extern fn g_match_info_get_string(p_match_info: *const MatchInfo) [*:0]const u8;
    pub const getString = g_match_info_get_string;

    /// Usually if the string passed to g_regex_match*() matches as far as
    /// it goes, but is too short to match the entire pattern, `FALSE` is
    /// returned. There are circumstances where it might be helpful to
    /// distinguish this case from other cases in which there is no match.
    ///
    /// Consider, for example, an application where a human is required to
    /// type in data for a field with specific formatting requirements. An
    /// example might be a date in the form ddmmmyy, defined by the pattern
    /// "^\d?\d(jan|feb|mar|apr|may|jun|jul|aug|sep|oct|nov|dec)\d\d$".
    /// If the application sees the user’s keystrokes one by one, and can
    /// check that what has been typed so far is potentially valid, it is
    /// able to raise an error as soon as a mistake is made.
    ///
    /// GRegex supports the concept of partial matching by means of the
    /// `G_REGEX_MATCH_PARTIAL_SOFT` and `G_REGEX_MATCH_PARTIAL_HARD` flags.
    /// When they are used, the return code for
    /// `glib.Regex.match` or `glib.Regex.matchFull` is, as usual, `TRUE`
    /// for a complete match, `FALSE` otherwise. But, when these functions
    /// return `FALSE`, you can check if the match was partial calling
    /// `glib.MatchInfo.isPartialMatch`.
    ///
    /// The difference between `G_REGEX_MATCH_PARTIAL_SOFT` and
    /// `G_REGEX_MATCH_PARTIAL_HARD` is that when a partial match is encountered
    /// with `G_REGEX_MATCH_PARTIAL_SOFT`, matching continues to search for a
    /// possible complete match, while with `G_REGEX_MATCH_PARTIAL_HARD` matching
    /// stops at the partial match.
    /// When both `G_REGEX_MATCH_PARTIAL_SOFT` and `G_REGEX_MATCH_PARTIAL_HARD`
    /// are set, the latter takes precedence.
    ///
    /// There were formerly some restrictions on the pattern for partial matching.
    /// The restrictions no longer apply.
    ///
    /// See pcrepartial(3) for more information on partial matching.
    extern fn g_match_info_is_partial_match(p_match_info: *const MatchInfo) c_int;
    pub const isPartialMatch = g_match_info_is_partial_match;

    /// Returns whether the previous match operation succeeded.
    extern fn g_match_info_matches(p_match_info: *const MatchInfo) c_int;
    pub const matches = g_match_info_matches;

    /// Scans for the next match using the same parameters of the previous
    /// call to `glib.Regex.matchFull` or `glib.Regex.match` that returned
    /// `match_info`.
    ///
    /// The match is done on the string passed to the match function, so you
    /// cannot free it before calling this function.
    extern fn g_match_info_next(p_match_info: *MatchInfo, p_error: ?*?*glib.Error) c_int;
    pub const next = g_match_info_next;

    /// Increases reference count of `match_info` by 1.
    extern fn g_match_info_ref(p_match_info: *MatchInfo) *glib.MatchInfo;
    pub const ref = g_match_info_ref;

    /// Decreases reference count of `match_info` by 1. When reference count drops
    /// to zero, it frees all the memory associated with the match_info structure.
    extern fn g_match_info_unref(p_match_info: *MatchInfo) void;
    pub const unref = g_match_info_unref;

    extern fn g_match_info_get_type() usize;
    pub const getGObjectType = g_match_info_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

pub const MemChunk = opaque {
    extern fn g_mem_chunk_info() void;
    pub const info = g_mem_chunk_info;

    extern fn g_mem_chunk_new(p_name: [*:0]const u8, p_atom_size: c_int, p_area_size: usize, p_type: c_int) *glib.MemChunk;
    pub const new = g_mem_chunk_new;

    extern fn g_mem_chunk_alloc(p_mem_chunk: *MemChunk) ?*anyopaque;
    pub const alloc = g_mem_chunk_alloc;

    extern fn g_mem_chunk_alloc0(p_mem_chunk: *MemChunk) ?*anyopaque;
    pub const alloc0 = g_mem_chunk_alloc0;

    extern fn g_mem_chunk_clean(p_mem_chunk: *MemChunk) void;
    pub const clean = g_mem_chunk_clean;

    extern fn g_mem_chunk_destroy(p_mem_chunk: *MemChunk) void;
    pub const destroy = g_mem_chunk_destroy;

    extern fn g_mem_chunk_free(p_mem_chunk: *MemChunk, p_mem: ?*anyopaque) void;
    pub const free = g_mem_chunk_free;

    extern fn g_mem_chunk_print(p_mem_chunk: *MemChunk) void;
    pub const print = g_mem_chunk_print;

    extern fn g_mem_chunk_reset(p_mem_chunk: *MemChunk) void;
    pub const reset = g_mem_chunk_reset;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A set of functions used to perform memory allocation. The same `glib.MemVTable` must
/// be used for all allocations in the same program; a call to `glib.memSetVtable`,
/// if it exists, should be prior to any use of GLib.
///
/// This functions related to this has been deprecated in 2.46, and no longer work.
pub const MemVTable = extern struct {
    /// function to use for allocating memory.
    f_malloc: ?*const fn (p_n_bytes: usize) callconv(.c) *anyopaque,
    /// function to use for reallocating memory.
    f_realloc: ?*const fn (p_mem: *anyopaque, p_n_bytes: usize) callconv(.c) *anyopaque,
    /// function to use to free memory.
    f_free: ?*const fn (p_mem: *anyopaque) callconv(.c) void,
    /// function to use for allocating zero-filled memory.
    f_calloc: ?*const fn (p_n_blocks: usize, p_n_block_bytes: usize) callconv(.c) *anyopaque,
    /// function to use for allocating memory without a default error handler.
    f_try_malloc: ?*const fn (p_n_bytes: usize) callconv(.c) *anyopaque,
    /// function to use for reallocating memory without a default error handler.
    f_try_realloc: ?*const fn (p_mem: *anyopaque, p_n_bytes: usize) callconv(.c) *anyopaque,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.Node` struct represents one node in a [n-ary tree](data-structures.html`n`-ary-trees).
pub const Node = extern struct {
    /// contains the actual data of the node.
    f_data: ?*anyopaque,
    /// points to the node's next sibling (a sibling is another
    ///        `glib.Node` with the same parent).
    f_next: ?*glib.Node,
    /// points to the node's previous sibling.
    f_prev: ?*glib.Node,
    /// points to the parent of the `glib.Node`, or is `NULL` if the
    ///          `glib.Node` is the root of the tree.
    f_parent: ?*glib.Node,
    /// points to the first child of the `glib.Node`.  The other
    ///            children are accessed by using the `next` pointer of each
    ///            child.
    f_children: ?*glib.Node,

    /// Creates a new `glib.Node` containing the given data.
    /// Used to create the first node in a tree.
    extern fn g_node_new(p_data: ?*anyopaque) *glib.Node;
    pub const new = g_node_new;

    extern fn g_node_pop_allocator() void;
    pub const popAllocator = g_node_pop_allocator;

    extern fn g_node_push_allocator(p_allocator: *glib.Allocator) void;
    pub const pushAllocator = g_node_push_allocator;

    /// Gets the position of the first child of a `glib.Node`
    /// which contains the given data.
    extern fn g_node_child_index(p_node: *Node, p_data: ?*anyopaque) c_int;
    pub const childIndex = g_node_child_index;

    /// Gets the position of a `glib.Node` with respect to its siblings.
    /// `child` must be a child of `node`. The first child is numbered 0,
    /// the second 1, and so on.
    extern fn g_node_child_position(p_node: *Node, p_child: *glib.Node) c_int;
    pub const childPosition = g_node_child_position;

    /// Calls a function for each of the children of a `glib.Node`. Note that it
    /// doesn't descend beneath the child nodes. `func` must not do anything
    /// that would modify the structure of the tree.
    extern fn g_node_children_foreach(p_node: *Node, p_flags: glib.TraverseFlags, p_func: glib.NodeForeachFunc, p_data: ?*anyopaque) void;
    pub const childrenForeach = g_node_children_foreach;

    /// Recursively copies a `glib.Node` (but does not deep-copy the data inside the
    /// nodes, see `glib.Node.copyDeep` if you need that).
    extern fn g_node_copy(p_node: *Node) *glib.Node;
    pub const copy = g_node_copy;

    /// Recursively copies a `glib.Node` and its data.
    extern fn g_node_copy_deep(p_node: *Node, p_copy_func: glib.CopyFunc, p_data: ?*anyopaque) *glib.Node;
    pub const copyDeep = g_node_copy_deep;

    /// Gets the depth of a `glib.Node`.
    ///
    /// If `node` is `NULL` the depth is 0. The root node has a depth of 1.
    /// For the children of the root node the depth is 2. And so on.
    extern fn g_node_depth(p_node: *Node) c_uint;
    pub const depth = g_node_depth;

    /// Removes `root` and its children from the tree, freeing any memory
    /// allocated.
    extern fn g_node_destroy(p_root: *Node) void;
    pub const destroy = g_node_destroy;

    /// Finds a `glib.Node` in a tree.
    extern fn g_node_find(p_root: *Node, p_order: glib.TraverseType, p_flags: glib.TraverseFlags, p_data: ?*anyopaque) *glib.Node;
    pub const find = g_node_find;

    /// Finds the first child of a `glib.Node` with the given data.
    extern fn g_node_find_child(p_node: *Node, p_flags: glib.TraverseFlags, p_data: ?*anyopaque) *glib.Node;
    pub const findChild = g_node_find_child;

    /// Gets the first sibling of a `glib.Node`.
    /// This could possibly be the node itself.
    extern fn g_node_first_sibling(p_node: *Node) *glib.Node;
    pub const firstSibling = g_node_first_sibling;

    /// Gets the root of a tree.
    extern fn g_node_get_root(p_node: *Node) *glib.Node;
    pub const getRoot = g_node_get_root;

    /// Inserts a `glib.Node` beneath the parent at the given position.
    extern fn g_node_insert(p_parent: *Node, p_position: c_int, p_node: *glib.Node) *glib.Node;
    pub const insert = g_node_insert;

    /// Inserts a `glib.Node` beneath the parent after the given sibling.
    extern fn g_node_insert_after(p_parent: *Node, p_sibling: *glib.Node, p_node: *glib.Node) *glib.Node;
    pub const insertAfter = g_node_insert_after;

    /// Inserts a `glib.Node` beneath the parent before the given sibling.
    extern fn g_node_insert_before(p_parent: *Node, p_sibling: *glib.Node, p_node: *glib.Node) *glib.Node;
    pub const insertBefore = g_node_insert_before;

    /// Returns `TRUE` if `node` is an ancestor of `descendant`.
    /// This is true if node is the parent of `descendant`,
    /// or if node is the grandparent of `descendant` etc.
    extern fn g_node_is_ancestor(p_node: *Node, p_descendant: *glib.Node) c_int;
    pub const isAncestor = g_node_is_ancestor;

    /// Gets the last child of a `glib.Node`.
    extern fn g_node_last_child(p_node: *Node) *glib.Node;
    pub const lastChild = g_node_last_child;

    /// Gets the last sibling of a `glib.Node`.
    /// This could possibly be the node itself.
    extern fn g_node_last_sibling(p_node: *Node) *glib.Node;
    pub const lastSibling = g_node_last_sibling;

    /// Gets the maximum height of all branches beneath a `glib.Node`.
    /// This is the maximum distance from the `glib.Node` to all leaf nodes.
    ///
    /// If `root` is `NULL`, 0 is returned. If `root` has no children,
    /// 1 is returned. If `root` has children, 2 is returned. And so on.
    extern fn g_node_max_height(p_root: *Node) c_uint;
    pub const maxHeight = g_node_max_height;

    /// Gets the number of children of a `glib.Node`.
    extern fn g_node_n_children(p_node: *Node) c_uint;
    pub const nChildren = g_node_n_children;

    /// Gets the number of nodes in a tree.
    extern fn g_node_n_nodes(p_root: *Node, p_flags: glib.TraverseFlags) c_uint;
    pub const nNodes = g_node_n_nodes;

    /// Gets a child of a `glib.Node`, using the given index.
    /// The first child is at index 0. If the index is
    /// too big, `NULL` is returned.
    extern fn g_node_nth_child(p_node: *Node, p_n: c_uint) *glib.Node;
    pub const nthChild = g_node_nth_child;

    /// Inserts a `glib.Node` as the first child of the given parent.
    extern fn g_node_prepend(p_parent: *Node, p_node: *glib.Node) *glib.Node;
    pub const prepend = g_node_prepend;

    /// Reverses the order of the children of a `glib.Node`.
    /// (It doesn't change the order of the grandchildren.)
    extern fn g_node_reverse_children(p_node: *Node) void;
    pub const reverseChildren = g_node_reverse_children;

    /// Traverses a tree starting at the given root `glib.Node`.
    /// It calls the given function for each node visited.
    /// The traversal can be halted at any point by returning `TRUE` from `func`.
    /// `func` must not do anything that would modify the structure of the tree.
    extern fn g_node_traverse(p_root: *Node, p_order: glib.TraverseType, p_flags: glib.TraverseFlags, p_max_depth: c_int, p_func: glib.NodeTraverseFunc, p_data: ?*anyopaque) void;
    pub const traverse = g_node_traverse;

    /// Unlinks a `glib.Node` from a tree, resulting in two separate trees.
    extern fn g_node_unlink(p_node: *Node) void;
    pub const unlink = g_node_unlink;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `glib.Once` struct controls a one-time initialization function. Any
/// one-time initialization function must have its own unique `glib.Once`
/// struct.
pub const Once = extern struct {
    /// the status of the `glib.Once`
    f_status: glib.OnceStatus,
    /// the value returned by the call to the function, if `status`
    ///          is `G_ONCE_STATUS_READY`
    f_retval: ?*anyopaque,

    /// Function to be called when starting a critical initialization
    /// section. The argument `location` must point to a static
    /// 0-initialized variable that will be set to a value other than 0 at
    /// the end of the initialization section. In combination with
    /// `glib.onceInitLeave` and the unique address `value_location`, it can
    /// be ensured that an initialization section will be executed only once
    /// during a program's life time, and that concurrent threads are
    /// blocked until initialization completed. To be used in constructs
    /// like this:
    ///
    /// ```
    ///   static gsize initialization_value = 0;
    ///
    ///   if (g_once_init_enter (&initialization_value))
    ///     {
    ///       gsize setup_value = 42; // initialization code here
    ///
    ///       g_once_init_leave (&initialization_value, setup_value);
    ///     }
    ///
    ///   // use initialization_value here
    /// ```
    ///
    /// While `location` has a `volatile` qualifier, this is a historical artifact and
    /// the pointer passed to it should not be `volatile`.
    extern fn g_once_init_enter(p_location: *anyopaque) c_int;
    pub const initEnter = g_once_init_enter;

    extern fn g_once_init_enter_impl(p_location: *usize) c_int;
    pub const initEnterImpl = g_once_init_enter_impl;

    /// This functions behaves in the same way as `glib.onceInitEnter`, but can
    /// can be used to initialize pointers (or `guintptr`) instead of `gsize`.
    ///
    /// ```
    ///   static MyStruct *interesting_struct = NULL;
    ///
    ///   if (g_once_init_enter_pointer (&interesting_struct))
    ///     {
    ///       MyStruct *setup_value = allocate_my_struct (); // initialization code here
    ///
    ///       g_once_init_leave_pointer (&interesting_struct, g_steal_pointer (&setup_value));
    ///     }
    ///
    ///   // use interesting_struct here
    /// ```
    extern fn g_once_init_enter_pointer(p_location: *anyopaque) c_int;
    pub const initEnterPointer = g_once_init_enter_pointer;

    /// Counterpart to `glib.onceInitEnter`. Expects a location of a static
    /// 0-initialized initialization variable, and an initialization value
    /// other than 0. Sets the variable to the initialization value, and
    /// releases concurrent threads blocking in `glib.onceInitEnter` on this
    /// initialization variable.
    ///
    /// While `location` has a `volatile` qualifier, this is a historical artifact and
    /// the pointer passed to it should not be `volatile`.
    extern fn g_once_init_leave(p_location: *anyopaque, p_result: usize) void;
    pub const initLeave = g_once_init_leave;

    /// Counterpart to `glib.onceInitEnterPointer`. Expects a location of a static
    /// `NULL`-initialized initialization variable, and an initialization value
    /// other than `NULL`. Sets the variable to the initialization value, and
    /// releases concurrent threads blocking in `glib.onceInitEnterPointer` on this
    /// initialization variable.
    ///
    /// This functions behaves in the same way as `glib.onceInitLeave`, but
    /// can be used to initialize pointers (or `guintptr`) instead of `gsize`.
    extern fn g_once_init_leave_pointer(p_location: *anyopaque, p_result: ?*anyopaque) void;
    pub const initLeavePointer = g_once_init_leave_pointer;

    extern fn g_once_impl(p_once: *Once, p_func: glib.ThreadFunc, p_arg: ?*anyopaque) ?*anyopaque;
    pub const impl = g_once_impl;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `GOptionContext` struct defines which options
/// are accepted by the commandline option parser. The struct has only private
/// fields and should not be directly accessed.
pub const OptionContext = opaque {
    /// Creates a new option context.
    ///
    /// The `parameter_string` can serve multiple purposes. It can be used
    /// to add descriptions for "rest" arguments, which are not parsed by
    /// the `glib.OptionContext`, typically something like "FILES" or
    /// "FILE1 FILE2...". If you are using `G_OPTION_REMAINING` for
    /// collecting "rest" arguments, GLib handles this automatically by
    /// using the `arg_description` of the corresponding `glib.OptionEntry` in
    /// the usage summary.
    ///
    /// Another usage is to give a short summary of the program
    /// functionality, like " - frob the strings", which will be displayed
    /// in the same line as the usage. For a longer description of the
    /// program functionality that should be displayed as a paragraph
    /// below the usage line, use `glib.OptionContext.setSummary`.
    ///
    /// Note that the `parameter_string` is translated using the
    /// function set with `glib.OptionContext.setTranslateFunc`, so
    /// it should normally be passed untranslated.
    extern fn g_option_context_new(p_parameter_string: ?[*:0]const u8) *glib.OptionContext;
    pub const new = g_option_context_new;

    /// Adds a `glib.OptionGroup` to the `context`, so that parsing with `context`
    /// will recognize the options in the group. Note that this will take
    /// ownership of the `group` and thus the `group` should not be freed.
    extern fn g_option_context_add_group(p_context: *OptionContext, p_group: *glib.OptionGroup) void;
    pub const addGroup = g_option_context_add_group;

    /// A convenience function which creates a main group if it doesn't
    /// exist, adds the `entries` to it and sets the translation domain.
    extern fn g_option_context_add_main_entries(p_context: *OptionContext, p_entries: [*]const glib.OptionEntry, p_translation_domain: ?[*:0]const u8) void;
    pub const addMainEntries = g_option_context_add_main_entries;

    /// Frees context and all the groups which have been
    /// added to it.
    ///
    /// Please note that parsed arguments need to be freed separately (see
    /// `glib.OptionEntry`).
    extern fn g_option_context_free(p_context: *OptionContext) void;
    pub const free = g_option_context_free;

    /// Returns the description. See `glib.OptionContext.setDescription`.
    extern fn g_option_context_get_description(p_context: *OptionContext) [*:0]const u8;
    pub const getDescription = g_option_context_get_description;

    /// Returns a formatted, translated help text for the given context.
    /// To obtain the text produced by `--help`, call
    /// `g_option_context_get_help (context, TRUE, NULL)`.
    /// To obtain the text produced by `--help-all`, call
    /// `g_option_context_get_help (context, FALSE, NULL)`.
    /// To obtain the help text for an option group, call
    /// `g_option_context_get_help (context, FALSE, group)`.
    extern fn g_option_context_get_help(p_context: *OptionContext, p_main_help: c_int, p_group: ?*glib.OptionGroup) [*:0]u8;
    pub const getHelp = g_option_context_get_help;

    /// Returns whether automatic `--help` generation
    /// is turned on for `context`. See `glib.OptionContext.setHelpEnabled`.
    extern fn g_option_context_get_help_enabled(p_context: *OptionContext) c_int;
    pub const getHelpEnabled = g_option_context_get_help_enabled;

    /// Returns whether unknown options are ignored or not. See
    /// `glib.OptionContext.setIgnoreUnknownOptions`.
    extern fn g_option_context_get_ignore_unknown_options(p_context: *OptionContext) c_int;
    pub const getIgnoreUnknownOptions = g_option_context_get_ignore_unknown_options;

    /// Returns a pointer to the main group of `context`.
    extern fn g_option_context_get_main_group(p_context: *OptionContext) *glib.OptionGroup;
    pub const getMainGroup = g_option_context_get_main_group;

    /// Returns whether strict POSIX code is enabled.
    ///
    /// See `glib.OptionContext.setStrictPosix` for more information.
    extern fn g_option_context_get_strict_posix(p_context: *OptionContext) c_int;
    pub const getStrictPosix = g_option_context_get_strict_posix;

    /// Returns the summary. See `glib.OptionContext.setSummary`.
    extern fn g_option_context_get_summary(p_context: *OptionContext) [*:0]const u8;
    pub const getSummary = g_option_context_get_summary;

    /// Parses the command line arguments, recognizing options
    /// which have been added to `context`. A side-effect of
    /// calling this function is that `glib.setPrgname` will be
    /// called.
    ///
    /// If the parsing is successful, any parsed arguments are
    /// removed from the array and `argc` and `argv` are updated
    /// accordingly. A '--' option is stripped from `argv`
    /// unless there are unparsed options before and after it,
    /// or some of the options after it start with '-'. In case
    /// of an error, `argc` and `argv` are left unmodified.
    ///
    /// If automatic `--help` support is enabled
    /// (see `glib.OptionContext.setHelpEnabled`), and the
    /// `argv` array contains one of the recognized help options,
    /// this function will produce help output to stdout and
    /// call `exit (0)`.
    ///
    /// Note that function depends on the
    /// [current locale](running.html`locale`) for automatic
    /// character set conversion of string and filename arguments.
    extern fn g_option_context_parse(p_context: *OptionContext, p_argc: ?*c_int, p_argv: ?*[*][*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const parse = g_option_context_parse;

    /// Parses the command line arguments.
    ///
    /// This function is similar to `glib.OptionContext.parse` except that it
    /// respects the normal memory rules when dealing with a strv instead of
    /// assuming that the passed-in array is the argv of the main function.
    ///
    /// In particular, strings that are removed from the arguments list will
    /// be freed using `glib.free`.
    ///
    /// On Windows, the strings are expected to be in UTF-8.  This is in
    /// contrast to `glib.OptionContext.parse` which expects them to be in the
    /// system codepage, which is how they are passed as `argv` to `main`.
    /// See `g_win32_get_command_line` for a solution.
    ///
    /// This function is useful if you are trying to use `glib.OptionContext` with
    /// `GApplication`.
    extern fn g_option_context_parse_strv(p_context: *OptionContext, p_arguments: ?*[*][*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const parseStrv = g_option_context_parse_strv;

    /// Adds a string to be displayed in `--help` output after the list
    /// of options. This text often includes a bug reporting address.
    ///
    /// Note that the summary is translated (see
    /// `glib.OptionContext.setTranslateFunc`).
    extern fn g_option_context_set_description(p_context: *OptionContext, p_description: ?[*:0]const u8) void;
    pub const setDescription = g_option_context_set_description;

    /// Enables or disables automatic generation of `--help` output.
    /// By default, `glib.OptionContext.parse` recognizes `--help`, `-h`,
    /// `-?`, `--help-all` and `--help-groupname` and creates suitable
    /// output to stdout.
    extern fn g_option_context_set_help_enabled(p_context: *OptionContext, p_help_enabled: c_int) void;
    pub const setHelpEnabled = g_option_context_set_help_enabled;

    /// Sets whether to ignore unknown options or not. If an argument is
    /// ignored, it is left in the `argv` array after parsing. By default,
    /// `glib.OptionContext.parse` treats unknown options as error.
    ///
    /// This setting does not affect non-option arguments (i.e. arguments
    /// which don't start with a dash). But note that GOption cannot reliably
    /// determine whether a non-option belongs to a preceding unknown option.
    extern fn g_option_context_set_ignore_unknown_options(p_context: *OptionContext, p_ignore_unknown: c_int) void;
    pub const setIgnoreUnknownOptions = g_option_context_set_ignore_unknown_options;

    /// Sets a `glib.OptionGroup` as main group of the `context`.
    /// This has the same effect as calling `glib.OptionContext.addGroup`,
    /// the only difference is that the options in the main group are
    /// treated differently when generating `--help` output.
    extern fn g_option_context_set_main_group(p_context: *OptionContext, p_group: *glib.OptionGroup) void;
    pub const setMainGroup = g_option_context_set_main_group;

    /// Sets strict POSIX mode.
    ///
    /// By default, this mode is disabled.
    ///
    /// In strict POSIX mode, the first non-argument parameter encountered
    /// (eg: filename) terminates argument processing.  Remaining arguments
    /// are treated as non-options and are not attempted to be parsed.
    ///
    /// If strict POSIX mode is disabled then parsing is done in the GNU way
    /// where option arguments can be freely mixed with non-options.
    ///
    /// As an example, consider "ls foo -l".  With GNU style parsing, this
    /// will list "foo" in long mode.  In strict POSIX style, this will list
    /// the files named "foo" and "-l".
    ///
    /// It may be useful to force strict POSIX mode when creating "verb
    /// style" command line tools.  For example, the "gsettings" command line
    /// tool supports the global option "--schemadir" as well as many
    /// subcommands ("get", "set", etc.) which each have their own set of
    /// arguments.  Using strict POSIX mode will allow parsing the global
    /// options up to the verb name while leaving the remaining options to be
    /// parsed by the relevant subcommand (which can be determined by
    /// examining the verb name, which should be present in argv[1] after
    /// parsing).
    extern fn g_option_context_set_strict_posix(p_context: *OptionContext, p_strict_posix: c_int) void;
    pub const setStrictPosix = g_option_context_set_strict_posix;

    /// Adds a string to be displayed in `--help` output before the list
    /// of options. This is typically a summary of the program functionality.
    ///
    /// Note that the summary is translated (see
    /// `glib.OptionContext.setTranslateFunc` and
    /// `glib.OptionContext.setTranslationDomain`).
    extern fn g_option_context_set_summary(p_context: *OptionContext, p_summary: ?[*:0]const u8) void;
    pub const setSummary = g_option_context_set_summary;

    /// Sets the function which is used to translate the contexts
    /// user-visible strings, for `--help` output. If `func` is `NULL`,
    /// strings are not translated.
    ///
    /// Note that option groups have their own translation functions,
    /// this function only affects the `parameter_string` (see `glib.OptionContext.new`),
    /// the summary (see `glib.OptionContext.setSummary`) and the description
    /// (see `glib.OptionContext.setDescription`).
    ///
    /// If you are using `gettext`, you only need to set the translation
    /// domain, see `glib.OptionContext.setTranslationDomain`.
    extern fn g_option_context_set_translate_func(p_context: *OptionContext, p_func: ?glib.TranslateFunc, p_data: ?*anyopaque, p_destroy_notify: ?glib.DestroyNotify) void;
    pub const setTranslateFunc = g_option_context_set_translate_func;

    /// A convenience function to use `gettext` for translating
    /// user-visible strings.
    extern fn g_option_context_set_translation_domain(p_context: *OptionContext, p_domain: [*:0]const u8) void;
    pub const setTranslationDomain = g_option_context_set_translation_domain;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// - `G_OPTION_ARG_NONE`: `gboolean`
///     - `G_OPTION_ARG_STRING`: `gchar`*
///     - `G_OPTION_ARG_INT`: `gint`
///     - `G_OPTION_ARG_FILENAME`: `gchar`*
///     - `G_OPTION_ARG_STRING_ARRAY`: `gchar`**
///     - `G_OPTION_ARG_FILENAME_ARRAY`: `gchar`**
///     - `G_OPTION_ARG_DOUBLE`: `gdouble`
///
///     If `arg` type is `G_OPTION_ARG_STRING` or `G_OPTION_ARG_FILENAME`,
///     the location will contain a newly allocated string if the option
///     was given. That string needs to be freed by the callee using `glib.free`.
///     Likewise if `arg` type is `G_OPTION_ARG_STRING_ARRAY` or
///     `G_OPTION_ARG_FILENAME_ARRAY`, the data should be freed using `glib.strfreev`.
/// A GOptionEntry struct defines a single option. To have an effect, they
/// must be added to a `glib.OptionGroup` with `glib.OptionContext.addMainEntries`
/// or `glib.OptionGroup.addEntries`.
pub const OptionEntry = extern struct {
    /// The long name of an option can be used to specify it
    ///     in a commandline as `--long_name`. Every option must have a
    ///     long name. To resolve conflicts if multiple option groups contain
    ///     the same long name, it is also possible to specify the option as
    ///     `--groupname-long_name`.
    f_long_name: ?[*:0]const u8,
    /// If an option has a short name, it can be specified
    ///     `-short_name` in a commandline. `short_name` must be  a printable
    ///     ASCII character different from '-', or zero if the option has no
    ///     short name.
    f_short_name: u8,
    /// Flags from `glib.OptionFlags`
    f_flags: c_int,
    /// The type of the option, as a `glib.OptionArg`
    f_arg: glib.OptionArg,
    /// If the `arg` type is `G_OPTION_ARG_CALLBACK`, then `arg_data`
    ///     must point to a `glib.OptionArgFunc` callback function, which will be
    ///     called to handle the extra argument. Otherwise, `arg_data` is a
    ///     pointer to a location to store the value, the required type of
    ///     the location depends on the `arg` type:
    f_arg_data: ?*anyopaque,
    /// the description for the option in `--help`
    ///     output. The `description` is translated using the `translate_func`
    ///     of the group, see `glib.OptionGroup.setTranslationDomain`.
    f_description: ?[*:0]const u8,
    /// The placeholder to use for the extra argument parsed
    ///     by the option in `--help` output. The `arg_description` is translated
    ///     using the `translate_func` of the group, see
    ///     `glib.OptionGroup.setTranslationDomain`.
    f_arg_description: ?[*:0]const u8,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `GOptionGroup` struct defines the options in a single
/// group. The struct has only private fields and should not be directly accessed.
///
/// All options in a group share the same translation function. Libraries which
/// need to parse commandline options are expected to provide a function for
/// getting a `GOptionGroup` holding their options, which
/// the application can then add to its `glib.OptionContext`.
pub const OptionGroup = opaque {
    /// Creates a new `glib.OptionGroup`.
    ///
    /// `description` is typically used to provide a title for the group. If so, it
    /// is recommended that it’s written in title case, and has a trailing colon so
    /// that it matches the style of built-in GLib group titles such as
    /// ‘Application Options:’.
    extern fn g_option_group_new(p_name: [*:0]const u8, p_description: [*:0]const u8, p_help_description: [*:0]const u8, p_user_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) *glib.OptionGroup;
    pub const new = g_option_group_new;

    /// Adds the options specified in `entries` to `group`.
    extern fn g_option_group_add_entries(p_group: *OptionGroup, p_entries: [*]const glib.OptionEntry) void;
    pub const addEntries = g_option_group_add_entries;

    /// Frees a `glib.OptionGroup`. Note that you must not free groups
    /// which have been added to a `glib.OptionContext`.
    extern fn g_option_group_free(p_group: *OptionGroup) void;
    pub const free = g_option_group_free;

    /// Increments the reference count of `group` by one.
    extern fn g_option_group_ref(p_group: *OptionGroup) *glib.OptionGroup;
    pub const ref = g_option_group_ref;

    /// Associates a function with `group` which will be called
    /// from `glib.OptionContext.parse` when an error occurs.
    ///
    /// Note that the user data to be passed to `error_func` can be
    /// specified when constructing the group with `glib.OptionGroup.new`.
    extern fn g_option_group_set_error_hook(p_group: *OptionGroup, p_error_func: glib.OptionErrorFunc) void;
    pub const setErrorHook = g_option_group_set_error_hook;

    /// Associates two functions with `group` which will be called
    /// from `glib.OptionContext.parse` before the first option is parsed
    /// and after the last option has been parsed, respectively.
    ///
    /// Note that the user data to be passed to `pre_parse_func` and
    /// `post_parse_func` can be specified when constructing the group
    /// with `glib.OptionGroup.new`.
    extern fn g_option_group_set_parse_hooks(p_group: *OptionGroup, p_pre_parse_func: ?glib.OptionParseFunc, p_post_parse_func: ?glib.OptionParseFunc) void;
    pub const setParseHooks = g_option_group_set_parse_hooks;

    /// Sets the function which is used to translate user-visible strings,
    /// for `--help` output. Different groups can use different
    /// `GTranslateFuncs`. If `func` is `NULL`, strings are not translated.
    ///
    /// If you are using `gettext`, you only need to set the translation
    /// domain, see `glib.OptionGroup.setTranslationDomain`.
    extern fn g_option_group_set_translate_func(p_group: *OptionGroup, p_func: ?glib.TranslateFunc, p_data: ?*anyopaque, p_destroy_notify: ?glib.DestroyNotify) void;
    pub const setTranslateFunc = g_option_group_set_translate_func;

    /// A convenience function to use `gettext` for translating
    /// user-visible strings.
    extern fn g_option_group_set_translation_domain(p_group: *OptionGroup, p_domain: [*:0]const u8) void;
    pub const setTranslationDomain = g_option_group_set_translation_domain;

    /// Decrements the reference count of `group` by one.
    /// If the reference count drops to 0, the `group` will be freed.
    /// and all memory allocated by the `group` is released.
    extern fn g_option_group_unref(p_group: *OptionGroup) void;
    pub const unref = g_option_group_unref;

    extern fn g_option_group_get_type() usize;
    pub const getGObjectType = g_option_group_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GPathBuf` is a helper type that allows you to easily build paths from
/// individual elements, using the platform specific conventions for path
/// separators.
///
/// ```c
/// g_auto (GPathBuf) path;
///
/// g_path_buf_init (&path);
///
/// g_path_buf_push (&path, "usr");
/// g_path_buf_push (&path, "bin");
/// g_path_buf_push (&path, "echo");
///
/// g_autofree char *echo = g_path_buf_to_path (&path);
/// g_assert_cmpstr (echo, ==, "/usr/bin/echo");
/// ```
///
/// You can also load a full path and then operate on its components:
///
/// ```c
/// g_auto (GPathBuf) path;
///
/// g_path_buf_init_from_path (&path, "/usr/bin/echo");
///
/// g_path_buf_pop (&path);
/// g_path_buf_push (&path, "sh");
///
/// g_autofree char *sh = g_path_buf_to_path (&path);
/// g_assert_cmpstr (sh, ==, "/usr/bin/sh");
/// ```
pub const PathBuf = extern struct {
    f_dummy: [8]*anyopaque,

    /// Compares two path buffers for equality and returns `TRUE`
    /// if they are equal.
    ///
    /// The path inside the paths buffers are not going to be normalized,
    /// so `X/Y/Z/A/..`, `X/./Y/Z` and `X/Y/Z` are not going to be considered
    /// equal.
    ///
    /// This function can be passed to `glib.HashTable.new` as the
    /// `key_equal_func` parameter.
    extern fn g_path_buf_equal(p_v1: *const anyopaque, p_v2: *const anyopaque) c_int;
    pub const equal = g_path_buf_equal;

    /// Allocates a new `GPathBuf`.
    extern fn g_path_buf_new() *glib.PathBuf;
    pub const new = g_path_buf_new;

    /// Allocates a new `GPathBuf` with the given `path`.
    extern fn g_path_buf_new_from_path(p_path: ?[*:0]const u8) *glib.PathBuf;
    pub const newFromPath = g_path_buf_new_from_path;

    /// Clears the contents of the path buffer.
    ///
    /// This function should be use to free the resources in a stack-allocated
    /// `GPathBuf` initialized using `glib.PathBuf.init` or
    /// `glib.PathBuf.initFromPath`.
    extern fn g_path_buf_clear(p_buf: *PathBuf) void;
    pub const clear = g_path_buf_clear;

    /// Clears the contents of the path buffer and returns the built path.
    ///
    /// This function returns `NULL` if the `GPathBuf` is empty.
    ///
    /// See also: `glib.PathBuf.toPath`
    extern fn g_path_buf_clear_to_path(p_buf: *PathBuf) ?[*:0]u8;
    pub const clearToPath = g_path_buf_clear_to_path;

    /// Copies the contents of a path buffer into a new `GPathBuf`.
    extern fn g_path_buf_copy(p_buf: *PathBuf) *glib.PathBuf;
    pub const copy = g_path_buf_copy;

    /// Frees a `GPathBuf` allocated by `glib.PathBuf.new`.
    extern fn g_path_buf_free(p_buf: *PathBuf) void;
    pub const free = g_path_buf_free;

    /// Frees a `GPathBuf` allocated by `glib.PathBuf.new`, and
    /// returns the path inside the buffer.
    ///
    /// This function returns `NULL` if the `GPathBuf` is empty.
    ///
    /// See also: `glib.PathBuf.toPath`
    extern fn g_path_buf_free_to_path(p_buf: *PathBuf) ?[*:0]u8;
    pub const freeToPath = g_path_buf_free_to_path;

    /// Initializes a `GPathBuf` instance.
    extern fn g_path_buf_init(p_buf: *PathBuf) *glib.PathBuf;
    pub const init = g_path_buf_init;

    /// Initializes a `GPathBuf` instance with the given path.
    extern fn g_path_buf_init_from_path(p_buf: *PathBuf, p_path: ?[*:0]const u8) *glib.PathBuf;
    pub const initFromPath = g_path_buf_init_from_path;

    /// Removes the last element of the path buffer.
    ///
    /// If there is only one element in the path buffer (for example, `/` on
    /// Unix-like operating systems or the drive on Windows systems), it will
    /// not be removed and `FALSE` will be returned instead.
    ///
    /// ```
    /// GPathBuf buf, cmp;
    ///
    /// g_path_buf_init_from_path (&buf, "/bin/sh");
    ///
    /// g_path_buf_pop (&buf);
    /// g_path_buf_init_from_path (&cmp, "/bin");
    /// g_assert_true (g_path_buf_equal (&buf, &cmp));
    /// g_path_buf_clear (&cmp);
    ///
    /// g_path_buf_pop (&buf);
    /// g_path_buf_init_from_path (&cmp, "/");
    /// g_assert_true (g_path_buf_equal (&buf, &cmp));
    /// g_path_buf_clear (&cmp);
    ///
    /// g_path_buf_clear (&buf);
    /// ```
    extern fn g_path_buf_pop(p_buf: *PathBuf) c_int;
    pub const pop = g_path_buf_pop;

    /// Extends the given path buffer with `path`.
    ///
    /// If `path` is absolute, it replaces the current path.
    ///
    /// If `path` contains a directory separator, the buffer is extended by
    /// as many elements the path provides.
    ///
    /// On Windows, both forward slashes and backslashes are treated as
    /// directory separators. On other platforms, `G_DIR_SEPARATOR_S` is the
    /// only directory separator.
    ///
    /// ```
    /// GPathBuf buf, cmp;
    ///
    /// g_path_buf_init_from_path (&buf, "/tmp");
    /// g_path_buf_push (&buf, ".X11-unix/X0");
    /// g_path_buf_init_from_path (&cmp, "/tmp/.X11-unix/X0");
    /// g_assert_true (g_path_buf_equal (&buf, &cmp));
    /// g_path_buf_clear (&cmp);
    ///
    /// g_path_buf_push (&buf, "/etc/locale.conf");
    /// g_path_buf_init_from_path (&cmp, "/etc/locale.conf");
    /// g_assert_true (g_path_buf_equal (&buf, &cmp));
    /// g_path_buf_clear (&cmp);
    ///
    /// g_path_buf_clear (&buf);
    /// ```
    extern fn g_path_buf_push(p_buf: *PathBuf, p_path: [*:0]const u8) *glib.PathBuf;
    pub const push = g_path_buf_push;

    /// Adds an extension to the file name in the path buffer.
    ///
    /// If `extension` is `NULL`, the extension will be unset.
    ///
    /// If the path buffer does not have a file name set, this function returns
    /// `FALSE` and leaves the path buffer unmodified.
    extern fn g_path_buf_set_extension(p_buf: *PathBuf, p_extension: ?[*:0]const u8) c_int;
    pub const setExtension = g_path_buf_set_extension;

    /// Sets the file name of the path.
    ///
    /// If the path buffer is empty, the filename is left unset and this
    /// function returns `FALSE`.
    ///
    /// If the path buffer only contains the root element (on Unix-like operating
    /// systems) or the drive (on Windows), this is the equivalent of pushing
    /// the new `file_name`.
    ///
    /// If the path buffer contains a path, this is the equivalent of
    /// popping the path buffer and pushing `file_name`, creating a
    /// sibling of the original path.
    ///
    /// ```
    /// GPathBuf buf, cmp;
    ///
    /// g_path_buf_init_from_path (&buf, "/");
    ///
    /// g_path_buf_set_filename (&buf, "bar");
    /// g_path_buf_init_from_path (&cmp, "/bar");
    /// g_assert_true (g_path_buf_equal (&buf, &cmp));
    /// g_path_buf_clear (&cmp);
    ///
    /// g_path_buf_set_filename (&buf, "baz.txt");
    /// g_path_buf_init_from_path (&cmp, "/baz.txt");
    /// g_assert_true (g_path_buf_equal (&buf, &cmp);
    /// g_path_buf_clear (&cmp);
    ///
    /// g_path_buf_clear (&buf);
    /// ```
    extern fn g_path_buf_set_filename(p_buf: *PathBuf, p_file_name: [*:0]const u8) c_int;
    pub const setFilename = g_path_buf_set_filename;

    /// Retrieves the built path from the path buffer.
    ///
    /// On Windows, the result contains backslashes as directory separators,
    /// even if forward slashes were used in input.
    ///
    /// If the path buffer is empty, this function returns `NULL`.
    extern fn g_path_buf_to_path(p_buf: *PathBuf) ?[*:0]u8;
    pub const toPath = g_path_buf_to_path;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `GPatternSpec` struct is the ‘compiled’ form of a glob-style pattern.
///
/// The `glib.patternMatchSimple` and `glib.PatternSpec.match` functions
/// match a string against a pattern containing `*` and `?` wildcards with similar
/// semantics as the standard ``glob`` function: `*` matches an arbitrary,
/// possibly empty, string, `?` matches an arbitrary character.
///
/// Note that in contrast to [``glob``](man:glob(3)), the `/` character can be
/// matched by the wildcards, there are no `[…]` character ranges and `*` and `?`
/// can not be escaped to include them literally in a pattern.
///
/// When multiple strings must be matched against the same pattern, it is better
/// to compile the pattern to a `glib.PatternSpec` using
/// `glib.PatternSpec.new` and use `glib.PatternSpec.matchString`
/// instead of `glib.patternMatchSimple`. This avoids the overhead of repeated
/// pattern compilation.
pub const PatternSpec = opaque {
    /// Compiles a pattern to a `glib.PatternSpec`.
    extern fn g_pattern_spec_new(p_pattern: [*:0]const u8) *glib.PatternSpec;
    pub const new = g_pattern_spec_new;

    /// Copies `pspec` in a new `glib.PatternSpec`.
    extern fn g_pattern_spec_copy(p_pspec: *PatternSpec) *glib.PatternSpec;
    pub const copy = g_pattern_spec_copy;

    /// Compares two compiled pattern specs and returns whether they will
    /// match the same set of strings.
    extern fn g_pattern_spec_equal(p_pspec1: *PatternSpec, p_pspec2: *glib.PatternSpec) c_int;
    pub const equal = g_pattern_spec_equal;

    /// Frees the memory allocated for the `glib.PatternSpec`.
    extern fn g_pattern_spec_free(p_pspec: *PatternSpec) void;
    pub const free = g_pattern_spec_free;

    /// Matches a string against a compiled pattern.
    ///
    /// Passing the correct
    /// length of the string given is mandatory. The reversed string can be
    /// omitted by passing `NULL`, this is more efficient if the reversed
    /// version of the string to be matched is not at hand, as
    /// `glib.PatternSpec.match` will only construct it if the compiled pattern
    /// requires reverse matches.
    ///
    /// Note that, if the user code will (possibly) match a string against a
    /// multitude of patterns containing wildcards, chances are high that
    /// some patterns will require a reversed string. In this case, it’s
    /// more efficient to provide the reversed string to avoid multiple
    /// constructions thereof in the various calls to `glib.PatternSpec.match`.
    ///
    /// Note also that the reverse of a UTF-8 encoded string can in general
    /// not be obtained by `glib.strreverse`. This works only if the string
    /// does not contain any multibyte characters. GLib offers the
    /// `glib.utf8Strreverse` function to reverse UTF-8 encoded strings.
    extern fn g_pattern_spec_match(p_pspec: *PatternSpec, p_string_length: usize, p_string: [*:0]const u8, p_string_reversed: ?[*:0]const u8) c_int;
    pub const match = g_pattern_spec_match;

    /// Matches a string against a compiled pattern.
    ///
    /// If the string is to be
    /// matched against more than one pattern, consider using
    /// `glib.PatternSpec.match` instead while supplying the reversed string.
    extern fn g_pattern_spec_match_string(p_pspec: *PatternSpec, p_string: [*:0]const u8) c_int;
    pub const matchString = g_pattern_spec_match_string;

    extern fn g_pattern_spec_get_type() usize;
    pub const getGObjectType = g_pattern_spec_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Represents a file descriptor, which events to poll for, and which events
/// occurred.
pub const PollFD = extern struct {
    /// the file descriptor to poll (or a HANDLE on Win32)
    f_fd: c_int,
    /// a bitwise combination from `glib.IOCondition`, specifying which
    ///     events should be polled for. Typically for reading from a file
    ///     descriptor you would use `G_IO_IN` | `G_IO_HUP` | `G_IO_ERR`, and
    ///     for writing you would use `G_IO_OUT` | `G_IO_ERR`.
    f_events: c_ushort,
    /// a bitwise combination of flags from `glib.IOCondition`, returned
    ///     from the `poll` function to indicate which events occurred.
    f_revents: c_ushort,

    extern fn g_pollfd_get_type() usize;
    pub const getGObjectType = g_pollfd_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.Private` struct is an opaque data structure to represent a
/// thread-local data key. It is approximately equivalent to the
/// `pthread_setspecific`/`pthread_getspecific` APIs on POSIX and to
/// `TlsSetValue`/`TlsGetValue` on Windows.
///
/// If you don't already know why you might want this functionality,
/// then you probably don't need it.
///
/// `glib.Private` is a very limited resource (as far as 128 per program,
/// shared between all libraries). It is also not possible to destroy a
/// `glib.Private` after it has been used. As such, it is only ever acceptable
/// to use `glib.Private` in static scope, and even then sparingly so.
///
/// See `G_PRIVATE_INIT` for a couple of examples.
///
/// The `glib.Private` structure should be considered opaque.  It should only
/// be accessed via the g_private_ functions.
pub const Private = extern struct {
    f_p: ?*anyopaque,
    f_notify: ?glib.DestroyNotify,
    f_future: [2]*anyopaque,

    /// Creates a new `glib.Private`.
    extern fn g_private_new(p_notify: glib.DestroyNotify) *glib.Private;
    pub const new = g_private_new;

    /// Returns the current value of the thread local variable `key`.
    ///
    /// If the value has not yet been set in this thread, `NULL` is returned.
    /// Values are never copied between threads (when a new thread is
    /// created, for example).
    extern fn g_private_get(p_key: *Private) ?*anyopaque;
    pub const get = g_private_get;

    /// Sets the thread local variable `key` to have the value `value` in the
    /// current thread.
    ///
    /// This function differs from `glib.Private.set` in the following way: if
    /// the previous value was non-`NULL` then the `glib.DestroyNotify` handler for
    /// `key` is run on it.
    extern fn g_private_replace(p_key: *Private, p_value: ?*anyopaque) void;
    pub const replace = g_private_replace;

    /// Sets the thread local variable `key` to have the value `value` in the
    /// current thread.
    ///
    /// This function differs from `glib.Private.replace` in the following way:
    /// the `glib.DestroyNotify` for `key` is not called on the old value.
    extern fn g_private_set(p_key: *Private, p_value: ?*anyopaque) void;
    pub const set = g_private_set;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Contains the public fields of a pointer array.
pub const PtrArray = extern struct {
    /// points to the array of pointers, which may be moved when the
    ///     array grows
    f_pdata: ?*anyopaque,
    /// number of pointers in the array
    f_len: c_uint,

    /// Adds a pointer to the end of the pointer array. The array will grow
    /// in size automatically if necessary.
    extern fn g_ptr_array_add(p_array: *glib.PtrArray, p_data: ?*anyopaque) void;
    pub const add = g_ptr_array_add;

    /// Makes a full (deep) copy of a `glib.PtrArray`.
    ///
    /// `func`, as a `glib.CopyFunc`, takes two arguments, the data to be copied
    /// and a `user_data` pointer. On common processor architectures, it's safe to
    /// pass `NULL` as `user_data` if the copy function takes only one argument. You
    /// may get compiler warnings from this though if compiling with GCC’s
    /// `-Wcast-function-type` warning.
    ///
    /// If `func` is `NULL`, then only the pointers (and not what they are
    /// pointing to) are copied to the new `glib.PtrArray`.
    ///
    /// The copy of `array` will have the same `glib.DestroyNotify` for its elements as
    /// `array`. The copy will also be `NULL` terminated if (and only if) the source
    /// array is.
    extern fn g_ptr_array_copy(p_array: *glib.PtrArray, p_func: ?glib.CopyFunc, p_user_data: ?*anyopaque) *glib.PtrArray;
    pub const copy = g_ptr_array_copy;

    /// Adds all pointers of `array` to the end of the array `array_to_extend`.
    /// The array will grow in size automatically if needed. `array_to_extend` is
    /// modified in-place.
    ///
    /// `func`, as a `glib.CopyFunc`, takes two arguments, the data to be copied
    /// and a `user_data` pointer. On common processor architectures, it's safe to
    /// pass `NULL` as `user_data` if the copy function takes only one argument. You
    /// may get compiler warnings from this though if compiling with GCC’s
    /// `-Wcast-function-type` warning.
    ///
    /// If `func` is `NULL`, then only the pointers (and not what they are
    /// pointing to) are copied to the new `glib.PtrArray`.
    ///
    /// Whether `array_to_extend` is `NULL` terminated stays unchanged by this function.
    extern fn g_ptr_array_extend(p_array_to_extend: *glib.PtrArray, p_array: *glib.PtrArray, p_func: ?glib.CopyFunc, p_user_data: ?*anyopaque) void;
    pub const extend = g_ptr_array_extend;

    /// Adds all the pointers in `array` to the end of `array_to_extend`, transferring
    /// ownership of each element from `array` to `array_to_extend` and modifying
    /// `array_to_extend` in-place. `array` is then freed.
    ///
    /// As with `glib.PtrArray.free`, `array` will be destroyed if its reference count
    /// is 1. If its reference count is higher, it will be decremented and the
    /// length of `array` set to zero.
    extern fn g_ptr_array_extend_and_steal(p_array_to_extend: *glib.PtrArray, p_array: *glib.PtrArray) void;
    pub const extendAndSteal = g_ptr_array_extend_and_steal;

    /// Checks whether `needle` exists in `haystack`. If the element is found, `TRUE` is
    /// returned and the element’s index is returned in `index_` (if non-`NULL`).
    /// Otherwise, `FALSE` is returned and `index_` is undefined. If `needle` exists
    /// multiple times in `haystack`, the index of the first instance is returned.
    ///
    /// This does pointer comparisons only. If you want to use more complex equality
    /// checks, such as string comparisons, use `glib.ptrArrayFindWithEqualFunc`.
    extern fn g_ptr_array_find(p_haystack: *glib.PtrArray, p_needle: ?*const anyopaque, p_index_: ?*c_uint) c_int;
    pub const find = g_ptr_array_find;

    /// Checks whether `needle` exists in `haystack`, using the given `equal_func`.
    /// If the element is found, `TRUE` is returned and the element’s index is
    /// returned in `index_` (if non-`NULL`). Otherwise, `FALSE` is returned and `index_`
    /// is undefined. If `needle` exists multiple times in `haystack`, the index of
    /// the first instance is returned.
    ///
    /// `equal_func` is called with the element from the array as its first parameter,
    /// and `needle` as its second parameter. If `equal_func` is `NULL`, pointer
    /// equality is used.
    extern fn g_ptr_array_find_with_equal_func(p_haystack: *glib.PtrArray, p_needle: ?*const anyopaque, p_equal_func: ?glib.EqualFunc, p_index_: ?*c_uint) c_int;
    pub const findWithEqualFunc = g_ptr_array_find_with_equal_func;

    /// Calls a function for each element of a `glib.PtrArray`. `func` must not
    /// add elements to or remove elements from the array.
    extern fn g_ptr_array_foreach(p_array: *glib.PtrArray, p_func: glib.Func, p_user_data: ?*anyopaque) void;
    pub const foreach = g_ptr_array_foreach;

    /// Frees the memory allocated for the `glib.PtrArray`. If `free_segment` is `TRUE`
    /// it frees the memory block holding the elements as well. Pass `FALSE`
    /// if you want to free the `glib.PtrArray` wrapper but preserve the
    /// underlying array for use elsewhere. If the reference count of `array`
    /// is greater than one, the `glib.PtrArray` wrapper is preserved but the
    /// size of `array` will be set to zero.
    ///
    /// If array contents point to dynamically-allocated memory, they should
    /// be freed separately if `free_segment` is `TRUE` and no `glib.DestroyNotify`
    /// function has been set for `array`.
    ///
    /// Note that if the array is `NULL` terminated and `free_segment` is `FALSE`
    /// then this will always return an allocated `NULL` terminated buffer.
    /// If pdata is previously `NULL`, a new buffer will be allocated.
    ///
    /// This function is not thread-safe. If using a `glib.PtrArray` from multiple
    /// threads, use only the atomic `glib.PtrArray.ref` and `glib.PtrArray.unref`
    /// functions.
    extern fn g_ptr_array_free(p_array: *glib.PtrArray, p_free_segment: c_int) ?*anyopaque;
    pub const free = g_ptr_array_free;

    /// Inserts an element into the pointer array at the given index. The
    /// array will grow in size automatically if necessary.
    extern fn g_ptr_array_insert(p_array: *glib.PtrArray, p_index_: c_int, p_data: ?*anyopaque) void;
    pub const insert = g_ptr_array_insert;

    /// Gets whether the `array` was constructed as `NULL`-terminated.
    ///
    /// This will only return `TRUE` for arrays constructed by passing `TRUE` to the
    /// `null_terminated` argument of `glib.PtrArray.newNullTerminated`. It will not
    /// return `TRUE` for normal arrays which have had a `NULL` element appended to
    /// them.
    extern fn g_ptr_array_is_null_terminated(p_array: *glib.PtrArray) c_int;
    pub const isNullTerminated = g_ptr_array_is_null_terminated;

    /// Creates a new `glib.PtrArray` with a reference count of 1.
    extern fn g_ptr_array_new() *glib.PtrArray;
    pub const new = g_ptr_array_new;

    /// Creates a new `glib.PtrArray`, copying `len` pointers from `data`, and setting
    /// the array’s reference count to 1.
    ///
    /// This avoids having to manually add each element one by one.
    ///
    /// If `copy_func` is provided, then it is used to copy each element before
    /// adding them to the new array. If it is `NULL` then the pointers are copied
    /// directly.
    ///
    /// It also sets `element_free_func` for freeing each element when the array is
    /// destroyed either via `glib.PtrArray.unref`, when `glib.PtrArray.free` is called
    /// with `free_segment` set to `TRUE` or when removing elements.
    ///
    /// Do not use it if `len` is greater than `G_MAXUINT`. `glib.PtrArray`
    /// stores the length of its data in `guint`, which may be shorter than
    /// `gsize`.
    extern fn g_ptr_array_new_from_array(p_data: ?[*]*anyopaque, p_len: usize, p_copy_func: ?glib.CopyFunc, p_copy_func_user_data: ?*anyopaque, p_element_free_func: ?glib.DestroyNotify) *glib.PtrArray;
    pub const newFromArray = g_ptr_array_new_from_array;

    /// Creates a new `glib.PtrArray` copying the pointers from `data` after having
    /// computed the length of it and with a reference count of 1.
    /// This avoids having to manually add each element one by one.
    /// If `copy_func` is provided, then it is used to copy the data in the new
    /// array.
    /// It also set `element_free_func` for freeing each element when the array is
    /// destroyed either via `glib.PtrArray.unref`, when `glib.PtrArray.free` is called
    /// with `free_segment` set to `TRUE` or when removing elements.
    ///
    /// Do not use it if the `data` has more than `G_MAXUINT` elements. `glib.PtrArray`
    /// stores the length of its data in `guint`, which may be shorter than
    /// `gsize`.
    extern fn g_ptr_array_new_from_null_terminated_array(p_data: ?[*]*anyopaque, p_copy_func: ?glib.CopyFunc, p_copy_func_user_data: ?*anyopaque, p_element_free_func: ?glib.DestroyNotify) *glib.PtrArray;
    pub const newFromNullTerminatedArray = g_ptr_array_new_from_null_terminated_array;

    /// Creates a new `glib.PtrArray` with `reserved_size` pointers preallocated
    /// and a reference count of 1. This avoids frequent reallocation, if
    /// you are going to add many pointers to the array. Note however that
    /// the size of the array is still 0. It also set `element_free_func`
    /// for freeing each element when the array is destroyed either via
    /// `glib.PtrArray.unref`, when `glib.PtrArray.free` is called with
    /// `free_segment` set to `TRUE` or when removing elements.
    extern fn g_ptr_array_new_full(p_reserved_size: c_uint, p_element_free_func: ?glib.DestroyNotify) *glib.PtrArray;
    pub const newFull = g_ptr_array_new_full;

    /// Like `glib.PtrArray.newFull` but also allows to set the array to
    /// be `NULL` terminated. A `NULL` terminated pointer array has an
    /// additional `NULL` pointer after the last element, beyond the
    /// current length.
    ///
    /// `glib.PtrArray` created by other constructors are not automatically `NULL`
    /// terminated.
    ///
    /// Note that if the `array`'s length is zero and currently no
    /// data array is allocated, then pdata will still be `NULL`.
    /// `glib.PtrArray` will only `NULL` terminate pdata, if an actual
    /// array is allocated. It does not guarantee that an array
    /// is always allocated. In other words, if the length is zero,
    /// then pdata may either point to a `NULL` terminated array of length
    /// zero or be `NULL`.
    extern fn g_ptr_array_new_null_terminated(p_reserved_size: c_uint, p_element_free_func: ?glib.DestroyNotify, p_null_terminated: c_int) *glib.PtrArray;
    pub const newNullTerminated = g_ptr_array_new_null_terminated;

    /// Creates a new `glib.PtrArray` with `data` as pointers, `len` as length and a
    /// reference count of 1.
    ///
    /// This avoids having to copy such data manually.
    /// After this call, `data` belongs to the `glib.PtrArray` and may no longer be
    /// modified by the caller. The memory of `data` has to be dynamically
    /// allocated and will eventually be freed with `glib.free`.
    ///
    /// It also sets `element_free_func` for freeing each element when the array is
    /// destroyed either via `glib.PtrArray.unref`, when `glib.PtrArray.free` is called
    /// with `free_segment` set to `TRUE` or when removing elements.
    ///
    /// Do not use it if `len` is greater than `G_MAXUINT`. `glib.PtrArray`
    /// stores the length of its data in `guint`, which may be shorter than
    /// `gsize`.
    extern fn g_ptr_array_new_take(p_data: ?[*]*anyopaque, p_len: usize, p_element_free_func: ?glib.DestroyNotify) *glib.PtrArray;
    pub const newTake = g_ptr_array_new_take;

    /// Creates a new `glib.PtrArray` with `data` as pointers, computing the length of it
    /// and setting the reference count to 1.
    ///
    /// This avoids having to copy such data manually.
    /// After this call, `data` belongs to the `glib.PtrArray` and may no longer be
    /// modified by the caller. The memory of `data` has to be dynamically
    /// allocated and will eventually be freed with `glib.free`.
    ///
    /// The length is calculated by iterating through `data` until the first `NULL`
    /// element is found.
    ///
    /// It also sets `element_free_func` for freeing each element when the array is
    /// destroyed either via `glib.PtrArray.unref`, when `glib.PtrArray.free` is called
    /// with `free_segment` set to `TRUE` or when removing elements.
    ///
    /// Do not use it if the `data` length is greater than `G_MAXUINT`. `glib.PtrArray`
    /// stores the length of its data in `guint`, which may be shorter than
    /// `gsize`.
    extern fn g_ptr_array_new_take_null_terminated(p_data: ?[*]*anyopaque, p_element_free_func: ?glib.DestroyNotify) *glib.PtrArray;
    pub const newTakeNullTerminated = g_ptr_array_new_take_null_terminated;

    /// Creates a new `glib.PtrArray` with a reference count of 1 and use
    /// `element_free_func` for freeing each element when the array is destroyed
    /// either via `glib.PtrArray.unref`, when `glib.PtrArray.free` is called with
    /// `free_segment` set to `TRUE` or when removing elements.
    extern fn g_ptr_array_new_with_free_func(p_element_free_func: ?glib.DestroyNotify) *glib.PtrArray;
    pub const newWithFreeFunc = g_ptr_array_new_with_free_func;

    /// Atomically increments the reference count of `array` by one.
    /// This function is thread-safe and may be called from any thread.
    extern fn g_ptr_array_ref(p_array: *glib.PtrArray) *glib.PtrArray;
    pub const ref = g_ptr_array_ref;

    /// Removes the first occurrence of the given pointer from the pointer
    /// array. The following elements are moved down one place. If `array`
    /// has a non-`NULL` `glib.DestroyNotify` function it is called for the
    /// removed element.
    ///
    /// It returns `TRUE` if the pointer was removed, or `FALSE` if the
    /// pointer was not found.
    extern fn g_ptr_array_remove(p_array: *glib.PtrArray, p_data: ?*anyopaque) c_int;
    pub const remove = g_ptr_array_remove;

    /// Removes the first occurrence of the given pointer from the pointer
    /// array. The last element in the array is used to fill in the space,
    /// so this function does not preserve the order of the array. But it
    /// is faster than `glib.PtrArray.remove`. If `array` has a non-`NULL`
    /// `glib.DestroyNotify` function it is called for the removed element.
    ///
    /// It returns `TRUE` if the pointer was removed, or `FALSE` if the
    /// pointer was not found.
    extern fn g_ptr_array_remove_fast(p_array: *glib.PtrArray, p_data: ?*anyopaque) c_int;
    pub const removeFast = g_ptr_array_remove_fast;

    /// Removes the pointer at the given index from the pointer array.
    /// The following elements are moved down one place. If `array` has
    /// a non-`NULL` `glib.DestroyNotify` function it is called for the removed
    /// element. If so, the return value from this function will potentially point
    /// to freed memory (depending on the `glib.DestroyNotify` implementation).
    extern fn g_ptr_array_remove_index(p_array: *glib.PtrArray, p_index_: c_uint) ?*anyopaque;
    pub const removeIndex = g_ptr_array_remove_index;

    /// Removes the pointer at the given index from the pointer array.
    /// The last element in the array is used to fill in the space, so
    /// this function does not preserve the order of the array. But it
    /// is faster than `glib.PtrArray.removeIndex`. If `array` has a non-`NULL`
    /// `glib.DestroyNotify` function it is called for the removed element. If so, the
    /// return value from this function will potentially point to freed memory
    /// (depending on the `glib.DestroyNotify` implementation).
    extern fn g_ptr_array_remove_index_fast(p_array: *glib.PtrArray, p_index_: c_uint) ?*anyopaque;
    pub const removeIndexFast = g_ptr_array_remove_index_fast;

    /// Removes the given number of pointers starting at the given index
    /// from a `glib.PtrArray`. The following elements are moved to close the
    /// gap. If `array` has a non-`NULL` `glib.DestroyNotify` function it is
    /// called for the removed elements.
    extern fn g_ptr_array_remove_range(p_array: *glib.PtrArray, p_index_: c_uint, p_length: c_uint) *glib.PtrArray;
    pub const removeRange = g_ptr_array_remove_range;

    /// Sets a function for freeing each element when `array` is destroyed
    /// either via `glib.PtrArray.unref`, when `glib.PtrArray.free` is called
    /// with `free_segment` set to `TRUE` or when removing elements.
    extern fn g_ptr_array_set_free_func(p_array: *glib.PtrArray, p_element_free_func: ?glib.DestroyNotify) void;
    pub const setFreeFunc = g_ptr_array_set_free_func;

    /// Sets the size of the array. When making the array larger,
    /// newly-added elements will be set to `NULL`. When making it smaller,
    /// if `array` has a non-`NULL` `glib.DestroyNotify` function then it will be
    /// called for the removed elements.
    extern fn g_ptr_array_set_size(p_array: *glib.PtrArray, p_length: c_int) void;
    pub const setSize = g_ptr_array_set_size;

    /// Creates a new `glib.PtrArray` with `reserved_size` pointers preallocated
    /// and a reference count of 1. This avoids frequent reallocation, if
    /// you are going to add many pointers to the array. Note however that
    /// the size of the array is still 0.
    extern fn g_ptr_array_sized_new(p_reserved_size: c_uint) *glib.PtrArray;
    pub const sizedNew = g_ptr_array_sized_new;

    /// Sorts the array, using `compare_func` which should be a `qsort`-style
    /// comparison function (returns less than zero for first arg is less
    /// than second arg, zero for equal, greater than zero if first arg is
    /// greater than second arg).
    ///
    /// Note that the comparison function for `glib.PtrArray.sort` doesn't
    /// take the pointers from the array as arguments, it takes pointers to
    /// the pointers in the array.
    ///
    /// Use `glib.PtrArray.sortValues` if you want to use normal
    /// `GCompareFuncs`, otherwise here is a full example of use:
    ///
    /// ```
    /// typedef struct
    /// {
    ///   gchar *name;
    ///   gint size;
    /// } FileListEntry;
    ///
    /// static gint
    /// sort_filelist (gconstpointer a, gconstpointer b)
    /// {
    ///   const FileListEntry *entry1 = *((FileListEntry **) a);
    ///   const FileListEntry *entry2 = *((FileListEntry **) b);
    ///
    ///   return g_ascii_strcasecmp (entry1->name, entry2->name);
    /// }
    ///
    /// …
    /// g_autoptr (GPtrArray) file_list = NULL;
    ///
    /// // initialize file_list array and load with many FileListEntry entries
    /// ...
    /// // now sort it with
    /// g_ptr_array_sort (file_list, sort_filelist);
    /// ```
    ///
    /// This is guaranteed to be a stable sort since version 2.32.
    extern fn g_ptr_array_sort(p_array: *glib.PtrArray, p_compare_func: glib.CompareFunc) void;
    pub const sort = g_ptr_array_sort;

    /// Sorts the array, using `compare_func` which should be a `qsort`-style
    /// comparison function (returns less than zero for first arg is less
    /// than second arg, zero for equal, greater than zero if first arg is
    /// greater than second arg).
    ///
    /// This is guaranteed to be a stable sort.
    extern fn g_ptr_array_sort_values(p_array: *glib.PtrArray, p_compare_func: glib.CompareFunc) void;
    pub const sortValues = g_ptr_array_sort_values;

    /// Like `glib.PtrArray.sortValues`, but the comparison function has an extra
    /// user data argument.
    ///
    /// This is guaranteed to be a stable sort.
    extern fn g_ptr_array_sort_values_with_data(p_array: *glib.PtrArray, p_compare_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const sortValuesWithData = g_ptr_array_sort_values_with_data;

    /// Like `glib.PtrArray.sort`, but the comparison function has an extra
    /// user data argument.
    ///
    /// Note that the comparison function for `glib.PtrArray.sortWithData`
    /// doesn't take the pointers from the array as arguments, it takes
    /// pointers to the pointers in the array.
    ///
    /// Use `glib.PtrArray.sortValuesWithData` if you want to use normal
    /// `GCompareDataFuncs`, otherwise here is a full example of use:
    ///
    /// ```
    /// typedef enum { SORT_NAME, SORT_SIZE } SortMode;
    ///
    /// typedef struct
    /// {
    ///   gchar *name;
    ///   gint size;
    /// } FileListEntry;
    ///
    /// static gint
    /// sort_filelist (gconstpointer a, gconstpointer b, gpointer user_data)
    /// {
    ///   gint order;
    ///   const SortMode sort_mode = GPOINTER_TO_INT (user_data);
    ///   const FileListEntry *entry1 = *((FileListEntry **) a);
    ///   const FileListEntry *entry2 = *((FileListEntry **) b);
    ///
    ///   switch (sort_mode)
    ///     {
    ///     case SORT_NAME:
    ///       order = g_ascii_strcasecmp (entry1->name, entry2->name);
    ///       break;
    ///     case SORT_SIZE:
    ///       order = entry1->size - entry2->size;
    ///       break;
    ///     default:
    ///       order = 0;
    ///       break;
    ///     }
    ///   return order;
    /// }
    ///
    /// ...
    /// g_autoptr (GPtrArray) file_list = NULL;
    /// SortMode sort_mode;
    ///
    /// // initialize file_list array and load with many FileListEntry entries
    /// ...
    /// // now sort it with
    /// sort_mode = SORT_NAME;
    /// g_ptr_array_sort_with_data (file_list,
    ///                             sort_filelist,
    ///                             GINT_TO_POINTER (sort_mode));
    /// ```
    ///
    /// This is guaranteed to be a stable sort since version 2.32.
    extern fn g_ptr_array_sort_with_data(p_array: *glib.PtrArray, p_compare_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const sortWithData = g_ptr_array_sort_with_data;

    /// Frees the data in the array and resets the size to zero, while
    /// the underlying array is preserved for use elsewhere and returned
    /// to the caller.
    ///
    /// Note that if the array is `NULL` terminated this may still return
    /// `NULL` if the length of the array was zero and pdata was not yet
    /// allocated.
    ///
    /// Even if set, the `glib.DestroyNotify` function will never be called
    /// on the current contents of the array and the caller is
    /// responsible for freeing the array elements.
    ///
    /// An example of use:
    /// ```
    /// g_autoptr(GPtrArray) chunk_buffer = g_ptr_array_new_with_free_func (g_bytes_unref);
    ///
    /// // Some part of your application appends a number of chunks to the pointer array.
    /// g_ptr_array_add (chunk_buffer, g_bytes_new_static ("hello", 5));
    /// g_ptr_array_add (chunk_buffer, g_bytes_new_static ("world", 5));
    ///
    /// …
    ///
    /// // Periodically, the chunks need to be sent as an array-and-length to some
    /// // other part of the program.
    /// GBytes **chunks;
    /// gsize n_chunks;
    ///
    /// chunks = g_ptr_array_steal (chunk_buffer, &n_chunks);
    /// for (gsize i = 0; i < n_chunks; i++)
    ///   {
    ///     // Do something with each chunk here, and then free them, since
    ///     // `glib.PtrArray.steal` transfers ownership of all the elements and the
    ///     // array to the caller.
    ///     …
    ///
    ///     g_bytes_unref (chunks[i]);
    ///   }
    ///
    /// g_free (chunks);
    ///
    /// // After calling `glib.PtrArray.steal`, the pointer array can be reused for the
    /// // next set of chunks.
    /// g_assert (chunk_buffer->len == 0);
    /// ```
    extern fn g_ptr_array_steal(p_array: *glib.PtrArray, p_len: ?*usize) ?*anyopaque;
    pub const steal = g_ptr_array_steal;

    /// Removes the pointer at the given index from the pointer array.
    /// The following elements are moved down one place. The `glib.DestroyNotify` for
    /// `array` is *not* called on the removed element; ownership is transferred to
    /// the caller of this function.
    extern fn g_ptr_array_steal_index(p_array: *glib.PtrArray, p_index_: c_uint) ?*anyopaque;
    pub const stealIndex = g_ptr_array_steal_index;

    /// Removes the pointer at the given index from the pointer array.
    /// The last element in the array is used to fill in the space, so
    /// this function does not preserve the order of the array. But it
    /// is faster than `glib.PtrArray.stealIndex`. The `glib.DestroyNotify` for `array` is
    /// *not* called on the removed element; ownership is transferred to the caller
    /// of this function.
    extern fn g_ptr_array_steal_index_fast(p_array: *glib.PtrArray, p_index_: c_uint) ?*anyopaque;
    pub const stealIndexFast = g_ptr_array_steal_index_fast;

    /// Atomically decrements the reference count of `array` by one. If the
    /// reference count drops to 0, the effect is the same as calling
    /// `glib.PtrArray.free` with `free_segment` set to `TRUE`. This function
    /// is thread-safe and may be called from any thread.
    extern fn g_ptr_array_unref(p_array: *glib.PtrArray) void;
    pub const unref = g_ptr_array_unref;

    extern fn g_ptr_array_get_type() usize;
    pub const getGObjectType = g_ptr_array_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Contains the public fields of a
/// [Queue](data-structures.html`double`-ended-queues).
pub const Queue = extern struct {
    /// a pointer to the first element of the queue
    f_head: ?*glib.List,
    /// a pointer to the last element of the queue
    f_tail: ?*glib.List,
    /// the number of elements in the queue
    f_length: c_uint,

    /// Creates a new `glib.Queue`.
    extern fn g_queue_new() *glib.Queue;
    pub const new = g_queue_new;

    /// Removes all the elements in `queue`. If queue elements contain
    /// dynamically-allocated memory, they should be freed first.
    extern fn g_queue_clear(p_queue: *Queue) void;
    pub const clear = g_queue_clear;

    /// Convenience method, which frees all the memory used by a `glib.Queue`,
    /// and calls the provided `free_func` on each item in the `glib.Queue`.
    extern fn g_queue_clear_full(p_queue: *Queue, p_free_func: ?glib.DestroyNotify) void;
    pub const clearFull = g_queue_clear_full;

    /// Copies a `queue`. Note that is a shallow copy. If the elements in the
    /// queue consist of pointers to data, the pointers are copied, but the
    /// actual data is not.
    extern fn g_queue_copy(p_queue: *Queue) *glib.Queue;
    pub const copy = g_queue_copy;

    /// Removes `link_` from `queue` and frees it.
    ///
    /// `link_` must be part of `queue`.
    extern fn g_queue_delete_link(p_queue: *Queue, p_link_: *glib.List) void;
    pub const deleteLink = g_queue_delete_link;

    /// Finds the first link in `queue` which contains `data`.
    extern fn g_queue_find(p_queue: *Queue, p_data: ?*const anyopaque) *glib.List;
    pub const find = g_queue_find;

    /// Finds an element in a `glib.Queue`, using a supplied function to find the
    /// desired element. It iterates over the queue, calling the given function
    /// which should return 0 when the desired element is found. The function
    /// takes two gconstpointer arguments, the `glib.Queue` element's data as the
    /// first argument and the given user data as the second argument.
    extern fn g_queue_find_custom(p_queue: *Queue, p_data: ?*const anyopaque, p_func: glib.CompareFunc) *glib.List;
    pub const findCustom = g_queue_find_custom;

    /// Calls `func` for each element in the queue passing `user_data` to the
    /// function.
    ///
    /// It is safe for `func` to remove the element from `queue`, but it must
    /// not modify any part of the queue after that element.
    extern fn g_queue_foreach(p_queue: *Queue, p_func: glib.Func, p_user_data: ?*anyopaque) void;
    pub const foreach = g_queue_foreach;

    /// Frees the memory allocated for the `glib.Queue`. Only call this function
    /// if `queue` was created with `glib.Queue.new`. If queue elements contain
    /// dynamically-allocated memory, they should be freed first.
    ///
    /// If queue elements contain dynamically-allocated memory, you should
    /// either use `glib.Queue.freeFull` or free them manually first.
    extern fn g_queue_free(p_queue: *Queue) void;
    pub const free = g_queue_free;

    /// Convenience method, which frees all the memory used by a `glib.Queue`,
    /// and calls the specified destroy function on every element's data.
    ///
    /// `free_func` should not modify the queue (eg, by removing the freed
    /// element from it).
    extern fn g_queue_free_full(p_queue: *Queue, p_free_func: glib.DestroyNotify) void;
    pub const freeFull = g_queue_free_full;

    /// Returns the number of items in `queue`.
    extern fn g_queue_get_length(p_queue: *Queue) c_uint;
    pub const getLength = g_queue_get_length;

    /// Returns the position of the first element in `queue` which contains `data`.
    extern fn g_queue_index(p_queue: *Queue, p_data: ?*const anyopaque) c_int;
    pub const index = g_queue_index;

    /// A statically-allocated `glib.Queue` must be initialized with this function
    /// before it can be used. Alternatively you can initialize it with
    /// `G_QUEUE_INIT`. It is not necessary to initialize queues created with
    /// `glib.Queue.new`.
    extern fn g_queue_init(p_queue: *Queue) void;
    pub const init = g_queue_init;

    /// Inserts `data` into `queue` after `sibling`.
    ///
    /// `sibling` must be part of `queue`. Since GLib 2.44 a `NULL` sibling pushes the
    /// data at the head of the queue.
    extern fn g_queue_insert_after(p_queue: *Queue, p_sibling: ?*glib.List, p_data: ?*anyopaque) void;
    pub const insertAfter = g_queue_insert_after;

    /// Inserts `link_` into `queue` after `sibling`.
    ///
    /// `sibling` must be part of `queue`.
    extern fn g_queue_insert_after_link(p_queue: *Queue, p_sibling: ?*glib.List, p_link_: *glib.List) void;
    pub const insertAfterLink = g_queue_insert_after_link;

    /// Inserts `data` into `queue` before `sibling`.
    ///
    /// `sibling` must be part of `queue`. Since GLib 2.44 a `NULL` sibling pushes the
    /// data at the tail of the queue.
    extern fn g_queue_insert_before(p_queue: *Queue, p_sibling: ?*glib.List, p_data: ?*anyopaque) void;
    pub const insertBefore = g_queue_insert_before;

    /// Inserts `link_` into `queue` before `sibling`.
    ///
    /// `sibling` must be part of `queue`.
    extern fn g_queue_insert_before_link(p_queue: *Queue, p_sibling: ?*glib.List, p_link_: *glib.List) void;
    pub const insertBeforeLink = g_queue_insert_before_link;

    /// Inserts `data` into `queue` using `func` to determine the new position.
    extern fn g_queue_insert_sorted(p_queue: *Queue, p_data: ?*anyopaque, p_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const insertSorted = g_queue_insert_sorted;

    /// Returns `TRUE` if the queue is empty.
    extern fn g_queue_is_empty(p_queue: *Queue) c_int;
    pub const isEmpty = g_queue_is_empty;

    /// Returns the position of `link_` in `queue`.
    extern fn g_queue_link_index(p_queue: *Queue, p_link_: *glib.List) c_int;
    pub const linkIndex = g_queue_link_index;

    /// Returns the first element of the queue.
    extern fn g_queue_peek_head(p_queue: *Queue) ?*anyopaque;
    pub const peekHead = g_queue_peek_head;

    /// Returns the first link in `queue`.
    extern fn g_queue_peek_head_link(p_queue: *Queue) *glib.List;
    pub const peekHeadLink = g_queue_peek_head_link;

    /// Returns the `n`'th element of `queue`.
    extern fn g_queue_peek_nth(p_queue: *Queue, p_n: c_uint) ?*anyopaque;
    pub const peekNth = g_queue_peek_nth;

    /// Returns the link at the given position
    extern fn g_queue_peek_nth_link(p_queue: *Queue, p_n: c_uint) *glib.List;
    pub const peekNthLink = g_queue_peek_nth_link;

    /// Returns the last element of the queue.
    extern fn g_queue_peek_tail(p_queue: *Queue) ?*anyopaque;
    pub const peekTail = g_queue_peek_tail;

    /// Returns the last link in `queue`.
    extern fn g_queue_peek_tail_link(p_queue: *Queue) *glib.List;
    pub const peekTailLink = g_queue_peek_tail_link;

    /// Removes the first element of the queue and returns its data.
    extern fn g_queue_pop_head(p_queue: *Queue) ?*anyopaque;
    pub const popHead = g_queue_pop_head;

    /// Removes and returns the first element of the queue.
    extern fn g_queue_pop_head_link(p_queue: *Queue) *glib.List;
    pub const popHeadLink = g_queue_pop_head_link;

    /// Removes the `n`'th element of `queue` and returns its data.
    extern fn g_queue_pop_nth(p_queue: *Queue, p_n: c_uint) ?*anyopaque;
    pub const popNth = g_queue_pop_nth;

    /// Removes and returns the link at the given position.
    extern fn g_queue_pop_nth_link(p_queue: *Queue, p_n: c_uint) *glib.List;
    pub const popNthLink = g_queue_pop_nth_link;

    /// Removes the last element of the queue and returns its data.
    extern fn g_queue_pop_tail(p_queue: *Queue) ?*anyopaque;
    pub const popTail = g_queue_pop_tail;

    /// Removes and returns the last element of the queue.
    extern fn g_queue_pop_tail_link(p_queue: *Queue) *glib.List;
    pub const popTailLink = g_queue_pop_tail_link;

    /// Adds a new element at the head of the queue.
    extern fn g_queue_push_head(p_queue: *Queue, p_data: ?*anyopaque) void;
    pub const pushHead = g_queue_push_head;

    /// Adds a new element at the head of the queue.
    extern fn g_queue_push_head_link(p_queue: *Queue, p_link_: *glib.List) void;
    pub const pushHeadLink = g_queue_push_head_link;

    /// Inserts a new element into `queue` at the given position.
    extern fn g_queue_push_nth(p_queue: *Queue, p_data: ?*anyopaque, p_n: c_int) void;
    pub const pushNth = g_queue_push_nth;

    /// Inserts `link` into `queue` at the given position.
    extern fn g_queue_push_nth_link(p_queue: *Queue, p_n: c_int, p_link_: *glib.List) void;
    pub const pushNthLink = g_queue_push_nth_link;

    /// Adds a new element at the tail of the queue.
    extern fn g_queue_push_tail(p_queue: *Queue, p_data: ?*anyopaque) void;
    pub const pushTail = g_queue_push_tail;

    /// Adds a new element at the tail of the queue.
    extern fn g_queue_push_tail_link(p_queue: *Queue, p_link_: *glib.List) void;
    pub const pushTailLink = g_queue_push_tail_link;

    /// Removes the first element in `queue` that contains `data`.
    extern fn g_queue_remove(p_queue: *Queue, p_data: ?*const anyopaque) c_int;
    pub const remove = g_queue_remove;

    /// Remove all elements whose data equals `data` from `queue`.
    extern fn g_queue_remove_all(p_queue: *Queue, p_data: ?*const anyopaque) c_uint;
    pub const removeAll = g_queue_remove_all;

    /// Reverses the order of the items in `queue`.
    extern fn g_queue_reverse(p_queue: *Queue) void;
    pub const reverse = g_queue_reverse;

    /// Sorts `queue` using `compare_func`.
    extern fn g_queue_sort(p_queue: *Queue, p_compare_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const sort = g_queue_sort;

    /// Unlinks `link_` so that it will no longer be part of `queue`.
    /// The link is not freed.
    ///
    /// `link_` must be part of `queue`.
    extern fn g_queue_unlink(p_queue: *Queue, p_link_: *glib.List) void;
    pub const unlink = g_queue_unlink;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The GRWLock struct is an opaque data structure to represent a
/// reader-writer lock. It is similar to a `glib.Mutex` in that it allows
/// multiple threads to coordinate access to a shared resource.
///
/// The difference to a mutex is that a reader-writer lock discriminates
/// between read-only ('reader') and full ('writer') access. While only
/// one thread at a time is allowed write access (by holding the 'writer'
/// lock via `glib.RWLock.writerLock`), multiple threads can gain
/// simultaneous read-only access (by holding the 'reader' lock via
/// `glib.RWLock.readerLock`).
///
/// It is unspecified whether readers or writers have priority in acquiring the
/// lock when a reader already holds the lock and a writer is queued to acquire
/// it.
///
/// Here is an example for an array with access functions:
/// ```
///   GRWLock lock;
///   GPtrArray *array;
///
///   gpointer
///   my_array_get (guint index)
///   {
///     gpointer retval = NULL;
///
///     if (!array)
///       return NULL;
///
///     g_rw_lock_reader_lock (&lock);
///     if (index < array->len)
///       retval = g_ptr_array_index (array, index);
///     g_rw_lock_reader_unlock (&lock);
///
///     return retval;
///   }
///
///   void
///   my_array_set (guint index, gpointer data)
///   {
///     g_rw_lock_writer_lock (&lock);
///
///     if (!array)
///       array = g_ptr_array_new ();
///
///     if (index >= array->len)
///       g_ptr_array_set_size (array, index+1);
///     g_ptr_array_index (array, index) = data;
///
///     g_rw_lock_writer_unlock (&lock);
///   }
/// ```
/// This example shows an array which can be accessed by many readers
/// (the `my_array_get` function) simultaneously, whereas the writers
/// (the `my_array_set` function) will only be allowed one at a time
/// and only if no readers currently access the array. This is because
/// of the potentially dangerous resizing of the array. Using these
/// functions is fully multi-thread safe now.
///
/// If a `glib.RWLock` is allocated in static storage then it can be used
/// without initialisation.  Otherwise, you should call
/// `glib.RWLock.init` on it and `glib.RWLock.clear` when done.
///
/// A GRWLock should only be accessed with the g_rw_lock_ functions.
pub const RWLock = extern struct {
    f_p: ?*anyopaque,
    f_i: [2]c_uint,

    /// Frees the resources allocated to a lock with `glib.RWLock.init`.
    ///
    /// This function should not be used with a `glib.RWLock` that has been
    /// statically allocated.
    ///
    /// Calling `glib.RWLock.clear` when any thread holds the lock
    /// leads to undefined behaviour.
    extern fn g_rw_lock_clear(p_rw_lock: *RWLock) void;
    pub const clear = g_rw_lock_clear;

    /// Initializes a `glib.RWLock` so that it can be used.
    ///
    /// This function is useful to initialize a lock that has been
    /// allocated on the stack, or as part of a larger structure.  It is not
    /// necessary to initialise a reader-writer lock that has been statically
    /// allocated.
    ///
    /// ```
    ///   typedef struct {
    ///     GRWLock l;
    ///     ...
    ///   } Blob;
    ///
    /// Blob *b;
    ///
    /// b = g_new (Blob, 1);
    /// g_rw_lock_init (&b->l);
    /// ```
    ///
    /// To undo the effect of `glib.RWLock.init` when a lock is no longer
    /// needed, use `glib.RWLock.clear`.
    ///
    /// Calling `glib.RWLock.init` on an already initialized `glib.RWLock` leads
    /// to undefined behaviour.
    extern fn g_rw_lock_init(p_rw_lock: *RWLock) void;
    pub const init = g_rw_lock_init;

    /// Obtain a read lock on `rw_lock`. If another thread currently holds
    /// the write lock on `rw_lock`, the current thread will block until the
    /// write lock was (held and) released. If another thread does not hold
    /// the write lock, but is waiting for it, it is implementation defined
    /// whether the reader or writer will block. Read locks can be taken
    /// recursively.
    ///
    /// Calling `glib.RWLock.readerLock` while the current thread already
    /// owns a write lock leads to undefined behaviour. Read locks however
    /// can be taken recursively, in which case you need to make sure to
    /// call `glib.RWLock.readerUnlock` the same amount of times.
    ///
    /// It is implementation-defined how many read locks are allowed to be
    /// held on the same lock simultaneously. If the limit is hit,
    /// or if a deadlock is detected, a critical warning will be emitted.
    extern fn g_rw_lock_reader_lock(p_rw_lock: *RWLock) void;
    pub const readerLock = g_rw_lock_reader_lock;

    /// Tries to obtain a read lock on `rw_lock` and returns `TRUE` if
    /// the read lock was successfully obtained. Otherwise it
    /// returns `FALSE`.
    extern fn g_rw_lock_reader_trylock(p_rw_lock: *RWLock) c_int;
    pub const readerTrylock = g_rw_lock_reader_trylock;

    /// Release a read lock on `rw_lock`.
    ///
    /// Calling `glib.RWLock.readerUnlock` on a lock that is not held
    /// by the current thread leads to undefined behaviour.
    extern fn g_rw_lock_reader_unlock(p_rw_lock: *RWLock) void;
    pub const readerUnlock = g_rw_lock_reader_unlock;

    /// Obtain a write lock on `rw_lock`. If another thread currently holds
    /// a read or write lock on `rw_lock`, the current thread will block
    /// until all other threads have dropped their locks on `rw_lock`.
    ///
    /// Calling `glib.RWLock.writerLock` while the current thread already
    /// owns a read or write lock on `rw_lock` leads to undefined behaviour.
    extern fn g_rw_lock_writer_lock(p_rw_lock: *RWLock) void;
    pub const writerLock = g_rw_lock_writer_lock;

    /// Tries to obtain a write lock on `rw_lock`. If another thread
    /// currently holds a read or write lock on `rw_lock`, it immediately
    /// returns `FALSE`.
    /// Otherwise it locks `rw_lock` and returns `TRUE`.
    extern fn g_rw_lock_writer_trylock(p_rw_lock: *RWLock) c_int;
    pub const writerTrylock = g_rw_lock_writer_trylock;

    /// Release a write lock on `rw_lock`.
    ///
    /// Calling `glib.RWLock.writerUnlock` on a lock that is not held
    /// by the current thread leads to undefined behaviour.
    extern fn g_rw_lock_writer_unlock(p_rw_lock: *RWLock) void;
    pub const writerUnlock = g_rw_lock_writer_unlock;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The GRand struct is an opaque data structure. It should only be
/// accessed through the g_rand_* functions.
pub const Rand = opaque {
    /// Creates a new random number generator initialized with a seed taken
    /// either from `/dev/urandom` (if existing) or from the current time
    /// (as a fallback).
    ///
    /// On Windows, the seed is taken from `rand_s`.
    extern fn g_rand_new() *glib.Rand;
    pub const new = g_rand_new;

    /// Creates a new random number generator initialized with `seed`.
    extern fn g_rand_new_with_seed(p_seed: u32) *glib.Rand;
    pub const newWithSeed = g_rand_new_with_seed;

    /// Creates a new random number generator initialized with `seed`.
    extern fn g_rand_new_with_seed_array(p_seed: *const u32, p_seed_length: c_uint) *glib.Rand;
    pub const newWithSeedArray = g_rand_new_with_seed_array;

    /// Copies a `glib.Rand` into a new one with the same exact state as before.
    /// This way you can take a snapshot of the random number generator for
    /// replaying later.
    extern fn g_rand_copy(p_rand_: *Rand) *glib.Rand;
    pub const copy = g_rand_copy;

    /// Returns the next random `gdouble` from `rand_` equally distributed over
    /// the range [0..1).
    extern fn g_rand_double(p_rand_: *Rand) f64;
    pub const double = g_rand_double;

    /// Returns the next random `gdouble` from `rand_` equally distributed over
    /// the range [`begin`..`end`).
    extern fn g_rand_double_range(p_rand_: *Rand, p_begin: f64, p_end: f64) f64;
    pub const doubleRange = g_rand_double_range;

    /// Frees the memory allocated for the `glib.Rand`.
    extern fn g_rand_free(p_rand_: *Rand) void;
    pub const free = g_rand_free;

    /// Returns the next random `guint32` from `rand_` equally distributed over
    /// the range [0..2^32-1].
    extern fn g_rand_int(p_rand_: *Rand) u32;
    pub const int = g_rand_int;

    /// Returns the next random `gint32` from `rand_` equally distributed over
    /// the range [`begin`..`end`-1].
    extern fn g_rand_int_range(p_rand_: *Rand, p_begin: i32, p_end: i32) i32;
    pub const intRange = g_rand_int_range;

    /// Sets the seed for the random number generator `glib.Rand` to `seed`.
    extern fn g_rand_set_seed(p_rand_: *Rand, p_seed: u32) void;
    pub const setSeed = g_rand_set_seed;

    /// Initializes the random number generator by an array of longs.
    /// Array can be of arbitrary size, though only the first 624 values
    /// are taken.  This function is useful if you have many low entropy
    /// seeds, or if you require more then 32 bits of actual entropy for
    /// your application.
    extern fn g_rand_set_seed_array(p_rand_: *Rand, p_seed: *const u32, p_seed_length: c_uint) void;
    pub const setSeedArray = g_rand_set_seed_array;

    extern fn g_rand_get_type() usize;
    pub const getGObjectType = g_rand_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The GRecMutex struct is an opaque data structure to represent a
/// recursive mutex. It is similar to a `glib.Mutex` with the difference
/// that it is possible to lock a GRecMutex multiple times in the same
/// thread without deadlock. When doing so, care has to be taken to
/// unlock the recursive mutex as often as it has been locked.
///
/// If a `glib.RecMutex` is allocated in static storage then it can be used
/// without initialisation.  Otherwise, you should call
/// `glib.RecMutex.init` on it and `glib.RecMutex.clear` when done.
///
/// A GRecMutex should only be accessed with the
/// g_rec_mutex_ functions.
pub const RecMutex = extern struct {
    f_p: ?*anyopaque,
    f_i: [2]c_uint,

    /// Frees the resources allocated to a recursive mutex with
    /// `glib.RecMutex.init`.
    ///
    /// This function should not be used with a `glib.RecMutex` that has been
    /// statically allocated.
    ///
    /// Calling `glib.RecMutex.clear` on a locked recursive mutex leads
    /// to undefined behaviour.
    extern fn g_rec_mutex_clear(p_rec_mutex: *RecMutex) void;
    pub const clear = g_rec_mutex_clear;

    /// Initializes a `glib.RecMutex` so that it can be used.
    ///
    /// This function is useful to initialize a recursive mutex
    /// that has been allocated on the stack, or as part of a larger
    /// structure.
    ///
    /// It is not necessary to initialise a recursive mutex that has been
    /// statically allocated.
    ///
    /// ```
    ///   typedef struct {
    ///     GRecMutex m;
    ///     ...
    ///   } Blob;
    ///
    /// Blob *b;
    ///
    /// b = g_new (Blob, 1);
    /// g_rec_mutex_init (&b->m);
    /// ```
    ///
    /// Calling `glib.RecMutex.init` on an already initialized `glib.RecMutex`
    /// leads to undefined behaviour.
    ///
    /// To undo the effect of `glib.RecMutex.init` when a recursive mutex
    /// is no longer needed, use `glib.RecMutex.clear`.
    extern fn g_rec_mutex_init(p_rec_mutex: *RecMutex) void;
    pub const init = g_rec_mutex_init;

    /// Locks `rec_mutex`. If `rec_mutex` is already locked by another
    /// thread, the current thread will block until `rec_mutex` is
    /// unlocked by the other thread. If `rec_mutex` is already locked
    /// by the current thread, the 'lock count' of `rec_mutex` is increased.
    /// The mutex will only become available again when it is unlocked
    /// as many times as it has been locked.
    extern fn g_rec_mutex_lock(p_rec_mutex: *RecMutex) void;
    pub const lock = g_rec_mutex_lock;

    /// Tries to lock `rec_mutex`. If `rec_mutex` is already locked
    /// by another thread, it immediately returns `FALSE`. Otherwise
    /// it locks `rec_mutex` and returns `TRUE`.
    extern fn g_rec_mutex_trylock(p_rec_mutex: *RecMutex) c_int;
    pub const trylock = g_rec_mutex_trylock;

    /// Unlocks `rec_mutex`. If another thread is blocked in a
    /// `glib.RecMutex.lock` call for `rec_mutex`, it will become unblocked
    /// and can lock `rec_mutex` itself.
    ///
    /// Calling `glib.RecMutex.unlock` on a recursive mutex that is not
    /// locked by the current thread leads to undefined behaviour.
    extern fn g_rec_mutex_unlock(p_rec_mutex: *RecMutex) void;
    pub const unlock = g_rec_mutex_unlock;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `GRegex` is the "compiled" form of a regular expression pattern.
///
/// `GRegex` implements regular expression pattern matching using syntax and
/// semantics similar to Perl regular expression. See the
/// [PCRE documentation](man:pcrepattern(3)) for the syntax definition.
///
/// Some functions accept a `start_position` argument, setting it differs
/// from just passing over a shortened string and setting `G_REGEX_MATCH_NOTBOL`
/// in the case of a pattern that begins with any kind of lookbehind assertion.
/// For example, consider the pattern "\Biss\B" which finds occurrences of "iss"
/// in the middle of words. ("\B" matches only if the current position in the
/// subject is not a word boundary.) When applied to the string "Mississipi"
/// from the fourth byte, namely "issipi", it does not match, because "\B" is
/// always false at the start of the subject, which is deemed to be a word
/// boundary. However, if the entire string is passed , but with
/// `start_position` set to 4, it finds the second occurrence of "iss" because
/// it is able to look behind the starting point to discover that it is
/// preceded by a letter.
///
/// Note that, unless you set the `G_REGEX_RAW` flag, all the strings passed
/// to these functions must be encoded in UTF-8. The lengths and the positions
/// inside the strings are in bytes and not in characters, so, for instance,
/// "\xc3\xa0" (i.e. "à") is two bytes long but it is treated as a
/// single character. If you set `G_REGEX_RAW` the strings can be non-valid
/// UTF-8 strings and a byte is treated as a character, so "\xc3\xa0" is two
/// bytes and two characters long.
///
/// When matching a pattern, "\n" matches only against a "\n" character in
/// the string, and "\r" matches only a "\r" character. To match any newline
/// sequence use "\R". This particular group matches either the two-character
/// sequence CR + LF ("\r\n"), or one of the single characters LF (linefeed,
/// U+000A, "\n"), VT vertical tab, U+000B, "\v"), FF (formfeed, U+000C, "\f"),
/// CR (carriage return, U+000D, "\r"), NEL (next line, U+0085), LS (line
/// separator, U+2028), or PS (paragraph separator, U+2029).
///
/// The behaviour of the dot, circumflex, and dollar metacharacters are
/// affected by newline characters, the default is to recognize any newline
/// character (the same characters recognized by "\R"). This can be changed
/// with `G_REGEX_NEWLINE_CR`, `G_REGEX_NEWLINE_LF` and `G_REGEX_NEWLINE_CRLF`
/// compile options, and with `G_REGEX_MATCH_NEWLINE_ANY`,
/// `G_REGEX_MATCH_NEWLINE_CR`, `G_REGEX_MATCH_NEWLINE_LF` and
/// `G_REGEX_MATCH_NEWLINE_CRLF` match options. These settings are also
/// relevant when compiling a pattern if `G_REGEX_EXTENDED` is set, and an
/// unescaped "#" outside a character class is encountered. This indicates
/// a comment that lasts until after the next newline.
///
/// Creating and manipulating the same `GRegex` structure from different
/// threads is not a problem as `GRegex` does not modify its internal
/// state between creation and destruction, on the other hand `GMatchInfo`
/// is not threadsafe.
///
/// The regular expressions low-level functionalities are obtained through
/// the excellent [PCRE](http://www.pcre.org/) library written by Philip Hazel.
pub const Regex = opaque {
    /// Checks whether `replacement` is a valid replacement string
    /// (see `glib.Regex.replace`), i.e. that all escape sequences in
    /// it are valid.
    ///
    /// If `has_references` is not `NULL` then `replacement` is checked
    /// for pattern references. For instance, replacement text 'foo\n'
    /// does not contain references and may be evaluated without information
    /// about actual match, but '\0\1' (whole match followed by first
    /// subpattern) requires valid `glib.MatchInfo` object.
    extern fn g_regex_check_replacement(p_replacement: [*:0]const u8, p_has_references: ?*c_int, p_error: ?*?*glib.Error) c_int;
    pub const checkReplacement = g_regex_check_replacement;

    extern fn g_regex_error_quark() glib.Quark;
    pub const errorQuark = g_regex_error_quark;

    /// Escapes the nul characters in `string` to "\x00".  It can be used
    /// to compile a regex with embedded nul characters.
    ///
    /// For completeness, `length` can be -1 for a nul-terminated string.
    /// In this case the output string will be of course equal to `string`.
    extern fn g_regex_escape_nul(p_string: [*:0]const u8, p_length: c_int) [*:0]u8;
    pub const escapeNul = g_regex_escape_nul;

    /// Escapes the special characters used for regular expressions
    /// in `string`, for instance "a.b*c" becomes "a\.b\*c". This
    /// function is useful to dynamically generate regular expressions.
    ///
    /// `string` can contain nul characters that are replaced with "\0",
    /// in this case remember to specify the correct length of `string`
    /// in `length`.
    extern fn g_regex_escape_string(p_string: [*:0]const u8, p_length: c_int) [*:0]u8;
    pub const escapeString = g_regex_escape_string;

    /// Scans for a match in `string` for `pattern`.
    ///
    /// This function is equivalent to `glib.Regex.match` but it does not
    /// require to compile the pattern with `glib.Regex.new`, avoiding some
    /// lines of code when you need just to do a match without extracting
    /// substrings, capture counts, and so on.
    ///
    /// If this function is to be called on the same `pattern` more than
    /// once, it's more efficient to compile the pattern once with
    /// `glib.Regex.new` and then use `glib.Regex.match`.
    extern fn g_regex_match_simple(p_pattern: [*:0]const u8, p_string: [*:0]const u8, p_compile_options: glib.RegexCompileFlags, p_match_options: glib.RegexMatchFlags) c_int;
    pub const matchSimple = g_regex_match_simple;

    /// Breaks the string on the pattern, and returns an array of
    /// the tokens. If the pattern contains capturing parentheses,
    /// then the text for each of the substrings will also be returned.
    /// If the pattern does not match anywhere in the string, then the
    /// whole string is returned as the first token.
    ///
    /// This function is equivalent to `glib.Regex.split` but it does
    /// not require to compile the pattern with `glib.Regex.new`, avoiding
    /// some lines of code when you need just to do a split without
    /// extracting substrings, capture counts, and so on.
    ///
    /// If this function is to be called on the same `pattern` more than
    /// once, it's more efficient to compile the pattern once with
    /// `glib.Regex.new` and then use `glib.Regex.split`.
    ///
    /// As a special case, the result of splitting the empty string ""
    /// is an empty vector, not a vector containing a single string.
    /// The reason for this special case is that being able to represent
    /// an empty vector is typically more useful than consistent handling
    /// of empty elements. If you do need to represent empty elements,
    /// you'll need to check for the empty string before calling this
    /// function.
    ///
    /// A pattern that can match empty strings splits `string` into
    /// separate characters wherever it matches the empty string between
    /// characters. For example splitting "ab c" using as a separator
    /// "\s*", you will get "a", "b" and "c".
    extern fn g_regex_split_simple(p_pattern: [*:0]const u8, p_string: [*:0]const u8, p_compile_options: glib.RegexCompileFlags, p_match_options: glib.RegexMatchFlags) [*:null]?[*:0]u8;
    pub const splitSimple = g_regex_split_simple;

    /// Compiles the regular expression to an internal form, and does
    /// the initial setup of the `glib.Regex` structure.
    extern fn g_regex_new(p_pattern: [*:0]const u8, p_compile_options: glib.RegexCompileFlags, p_match_options: glib.RegexMatchFlags, p_error: ?*?*glib.Error) ?*glib.Regex;
    pub const new = g_regex_new;

    /// Returns the number of capturing subpatterns in the pattern.
    extern fn g_regex_get_capture_count(p_regex: *const Regex) c_int;
    pub const getCaptureCount = g_regex_get_capture_count;

    /// Returns the compile options that `regex` was created with.
    ///
    /// Depending on the version of PCRE that is used, this may or may not
    /// include flags set by option expressions such as `(?i)` found at the
    /// top-level within the compiled pattern.
    extern fn g_regex_get_compile_flags(p_regex: *const Regex) glib.RegexCompileFlags;
    pub const getCompileFlags = g_regex_get_compile_flags;

    /// Checks whether the pattern contains explicit CR or LF references.
    extern fn g_regex_get_has_cr_or_lf(p_regex: *const Regex) c_int;
    pub const getHasCrOrLf = g_regex_get_has_cr_or_lf;

    /// Returns the match options that `regex` was created with.
    extern fn g_regex_get_match_flags(p_regex: *const Regex) glib.RegexMatchFlags;
    pub const getMatchFlags = g_regex_get_match_flags;

    /// Returns the number of the highest back reference
    /// in the pattern, or 0 if the pattern does not contain
    /// back references.
    extern fn g_regex_get_max_backref(p_regex: *const Regex) c_int;
    pub const getMaxBackref = g_regex_get_max_backref;

    /// Gets the number of characters in the longest lookbehind assertion in the
    /// pattern. This information is useful when doing multi-segment matching using
    /// the partial matching facilities.
    extern fn g_regex_get_max_lookbehind(p_regex: *const Regex) c_int;
    pub const getMaxLookbehind = g_regex_get_max_lookbehind;

    /// Gets the pattern string associated with `regex`, i.e. a copy of
    /// the string passed to `glib.Regex.new`.
    extern fn g_regex_get_pattern(p_regex: *const Regex) [*:0]const u8;
    pub const getPattern = g_regex_get_pattern;

    /// Retrieves the number of the subexpression named `name`.
    extern fn g_regex_get_string_number(p_regex: *const Regex, p_name: [*:0]const u8) c_int;
    pub const getStringNumber = g_regex_get_string_number;

    /// Scans for a match in `string` for the pattern in `regex`.
    /// The `match_options` are combined with the match options specified
    /// when the `regex` structure was created, letting you have more
    /// flexibility in reusing `glib.Regex` structures.
    ///
    /// Unless `G_REGEX_RAW` is specified in the options, `string` must be valid UTF-8.
    ///
    /// A `glib.MatchInfo` structure, used to get information on the match,
    /// is stored in `match_info` if not `NULL`. Note that if `match_info`
    /// is not `NULL` then it is created even if the function returns `FALSE`,
    /// i.e. you must free it regardless if regular expression actually matched.
    ///
    /// To retrieve all the non-overlapping matches of the pattern in
    /// string you can use `glib.MatchInfo.next`.
    ///
    /// ```
    /// static void
    /// print_uppercase_words (const gchar *string)
    /// {
    ///   // Print all uppercase-only words.
    ///   GRegex *regex;
    ///   GMatchInfo *match_info;
    ///
    ///   regex = g_regex_new ("[A-Z]+", G_REGEX_DEFAULT, G_REGEX_MATCH_DEFAULT, NULL);
    ///   g_regex_match (regex, string, 0, &match_info);
    ///   while (g_match_info_matches (match_info))
    ///     {
    ///       gchar *word = g_match_info_fetch (match_info, 0);
    ///       g_print ("Found: `s`\n", word);
    ///       g_free (word);
    ///       g_match_info_next (match_info, NULL);
    ///     }
    ///   g_match_info_free (match_info);
    ///   g_regex_unref (regex);
    /// }
    /// ```
    ///
    /// `string` is not copied and is used in `glib.MatchInfo` internally. If
    /// you use any `glib.MatchInfo` method (except `glib.MatchInfo.free`) after
    /// freeing or modifying `string` then the behaviour is undefined.
    extern fn g_regex_match(p_regex: *const Regex, p_string: [*:0]const u8, p_match_options: glib.RegexMatchFlags, p_match_info: ?**glib.MatchInfo) c_int;
    pub const match = g_regex_match;

    /// Using the standard algorithm for regular expression matching only
    /// the longest match in the string is retrieved. This function uses
    /// a different algorithm so it can retrieve all the possible matches.
    /// For more documentation see `glib.Regex.matchAllFull`.
    ///
    /// A `glib.MatchInfo` structure, used to get information on the match, is
    /// stored in `match_info` if not `NULL`. Note that if `match_info` is
    /// not `NULL` then it is created even if the function returns `FALSE`,
    /// i.e. you must free it regardless if regular expression actually
    /// matched.
    ///
    /// `string` is not copied and is used in `glib.MatchInfo` internally. If
    /// you use any `glib.MatchInfo` method (except `glib.MatchInfo.free`) after
    /// freeing or modifying `string` then the behaviour is undefined.
    extern fn g_regex_match_all(p_regex: *const Regex, p_string: [*:0]const u8, p_match_options: glib.RegexMatchFlags, p_match_info: ?**glib.MatchInfo) c_int;
    pub const matchAll = g_regex_match_all;

    /// Using the standard algorithm for regular expression matching only
    /// the longest match in the `string` is retrieved, it is not possible
    /// to obtain all the available matches. For instance matching
    /// `"<a> <b> <c>"` against the pattern `"<.*>"`
    /// you get `"<a> <b> <c>"`.
    ///
    /// This function uses a different algorithm (called DFA, i.e. deterministic
    /// finite automaton), so it can retrieve all the possible matches, all
    /// starting at the same point in the string. For instance matching
    /// `"<a> <b> <c>"` against the pattern `"<.*>"`
    /// you would obtain three matches: `"<a> <b> <c>"`,
    /// `"<a> <b>"` and `"<a>"`.
    ///
    /// The number of matched strings is retrieved using
    /// `glib.MatchInfo.getMatchCount`. To obtain the matched strings and
    /// their position you can use, respectively, `glib.MatchInfo.fetch` and
    /// `glib.MatchInfo.fetchPos`. Note that the strings are returned in
    /// reverse order of length; that is, the longest matching string is
    /// given first.
    ///
    /// Note that the DFA algorithm is slower than the standard one and it
    /// is not able to capture substrings, so backreferences do not work.
    ///
    /// Setting `start_position` differs from just passing over a shortened
    /// string and setting `G_REGEX_MATCH_NOTBOL` in the case of a pattern
    /// that begins with any kind of lookbehind assertion, such as "\b".
    ///
    /// Unless `G_REGEX_RAW` is specified in the options, `string` must be valid UTF-8.
    ///
    /// A `glib.MatchInfo` structure, used to get information on the match, is
    /// stored in `match_info` if not `NULL`. Note that if `match_info` is
    /// not `NULL` then it is created even if the function returns `FALSE`,
    /// i.e. you must free it regardless if regular expression actually
    /// matched.
    ///
    /// `string` is not copied and is used in `glib.MatchInfo` internally. If
    /// you use any `glib.MatchInfo` method (except `glib.MatchInfo.free`) after
    /// freeing or modifying `string` then the behaviour is undefined.
    extern fn g_regex_match_all_full(p_regex: *const Regex, p_string: [*]const u8, p_string_len: isize, p_start_position: c_int, p_match_options: glib.RegexMatchFlags, p_match_info: ?**glib.MatchInfo, p_error: ?*?*glib.Error) c_int;
    pub const matchAllFull = g_regex_match_all_full;

    /// Scans for a match in `string` for the pattern in `regex`.
    /// The `match_options` are combined with the match options specified
    /// when the `regex` structure was created, letting you have more
    /// flexibility in reusing `glib.Regex` structures.
    ///
    /// Setting `start_position` differs from just passing over a shortened
    /// string and setting `G_REGEX_MATCH_NOTBOL` in the case of a pattern
    /// that begins with any kind of lookbehind assertion, such as "\b".
    ///
    /// Unless `G_REGEX_RAW` is specified in the options, `string` must be valid UTF-8.
    ///
    /// A `glib.MatchInfo` structure, used to get information on the match, is
    /// stored in `match_info` if not `NULL`. Note that if `match_info` is
    /// not `NULL` then it is created even if the function returns `FALSE`,
    /// i.e. you must free it regardless if regular expression actually
    /// matched.
    ///
    /// `string` is not copied and is used in `glib.MatchInfo` internally. If
    /// you use any `glib.MatchInfo` method (except `glib.MatchInfo.free`) after
    /// freeing or modifying `string` then the behaviour is undefined.
    ///
    /// To retrieve all the non-overlapping matches of the pattern in
    /// string you can use `glib.MatchInfo.next`.
    ///
    /// ```
    /// static void
    /// print_uppercase_words (const gchar *string)
    /// {
    ///   // Print all uppercase-only words.
    ///   GRegex *regex;
    ///   GMatchInfo *match_info;
    ///   GError *error = NULL;
    ///
    ///   regex = g_regex_new ("[A-Z]+", G_REGEX_DEFAULT, G_REGEX_MATCH_DEFAULT, NULL);
    ///   g_regex_match_full (regex, string, -1, 0, 0, &match_info, &error);
    ///   while (g_match_info_matches (match_info))
    ///     {
    ///       gchar *word = g_match_info_fetch (match_info, 0);
    ///       g_print ("Found: `s`\n", word);
    ///       g_free (word);
    ///       g_match_info_next (match_info, &error);
    ///     }
    ///   g_match_info_free (match_info);
    ///   g_regex_unref (regex);
    ///   if (error != NULL)
    ///     {
    ///       g_printerr ("Error while matching: `s`\n", error->message);
    ///       g_error_free (error);
    ///     }
    /// }
    /// ```
    extern fn g_regex_match_full(p_regex: *const Regex, p_string: [*]const u8, p_string_len: isize, p_start_position: c_int, p_match_options: glib.RegexMatchFlags, p_match_info: ?**glib.MatchInfo, p_error: ?*?*glib.Error) c_int;
    pub const matchFull = g_regex_match_full;

    /// Increases reference count of `regex` by 1.
    extern fn g_regex_ref(p_regex: *Regex) *glib.Regex;
    pub const ref = g_regex_ref;

    /// Replaces all occurrences of the pattern in `regex` with the
    /// replacement text. Backreferences of the form `\number` or
    /// `\g<number>` in the replacement text are interpolated by the
    /// number-th captured subexpression of the match, `\g<name>` refers
    /// to the captured subexpression with the given name. `\0` refers
    /// to the complete match, but `\0` followed by a number is the octal
    /// representation of a character. To include a literal `\` in the
    /// replacement, write `\\\\`.
    ///
    /// There are also escapes that changes the case of the following text:
    ///
    /// - \l: Convert to lower case the next character
    /// - \u: Convert to upper case the next character
    /// - \L: Convert to lower case till \E
    /// - \U: Convert to upper case till \E
    /// - \E: End case modification
    ///
    /// If you do not need to use backreferences use `glib.Regex.replaceLiteral`.
    ///
    /// The `replacement` string must be UTF-8 encoded even if `G_REGEX_RAW` was
    /// passed to `glib.Regex.new`. If you want to use not UTF-8 encoded strings
    /// you can use `glib.Regex.replaceLiteral`.
    ///
    /// Setting `start_position` differs from just passing over a shortened
    /// string and setting `G_REGEX_MATCH_NOTBOL` in the case of a pattern that
    /// begins with any kind of lookbehind assertion, such as "\b".
    extern fn g_regex_replace(p_regex: *const Regex, p_string: [*]const u8, p_string_len: isize, p_start_position: c_int, p_replacement: [*:0]const u8, p_match_options: glib.RegexMatchFlags, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const replace = g_regex_replace;

    /// Replaces occurrences of the pattern in regex with the output of
    /// `eval` for that occurrence.
    ///
    /// Setting `start_position` differs from just passing over a shortened
    /// string and setting `G_REGEX_MATCH_NOTBOL` in the case of a pattern
    /// that begins with any kind of lookbehind assertion, such as "\b".
    ///
    /// The following example uses `glib.Regex.replaceEval` to replace multiple
    /// strings at once:
    /// ```
    /// static gboolean
    /// eval_cb (const GMatchInfo *info,
    ///          GString          *res,
    ///          gpointer          data)
    /// {
    ///   gchar *match;
    ///   gchar *r;
    ///
    ///    match = g_match_info_fetch (info, 0);
    ///    r = g_hash_table_lookup ((GHashTable *)data, match);
    ///    g_string_append (res, r);
    ///    g_free (match);
    ///
    ///    return FALSE;
    /// }
    ///
    /// ...
    ///
    /// GRegex *reg;
    /// GHashTable *h;
    /// gchar *res;
    ///
    /// h = g_hash_table_new (g_str_hash, g_str_equal);
    ///
    /// g_hash_table_insert (h, "1", "ONE");
    /// g_hash_table_insert (h, "2", "TWO");
    /// g_hash_table_insert (h, "3", "THREE");
    /// g_hash_table_insert (h, "4", "FOUR");
    ///
    /// reg = g_regex_new ("1|2|3|4", G_REGEX_DEFAULT, G_REGEX_MATCH_DEFAULT, NULL);
    /// res = g_regex_replace_eval (reg, text, -1, 0, 0, eval_cb, h, NULL);
    /// g_hash_table_destroy (h);
    ///
    /// ...
    /// ```
    extern fn g_regex_replace_eval(p_regex: *const Regex, p_string: [*]const u8, p_string_len: isize, p_start_position: c_int, p_match_options: glib.RegexMatchFlags, p_eval: glib.RegexEvalCallback, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const replaceEval = g_regex_replace_eval;

    /// Replaces all occurrences of the pattern in `regex` with the
    /// replacement text. `replacement` is replaced literally, to
    /// include backreferences use `glib.Regex.replace`.
    ///
    /// Setting `start_position` differs from just passing over a
    /// shortened string and setting `G_REGEX_MATCH_NOTBOL` in the
    /// case of a pattern that begins with any kind of lookbehind
    /// assertion, such as "\b".
    extern fn g_regex_replace_literal(p_regex: *const Regex, p_string: [*]const u8, p_string_len: isize, p_start_position: c_int, p_replacement: [*:0]const u8, p_match_options: glib.RegexMatchFlags, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const replaceLiteral = g_regex_replace_literal;

    /// Breaks the string on the pattern, and returns an array of the tokens.
    /// If the pattern contains capturing parentheses, then the text for each
    /// of the substrings will also be returned. If the pattern does not match
    /// anywhere in the string, then the whole string is returned as the first
    /// token.
    ///
    /// As a special case, the result of splitting the empty string "" is an
    /// empty vector, not a vector containing a single string. The reason for
    /// this special case is that being able to represent an empty vector is
    /// typically more useful than consistent handling of empty elements. If
    /// you do need to represent empty elements, you'll need to check for the
    /// empty string before calling this function.
    ///
    /// A pattern that can match empty strings splits `string` into separate
    /// characters wherever it matches the empty string between characters.
    /// For example splitting "ab c" using as a separator "\s*", you will get
    /// "a", "b" and "c".
    extern fn g_regex_split(p_regex: *const Regex, p_string: [*:0]const u8, p_match_options: glib.RegexMatchFlags) [*:null]?[*:0]u8;
    pub const split = g_regex_split;

    /// Breaks the string on the pattern, and returns an array of the tokens.
    /// If the pattern contains capturing parentheses, then the text for each
    /// of the substrings will also be returned. If the pattern does not match
    /// anywhere in the string, then the whole string is returned as the first
    /// token.
    ///
    /// As a special case, the result of splitting the empty string "" is an
    /// empty vector, not a vector containing a single string. The reason for
    /// this special case is that being able to represent an empty vector is
    /// typically more useful than consistent handling of empty elements. If
    /// you do need to represent empty elements, you'll need to check for the
    /// empty string before calling this function.
    ///
    /// A pattern that can match empty strings splits `string` into separate
    /// characters wherever it matches the empty string between characters.
    /// For example splitting "ab c" using as a separator "\s*", you will get
    /// "a", "b" and "c".
    ///
    /// Setting `start_position` differs from just passing over a shortened
    /// string and setting `G_REGEX_MATCH_NOTBOL` in the case of a pattern
    /// that begins with any kind of lookbehind assertion, such as "\b".
    extern fn g_regex_split_full(p_regex: *const Regex, p_string: [*]const u8, p_string_len: isize, p_start_position: c_int, p_match_options: glib.RegexMatchFlags, p_max_tokens: c_int, p_error: ?*?*glib.Error) ?[*:null]?[*:0]u8;
    pub const splitFull = g_regex_split_full;

    /// Decreases reference count of `regex` by 1. When reference count drops
    /// to zero, it frees all the memory associated with the regex structure.
    extern fn g_regex_unref(p_regex: *Regex) void;
    pub const unref = g_regex_unref;

    extern fn g_regex_get_type() usize;
    pub const getGObjectType = g_regex_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `GRelation` is a table of data which can be indexed on any number
/// of fields, rather like simple database tables. A `GRelation` contains
/// a number of records, called tuples. Each record contains a number of
/// fields. Records are not ordered, so it is not possible to find the
/// record at a particular index.
///
/// Note that `GRelation` tables are currently limited to 2 fields.
///
/// To create a `GRelation`, use `glib.Relation.new`.
///
/// To specify which fields should be indexed, use `glib.Relation.index`.
/// Note that this must be called before any tuples are added to the
/// `GRelation`.
///
/// To add records to a `GRelation` use `glib.Relation.insert`.
///
/// To determine if a given record appears in a `GRelation`, use
/// `glib.Relation.exists`. Note that fields are compared directly, so
/// pointers must point to the exact same position (i.e. different
/// copies of the same string will not match.)
///
/// To count the number of records which have a particular value in a
/// given field, use `glib.Relation.count`.
///
/// To get all the records which have a particular value in a given
/// field, use `glib.Relation.select`. To access fields of the resulting
/// records, use `glib.Tuples.index`. To free the resulting records use
/// `glib.Tuples.destroy`.
///
/// To delete all records which have a particular value in a given
/// field, use `glib.Relation.delete`.
///
/// To destroy the `GRelation`, use `glib.Relation.destroy`.
///
/// To help debug `GRelation` objects, use `glib.Relation.print`.
///
/// `GRelation` has been marked as deprecated, since this API has never
/// been fully implemented, is not very actively maintained and rarely
/// used.
pub const Relation = opaque {
    /// Creates a new `glib.Relation` with the given number of fields. Note that
    /// currently the number of fields must be 2.
    extern fn g_relation_new(p_fields: c_int) *glib.Relation;
    pub const new = g_relation_new;

    /// Returns the number of tuples in a `glib.Relation` that have the given
    /// value in the given field.
    extern fn g_relation_count(p_relation: *Relation, p_key: ?*const anyopaque, p_field: c_int) c_int;
    pub const count = g_relation_count;

    /// Deletes any records from a `glib.Relation` that have the given key value
    /// in the given field.
    extern fn g_relation_delete(p_relation: *Relation, p_key: ?*const anyopaque, p_field: c_int) c_int;
    pub const delete = g_relation_delete;

    /// Destroys the `glib.Relation`, freeing all memory allocated. However, it
    /// does not free memory allocated for the tuple data, so you should
    /// free that first if appropriate.
    extern fn g_relation_destroy(p_relation: *Relation) void;
    pub const destroy = g_relation_destroy;

    /// Returns `TRUE` if a record with the given values exists in a
    /// `glib.Relation`. Note that the values are compared directly, so that, for
    /// example, two copies of the same string will not match.
    extern fn g_relation_exists(p_relation: *Relation, ...) c_int;
    pub const exists = g_relation_exists;

    /// Creates an index on the given field. Note that this must be called
    /// before any records are added to the `glib.Relation`.
    extern fn g_relation_index(p_relation: *Relation, p_field: c_int, p_hash_func: glib.HashFunc, p_key_equal_func: glib.EqualFunc) void;
    pub const index = g_relation_index;

    /// Inserts a record into a `glib.Relation`.
    extern fn g_relation_insert(p_relation: *Relation, ...) void;
    pub const insert = g_relation_insert;

    /// Outputs information about all records in a `glib.Relation`, as well as
    /// the indexes. It is for debugging.
    extern fn g_relation_print(p_relation: *Relation) void;
    pub const print = g_relation_print;

    /// Returns all of the tuples which have the given key in the given
    /// field. Use `glib.Tuples.index` to access the returned records. The
    /// returned records should be freed with `glib.Tuples.destroy`.
    extern fn g_relation_select(p_relation: *Relation, p_key: ?*const anyopaque, p_field: c_int) *glib.Tuples;
    pub const select = g_relation_select;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.SList` struct is used for each element in the singly-linked
/// list.
pub const SList = extern struct {
    /// holds the element's data, which can be a pointer to any kind
    ///        of data, or any integer value using the
    ///        [Type Conversion Macros](conversion-macros.html`conversion`-macros)
    f_data: ?*anyopaque,
    /// contains the link to the next element in the list.
    f_next: ?*glib.SList,

    /// Allocates space for one `glib.SList` element. It is called by the
    /// `glib.SList.append`, `glib.SList.prepend`, `glib.SList.insert` and
    /// `glib.SList.insertSorted` functions and so is rarely used on its own.
    extern fn g_slist_alloc() *glib.SList;
    pub const alloc = g_slist_alloc;

    /// Adds a new element on to the end of the list.
    ///
    /// The return value is the new start of the list, which may
    /// have changed, so make sure you store the new value.
    ///
    /// Note that `glib.SList.append` has to traverse the entire list
    /// to find the end, which is inefficient when adding multiple
    /// elements. A common idiom to avoid the inefficiency is to prepend
    /// the elements and reverse the list when all elements have been added.
    ///
    /// ```
    /// // Notice that these are initialized to the empty list.
    /// GSList *list = NULL, *number_list = NULL;
    ///
    /// // This is a list of strings.
    /// list = g_slist_append (list, "first");
    /// list = g_slist_append (list, "second");
    ///
    /// // This is a list of integers.
    /// number_list = g_slist_append (number_list, GINT_TO_POINTER (27));
    /// number_list = g_slist_append (number_list, GINT_TO_POINTER (14));
    /// ```
    extern fn g_slist_append(p_list: *glib.SList, p_data: ?*anyopaque) *glib.SList;
    pub const append = g_slist_append;

    /// Adds the second `glib.SList` onto the end of the first `glib.SList`.
    /// Note that the elements of the second `glib.SList` are not copied.
    /// They are used directly.
    extern fn g_slist_concat(p_list1: *glib.SList, p_list2: *glib.SList) *glib.SList;
    pub const concat = g_slist_concat;

    /// Copies a `glib.SList`.
    ///
    /// Note that this is a "shallow" copy. If the list elements
    /// consist of pointers to data, the pointers are copied but
    /// the actual data isn't. See `glib.SList.copyDeep` if you need
    /// to copy the data as well.
    extern fn g_slist_copy(p_list: *glib.SList) *glib.SList;
    pub const copy = g_slist_copy;

    /// Makes a full (deep) copy of a `glib.SList`.
    ///
    /// In contrast with `glib.SList.copy`, this function uses `func` to make a copy of
    /// each list element, in addition to copying the list container itself.
    ///
    /// `func`, as a `glib.CopyFunc`, takes two arguments, the data to be copied
    /// and a `user_data` pointer. On common processor architectures, it's safe to
    /// pass `NULL` as `user_data` if the copy function takes only one argument. You
    /// may get compiler warnings from this though if compiling with GCC’s
    /// `-Wcast-function-type` warning.
    ///
    /// For instance, if `list` holds a list of GObjects, you can do:
    /// ```
    /// another_list = g_slist_copy_deep (list, (GCopyFunc) g_object_ref, NULL);
    /// ```
    ///
    /// And, to entirely free the new list, you could do:
    /// ```
    /// g_slist_free_full (another_list, g_object_unref);
    /// ```
    extern fn g_slist_copy_deep(p_list: *glib.SList, p_func: glib.CopyFunc, p_user_data: ?*anyopaque) *glib.SList;
    pub const copyDeep = g_slist_copy_deep;

    /// Removes the node link_ from the list and frees it.
    /// Compare this to `glib.SList.removeLink` which removes the node
    /// without freeing it.
    ///
    /// Removing arbitrary nodes from a singly-linked list requires time
    /// that is proportional to the length of the list (ie. O(n)). If you
    /// find yourself using `glib.SList.deleteLink` frequently, you should
    /// consider a different data structure, such as the doubly-linked
    /// `glib.List`.
    extern fn g_slist_delete_link(p_list: *glib.SList, p_link_: *glib.SList) *glib.SList;
    pub const deleteLink = g_slist_delete_link;

    /// Finds the element in a `glib.SList` which
    /// contains the given data.
    extern fn g_slist_find(p_list: *glib.SList, p_data: ?*const anyopaque) *glib.SList;
    pub const find = g_slist_find;

    /// Finds an element in a `glib.SList`, using a supplied function to
    /// find the desired element. It iterates over the list, calling
    /// the given function which should return 0 when the desired
    /// element is found. The function takes two `gconstpointer` arguments,
    /// the `glib.SList` element's data as the first argument and the
    /// given user data.
    extern fn g_slist_find_custom(p_list: *glib.SList, p_data: ?*const anyopaque, p_func: glib.CompareFunc) *glib.SList;
    pub const findCustom = g_slist_find_custom;

    /// Calls a function for each element of a `glib.SList`.
    ///
    /// It is safe for `func` to remove the element from `list`, but it must
    /// not modify any part of the list after that element.
    extern fn g_slist_foreach(p_list: *glib.SList, p_func: glib.Func, p_user_data: ?*anyopaque) void;
    pub const foreach = g_slist_foreach;

    /// Frees all of the memory used by a `glib.SList`.
    /// The freed elements are returned to the slice allocator.
    ///
    /// If list elements contain dynamically-allocated memory,
    /// you should either use `glib.SList.freeFull` or free them manually
    /// first.
    ///
    /// It can be combined with `g_steal_pointer` to ensure the list head pointer
    /// is not left dangling:
    /// ```
    /// GSList *list_of_borrowed_things = …;  /<!-- -->* (transfer container) *<!-- -->/
    /// g_slist_free (g_steal_pointer (&list_of_borrowed_things));
    /// ```
    extern fn g_slist_free(p_list: *glib.SList) void;
    pub const free = g_slist_free;

    /// Frees one `glib.SList` element.
    /// It is usually used after `glib.SList.removeLink`.
    extern fn g_slist_free_1(p_list: *glib.SList) void;
    pub const free1 = g_slist_free_1;

    /// Convenience method, which frees all the memory used by a `glib.SList`, and
    /// calls the specified destroy function on every element's data.
    ///
    /// `free_func` must not modify the list (eg, by removing the freed
    /// element from it).
    ///
    /// It can be combined with `g_steal_pointer` to ensure the list head pointer
    /// is not left dangling ­— this also has the nice property that the head pointer
    /// is cleared before any of the list elements are freed, to prevent double frees
    /// from `free_func`:
    /// ```
    /// GSList *list_of_owned_things = …;  /<!-- -->* (transfer full) (element-type GObject) *<!-- -->/
    /// g_slist_free_full (g_steal_pointer (&list_of_owned_things), g_object_unref);
    /// ```
    extern fn g_slist_free_full(p_list: *glib.SList, p_free_func: glib.DestroyNotify) void;
    pub const freeFull = g_slist_free_full;

    /// Gets the position of the element containing
    /// the given data (starting from 0).
    extern fn g_slist_index(p_list: *glib.SList, p_data: ?*const anyopaque) c_int;
    pub const index = g_slist_index;

    /// Inserts a new element into the list at the given position.
    extern fn g_slist_insert(p_list: *glib.SList, p_data: ?*anyopaque, p_position: c_int) *glib.SList;
    pub const insert = g_slist_insert;

    /// Inserts a node before `sibling` containing `data`.
    extern fn g_slist_insert_before(p_slist: *glib.SList, p_sibling: *glib.SList, p_data: ?*anyopaque) *glib.SList;
    pub const insertBefore = g_slist_insert_before;

    /// Inserts a new element into the list, using the given
    /// comparison function to determine its position.
    extern fn g_slist_insert_sorted(p_list: *glib.SList, p_data: ?*anyopaque, p_func: glib.CompareFunc) *glib.SList;
    pub const insertSorted = g_slist_insert_sorted;

    /// Inserts a new element into the list, using the given
    /// comparison function to determine its position.
    extern fn g_slist_insert_sorted_with_data(p_list: *glib.SList, p_data: ?*anyopaque, p_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) *glib.SList;
    pub const insertSortedWithData = g_slist_insert_sorted_with_data;

    /// Gets the last element in a `glib.SList`.
    ///
    /// This function iterates over the whole list.
    extern fn g_slist_last(p_list: *glib.SList) *glib.SList;
    pub const last = g_slist_last;

    /// Gets the number of elements in a `glib.SList`.
    ///
    /// This function iterates over the whole list to
    /// count its elements. To check whether the list is non-empty, it is faster to
    /// check `list` against `NULL`.
    extern fn g_slist_length(p_list: *glib.SList) c_uint;
    pub const length = g_slist_length;

    /// Gets the element at the given position in a `glib.SList`.
    extern fn g_slist_nth(p_list: *glib.SList, p_n: c_uint) *glib.SList;
    pub const nth = g_slist_nth;

    /// Gets the data of the element at the given position.
    extern fn g_slist_nth_data(p_list: *glib.SList, p_n: c_uint) ?*anyopaque;
    pub const nthData = g_slist_nth_data;

    extern fn g_slist_pop_allocator() void;
    pub const popAllocator = g_slist_pop_allocator;

    /// Gets the position of the given element
    /// in the `glib.SList` (starting from 0).
    extern fn g_slist_position(p_list: *glib.SList, p_llink: *glib.SList) c_int;
    pub const position = g_slist_position;

    /// Adds a new element on to the start of the list.
    ///
    /// The return value is the new start of the list, which
    /// may have changed, so make sure you store the new value.
    ///
    /// ```
    /// // Notice that it is initialized to the empty list.
    /// GSList *list = NULL;
    /// list = g_slist_prepend (list, "last");
    /// list = g_slist_prepend (list, "first");
    /// ```
    extern fn g_slist_prepend(p_list: *glib.SList, p_data: ?*anyopaque) *glib.SList;
    pub const prepend = g_slist_prepend;

    extern fn g_slist_push_allocator(p_allocator: *glib.Allocator) void;
    pub const pushAllocator = g_slist_push_allocator;

    /// Removes an element from a `glib.SList`.
    /// If two elements contain the same data, only the first is removed.
    /// If none of the elements contain the data, the `glib.SList` is unchanged.
    extern fn g_slist_remove(p_list: *glib.SList, p_data: ?*const anyopaque) *glib.SList;
    pub const remove = g_slist_remove;

    /// Removes all list nodes with data equal to `data`.
    /// Returns the new head of the list. Contrast with
    /// `glib.SList.remove` which removes only the first node
    /// matching the given data.
    extern fn g_slist_remove_all(p_list: *glib.SList, p_data: ?*const anyopaque) *glib.SList;
    pub const removeAll = g_slist_remove_all;

    /// Removes an element from a `glib.SList`, without
    /// freeing the element. The removed element's next
    /// link is set to `NULL`, so that it becomes a
    /// self-contained list with one element.
    ///
    /// Removing arbitrary nodes from a singly-linked list
    /// requires time that is proportional to the length of the list
    /// (ie. O(n)). If you find yourself using `glib.SList.removeLink`
    /// frequently, you should consider a different data structure,
    /// such as the doubly-linked `glib.List`.
    extern fn g_slist_remove_link(p_list: *glib.SList, p_link_: *glib.SList) *glib.SList;
    pub const removeLink = g_slist_remove_link;

    /// Reverses a `glib.SList`.
    extern fn g_slist_reverse(p_list: *glib.SList) *glib.SList;
    pub const reverse = g_slist_reverse;

    /// Sorts a `glib.SList` using the given comparison function. The algorithm
    /// used is a stable sort.
    extern fn g_slist_sort(p_list: *glib.SList, p_compare_func: glib.CompareFunc) *glib.SList;
    pub const sort = g_slist_sort;

    /// Like `glib.SList.sort`, but the sort function accepts a user data argument.
    extern fn g_slist_sort_with_data(p_list: *glib.SList, p_compare_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) *glib.SList;
    pub const sortWithData = g_slist_sort_with_data;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GScanner` provides a general-purpose lexical scanner.
///
/// You should set `input_name` after creating the scanner, since
/// it is used by the default message handler when displaying
/// warnings and errors. If you are scanning a file, the filename
/// would be a good choice.
///
/// The `user_data` and `max_parse_errors` fields are not used.
/// If you need to associate extra data with the scanner you
/// can place them here.
///
/// If you want to use your own message handler you can set the
/// `msg_handler` field. The type of the message handler function
/// is declared by `glib.ScannerMsgFunc`.
pub const Scanner = extern struct {
    /// unused
    f_user_data: ?*anyopaque,
    /// unused
    f_max_parse_errors: c_uint,
    /// `glib.Scanner.@"error"` increments this field
    f_parse_errors: c_uint,
    /// name of input stream, featured by the default message handler
    f_input_name: ?[*:0]const u8,
    /// quarked data
    f_qdata: ?*glib.Data,
    /// link into the scanner configuration
    f_config: ?*glib.ScannerConfig,
    /// token parsed by the last `glib.Scanner.getNextToken`
    f_token: glib.TokenType,
    /// value of the last token from `glib.Scanner.getNextToken`
    f_value: glib.TokenValue,
    /// line number of the last token from `glib.Scanner.getNextToken`
    f_line: c_uint,
    /// char number of the last token from `glib.Scanner.getNextToken`
    f_position: c_uint,
    /// token parsed by the last `glib.Scanner.peekNextToken`
    f_next_token: glib.TokenType,
    /// value of the last token from `glib.Scanner.peekNextToken`
    f_next_value: glib.TokenValue,
    /// line number of the last token from `glib.Scanner.peekNextToken`
    f_next_line: c_uint,
    /// char number of the last token from `glib.Scanner.peekNextToken`
    f_next_position: c_uint,
    f_symbol_table: ?*glib.HashTable,
    f_input_fd: c_int,
    f_text: ?[*:0]const u8,
    f_text_end: ?[*:0]const u8,
    f_buffer: ?[*:0]u8,
    f_scope_id: c_uint,
    /// handler function for _warn and _error
    f_msg_handler: ?glib.ScannerMsgFunc,

    /// Creates a new `glib.Scanner`.
    ///
    /// The `config_templ` structure specifies the initial settings
    /// of the scanner, which are copied into the `glib.Scanner`
    /// `config` field. If you pass `NULL` then the default settings
    /// are used.
    extern fn g_scanner_new(p_config_templ: *const glib.ScannerConfig) *glib.Scanner;
    pub const new = g_scanner_new;

    /// Returns the current line in the input stream (counting
    /// from 1). This is the line of the last token parsed via
    /// `glib.Scanner.getNextToken`.
    extern fn g_scanner_cur_line(p_scanner: *Scanner) c_uint;
    pub const curLine = g_scanner_cur_line;

    /// Returns the current position in the current line (counting
    /// from 0). This is the position of the last token parsed via
    /// `glib.Scanner.getNextToken`.
    extern fn g_scanner_cur_position(p_scanner: *Scanner) c_uint;
    pub const curPosition = g_scanner_cur_position;

    /// Gets the current token type. This is simply the `token`
    /// field in the `glib.Scanner` structure.
    extern fn g_scanner_cur_token(p_scanner: *Scanner) glib.TokenType;
    pub const curToken = g_scanner_cur_token;

    /// Gets the current token value. This is simply the `value`
    /// field in the `glib.Scanner` structure.
    extern fn g_scanner_cur_value(p_scanner: *Scanner) glib.TokenValue;
    pub const curValue = g_scanner_cur_value;

    /// Frees all memory used by the `glib.Scanner`.
    extern fn g_scanner_destroy(p_scanner: *Scanner) void;
    pub const destroy = g_scanner_destroy;

    /// Returns `TRUE` if the scanner has reached the end of
    /// the file or text buffer.
    extern fn g_scanner_eof(p_scanner: *Scanner) c_int;
    pub const eof = g_scanner_eof;

    /// Outputs an error message, via the `glib.Scanner` message handler.
    extern fn g_scanner_error(p_scanner: *Scanner, p_format: [*:0]const u8, ...) void;
    pub const @"error" = g_scanner_error;

    /// Parses the next token just like `glib.Scanner.peekNextToken`
    /// and also removes it from the input stream. The token data is
    /// placed in the `token`, `value`, `line`, and `position` fields of
    /// the `glib.Scanner` structure.
    extern fn g_scanner_get_next_token(p_scanner: *Scanner) glib.TokenType;
    pub const getNextToken = g_scanner_get_next_token;

    /// Prepares to scan a file.
    extern fn g_scanner_input_file(p_scanner: *Scanner, p_input_fd: c_int) void;
    pub const inputFile = g_scanner_input_file;

    /// Prepares to scan a text buffer.
    extern fn g_scanner_input_text(p_scanner: *Scanner, p_text: [*:0]const u8, p_text_len: c_uint) void;
    pub const inputText = g_scanner_input_text;

    /// Looks up a symbol in the current scope and return its value.
    /// If the symbol is not bound in the current scope, `NULL` is
    /// returned.
    extern fn g_scanner_lookup_symbol(p_scanner: *Scanner, p_symbol: [*:0]const u8) ?*anyopaque;
    pub const lookupSymbol = g_scanner_lookup_symbol;

    /// Parses the next token, without removing it from the input stream.
    /// The token data is placed in the `next_token`, `next_value`, `next_line`,
    /// and `next_position` fields of the `glib.Scanner` structure.
    ///
    /// Note that, while the token is not removed from the input stream
    /// (i.e. the next call to `glib.Scanner.getNextToken` will return the
    /// same token), it will not be reevaluated. This can lead to surprising
    /// results when changing scope or the scanner configuration after peeking
    /// the next token. Getting the next token after switching the scope or
    /// configuration will return whatever was peeked before, regardless of
    /// any symbols that may have been added or removed in the new scope.
    extern fn g_scanner_peek_next_token(p_scanner: *Scanner) glib.TokenType;
    pub const peekNextToken = g_scanner_peek_next_token;

    /// Adds a symbol to the given scope.
    extern fn g_scanner_scope_add_symbol(p_scanner: *Scanner, p_scope_id: c_uint, p_symbol: [*:0]const u8, p_value: ?*anyopaque) void;
    pub const scopeAddSymbol = g_scanner_scope_add_symbol;

    /// Calls the given function for each of the symbol/value pairs
    /// in the given scope of the `glib.Scanner`. The function is passed
    /// the symbol and value of each pair, and the given `user_data`
    /// parameter.
    extern fn g_scanner_scope_foreach_symbol(p_scanner: *Scanner, p_scope_id: c_uint, p_func: glib.HFunc, p_user_data: ?*anyopaque) void;
    pub const scopeForeachSymbol = g_scanner_scope_foreach_symbol;

    /// Looks up a symbol in a scope and return its value. If the
    /// symbol is not bound in the scope, `NULL` is returned.
    extern fn g_scanner_scope_lookup_symbol(p_scanner: *Scanner, p_scope_id: c_uint, p_symbol: [*:0]const u8) ?*anyopaque;
    pub const scopeLookupSymbol = g_scanner_scope_lookup_symbol;

    /// Removes a symbol from a scope.
    extern fn g_scanner_scope_remove_symbol(p_scanner: *Scanner, p_scope_id: c_uint, p_symbol: [*:0]const u8) void;
    pub const scopeRemoveSymbol = g_scanner_scope_remove_symbol;

    /// Sets the current scope.
    extern fn g_scanner_set_scope(p_scanner: *Scanner, p_scope_id: c_uint) c_uint;
    pub const setScope = g_scanner_set_scope;

    /// Rewinds the filedescriptor to the current buffer position
    /// and blows the file read ahead buffer. This is useful for
    /// third party uses of the scanners filedescriptor, which hooks
    /// onto the current scanning position.
    extern fn g_scanner_sync_file_offset(p_scanner: *Scanner) void;
    pub const syncFileOffset = g_scanner_sync_file_offset;

    /// Outputs a message through the scanner's msg_handler,
    /// resulting from an unexpected token in the input stream.
    /// Note that you should not call `glib.Scanner.peekNextToken`
    /// followed by `glib.Scanner.unexpToken` without an intermediate
    /// call to `glib.Scanner.getNextToken`, as `glib.Scanner.unexpToken`
    /// evaluates the scanner's current token (not the peeked token)
    /// to construct part of the message.
    extern fn g_scanner_unexp_token(p_scanner: *Scanner, p_expected_token: glib.TokenType, p_identifier_spec: [*:0]const u8, p_symbol_spec: [*:0]const u8, p_symbol_name: [*:0]const u8, p_message: [*:0]const u8, p_is_error: c_int) void;
    pub const unexpToken = g_scanner_unexp_token;

    /// Outputs a warning message, via the `glib.Scanner` message handler.
    extern fn g_scanner_warn(p_scanner: *Scanner, p_format: [*:0]const u8, ...) void;
    pub const warn = g_scanner_warn;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Specifies the `glib.Scanner` parser configuration. Most settings can
/// be changed during the parsing phase and will affect the lexical
/// parsing of the next unpeeked token.
pub const ScannerConfig = extern struct {
    /// specifies which characters should be skipped
    ///     by the scanner (the default is the whitespace characters: space,
    ///     tab, carriage-return and line-feed).
    f_cset_skip_characters: ?[*:0]u8,
    /// specifies the characters which can start
    ///     identifiers (the default is `G_CSET_a_2_z`, "_", and `G_CSET_A_2_Z`).
    f_cset_identifier_first: ?[*:0]u8,
    /// specifies the characters which can be used
    ///     in identifiers, after the first character (the default is
    ///     `G_CSET_a_2_z`, "_0123456789", `G_CSET_A_2_Z`, `G_CSET_LATINS`,
    ///     `G_CSET_LATINC`).
    f_cset_identifier_nth: ?[*:0]u8,
    /// specifies the characters at the start and
    ///     end of single-line comments. The default is "#\n" which means
    ///     that single-line comments start with a '#' and continue until
    ///     a '\n' (end of line).
    f_cpair_comment_single: ?[*:0]u8,
    bitfields0: packed struct(c_uint) {
        /// specifies if symbols are case sensitive (the
        ///     default is `FALSE`).
        f_case_sensitive: u1,
        /// specifies if multi-line comments are skipped
        ///     and not returned as tokens (the default is `TRUE`).
        f_skip_comment_multi: u1,
        /// specifies if single-line comments are skipped
        ///     and not returned as tokens (the default is `TRUE`).
        f_skip_comment_single: u1,
        /// specifies if multi-line comments are recognized
        ///     (the default is `TRUE`).
        f_scan_comment_multi: u1,
        /// specifies if identifiers are recognized (the
        ///     default is `TRUE`).
        f_scan_identifier: u1,
        /// specifies if single-character
        ///     identifiers are recognized (the default is `FALSE`).
        f_scan_identifier_1char: u1,
        /// specifies if `NULL` is reported as
        ///     `G_TOKEN_IDENTIFIER_NULL` (the default is `FALSE`).
        f_scan_identifier_NULL: u1,
        /// specifies if symbols are recognized (the default
        ///     is `TRUE`).
        f_scan_symbols: u1,
        /// specifies if binary numbers are recognized (the
        ///     default is `FALSE`).
        f_scan_binary: u1,
        /// specifies if octal numbers are recognized (the
        ///     default is `TRUE`).
        f_scan_octal: u1,
        /// specifies if floating point numbers are recognized
        ///     (the default is `TRUE`).
        f_scan_float: u1,
        /// specifies if hexadecimal numbers are recognized (the
        ///     default is `TRUE`).
        f_scan_hex: u1,
        /// specifies if '$' is recognized as a prefix for
        ///     hexadecimal numbers (the default is `FALSE`).
        f_scan_hex_dollar: u1,
        /// specifies if strings can be enclosed in single
        ///     quotes (the default is `TRUE`).
        f_scan_string_sq: u1,
        /// specifies if strings can be enclosed in double
        ///     quotes (the default is `TRUE`).
        f_scan_string_dq: u1,
        /// specifies if binary, octal and hexadecimal numbers
        ///     are reported as `G_TOKEN_INT` (the default is `TRUE`).
        f_numbers_2_int: u1,
        /// specifies if all numbers are reported as `G_TOKEN_FLOAT`
        ///     (the default is `FALSE`).
        f_int_2_float: u1,
        /// specifies if identifiers are reported as strings
        ///     (the default is `FALSE`).
        f_identifier_2_string: u1,
        /// specifies if characters are reported by setting
        ///     `token = ch` or as `G_TOKEN_CHAR` (the default is `TRUE`).
        f_char_2_token: u1,
        /// specifies if symbols are reported by setting
        ///     `token = v_symbol` or as `G_TOKEN_SYMBOL` (the default is `FALSE`).
        f_symbol_2_token: u1,
        /// specifies if a symbol is searched for in the
        ///     default scope in addition to the current scope (the default is `FALSE`).
        f_scope_0_fallback: u1,
        /// use value.v_int64 rather than v_int
        f_store_int64: u1,
        _: u10,
    },
    f_padding_dummy: c_uint,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.Sequence` struct is an opaque data type representing a
/// [sequence](data-structures.html`scalable`-lists) data type.
pub const Sequence = opaque {
    /// Calls `func` for each item in the range (`begin`, `end`) passing
    /// `user_data` to the function. `func` must not modify the sequence
    /// itself.
    extern fn g_sequence_foreach_range(p_begin: *glib.SequenceIter, p_end: *glib.SequenceIter, p_func: glib.Func, p_user_data: ?*anyopaque) void;
    pub const foreachRange = g_sequence_foreach_range;

    /// Returns the data that `iter` points to.
    extern fn g_sequence_get(p_iter: *glib.SequenceIter) ?*anyopaque;
    pub const get = g_sequence_get;

    /// Inserts a new item just before the item pointed to by `iter`.
    extern fn g_sequence_insert_before(p_iter: *glib.SequenceIter, p_data: ?*anyopaque) *glib.SequenceIter;
    pub const insertBefore = g_sequence_insert_before;

    /// Moves the item pointed to by `src` to the position indicated by `dest`.
    /// After calling this function `dest` will point to the position immediately
    /// after `src`. It is allowed for `src` and `dest` to point into different
    /// sequences.
    extern fn g_sequence_move(p_src: *glib.SequenceIter, p_dest: *glib.SequenceIter) void;
    pub const move = g_sequence_move;

    /// Inserts the (`begin`, `end`) range at the destination pointed to by `dest`.
    /// The `begin` and `end` iters must point into the same sequence. It is
    /// allowed for `dest` to point to a different sequence than the one pointed
    /// into by `begin` and `end`.
    ///
    /// If `dest` is `NULL`, the range indicated by `begin` and `end` is
    /// removed from the sequence. If `dest` points to a place within
    /// the (`begin`, `end`) range, the range does not move.
    extern fn g_sequence_move_range(p_dest: *glib.SequenceIter, p_begin: *glib.SequenceIter, p_end: *glib.SequenceIter) void;
    pub const moveRange = g_sequence_move_range;

    /// Creates a new GSequence. The `data_destroy` function, if non-`NULL` will
    /// be called on all items when the sequence is destroyed and on items that
    /// are removed from the sequence.
    extern fn g_sequence_new(p_data_destroy: ?glib.DestroyNotify) *glib.Sequence;
    pub const new = g_sequence_new;

    /// Finds an iterator somewhere in the range (`begin`, `end`). This
    /// iterator will be close to the middle of the range, but is not
    /// guaranteed to be exactly in the middle.
    ///
    /// The `begin` and `end` iterators must both point to the same sequence
    /// and `begin` must come before or be equal to `end` in the sequence.
    extern fn g_sequence_range_get_midpoint(p_begin: *glib.SequenceIter, p_end: *glib.SequenceIter) *glib.SequenceIter;
    pub const rangeGetMidpoint = g_sequence_range_get_midpoint;

    /// Removes the item pointed to by `iter`. It is an error to pass the
    /// end iterator to this function.
    ///
    /// If the sequence has a data destroy function associated with it, this
    /// function is called on the data for the removed item.
    extern fn g_sequence_remove(p_iter: *glib.SequenceIter) void;
    pub const remove = g_sequence_remove;

    /// Removes all items in the (`begin`, `end`) range.
    ///
    /// If the sequence has a data destroy function associated with it, this
    /// function is called on the data for the removed items.
    extern fn g_sequence_remove_range(p_begin: *glib.SequenceIter, p_end: *glib.SequenceIter) void;
    pub const removeRange = g_sequence_remove_range;

    /// Changes the data for the item pointed to by `iter` to be `data`. If
    /// the sequence has a data destroy function associated with it, that
    /// function is called on the existing data that `iter` pointed to.
    extern fn g_sequence_set(p_iter: *glib.SequenceIter, p_data: ?*anyopaque) void;
    pub const set = g_sequence_set;

    /// Moves the data pointed to by `iter` to a new position as indicated by
    /// `cmp_func`. This
    /// function should be called for items in a sequence already sorted according
    /// to `cmp_func` whenever some aspect of an item changes so that `cmp_func`
    /// may return different values for that item.
    ///
    /// `cmp_func` is called with two items of the `seq`, and `cmp_data`.
    /// It should return 0 if the items are equal, a negative value if
    /// the first item comes before the second, and a positive value if
    /// the second item comes before the first.
    extern fn g_sequence_sort_changed(p_iter: *glib.SequenceIter, p_cmp_func: glib.CompareDataFunc, p_cmp_data: ?*anyopaque) void;
    pub const sortChanged = g_sequence_sort_changed;

    /// Like `glib.sequenceSortChanged`, but uses
    /// a `glib.SequenceIterCompareFunc` instead of a `glib.CompareDataFunc` as
    /// the compare function.
    ///
    /// `iter_cmp` is called with two iterators pointing into the `glib.Sequence` that
    /// `iter` points into. It should
    /// return 0 if the iterators are equal, a negative value if the first
    /// iterator comes before the second, and a positive value if the second
    /// iterator comes before the first.
    extern fn g_sequence_sort_changed_iter(p_iter: *glib.SequenceIter, p_iter_cmp: glib.SequenceIterCompareFunc, p_cmp_data: ?*anyopaque) void;
    pub const sortChangedIter = g_sequence_sort_changed_iter;

    /// Swaps the items pointed to by `a` and `b`. It is allowed for `a` and `b`
    /// to point into difference sequences.
    extern fn g_sequence_swap(p_a: *glib.SequenceIter, p_b: *glib.SequenceIter) void;
    pub const swap = g_sequence_swap;

    /// Adds a new item to the end of `seq`.
    extern fn g_sequence_append(p_seq: *Sequence, p_data: ?*anyopaque) *glib.SequenceIter;
    pub const append = g_sequence_append;

    /// Calls `func` for each item in the sequence passing `user_data`
    /// to the function. `func` must not modify the sequence itself.
    extern fn g_sequence_foreach(p_seq: *Sequence, p_func: glib.Func, p_user_data: ?*anyopaque) void;
    pub const foreach = g_sequence_foreach;

    /// Frees the memory allocated for `seq`. If `seq` has a data destroy
    /// function associated with it, that function is called on all items
    /// in `seq`.
    extern fn g_sequence_free(p_seq: *Sequence) void;
    pub const free = g_sequence_free;

    /// Returns the begin iterator for `seq`.
    extern fn g_sequence_get_begin_iter(p_seq: *Sequence) *glib.SequenceIter;
    pub const getBeginIter = g_sequence_get_begin_iter;

    /// Returns the end iterator for `seg`
    extern fn g_sequence_get_end_iter(p_seq: *Sequence) *glib.SequenceIter;
    pub const getEndIter = g_sequence_get_end_iter;

    /// Returns the iterator at position `pos`. If `pos` is negative or larger
    /// than the number of items in `seq`, the end iterator is returned.
    extern fn g_sequence_get_iter_at_pos(p_seq: *Sequence, p_pos: c_int) *glib.SequenceIter;
    pub const getIterAtPos = g_sequence_get_iter_at_pos;

    /// Returns the positive length (>= 0) of `seq`. Note that this method is
    /// O(h) where `h' is the height of the tree. It is thus more efficient
    /// to use `glib.Sequence.isEmpty` when comparing the length to zero.
    extern fn g_sequence_get_length(p_seq: *Sequence) c_int;
    pub const getLength = g_sequence_get_length;

    /// Inserts `data` into `seq` using `cmp_func` to determine the new
    /// position. The sequence must already be sorted according to `cmp_func`;
    /// otherwise the new position of `data` is undefined.
    ///
    /// `cmp_func` is called with two items of the `seq`, and `cmp_data`.
    /// It should return 0 if the items are equal, a negative value
    /// if the first item comes before the second, and a positive value
    /// if the second item comes before the first.
    ///
    /// Note that when adding a large amount of data to a `glib.Sequence`,
    /// it is more efficient to do unsorted insertions and then call
    /// `glib.Sequence.sort` or `glib.Sequence.sortIter`.
    extern fn g_sequence_insert_sorted(p_seq: *Sequence, p_data: ?*anyopaque, p_cmp_func: glib.CompareDataFunc, p_cmp_data: ?*anyopaque) *glib.SequenceIter;
    pub const insertSorted = g_sequence_insert_sorted;

    /// Like `glib.Sequence.insertSorted`, but uses
    /// a `glib.SequenceIterCompareFunc` instead of a `glib.CompareDataFunc` as
    /// the compare function.
    ///
    /// `iter_cmp` is called with two iterators pointing into `seq`.
    /// It should return 0 if the iterators are equal, a negative
    /// value if the first iterator comes before the second, and a
    /// positive value if the second iterator comes before the first.
    ///
    /// Note that when adding a large amount of data to a `glib.Sequence`,
    /// it is more efficient to do unsorted insertions and then call
    /// `glib.Sequence.sort` or `glib.Sequence.sortIter`.
    extern fn g_sequence_insert_sorted_iter(p_seq: *Sequence, p_data: ?*anyopaque, p_iter_cmp: glib.SequenceIterCompareFunc, p_cmp_data: ?*anyopaque) *glib.SequenceIter;
    pub const insertSortedIter = g_sequence_insert_sorted_iter;

    /// Returns `TRUE` if the sequence contains zero items.
    ///
    /// This function is functionally identical to checking the result of
    /// `glib.Sequence.getLength` being equal to zero. However this function is
    /// implemented in O(1) running time.
    extern fn g_sequence_is_empty(p_seq: *Sequence) c_int;
    pub const isEmpty = g_sequence_is_empty;

    /// Returns an iterator pointing to the position of the first item found
    /// equal to `data` according to `cmp_func` and `cmp_data`. If more than one
    /// item is equal, it is not guaranteed that it is the first which is
    /// returned. In that case, you can use `glib.SequenceIter.next` and
    /// `glib.SequenceIter.prev` to get others.
    ///
    /// `cmp_func` is called with two items of the `seq`, and `cmp_data`.
    /// It should return 0 if the items are equal, a negative value if
    /// the first item comes before the second, and a positive value if
    /// the second item comes before the first.
    ///
    /// This function will fail if the data contained in the sequence is
    /// unsorted.
    extern fn g_sequence_lookup(p_seq: *Sequence, p_data: ?*anyopaque, p_cmp_func: glib.CompareDataFunc, p_cmp_data: ?*anyopaque) ?*glib.SequenceIter;
    pub const lookup = g_sequence_lookup;

    /// Like `glib.Sequence.lookup`, but uses a `glib.SequenceIterCompareFunc`
    /// instead of a `glib.CompareDataFunc` as the compare function.
    ///
    /// `iter_cmp` is called with two iterators pointing into `seq`.
    /// It should return 0 if the iterators are equal, a negative value
    /// if the first iterator comes before the second, and a positive
    /// value if the second iterator comes before the first.
    ///
    /// This function will fail if the data contained in the sequence is
    /// unsorted.
    extern fn g_sequence_lookup_iter(p_seq: *Sequence, p_data: ?*anyopaque, p_iter_cmp: glib.SequenceIterCompareFunc, p_cmp_data: ?*anyopaque) ?*glib.SequenceIter;
    pub const lookupIter = g_sequence_lookup_iter;

    /// Adds a new item to the front of `seq`
    extern fn g_sequence_prepend(p_seq: *Sequence, p_data: ?*anyopaque) *glib.SequenceIter;
    pub const prepend = g_sequence_prepend;

    /// Returns an iterator pointing to the position where `data` would
    /// be inserted according to `cmp_func` and `cmp_data`.
    ///
    /// `cmp_func` is called with two items of the `seq`, and `cmp_data`.
    /// It should return 0 if the items are equal, a negative value if
    /// the first item comes before the second, and a positive value if
    /// the second item comes before the first.
    ///
    /// If you are simply searching for an existing element of the sequence,
    /// consider using `glib.Sequence.lookup`.
    ///
    /// This function will fail if the data contained in the sequence is
    /// unsorted.
    extern fn g_sequence_search(p_seq: *Sequence, p_data: ?*anyopaque, p_cmp_func: glib.CompareDataFunc, p_cmp_data: ?*anyopaque) *glib.SequenceIter;
    pub const search = g_sequence_search;

    /// Like `glib.Sequence.search`, but uses a `glib.SequenceIterCompareFunc`
    /// instead of a `glib.CompareDataFunc` as the compare function.
    ///
    /// `iter_cmp` is called with two iterators pointing into `seq`.
    /// It should return 0 if the iterators are equal, a negative value
    /// if the first iterator comes before the second, and a positive
    /// value if the second iterator comes before the first.
    ///
    /// If you are simply searching for an existing element of the sequence,
    /// consider using `glib.Sequence.lookupIter`.
    ///
    /// This function will fail if the data contained in the sequence is
    /// unsorted.
    extern fn g_sequence_search_iter(p_seq: *Sequence, p_data: ?*anyopaque, p_iter_cmp: glib.SequenceIterCompareFunc, p_cmp_data: ?*anyopaque) *glib.SequenceIter;
    pub const searchIter = g_sequence_search_iter;

    /// Sorts `seq` using `cmp_func`.
    ///
    /// `cmp_func` is passed two items of `seq` and should
    /// return 0 if they are equal, a negative value if the
    /// first comes before the second, and a positive value
    /// if the second comes before the first.
    extern fn g_sequence_sort(p_seq: *Sequence, p_cmp_func: glib.CompareDataFunc, p_cmp_data: ?*anyopaque) void;
    pub const sort = g_sequence_sort;

    /// Like `glib.Sequence.sort`, but uses a `glib.SequenceIterCompareFunc` instead
    /// of a `glib.CompareDataFunc` as the compare function
    ///
    /// `cmp_func` is called with two iterators pointing into `seq`. It should
    /// return 0 if the iterators are equal, a negative value if the first
    /// iterator comes before the second, and a positive value if the second
    /// iterator comes before the first.
    extern fn g_sequence_sort_iter(p_seq: *Sequence, p_cmp_func: glib.SequenceIterCompareFunc, p_cmp_data: ?*anyopaque) void;
    pub const sortIter = g_sequence_sort_iter;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.SequenceIter` struct is an opaque data type representing an
/// iterator pointing into a `glib.Sequence`.
pub const SequenceIter = opaque {
    /// Returns a negative number if `a` comes before `b`, 0 if they are equal,
    /// and a positive number if `a` comes after `b`.
    ///
    /// The `a` and `b` iterators must point into the same sequence.
    extern fn g_sequence_iter_compare(p_a: *SequenceIter, p_b: *glib.SequenceIter) c_int;
    pub const compare = g_sequence_iter_compare;

    /// Returns the position of `iter`
    extern fn g_sequence_iter_get_position(p_iter: *SequenceIter) c_int;
    pub const getPosition = g_sequence_iter_get_position;

    /// Returns the `glib.Sequence` that `iter` points into.
    extern fn g_sequence_iter_get_sequence(p_iter: *SequenceIter) *glib.Sequence;
    pub const getSequence = g_sequence_iter_get_sequence;

    /// Returns whether `iter` is the begin iterator
    extern fn g_sequence_iter_is_begin(p_iter: *SequenceIter) c_int;
    pub const isBegin = g_sequence_iter_is_begin;

    /// Returns whether `iter` is the end iterator
    extern fn g_sequence_iter_is_end(p_iter: *SequenceIter) c_int;
    pub const isEnd = g_sequence_iter_is_end;

    /// Returns the `glib.SequenceIter` which is `delta` positions away from `iter`.
    /// If `iter` is closer than -`delta` positions to the beginning of the sequence,
    /// the begin iterator is returned. If `iter` is closer than `delta` positions
    /// to the end of the sequence, the end iterator is returned.
    extern fn g_sequence_iter_move(p_iter: *SequenceIter, p_delta: c_int) *glib.SequenceIter;
    pub const move = g_sequence_iter_move;

    /// Returns an iterator pointing to the next position after `iter`.
    /// If `iter` is the end iterator, the end iterator is returned.
    extern fn g_sequence_iter_next(p_iter: *SequenceIter) *glib.SequenceIter;
    pub const next = g_sequence_iter_next;

    /// Returns an iterator pointing to the previous position before `iter`.
    /// If `iter` is the begin iterator, the begin iterator is returned.
    extern fn g_sequence_iter_prev(p_iter: *SequenceIter) *glib.SequenceIter;
    pub const prev = g_sequence_iter_prev;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `GSource` struct is an opaque data type
/// representing an event source.
pub const Source = extern struct {
    f_callback_data: ?*anyopaque,
    f_callback_funcs: ?*glib.SourceCallbackFuncs,
    f_source_funcs: ?*const glib.SourceFuncs,
    f_ref_count: c_uint,
    f_context: ?*glib.MainContext,
    f_priority: c_int,
    f_flags: c_uint,
    f_source_id: c_uint,
    f_poll_fds: ?*glib.SList,
    f_prev: ?*glib.Source,
    f_next: ?*glib.Source,
    f_name: ?[*:0]u8,
    f_priv: ?*glib.SourcePrivate,

    /// Removes the source with the given ID from the default main context. You must
    /// use `glib.Source.destroy` for sources added to a non-default main context.
    ///
    /// The ID of a `glib.Source` is given by `glib.Source.getId`, or will be
    /// returned by the functions `glib.Source.attach`, `glib.idleAdd`,
    /// `glib.idleAddFull`, `glib.timeoutAdd`,
    /// `glib.timeoutAddFull`, `glib.childWatchAdd`,
    /// `glib.childWatchAddFull`, `glib.ioAddWatch`, and
    /// `glib.ioAddWatchFull`.
    ///
    /// It is a programmer error to attempt to remove a non-existent source.
    ///
    /// More specifically: source IDs can be reissued after a source has been
    /// destroyed and therefore it is never valid to use this function with a
    /// source ID which may have already been removed.  An example is when
    /// scheduling an idle to run in another thread with `glib.idleAdd`: the
    /// idle may already have run and been removed by the time this function
    /// is called on its (now invalid) source ID.  This source ID may have
    /// been reissued, leading to the operation being performed against the
    /// wrong source.
    extern fn g_source_remove(p_tag: c_uint) c_int;
    pub const remove = g_source_remove;

    /// Removes a source from the default main loop context given the
    /// source functions and user data. If multiple sources exist with the
    /// same source functions and user data, only one will be destroyed.
    extern fn g_source_remove_by_funcs_user_data(p_funcs: *glib.SourceFuncs, p_user_data: ?*anyopaque) c_int;
    pub const removeByFuncsUserData = g_source_remove_by_funcs_user_data;

    /// Removes a source from the default main loop context given the user
    /// data for the callback. If multiple sources exist with the same user
    /// data, only one will be destroyed.
    extern fn g_source_remove_by_user_data(p_user_data: ?*anyopaque) c_int;
    pub const removeByUserData = g_source_remove_by_user_data;

    /// Sets the name of a source using its ID.
    ///
    /// This is a convenience utility to set source names from the return
    /// value of `glib.idleAdd`, `glib.timeoutAdd`, etc.
    ///
    /// It is a programmer error to attempt to set the name of a non-existent
    /// source.
    ///
    /// More specifically: source IDs can be reissued after a source has been
    /// destroyed and therefore it is never valid to use this function with a
    /// source ID which may have already been removed.  An example is when
    /// scheduling an idle to run in another thread with `glib.idleAdd`: the
    /// idle may already have run and been removed by the time this function
    /// is called on its (now invalid) source ID.  This source ID may have
    /// been reissued, leading to the operation being performed against the
    /// wrong source.
    extern fn g_source_set_name_by_id(p_tag: c_uint, p_name: [*:0]const u8) void;
    pub const setNameById = g_source_set_name_by_id;

    /// Creates a new `glib.Source` structure. The size is specified to
    /// allow creating structures derived from `glib.Source` that contain
    /// additional data. The size passed in must be at least
    /// `sizeof (GSource)`.
    ///
    /// The source will not initially be associated with any `glib.MainContext`
    /// and must be added to one with `glib.Source.attach` before it will be
    /// executed.
    extern fn g_source_new(p_source_funcs: *glib.SourceFuncs, p_struct_size: c_uint) *glib.Source;
    pub const new = g_source_new;

    /// Adds `child_source` to `source` as a "polled" source; when `source` is
    /// added to a `glib.MainContext`, `child_source` will be automatically
    /// added with the same priority, when `child_source` is triggered, it will
    /// cause `source` to dispatch (in addition to calling its own
    /// callback), and when `source` is destroyed, it will destroy
    /// `child_source` as well. (`source` will also still be dispatched if
    /// its own prepare/check functions indicate that it is ready.)
    ///
    /// If you don't need `child_source` to do anything on its own when it
    /// triggers, you can call `g_source_set_dummy_callback` on it to set a
    /// callback that does nothing (except return `TRUE` if appropriate).
    ///
    /// `source` will hold a reference on `child_source` while `child_source`
    /// is attached to it.
    ///
    /// This API is only intended to be used by implementations of
    /// `glib.Source`. Do not call this API on a `glib.Source` that
    /// you did not create.
    extern fn g_source_add_child_source(p_source: *Source, p_child_source: *glib.Source) void;
    pub const addChildSource = g_source_add_child_source;

    /// Adds a file descriptor to the set of file descriptors polled for
    /// this source. This is usually combined with `glib.Source.new` to add an
    /// event source. The event source's check function will typically test
    /// the `revents` field in the `glib.PollFD` struct and return `TRUE` if events need
    /// to be processed.
    ///
    /// This API is only intended to be used by implementations of `glib.Source`.
    /// Do not call this API on a `glib.Source` that you did not create.
    ///
    /// Using this API forces the linear scanning of event sources on each
    /// main loop iteration.  Newly-written event sources should try to use
    /// `g_source_add_unix_fd` instead of this API.
    extern fn g_source_add_poll(p_source: *Source, p_fd: *glib.PollFD) void;
    pub const addPoll = g_source_add_poll;

    /// Monitors `fd` for the IO events in `events`.
    ///
    /// The tag returned by this function can be used to remove or modify the
    /// monitoring of the fd using `glib.Source.removeUnixFd` or
    /// `glib.Source.modifyUnixFd`.
    ///
    /// It is not necessary to remove the fd before destroying the source; it
    /// will be cleaned up automatically.
    ///
    /// This API is only intended to be used by implementations of `glib.Source`.
    /// Do not call this API on a `glib.Source` that you did not create.
    ///
    /// As the name suggests, this function is not available on Windows.
    extern fn g_source_add_unix_fd(p_source: *Source, p_fd: c_int, p_events: glib.IOCondition) *anyopaque;
    pub const addUnixFd = g_source_add_unix_fd;

    /// Adds a `glib.Source` to a `context` so that it will be executed within
    /// that context. Remove it by calling `glib.Source.destroy`.
    ///
    /// This function is safe to call from any thread, regardless of which thread
    /// the `context` is running in.
    extern fn g_source_attach(p_source: *Source, p_context: ?*glib.MainContext) c_uint;
    pub const attach = g_source_attach;

    /// Removes a source from its `glib.MainContext`, if any, and mark it as
    /// destroyed.  The source cannot be subsequently added to another
    /// context. It is safe to call this on sources which have already been
    /// removed from their context.
    ///
    /// This does not unref the `glib.Source`: if you still hold a reference,
    /// use `glib.Source.unref` to drop it.
    ///
    /// This function is safe to call from any thread, regardless of which thread
    /// the `glib.MainContext` is running in.
    ///
    /// If the source is currently attached to a `glib.MainContext`,
    /// destroying it will effectively unset the callback similar to calling
    /// `glib.Source.setCallback`. This can mean, that the data's
    /// `glib.DestroyNotify` gets called right away.
    extern fn g_source_destroy(p_source: *Source) void;
    pub const destroy = g_source_destroy;

    /// Checks whether a source is allowed to be called recursively.
    /// see `glib.Source.setCanRecurse`.
    extern fn g_source_get_can_recurse(p_source: *Source) c_int;
    pub const getCanRecurse = g_source_get_can_recurse;

    /// Gets the `glib.MainContext` with which the source is associated.
    ///
    /// You can call this on a source that has been destroyed, provided
    /// that the `glib.MainContext` it was attached to still exists (in which
    /// case it will return that `glib.MainContext`). In particular, you can
    /// always call this function on the source returned from
    /// `glib.mainCurrentSource`. But calling this function on a source
    /// whose `glib.MainContext` has been destroyed is an error.
    extern fn g_source_get_context(p_source: *Source) ?*glib.MainContext;
    pub const getContext = g_source_get_context;

    /// This function ignores `source` and is otherwise the same as
    /// `glib.getCurrentTime`.
    extern fn g_source_get_current_time(p_source: *Source, p_timeval: *glib.TimeVal) void;
    pub const getCurrentTime = g_source_get_current_time;

    /// Returns the numeric ID for a particular source. The ID of a source
    /// is a positive integer which is unique within a particular main loop
    /// context. The reverse mapping from ID to source is done by
    /// `glib.MainContext.findSourceById`.
    ///
    /// You can only call this function while the source is associated to a
    /// `glib.MainContext` instance; calling this function before
    /// `glib.Source.attach` or after `glib.Source.destroy` yields
    /// undefined behavior. The ID returned is unique within the
    /// `glib.MainContext` instance passed to `glib.Source.attach`.
    extern fn g_source_get_id(p_source: *Source) c_uint;
    pub const getId = g_source_get_id;

    /// Gets a name for the source, used in debugging and profiling.  The
    /// name may be `NULL` if it has never been set with `glib.Source.setName`.
    extern fn g_source_get_name(p_source: *Source) ?[*:0]const u8;
    pub const getName = g_source_get_name;

    /// Gets the priority of a source.
    extern fn g_source_get_priority(p_source: *Source) c_int;
    pub const getPriority = g_source_get_priority;

    /// Gets the "ready time" of `source`, as set by
    /// `glib.Source.setReadyTime`.
    ///
    /// Any time before or equal to the current monotonic time (including 0)
    /// is an indication that the source will fire immediately.
    extern fn g_source_get_ready_time(p_source: *Source) i64;
    pub const getReadyTime = g_source_get_ready_time;

    /// Gets the time to be used when checking this source. The advantage of
    /// calling this function over calling `glib.getMonotonicTime` directly is
    /// that when checking multiple sources, GLib can cache a single value
    /// instead of having to repeatedly get the system monotonic time.
    ///
    /// The time here is the system monotonic time, if available, or some
    /// other reasonable alternative otherwise.  See `glib.getMonotonicTime`.
    extern fn g_source_get_time(p_source: *Source) i64;
    pub const getTime = g_source_get_time;

    /// Returns whether `source` has been destroyed.
    ///
    /// This is important when you operate upon your objects
    /// from within idle handlers, but may have freed the object
    /// before the dispatch of your idle handler.
    ///
    /// ```
    /// static gboolean
    /// idle_callback (gpointer data)
    /// {
    ///   SomeWidget *self = data;
    ///
    ///   g_mutex_lock (&self->idle_id_mutex);
    ///   // do stuff with self
    ///   self->idle_id = 0;
    ///   g_mutex_unlock (&self->idle_id_mutex);
    ///
    ///   return G_SOURCE_REMOVE;
    /// }
    ///
    /// static void
    /// some_widget_do_stuff_later (SomeWidget *self)
    /// {
    ///   g_mutex_lock (&self->idle_id_mutex);
    ///   self->idle_id = g_idle_add (idle_callback, self);
    ///   g_mutex_unlock (&self->idle_id_mutex);
    /// }
    ///
    /// static void
    /// some_widget_init (SomeWidget *self)
    /// {
    ///   g_mutex_init (&self->idle_id_mutex);
    ///
    ///   // ...
    /// }
    ///
    /// static void
    /// some_widget_finalize (GObject *object)
    /// {
    ///   SomeWidget *self = SOME_WIDGET (object);
    ///
    ///   if (self->idle_id)
    ///     g_source_remove (self->idle_id);
    ///
    ///   g_mutex_clear (&self->idle_id_mutex);
    ///
    ///   G_OBJECT_CLASS (parent_class)->finalize (object);
    /// }
    /// ```
    ///
    /// This will fail in a multi-threaded application if the
    /// widget is destroyed before the idle handler fires due
    /// to the use after free in the callback. A solution, to
    /// this particular problem, is to check to if the source
    /// has already been destroy within the callback.
    ///
    /// ```
    /// static gboolean
    /// idle_callback (gpointer data)
    /// {
    ///   SomeWidget *self = data;
    ///
    ///   g_mutex_lock (&self->idle_id_mutex);
    ///   if (!g_source_is_destroyed (g_main_current_source ()))
    ///     {
    ///       // do stuff with self
    ///     }
    ///   g_mutex_unlock (&self->idle_id_mutex);
    ///
    ///   return FALSE;
    /// }
    /// ```
    ///
    /// Calls to this function from a thread other than the one acquired by the
    /// `glib.MainContext` the `glib.Source` is attached to are typically
    /// redundant, as the source could be destroyed immediately after this function
    /// returns. However, once a source is destroyed it cannot be un-destroyed, so
    /// this function can be used for opportunistic checks from any thread.
    extern fn g_source_is_destroyed(p_source: *Source) c_int;
    pub const isDestroyed = g_source_is_destroyed;

    /// Updates the event mask to watch for the fd identified by `tag`.
    ///
    /// `tag` is the tag returned from `glib.Source.addUnixFd`.
    ///
    /// If you want to remove a fd, don't set its event mask to zero.
    /// Instead, call `glib.Source.removeUnixFd`.
    ///
    /// This API is only intended to be used by implementations of `glib.Source`.
    /// Do not call this API on a `glib.Source` that you did not create.
    ///
    /// As the name suggests, this function is not available on Windows.
    extern fn g_source_modify_unix_fd(p_source: *Source, p_tag: *anyopaque, p_new_events: glib.IOCondition) void;
    pub const modifyUnixFd = g_source_modify_unix_fd;

    /// Queries the events reported for the fd corresponding to `tag` on
    /// `source` during the last poll.
    ///
    /// The return value of this function is only defined when the function
    /// is called from the check or dispatch functions for `source`.
    ///
    /// This API is only intended to be used by implementations of `glib.Source`.
    /// Do not call this API on a `glib.Source` that you did not create.
    ///
    /// As the name suggests, this function is not available on Windows.
    extern fn g_source_query_unix_fd(p_source: *Source, p_tag: *anyopaque) glib.IOCondition;
    pub const queryUnixFd = g_source_query_unix_fd;

    /// Increases the reference count on a source by one.
    extern fn g_source_ref(p_source: *Source) *glib.Source;
    pub const ref = g_source_ref;

    /// Detaches `child_source` from `source` and destroys it.
    ///
    /// This API is only intended to be used by implementations of `glib.Source`.
    /// Do not call this API on a `glib.Source` that you did not create.
    extern fn g_source_remove_child_source(p_source: *Source, p_child_source: *glib.Source) void;
    pub const removeChildSource = g_source_remove_child_source;

    /// Removes a file descriptor from the set of file descriptors polled for
    /// this source.
    ///
    /// This API is only intended to be used by implementations of `glib.Source`.
    /// Do not call this API on a `glib.Source` that you did not create.
    extern fn g_source_remove_poll(p_source: *Source, p_fd: *glib.PollFD) void;
    pub const removePoll = g_source_remove_poll;

    /// Reverses the effect of a previous call to `glib.Source.addUnixFd`.
    ///
    /// You only need to call this if you want to remove an fd from being
    /// watched while keeping the same source around.  In the normal case you
    /// will just want to destroy the source.
    ///
    /// This API is only intended to be used by implementations of `glib.Source`.
    /// Do not call this API on a `glib.Source` that you did not create.
    ///
    /// As the name suggests, this function is not available on Windows.
    extern fn g_source_remove_unix_fd(p_source: *Source, p_tag: *anyopaque) void;
    pub const removeUnixFd = g_source_remove_unix_fd;

    /// Sets the callback function for a source. The callback for a source is
    /// called from the source's dispatch function.
    ///
    /// The exact type of `func` depends on the type of source; ie. you
    /// should not count on `func` being called with `data` as its first
    /// parameter. Cast `func` with `glib.SOURCEFUNC` to avoid warnings about
    /// incompatible function types.
    ///
    /// See [mainloop memory management](main-loop.html`memory`-management-of-sources) for details
    /// on how to handle memory management of `data`.
    ///
    /// Typically, you won't use this function. Instead use functions specific
    /// to the type of source you are using, such as `glib.idleAdd` or
    /// `glib.timeoutAdd`.
    ///
    /// It is safe to call this function multiple times on a source which has already
    /// been attached to a context. The changes will take effect for the next time
    /// the source is dispatched after this call returns.
    ///
    /// Note that `glib.Source.destroy` for a currently attached source has the effect
    /// of also unsetting the callback.
    extern fn g_source_set_callback(p_source: *Source, p_func: glib.SourceFunc, p_data: ?*anyopaque, p_notify: ?glib.DestroyNotify) void;
    pub const setCallback = g_source_set_callback;

    /// Sets the callback function storing the data as a refcounted callback
    /// "object". This is used internally. Note that calling
    /// `glib.Source.setCallbackIndirect` assumes
    /// an initial reference count on `callback_data`, and thus
    /// `callback_funcs`->unref will eventually be called once more
    /// than `callback_funcs`->ref.
    ///
    /// It is safe to call this function multiple times on a source which has already
    /// been attached to a context. The changes will take effect for the next time
    /// the source is dispatched after this call returns.
    extern fn g_source_set_callback_indirect(p_source: *Source, p_callback_data: ?*anyopaque, p_callback_funcs: *glib.SourceCallbackFuncs) void;
    pub const setCallbackIndirect = g_source_set_callback_indirect;

    /// Sets whether a source can be called recursively. If `can_recurse` is
    /// `TRUE`, then while the source is being dispatched then this source
    /// will be processed normally. Otherwise, all processing of this
    /// source is blocked until the dispatch function returns.
    extern fn g_source_set_can_recurse(p_source: *Source, p_can_recurse: c_int) void;
    pub const setCanRecurse = g_source_set_can_recurse;

    /// Set `dispose` as dispose function on `source`. `dispose` will be called once
    /// the reference count of `source` reaches 0 but before any of the state of the
    /// source is freed, especially before the finalize function is called.
    ///
    /// This means that at this point `source` is still a valid `glib.Source`
    /// and it is allow for the reference count to increase again until `dispose`
    /// returns.
    ///
    /// The dispose function can be used to clear any "weak" references to the
    /// `source` in other data structures in a thread-safe way where it is possible
    /// for another thread to increase the reference count of `source` again while
    /// it is being freed.
    ///
    /// The finalize function can not be used for this purpose as at that point
    /// `source` is already partially freed and not valid anymore.
    ///
    /// This should only ever be called from `glib.Source` implementations.
    extern fn g_source_set_dispose_function(p_source: *Source, p_dispose: glib.SourceDisposeFunc) void;
    pub const setDisposeFunction = g_source_set_dispose_function;

    /// Sets the source functions (can be used to override
    /// default implementations) of an unattached source.
    extern fn g_source_set_funcs(p_source: *Source, p_funcs: *glib.SourceFuncs) void;
    pub const setFuncs = g_source_set_funcs;

    /// Sets a name for the source, used in debugging and profiling.
    /// The name defaults to `NULL`.
    ///
    /// The source name should describe in a human-readable way
    /// what the source does. For example, "X11 event queue"
    /// or "GTK repaint idle handler" or whatever it is.
    ///
    /// It is permitted to call this function multiple times, but is not
    /// recommended due to the potential performance impact.  For example,
    /// one could change the name in the "check" function of a `glib.SourceFuncs`
    /// to include details like the event type in the source name.
    ///
    /// Use caution if changing the name while another thread may be
    /// accessing it with `glib.Source.getName`; that function does not copy
    /// the value, and changing the value will free it while the other thread
    /// may be attempting to use it.
    ///
    /// Also see `glib.Source.setStaticName`.
    extern fn g_source_set_name(p_source: *Source, p_name: [*:0]const u8) void;
    pub const setName = g_source_set_name;

    /// Sets the priority of a source. While the main loop is being run, a
    /// source will be dispatched if it is ready to be dispatched and no
    /// sources at a higher (numerically smaller) priority are ready to be
    /// dispatched.
    ///
    /// A child source always has the same priority as its parent.  It is not
    /// permitted to change the priority of a source once it has been added
    /// as a child of another source.
    extern fn g_source_set_priority(p_source: *Source, p_priority: c_int) void;
    pub const setPriority = g_source_set_priority;

    /// Sets a `glib.Source` to be dispatched when the given monotonic time is
    /// reached (or passed).  If the monotonic time is in the past (as it
    /// always will be if `ready_time` is 0) then the source will be
    /// dispatched immediately.
    ///
    /// If `ready_time` is -1 then the source is never woken up on the basis
    /// of the passage of time.
    ///
    /// Dispatching the source does not reset the ready time.  You should do
    /// so yourself, from the source dispatch function.
    ///
    /// Note that if you have a pair of sources where the ready time of one
    /// suggests that it will be delivered first but the priority for the
    /// other suggests that it would be delivered first, and the ready time
    /// for both sources is reached during the same main context iteration,
    /// then the order of dispatch is undefined.
    ///
    /// It is a no-op to call this function on a `glib.Source` which has already been
    /// destroyed with `glib.Source.destroy`.
    ///
    /// This API is only intended to be used by implementations of `glib.Source`.
    /// Do not call this API on a `glib.Source` that you did not create.
    extern fn g_source_set_ready_time(p_source: *Source, p_ready_time: i64) void;
    pub const setReadyTime = g_source_set_ready_time;

    /// A variant of `glib.Source.setName` that does not
    /// duplicate the `name`, and can only be used with
    /// string literals.
    extern fn g_source_set_static_name(p_source: *Source, p_name: [*:0]const u8) void;
    pub const setStaticName = g_source_set_static_name;

    /// Decreases the reference count of a source by one. If the
    /// resulting reference count is zero the source and associated
    /// memory will be destroyed.
    extern fn g_source_unref(p_source: *Source) void;
    pub const unref = g_source_unref;

    extern fn g_source_get_type() usize;
    pub const getGObjectType = g_source_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `GSourceCallbackFuncs` struct contains
/// functions for managing callback objects.
pub const SourceCallbackFuncs = extern struct {
    /// Called when a reference is added to the callback object
    f_ref: ?*const fn (p_cb_data: *anyopaque) callconv(.c) void,
    /// Called when a reference to the callback object is dropped
    f_unref: ?*const fn (p_cb_data: *anyopaque) callconv(.c) void,
    /// Called to extract the callback function and data from the
    ///     callback object.
    f_get: ?*const fn (p_cb_data: *anyopaque, p_source: *glib.Source, p_func: *glib.SourceFunc, p_data: ?*anyopaque) callconv(.c) void,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `GSourceFuncs` struct contains a table of
/// functions used to handle event sources in a generic manner.
///
/// For idle sources, the prepare and check functions always return `TRUE`
/// to indicate that the source is always ready to be processed. The prepare
/// function also returns a timeout value of 0 to ensure that the `poll` call
/// doesn't block (since that would be time wasted which could have been spent
/// running the idle function).
///
/// For timeout sources, the prepare and check functions both return `TRUE`
/// if the timeout interval has expired. The prepare function also returns
/// a timeout value to ensure that the `poll` call doesn't block too long
/// and miss the next timeout.
///
/// For file descriptor sources, the prepare function typically returns `FALSE`,
/// since it must wait until `poll` has been called before it knows whether
/// any events need to be processed. It sets the returned timeout to -1 to
/// indicate that it doesn't mind how long the `poll` call blocks. In the
/// check function, it tests the results of the `poll` call to see if the
/// required condition has been met, and returns `TRUE` if so.
pub const SourceFuncs = extern struct {
    /// Called before all the file descriptors are polled. If the
    ///     source can determine that it is ready here (without waiting for the
    ///     results of the `poll` call) it should return `TRUE`. It can also return
    ///     a `timeout_` value which should be the maximum timeout (in milliseconds)
    ///     which should be passed to the `poll` call. The actual timeout used will
    ///     be -1 if all sources returned -1, or it will be the minimum of all
    ///     the `timeout_` values returned which were >= 0.  Since 2.36 this may
    ///     be `NULL`, in which case the effect is as if the function always returns
    ///     `FALSE` with a timeout of -1.  If `prepare` returns a
    ///     timeout and the source also has a ready time set, then the
    ///     lower of the two will be used.
    f_prepare: ?glib.SourceFuncsPrepareFunc,
    /// Called after all the file descriptors are polled. The source
    ///     should return `TRUE` if it is ready to be dispatched. Note that some
    ///     time may have passed since the previous prepare function was called,
    ///     so the source should be checked again here.  Since 2.36 this may
    ///     be `NULL`, in which case the effect is as if the function always returns
    ///     `FALSE`.
    f_check: ?glib.SourceFuncsCheckFunc,
    /// Called to dispatch the event source, after it has returned
    ///     `TRUE` in either its `prepare` or its `check` function, or if a ready time
    ///     has been reached. The `dispatch` function receives a callback function and
    ///     user data. The callback function may be `NULL` if the source was never
    ///     connected to a callback using `glib.Source.setCallback`. The
    ///     `dispatch` function should call the callback function with `user_data` and
    ///     whatever additional parameters are needed for this type of event source.
    ///     The return value of the `dispatch` function should be
    ///     `glib.SOURCE_REMOVE` if the source should be removed or
    ///     `glib.SOURCE_CONTINUE` to keep it.
    f_dispatch: ?glib.SourceFuncsDispatchFunc,
    /// Called when the source is finalized. At this point, the source
    ///     will have been destroyed, had its callback cleared, and have been removed
    ///     from its `glib.MainContext`, but it will still have its final
    ///     reference count, so methods can be called on it from within this
    ///     function.
    f_finalize: ?glib.SourceFuncsFinalizeFunc,
    f_closure_callback: ?glib.SourceFunc,
    f_closure_marshal: ?glib.SourceDummyMarshal,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

pub const SourcePrivate = opaque {
    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A type corresponding to the appropriate struct type for the `stat`
/// system call, depending on the platform and/or compiler being used.
///
/// See `glib.stat` for more information.
pub const StatBuf = opaque {
    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `glib.StaticMutex` works like a `glib.Mutex`.
///
/// Prior to GLib 2.32, GStaticMutex had the significant advantage
/// that it doesn't need to be created at run-time, but can be defined
/// at compile-time. Since 2.32, `glib.Mutex` can be statically allocated
/// as well, and GStaticMutex has been deprecated.
///
/// Here is a version of our `give_me_next_number` example using
/// a GStaticMutex:
/// ```
///   int
///   give_me_next_number (void)
///   {
///     static int current_number = 0;
///     int ret_val;
///     static GStaticMutex mutex = G_STATIC_MUTEX_INIT;
///
///     g_static_mutex_lock (&mutex);
///     ret_val = current_number = calc_next_number (current_number);
///     g_static_mutex_unlock (&mutex);
///
///     return ret_val;
///   }
/// ```
///
/// Sometimes you would like to dynamically create a mutex. If you don't
/// want to require prior calling to `glib.threadInit`, because your code
/// should also be usable in non-threaded programs, you are not able to
/// use `glib.mutexNew` and thus `glib.Mutex`, as that requires a prior call to
/// `glib.threadInit`. In these cases you can also use a `glib.StaticMutex`.
/// It must be initialized with `glib.StaticMutex.init` before using it
/// and freed with with `glib.StaticMutex.free` when not needed anymore to
/// free up any allocated resources.
///
/// Even though `glib.StaticMutex` is not opaque, it should only be used with
/// the following functions, as it is defined differently on different
/// platforms.
///
/// All of the g_static_mutex_* functions apart from
/// `g_static_mutex_get_mutex` can also be used even if `glib.threadInit`
/// has not yet been called. Then they do nothing, apart from
/// `g_static_mutex_trylock` which does nothing but returning `TRUE`.
///
/// All of the g_static_mutex_* functions are actually macros. Apart from
/// taking their addresses, you can however use them as if they were
/// functions.
pub const StaticMutex = extern struct {
    f_mutex: ?*glib.Mutex,

    /// Releases all resources allocated to `mutex`.
    ///
    /// You don't have to call this functions for a `glib.StaticMutex` with an
    /// unbounded lifetime, i.e. objects declared 'static', but if you have
    /// a `glib.StaticMutex` as a member of a structure and the structure is
    /// freed, you should also free the `glib.StaticMutex`.
    ///
    /// Calling `glib.StaticMutex.free` on a locked mutex may result in
    /// undefined behaviour.
    extern fn g_static_mutex_free(p_mutex: *StaticMutex) void;
    pub const free = g_static_mutex_free;

    extern fn g_static_mutex_get_mutex_impl(p_mutex: *StaticMutex) *glib.Mutex;
    pub const getMutexImpl = g_static_mutex_get_mutex_impl;

    /// Initializes `mutex`.
    /// Alternatively you can initialize it with `G_STATIC_MUTEX_INIT`.
    extern fn g_static_mutex_init(p_mutex: *StaticMutex) void;
    pub const init = g_static_mutex_init;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `glib.StaticPrivate` works almost like a `glib.Private`, but it has one
/// significant advantage. It doesn't need to be created at run-time
/// like a `glib.Private`, but can be defined at compile-time. This is
/// similar to the difference between `glib.Mutex` and `glib.StaticMutex`.
///
/// Now look at our `give_me_next_number` example with `glib.StaticPrivate`:
/// ```
///   int
///   give_me_next_number ()
///   {
///     static GStaticPrivate current_number_key = G_STATIC_PRIVATE_INIT;
///     int *current_number = g_static_private_get (&current_number_key);
///
///     if (!current_number)
///       {
///         current_number = g_new (int, 1);
///         *current_number = 0;
///         g_static_private_set (&current_number_key, current_number, g_free);
///       }
///
///     *current_number = calc_next_number (*current_number);
///
///     return *current_number;
///   }
/// ```
pub const StaticPrivate = extern struct {
    f_index: c_uint,

    /// Releases all resources allocated to `private_key`.
    ///
    /// You don't have to call this functions for a `glib.StaticPrivate` with an
    /// unbounded lifetime, i.e. objects declared 'static', but if you have
    /// a `glib.StaticPrivate` as a member of a structure and the structure is
    /// freed, you should also free the `glib.StaticPrivate`.
    extern fn g_static_private_free(p_private_key: *StaticPrivate) void;
    pub const free = g_static_private_free;

    /// Works like `glib.Private.get` only for a `glib.StaticPrivate`.
    ///
    /// This function works even if `glib.threadInit` has not yet been called.
    extern fn g_static_private_get(p_private_key: *StaticPrivate) ?*anyopaque;
    pub const get = g_static_private_get;

    /// Initializes `private_key`. Alternatively you can initialize it with
    /// `G_STATIC_PRIVATE_INIT`.
    extern fn g_static_private_init(p_private_key: *StaticPrivate) void;
    pub const init = g_static_private_init;

    /// Sets the pointer keyed to `private_key` for the current thread and
    /// the function `notify` to be called with that pointer (`NULL` or
    /// non-`NULL`), whenever the pointer is set again or whenever the
    /// current thread ends.
    ///
    /// This function works even if `glib.threadInit` has not yet been called.
    /// If `glib.threadInit` is called later, the `data` keyed to `private_key`
    /// will be inherited only by the main thread, i.e. the one that called
    /// `glib.threadInit`.
    ///
    /// `notify` is used quite differently from `destructor` in `glib.privateNew`.
    extern fn g_static_private_set(p_private_key: *StaticPrivate, p_data: ?*anyopaque, p_notify: glib.DestroyNotify) void;
    pub const set = g_static_private_set;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.StaticRWLock` struct represents a read-write lock. A read-write
/// lock can be used for protecting data that some portions of code only
/// read from, while others also write. In such situations it is
/// desirable that several readers can read at once, whereas of course
/// only one writer may write at a time.
///
/// Take a look at the following example:
/// ```
///   GStaticRWLock rwlock = G_STATIC_RW_LOCK_INIT;
///   GPtrArray *array;
///
///   gpointer
///   my_array_get (guint index)
///   {
///     gpointer retval = NULL;
///
///     if (!array)
///       return NULL;
///
///     g_static_rw_lock_reader_lock (&rwlock);
///     if (index < array->len)
///       retval = g_ptr_array_index (array, index);
///     g_static_rw_lock_reader_unlock (&rwlock);
///
///     return retval;
///   }
///
///   void
///   my_array_set (guint index, gpointer data)
///   {
///     g_static_rw_lock_writer_lock (&rwlock);
///
///     if (!array)
///       array = g_ptr_array_new ();
///
///     if (index >= array->len)
///       g_ptr_array_set_size (array, index + 1);
///     g_ptr_array_index (array, index) = data;
///
///     g_static_rw_lock_writer_unlock (&rwlock);
///   }
/// ```
///
/// This example shows an array which can be accessed by many readers
/// (the `my_array_get` function) simultaneously, whereas the writers
/// (the `my_array_set` function) will only be allowed once at a time
/// and only if no readers currently access the array. This is because
/// of the potentially dangerous resizing of the array. Using these
/// functions is fully multi-thread safe now.
///
/// Most of the time, writers should have precedence over readers. That
/// means, for this implementation, that as soon as a writer wants to
/// lock the data, no other reader is allowed to lock the data, whereas,
/// of course, the readers that already have locked the data are allowed
/// to finish their operation. As soon as the last reader unlocks the
/// data, the writer will lock it.
///
/// Even though `glib.StaticRWLock` is not opaque, it should only be used
/// with the following functions.
///
/// All of the g_static_rw_lock_* functions can be used even if
/// `glib.threadInit` has not been called. Then they do nothing, apart
/// from g_static_rw_lock_*_trylock, which does nothing but returning `TRUE`.
///
/// A read-write lock has a higher overhead than a mutex. For example, both
/// `glib.StaticRWLock.readerLock` and `glib.StaticRWLock.readerUnlock` have
/// to lock and unlock a `glib.StaticMutex`, so it takes at least twice the time
/// to lock and unlock a `glib.StaticRWLock` that it does to lock and unlock a
/// `glib.StaticMutex`. So only data structures that are accessed by multiple
/// readers, and which keep the lock for a considerable time justify a
/// `glib.StaticRWLock`. The above example most probably would fare better with a
/// `glib.StaticMutex`.
pub const StaticRWLock = extern struct {
    f_mutex: glib.StaticMutex,
    f_read_cond: ?*glib.Cond,
    f_write_cond: ?*glib.Cond,
    f_read_counter: c_uint,
    f_have_writer: c_int,
    f_want_to_read: c_uint,
    f_want_to_write: c_uint,

    /// Releases all resources allocated to `lock`.
    ///
    /// You don't have to call this functions for a `glib.StaticRWLock` with an
    /// unbounded lifetime, i.e. objects declared 'static', but if you have
    /// a `glib.StaticRWLock` as a member of a structure, and the structure is
    /// freed, you should also free the `glib.StaticRWLock`.
    extern fn g_static_rw_lock_free(p_lock: *StaticRWLock) void;
    pub const free = g_static_rw_lock_free;

    /// A `glib.StaticRWLock` must be initialized with this function before it
    /// can be used. Alternatively you can initialize it with
    /// `G_STATIC_RW_LOCK_INIT`.
    extern fn g_static_rw_lock_init(p_lock: *StaticRWLock) void;
    pub const init = g_static_rw_lock_init;

    /// Locks `lock` for reading. There may be unlimited concurrent locks for
    /// reading of a `glib.StaticRWLock` at the same time.  If `lock` is already
    /// locked for writing by another thread or if another thread is already
    /// waiting to lock `lock` for writing, this function will block until
    /// `lock` is unlocked by the other writing thread and no other writing
    /// threads want to lock `lock`. This lock has to be unlocked by
    /// `glib.StaticRWLock.readerUnlock`.
    ///
    /// `glib.StaticRWLock` is not recursive. It might seem to be possible to
    /// recursively lock for reading, but that can result in a deadlock, due
    /// to writer preference.
    extern fn g_static_rw_lock_reader_lock(p_lock: *StaticRWLock) void;
    pub const readerLock = g_static_rw_lock_reader_lock;

    /// Tries to lock `lock` for reading. If `lock` is already locked for
    /// writing by another thread or if another thread is already waiting to
    /// lock `lock` for writing, immediately returns `FALSE`. Otherwise locks
    /// `lock` for reading and returns `TRUE`. This lock has to be unlocked by
    /// `glib.StaticRWLock.readerUnlock`.
    extern fn g_static_rw_lock_reader_trylock(p_lock: *StaticRWLock) c_int;
    pub const readerTrylock = g_static_rw_lock_reader_trylock;

    /// Unlocks `lock`. If a thread waits to lock `lock` for writing and all
    /// locks for reading have been unlocked, the waiting thread is woken up
    /// and can lock `lock` for writing.
    extern fn g_static_rw_lock_reader_unlock(p_lock: *StaticRWLock) void;
    pub const readerUnlock = g_static_rw_lock_reader_unlock;

    /// Locks `lock` for writing. If `lock` is already locked for writing or
    /// reading by other threads, this function will block until `lock` is
    /// completely unlocked and then lock `lock` for writing. While this
    /// functions waits to lock `lock`, no other thread can lock `lock` for
    /// reading. When `lock` is locked for writing, no other thread can lock
    /// `lock` (neither for reading nor writing). This lock has to be
    /// unlocked by `glib.StaticRWLock.writerUnlock`.
    extern fn g_static_rw_lock_writer_lock(p_lock: *StaticRWLock) void;
    pub const writerLock = g_static_rw_lock_writer_lock;

    /// Tries to lock `lock` for writing. If `lock` is already locked (for
    /// either reading or writing) by another thread, it immediately returns
    /// `FALSE`. Otherwise it locks `lock` for writing and returns `TRUE`. This
    /// lock has to be unlocked by `glib.StaticRWLock.writerUnlock`.
    extern fn g_static_rw_lock_writer_trylock(p_lock: *StaticRWLock) c_int;
    pub const writerTrylock = g_static_rw_lock_writer_trylock;

    /// Unlocks `lock`. If a thread is waiting to lock `lock` for writing and
    /// all locks for reading have been unlocked, the waiting thread is
    /// woken up and can lock `lock` for writing. If no thread is waiting to
    /// lock `lock` for writing, and some thread or threads are waiting to
    /// lock `lock` for reading, the waiting threads are woken up and can
    /// lock `lock` for reading.
    extern fn g_static_rw_lock_writer_unlock(p_lock: *StaticRWLock) void;
    pub const writerUnlock = g_static_rw_lock_writer_unlock;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `glib.StaticRecMutex` works like a `glib.StaticMutex`, but it can be locked
/// multiple times by one thread. If you enter it n times, you have to
/// unlock it n times again to let other threads lock it. An exception
/// is the function `glib.StaticRecMutex.unlockFull`: that allows you to
/// unlock a `glib.StaticRecMutex` completely returning the depth, (i.e. the
/// number of times this mutex was locked). The depth can later be used
/// to restore the state of the `glib.StaticRecMutex` by calling
/// `glib.StaticRecMutex.lockFull`. In GLib 2.32, `glib.StaticRecMutex` has
/// been deprecated in favor of `glib.RecMutex`.
///
/// Even though `glib.StaticRecMutex` is not opaque, it should only be used
/// with the following functions.
///
/// All of the g_static_rec_mutex_* functions can be used even if
/// `glib.threadInit` has not been called. Then they do nothing, apart
/// from `glib.StaticRecMutex.trylock`, which does nothing but returning
/// `TRUE`.
pub const StaticRecMutex = extern struct {
    f_mutex: glib.StaticMutex,
    f_depth: c_uint,

    /// Releases all resources allocated to a `glib.StaticRecMutex`.
    ///
    /// You don't have to call this functions for a `glib.StaticRecMutex` with an
    /// unbounded lifetime, i.e. objects declared 'static', but if you have
    /// a `glib.StaticRecMutex` as a member of a structure and the structure is
    /// freed, you should also free the `glib.StaticRecMutex`.
    extern fn g_static_rec_mutex_free(p_mutex: *StaticRecMutex) void;
    pub const free = g_static_rec_mutex_free;

    /// A `glib.StaticRecMutex` must be initialized with this function before it
    /// can be used. Alternatively you can initialize it with
    /// `G_STATIC_REC_MUTEX_INIT`.
    extern fn g_static_rec_mutex_init(p_mutex: *StaticRecMutex) void;
    pub const init = g_static_rec_mutex_init;

    /// Locks `mutex`. If `mutex` is already locked by another thread, the
    /// current thread will block until `mutex` is unlocked by the other
    /// thread. If `mutex` is already locked by the calling thread, this
    /// functions increases the depth of `mutex` and returns immediately.
    extern fn g_static_rec_mutex_lock(p_mutex: *StaticRecMutex) void;
    pub const lock = g_static_rec_mutex_lock;

    /// Works like calling `glib.StaticRecMutex.lock` for `mutex` `depth` times.
    extern fn g_static_rec_mutex_lock_full(p_mutex: *StaticRecMutex, p_depth: c_uint) void;
    pub const lockFull = g_static_rec_mutex_lock_full;

    /// Tries to lock `mutex`. If `mutex` is already locked by another thread,
    /// it immediately returns `FALSE`. Otherwise it locks `mutex` and returns
    /// `TRUE`. If `mutex` is already locked by the calling thread, this
    /// functions increases the depth of `mutex` and immediately returns
    /// `TRUE`.
    extern fn g_static_rec_mutex_trylock(p_mutex: *StaticRecMutex) c_int;
    pub const trylock = g_static_rec_mutex_trylock;

    /// Unlocks `mutex`. Another thread will be allowed to lock `mutex` only
    /// when it has been unlocked as many times as it had been locked
    /// before. If `mutex` is completely unlocked and another thread is
    /// blocked in a `glib.StaticRecMutex.lock` call for `mutex`, it will be
    /// woken and can lock `mutex` itself.
    extern fn g_static_rec_mutex_unlock(p_mutex: *StaticRecMutex) void;
    pub const unlock = g_static_rec_mutex_unlock;

    /// Completely unlocks `mutex`. If another thread is blocked in a
    /// `glib.StaticRecMutex.lock` call for `mutex`, it will be woken and can
    /// lock `mutex` itself. This function returns the number of times that
    /// `mutex` has been locked by the current thread. To restore the state
    /// before the call to `glib.StaticRecMutex.unlockFull` you can call
    /// `glib.StaticRecMutex.lockFull` with the depth returned by this
    /// function.
    extern fn g_static_rec_mutex_unlock_full(p_mutex: *StaticRecMutex) c_uint;
    pub const unlockFull = g_static_rec_mutex_unlock_full;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `GString` is an object that handles the memory management of a C string.
///
/// The emphasis of `GString` is on text, typically UTF-8. Crucially, the "str" member
/// of a `GString` is guaranteed to have a trailing nul character, and it is therefore
/// always safe to call functions such as ``strchr`` or ``strdup`` on it.
///
/// However, a `GString` can also hold arbitrary binary data, because it has a "len" member,
/// which includes any possible embedded nul characters in the data. Conceptually then,
/// `GString` is like a `GByteArray` with the addition of many convenience methods for
/// text, and a guaranteed nul terminator.
pub const String = extern struct {
    /// points to the character data. It may move as text is added.
    ///   The `str` field is null-terminated and so
    ///   can be used as an ordinary C string.
    f_str: ?[*:0]u8,
    /// contains the length of the string, not including the
    ///   terminating nul byte.
    f_len: usize,
    /// the number of bytes that can be stored in the
    ///   string before it needs to be reallocated. May be larger than `len`.
    f_allocated_len: usize,

    /// Creates a new `glib.String`, initialized with the given string.
    extern fn g_string_new(p_init: ?[*:0]const u8) *glib.String;
    pub const new = g_string_new;

    /// Creates a new `glib.String` with `len` bytes of the `init` buffer.
    /// Because a length is provided, `init` need not be nul-terminated,
    /// and can contain embedded nul bytes.
    ///
    /// Since this function does not stop at nul bytes, it is the caller's
    /// responsibility to ensure that `init` has at least `len` addressable
    /// bytes.
    extern fn g_string_new_len(p_init: [*:0]const u8, p_len: isize) *glib.String;
    pub const newLen = g_string_new_len;

    /// Creates a new `glib.String`, initialized with the given string.
    ///
    /// After this call, `init` belongs to the `glib.String` and may no longer be
    /// modified by the caller. The memory of `data` has to be dynamically
    /// allocated and will eventually be freed with `glib.free`.
    extern fn g_string_new_take(p_init: ?[*:0]u8) *glib.String;
    pub const newTake = g_string_new_take;

    /// Creates a new `glib.String`, with enough space for `dfl_size`
    /// bytes. This is useful if you are going to add a lot of
    /// text to the string and don't want it to be reallocated
    /// too often.
    extern fn g_string_sized_new(p_dfl_size: usize) *glib.String;
    pub const sizedNew = g_string_sized_new;

    /// Adds a string onto the end of a `glib.String`, expanding
    /// it if necessary.
    extern fn g_string_append(p_string: *String, p_val: [*:0]const u8) *glib.String;
    pub const append = g_string_append;

    /// Adds a byte onto the end of a `glib.String`, expanding
    /// it if necessary.
    extern fn g_string_append_c(p_string: *String, p_c: u8) *glib.String;
    pub const appendC = g_string_append_c;

    /// Appends `len` bytes of `val` to `string`.
    ///
    /// If `len` is positive, `val` may contain embedded nuls and need
    /// not be nul-terminated. It is the caller's responsibility to
    /// ensure that `val` has at least `len` addressable bytes.
    ///
    /// If `len` is negative, `val` must be nul-terminated and `len`
    /// is considered to request the entire string length. This
    /// makes `glib.String.appendLen` equivalent to `glib.String.append`.
    extern fn g_string_append_len(p_string: *String, p_val: [*:0]const u8, p_len: isize) *glib.String;
    pub const appendLen = g_string_append_len;

    /// Appends a formatted string onto the end of a `glib.String`.
    /// This function is similar to `glib.String.printf` except
    /// that the text is appended to the `glib.String`.
    extern fn g_string_append_printf(p_string: *String, p_format: [*:0]const u8, ...) void;
    pub const appendPrintf = g_string_append_printf;

    /// Converts a Unicode character into UTF-8, and appends it
    /// to the string.
    extern fn g_string_append_unichar(p_string: *String, p_wc: u32) *glib.String;
    pub const appendUnichar = g_string_append_unichar;

    /// Appends `unescaped` to `string`, escaping any characters that
    /// are reserved in URIs using URI-style escape sequences.
    extern fn g_string_append_uri_escaped(p_string: *String, p_unescaped: [*:0]const u8, p_reserved_chars_allowed: [*:0]const u8, p_allow_utf8: c_int) *glib.String;
    pub const appendUriEscaped = g_string_append_uri_escaped;

    /// Appends a formatted string onto the end of a `glib.String`.
    /// This function is similar to `glib.String.appendPrintf`
    /// except that the arguments to the format string are passed
    /// as a va_list.
    extern fn g_string_append_vprintf(p_string: *String, p_format: [*:0]const u8, p_args: std.builtin.VaList) void;
    pub const appendVprintf = g_string_append_vprintf;

    /// Converts all uppercase ASCII letters to lowercase ASCII letters.
    extern fn g_string_ascii_down(p_string: *String) *glib.String;
    pub const asciiDown = g_string_ascii_down;

    /// Converts all lowercase ASCII letters to uppercase ASCII letters.
    extern fn g_string_ascii_up(p_string: *String) *glib.String;
    pub const asciiUp = g_string_ascii_up;

    /// Copies the bytes from a string into a `glib.String`,
    /// destroying any previous contents. It is rather like
    /// the standard `strcpy` function, except that you do not
    /// have to worry about having enough space to copy the string.
    extern fn g_string_assign(p_string: *String, p_rval: [*:0]const u8) *glib.String;
    pub const assign = g_string_assign;

    /// Converts a `glib.String` to lowercase.
    extern fn g_string_down(p_string: *String) *glib.String;
    pub const down = g_string_down;

    /// Compares two strings for equality, returning `TRUE` if they are equal.
    /// For use with `glib.HashTable`.
    extern fn g_string_equal(p_v: *const String, p_v2: *const glib.String) c_int;
    pub const equal = g_string_equal;

    /// Removes `len` bytes from a `glib.String`, starting at position `pos`.
    /// The rest of the `glib.String` is shifted down to fill the gap.
    extern fn g_string_erase(p_string: *String, p_pos: isize, p_len: isize) *glib.String;
    pub const erase = g_string_erase;

    /// Frees the memory allocated for the `glib.String`.
    /// If `free_segment` is `TRUE` it also frees the character data.  If
    /// it's `FALSE`, the caller gains ownership of the buffer and must
    /// free it after use with `glib.free`.
    ///
    /// Instead of passing `FALSE` to this function, consider using
    /// `glib.String.freeAndSteal`.
    extern fn g_string_free(p_string: *String, p_free_segment: c_int) ?[*:0]u8;
    pub const free = g_string_free;

    /// Frees the memory allocated for the `glib.String`.
    ///
    /// The caller gains ownership of the buffer and
    /// must free it after use with `glib.free`.
    extern fn g_string_free_and_steal(p_string: *String) [*:0]u8;
    pub const freeAndSteal = g_string_free_and_steal;

    /// Transfers ownership of the contents of `string` to a newly allocated
    /// `glib.Bytes`.  The `glib.String` structure itself is deallocated, and it is
    /// therefore invalid to use `string` after invoking this function.
    ///
    /// Note that while `glib.String` ensures that its buffer always has a
    /// trailing nul character (not reflected in its "len"), the returned
    /// `glib.Bytes` does not include this extra nul; i.e. it has length exactly
    /// equal to the "len" member.
    extern fn g_string_free_to_bytes(p_string: *String) *glib.Bytes;
    pub const freeToBytes = g_string_free_to_bytes;

    /// Creates a hash code for `str`; for use with `glib.HashTable`.
    extern fn g_string_hash(p_str: *const String) c_uint;
    pub const hash = g_string_hash;

    /// Inserts a copy of a string into a `glib.String`,
    /// expanding it if necessary.
    extern fn g_string_insert(p_string: *String, p_pos: isize, p_val: [*:0]const u8) *glib.String;
    pub const insert = g_string_insert;

    /// Inserts a byte into a `glib.String`, expanding it if necessary.
    extern fn g_string_insert_c(p_string: *String, p_pos: isize, p_c: u8) *glib.String;
    pub const insertC = g_string_insert_c;

    /// Inserts `len` bytes of `val` into `string` at `pos`.
    ///
    /// If `len` is positive, `val` may contain embedded nuls and need
    /// not be nul-terminated. It is the caller's responsibility to
    /// ensure that `val` has at least `len` addressable bytes.
    ///
    /// If `len` is negative, `val` must be nul-terminated and `len`
    /// is considered to request the entire string length.
    ///
    /// If `pos` is -1, bytes are inserted at the end of the string.
    extern fn g_string_insert_len(p_string: *String, p_pos: isize, p_val: [*:0]const u8, p_len: isize) *glib.String;
    pub const insertLen = g_string_insert_len;

    /// Converts a Unicode character into UTF-8, and insert it
    /// into the string at the given position.
    extern fn g_string_insert_unichar(p_string: *String, p_pos: isize, p_wc: u32) *glib.String;
    pub const insertUnichar = g_string_insert_unichar;

    /// Overwrites part of a string, lengthening it if necessary.
    extern fn g_string_overwrite(p_string: *String, p_pos: usize, p_val: [*:0]const u8) *glib.String;
    pub const overwrite = g_string_overwrite;

    /// Overwrites part of a string, lengthening it if necessary.
    /// This function will work with embedded nuls.
    extern fn g_string_overwrite_len(p_string: *String, p_pos: usize, p_val: [*:0]const u8, p_len: isize) *glib.String;
    pub const overwriteLen = g_string_overwrite_len;

    /// Adds a string on to the start of a `glib.String`,
    /// expanding it if necessary.
    extern fn g_string_prepend(p_string: *String, p_val: [*:0]const u8) *glib.String;
    pub const prepend = g_string_prepend;

    /// Adds a byte onto the start of a `glib.String`,
    /// expanding it if necessary.
    extern fn g_string_prepend_c(p_string: *String, p_c: u8) *glib.String;
    pub const prependC = g_string_prepend_c;

    /// Prepends `len` bytes of `val` to `string`.
    ///
    /// If `len` is positive, `val` may contain embedded nuls and need
    /// not be nul-terminated. It is the caller's responsibility to
    /// ensure that `val` has at least `len` addressable bytes.
    ///
    /// If `len` is negative, `val` must be nul-terminated and `len`
    /// is considered to request the entire string length. This
    /// makes `glib.String.prependLen` equivalent to `glib.String.prepend`.
    extern fn g_string_prepend_len(p_string: *String, p_val: [*:0]const u8, p_len: isize) *glib.String;
    pub const prependLen = g_string_prepend_len;

    /// Converts a Unicode character into UTF-8, and prepends it
    /// to the string.
    extern fn g_string_prepend_unichar(p_string: *String, p_wc: u32) *glib.String;
    pub const prependUnichar = g_string_prepend_unichar;

    /// Writes a formatted string into a `glib.String`.
    /// This is similar to the standard `sprintf` function,
    /// except that the `glib.String` buffer automatically expands
    /// to contain the results. The previous contents of the
    /// `glib.String` are destroyed.
    extern fn g_string_printf(p_string: *String, p_format: [*:0]const u8, ...) void;
    pub const printf = g_string_printf;

    /// Replaces the string `find` with the string `replace` in a `glib.String` up to
    /// `limit` times. If the number of instances of `find` in the `glib.String` is
    /// less than `limit`, all instances are replaced. If `limit` is `0`,
    /// all instances of `find` are replaced.
    ///
    /// If `find` is the empty string, since versions 2.69.1 and 2.68.4 the
    /// replacement will be inserted no more than once per possible position
    /// (beginning of string, end of string and between characters). This did
    /// not work correctly in earlier versions.
    extern fn g_string_replace(p_string: *String, p_find: [*:0]const u8, p_replace: [*:0]const u8, p_limit: c_uint) c_uint;
    pub const replace = g_string_replace;

    /// Sets the length of a `glib.String`. If the length is less than
    /// the current length, the string will be truncated. If the
    /// length is greater than the current length, the contents
    /// of the newly added area are undefined. (However, as
    /// always, string->str[string->len] will be a nul byte.)
    extern fn g_string_set_size(p_string: *String, p_len: usize) *glib.String;
    pub const setSize = g_string_set_size;

    /// Cuts off the end of the GString, leaving the first `len` bytes.
    extern fn g_string_truncate(p_string: *String, p_len: usize) *glib.String;
    pub const truncate = g_string_truncate;

    /// Converts a `glib.String` to uppercase.
    extern fn g_string_up(p_string: *String) *glib.String;
    pub const up = g_string_up;

    /// Writes a formatted string into a `glib.String`.
    /// This function is similar to `glib.String.printf` except that
    /// the arguments to the format string are passed as a va_list.
    extern fn g_string_vprintf(p_string: *String, p_format: [*:0]const u8, p_args: std.builtin.VaList) void;
    pub const vprintf = g_string_vprintf;

    extern fn g_gstring_get_type() usize;
    pub const getGObjectType = g_gstring_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GStringChunk` provides efficient storage of groups of strings
///
/// String chunks are used to store groups of strings. Memory is
/// allocated in blocks, and as strings are added to the `GStringChunk`
/// they are copied into the next free position in a block. When a block
/// is full a new block is allocated.
///
/// When storing a large number of strings, string chunks are more
/// efficient than using `glib.strdup` since fewer calls to ``malloc``
/// are needed, and less memory is wasted in memory allocation overheads.
///
/// By adding strings with `glib.StringChunk.insertConst` it is also
/// possible to remove duplicates.
///
/// To create a new `GStringChunk` use `glib.StringChunk.new`.
///
/// To add strings to a `GStringChunk` use `glib.StringChunk.insert`.
///
/// To add strings to a `GStringChunk`, but without duplicating strings
/// which are already in the `GStringChunk`, use `glib.StringChunk.insertConst`.
///
/// To free the entire `GStringChunk` use `glib.StringChunk.free`.
/// It is not possible to free individual strings.
pub const StringChunk = opaque {
    /// Creates a new `glib.StringChunk`.
    extern fn g_string_chunk_new(p_size: usize) *glib.StringChunk;
    pub const new = g_string_chunk_new;

    /// Frees all strings contained within the `glib.StringChunk`.
    /// After calling `glib.StringChunk.clear` it is not safe to
    /// access any of the strings which were contained within it.
    extern fn g_string_chunk_clear(p_chunk: *StringChunk) void;
    pub const clear = g_string_chunk_clear;

    /// Frees all memory allocated by the `glib.StringChunk`.
    /// After calling `glib.StringChunk.free` it is not safe to
    /// access any of the strings which were contained within it.
    extern fn g_string_chunk_free(p_chunk: *StringChunk) void;
    pub const free = g_string_chunk_free;

    /// Adds a copy of `string` to the `glib.StringChunk`.
    /// It returns a pointer to the new copy of the string
    /// in the `glib.StringChunk`. The characters in the string
    /// can be changed, if necessary, though you should not
    /// change anything after the end of the string.
    ///
    /// Unlike `glib.StringChunk.insertConst`, this function
    /// does not check for duplicates. Also strings added
    /// with `glib.StringChunk.insert` will not be searched
    /// by `glib.StringChunk.insertConst` when looking for
    /// duplicates.
    extern fn g_string_chunk_insert(p_chunk: *StringChunk, p_string: [*:0]const u8) [*:0]u8;
    pub const insert = g_string_chunk_insert;

    /// Adds a copy of `string` to the `glib.StringChunk`, unless the same
    /// string has already been added to the `glib.StringChunk` with
    /// `glib.StringChunk.insertConst`.
    ///
    /// This function is useful if you need to copy a large number
    /// of strings but do not want to waste space storing duplicates.
    /// But you must remember that there may be several pointers to
    /// the same string, and so any changes made to the strings
    /// should be done very carefully.
    ///
    /// Note that `glib.StringChunk.insertConst` will not return a
    /// pointer to a string added with `glib.StringChunk.insert`, even
    /// if they do match.
    extern fn g_string_chunk_insert_const(p_chunk: *StringChunk, p_string: [*:0]const u8) [*:0]u8;
    pub const insertConst = g_string_chunk_insert_const;

    /// Adds a copy of the first `len` bytes of `string` to the `glib.StringChunk`.
    /// The copy is nul-terminated.
    ///
    /// Since this function does not stop at nul bytes, it is the caller's
    /// responsibility to ensure that `string` has at least `len` addressable
    /// bytes.
    ///
    /// The characters in the returned string can be changed, if necessary,
    /// though you should not change anything after the end of the string.
    extern fn g_string_chunk_insert_len(p_chunk: *StringChunk, p_string: [*:0]const u8, p_len: isize) [*:0]u8;
    pub const insertLen = g_string_chunk_insert_len;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GStrvBuilder` is a helper object to build a `NULL`-terminated string arrays.
///
/// The following example shows how to build a two element array:
///
/// ```c
///   g_autoptr(GStrvBuilder) builder = g_strv_builder_new ();
///   g_strv_builder_add (builder, "hello");
///   g_strv_builder_add (builder, "world");
///
///   g_auto(GStrv) array = g_strv_builder_end (builder);
///
///   g_assert_true (g_strv_equal (array, (const char *[]) { "hello", "world", NULL }));
/// ```
pub const StrvBuilder = opaque {
    /// Creates a new `glib.StrvBuilder` with a reference count of 1.
    /// Use `glib.StrvBuilder.unref` on the returned value when no longer needed.
    extern fn g_strv_builder_new() *glib.StrvBuilder;
    pub const new = g_strv_builder_new;

    /// Add a string to the end of the array.
    ///
    /// Since 2.68
    extern fn g_strv_builder_add(p_builder: *StrvBuilder, p_value: [*:0]const u8) void;
    pub const add = g_strv_builder_add;

    /// Appends all the given strings to the builder.
    ///
    /// Since 2.70
    extern fn g_strv_builder_add_many(p_builder: *StrvBuilder, ...) void;
    pub const addMany = g_strv_builder_add_many;

    /// Appends all the strings in the given vector to the builder.
    ///
    /// Since 2.70
    extern fn g_strv_builder_addv(p_builder: *StrvBuilder, p_value: [*][*:0]const u8) void;
    pub const addv = g_strv_builder_addv;

    /// Ends the builder process and returns the constructed NULL-terminated string
    /// array. The returned value should be freed with `glib.strfreev` when no longer
    /// needed.
    extern fn g_strv_builder_end(p_builder: *StrvBuilder) [*][*:0]u8;
    pub const end = g_strv_builder_end;

    /// Atomically increments the reference count of `builder` by one.
    /// This function is thread-safe and may be called from any thread.
    extern fn g_strv_builder_ref(p_builder: *StrvBuilder) *glib.StrvBuilder;
    pub const ref = g_strv_builder_ref;

    /// Add a string to the end of the array. After `value` belongs to the
    /// `glib.StrvBuilder` and may no longer be modified by the caller.
    ///
    /// Since 2.80
    extern fn g_strv_builder_take(p_builder: *StrvBuilder, p_value: [*:0]u8) void;
    pub const take = g_strv_builder_take;

    /// Decreases the reference count on `builder`.
    ///
    /// In the event that there are no more references, releases all memory
    /// associated with the `glib.StrvBuilder`.
    extern fn g_strv_builder_unref(p_builder: *StrvBuilder) void;
    pub const unref = g_strv_builder_unref;

    /// Decreases the reference count on the string vector builder, and returns
    /// its contents as a `NULL`-terminated string array.
    ///
    /// This function is especially useful for cases where it's not possible
    /// to use ``g_autoptr``.
    ///
    /// ```c
    /// GStrvBuilder *builder = g_strv_builder_new ();
    /// g_strv_builder_add (builder, "hello");
    /// g_strv_builder_add (builder, "world");
    ///
    /// GStrv array = g_strv_builder_unref_to_strv (builder);
    ///
    /// g_assert_true (g_strv_equal (array, (const char *[]) { "hello", "world", NULL }));
    ///
    /// g_strfreev (array);
    /// ```
    extern fn g_strv_builder_unref_to_strv(p_builder: *StrvBuilder) [*][*:0]u8;
    pub const unrefToStrv = g_strv_builder_unref_to_strv;

    extern fn g_strv_builder_get_type() usize;
    pub const getGObjectType = g_strv_builder_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// An opaque structure representing a test case.
pub const TestCase = opaque {
    /// Free the `test_case`.
    extern fn g_test_case_free(p_test_case: *TestCase) void;
    pub const free = g_test_case_free;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

pub const TestConfig = extern struct {
    f_test_initialized: c_int,
    f_test_quick: c_int,
    f_test_perf: c_int,
    f_test_verbose: c_int,
    f_test_quiet: c_int,
    f_test_undefined: c_int,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

pub const TestLogBuffer = extern struct {
    f_data: ?*glib.String,
    f_msgs: ?*glib.SList,

    /// Internal function for gtester to decode test log messages, no ABI guarantees provided.
    extern fn g_test_log_buffer_new() *glib.TestLogBuffer;
    pub const new = g_test_log_buffer_new;

    /// Internal function for gtester to free test log messages, no ABI guarantees provided.
    extern fn g_test_log_buffer_free(p_tbuffer: *TestLogBuffer) void;
    pub const free = g_test_log_buffer_free;

    /// Internal function for gtester to retrieve test log messages, no ABI guarantees provided.
    extern fn g_test_log_buffer_pop(p_tbuffer: *TestLogBuffer) *glib.TestLogMsg;
    pub const pop = g_test_log_buffer_pop;

    /// Internal function for gtester to decode test log messages, no ABI guarantees provided.
    extern fn g_test_log_buffer_push(p_tbuffer: *TestLogBuffer, p_n_bytes: c_uint, p_bytes: *const u8) void;
    pub const push = g_test_log_buffer_push;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

pub const TestLogMsg = extern struct {
    f_log_type: glib.TestLogType,
    f_n_strings: c_uint,
    f_strings: ?*[*:0]u8,
    f_n_nums: c_uint,
    f_nums: ?*c_longdouble,

    /// Internal function for gtester to free test log messages, no ABI guarantees provided.
    extern fn g_test_log_msg_free(p_tmsg: *TestLogMsg) void;
    pub const free = g_test_log_msg_free;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// An opaque structure representing a test suite.
pub const TestSuite = opaque {
    /// Adds `test_case` to `suite`.
    extern fn g_test_suite_add(p_suite: *TestSuite, p_test_case: *glib.TestCase) void;
    pub const add = g_test_suite_add;

    /// Adds `nestedsuite` to `suite`.
    extern fn g_test_suite_add_suite(p_suite: *TestSuite, p_nestedsuite: *glib.TestSuite) void;
    pub const addSuite = g_test_suite_add_suite;

    /// Frees the `suite` and all nested suites.
    extern fn g_test_suite_free(p_suite: *TestSuite) void;
    pub const free = g_test_suite_free;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.Thread` struct represents a running thread. This struct
/// is returned by `glib.Thread.new` or `glib.Thread.tryNew`. You can
/// obtain the `glib.Thread` struct representing the current thread by
/// calling `glib.threadSelf`.
///
/// GThread is refcounted, see `glib.Thread.ref` and `glib.Thread.unref`.
/// The thread represented by it holds a reference while it is running,
/// and `glib.Thread.join` consumes the reference that it is given, so
/// it is normally not necessary to manage GThread references
/// explicitly.
///
/// The structure is opaque -- none of its fields may be directly
/// accessed.
pub const Thread = extern struct {
    f_func: ?glib.ThreadFunc,
    f_data: ?*anyopaque,
    f_joinable: c_int,
    f_priority: glib.ThreadPriority,

    /// This function creates a new thread.
    ///
    /// The new thread executes the function `func` with the argument `data`.
    /// If the thread was created successfully, it is returned.
    ///
    /// `error` can be `NULL` to ignore errors, or non-`NULL` to report errors.
    /// The error is set, if and only if the function returns `NULL`.
    ///
    /// This function returns a reference to the created thread only if
    /// `joinable` is `TRUE`.  In that case, you must free this reference by
    /// calling `glib.Thread.unref` or `glib.Thread.join`.  If `joinable` is `FALSE`
    /// then you should probably not touch the return value.
    extern fn g_thread_create(p_func: glib.ThreadFunc, p_data: ?*anyopaque, p_joinable: c_int, p_error: ?*?*glib.Error) ?*glib.Thread;
    pub const create = g_thread_create;

    /// This function creates a new thread.
    extern fn g_thread_create_full(p_func: glib.ThreadFunc, p_data: ?*anyopaque, p_stack_size: c_ulong, p_joinable: c_int, p_bound: c_int, p_priority: glib.ThreadPriority, p_error: ?*?*glib.Error) ?*glib.Thread;
    pub const createFull = g_thread_create_full;

    extern fn g_thread_error_quark() glib.Quark;
    pub const errorQuark = g_thread_error_quark;

    /// Terminates the current thread.
    ///
    /// If another thread is waiting for us using `glib.Thread.join` then the
    /// waiting thread will be woken up and get `retval` as the return value
    /// of `glib.Thread.join`.
    ///
    /// Calling `glib.threadExit` with a parameter `retval` is equivalent to
    /// returning `retval` from the function `func`, as given to `glib.Thread.new`.
    ///
    /// You must only call `glib.threadExit` from a thread that you created
    /// yourself with `glib.Thread.new` or related APIs. You must not call
    /// this function from a thread created with another threading library
    /// or or from within a `glib.ThreadPool`.
    extern fn g_thread_exit(p_retval: ?*anyopaque) void;
    pub const exit = g_thread_exit;

    /// Call `thread_func` on all `GThreads` that have been
    /// created with `glib.threadCreate`.
    ///
    /// Note that threads may decide to exit while `thread_func` is
    /// running, so without intimate knowledge about the lifetime of
    /// foreign threads, `thread_func` shouldn't access the GThread*
    /// pointer passed in as first argument. However, `thread_func` will
    /// not be called for threads which are known to have exited already.
    ///
    /// Due to thread lifetime checks, this function has an execution complexity
    /// which is quadratic in the number of existing threads.
    extern fn g_thread_foreach(p_thread_func: glib.Func, p_user_data: ?*anyopaque) void;
    pub const foreach = g_thread_foreach;

    /// Indicates if `glib.threadInit` has been called.
    extern fn g_thread_get_initialized() c_int;
    pub const getInitialized = g_thread_get_initialized;

    /// If you use GLib from more than one thread, you must initialize the
    /// thread system by calling `glib.threadInit`.
    ///
    /// Since version 2.24, calling `glib.threadInit` multiple times is allowed,
    /// but nothing happens except for the first call.
    ///
    /// Since version 2.32, GLib does not support custom thread implementations
    /// anymore and the `vtable` parameter is ignored and you should pass `NULL`.
    ///
    /// ::: note
    ///     `glib.threadInit` must not be called directly or indirectly in a
    ///     callback from GLib. Also no mutexes may be currently locked
    ///     while calling `glib.threadInit`.
    ///
    /// ::: note
    ///     To use `glib.threadInit` in your program, you have to link with
    ///     the libraries that the command `pkg-config --libs gthread-2.0`
    ///     outputs. This is not the case for all the other thread-related
    ///     functions of GLib. Those can be used without having to link
    ///     with the thread libraries.
    extern fn g_thread_init(p_vtable: ?*anyopaque) void;
    pub const init = g_thread_init;

    extern fn g_thread_init_with_errorcheck_mutexes(p_vtable: ?*anyopaque) void;
    pub const initWithErrorcheckMutexes = g_thread_init_with_errorcheck_mutexes;

    /// This function returns the `glib.Thread` corresponding to the
    /// current thread. Note that this function does not increase
    /// the reference count of the returned struct.
    ///
    /// This function will return a `glib.Thread` even for threads that
    /// were not created by GLib (i.e. those created by other threading
    /// APIs). This may be useful for thread identification purposes
    /// (i.e. comparisons) but you must not use GLib functions (such
    /// as `glib.Thread.join`) on these threads.
    extern fn g_thread_self() *glib.Thread;
    pub const self = g_thread_self;

    /// Causes the calling thread to voluntarily relinquish the CPU, so
    /// that other threads can run.
    ///
    /// This function is often used as a method to make busy wait less evil.
    extern fn g_thread_yield() void;
    pub const yield = g_thread_yield;

    /// This function creates a new thread. The new thread starts by invoking
    /// `func` with the argument data. The thread will run until `func` returns
    /// or until `glib.threadExit` is called from the new thread. The return value
    /// of `func` becomes the return value of the thread, which can be obtained
    /// with `glib.Thread.join`.
    ///
    /// The `name` can be useful for discriminating threads in a debugger.
    /// It is not used for other purposes and does not have to be unique.
    /// Some systems restrict the length of `name` to 16 bytes.
    ///
    /// If the thread can not be created the program aborts. See
    /// `glib.Thread.tryNew` if you want to attempt to deal with failures.
    ///
    /// If you are using threads to offload (potentially many) short-lived tasks,
    /// `glib.ThreadPool` may be more appropriate than manually spawning and tracking
    /// multiple `GThreads`.
    ///
    /// To free the struct returned by this function, use `glib.Thread.unref`.
    /// Note that `glib.Thread.join` implicitly unrefs the `glib.Thread` as well.
    ///
    /// New threads by default inherit their scheduler policy (POSIX) or thread
    /// priority (Windows) of the thread creating the new thread.
    ///
    /// This behaviour changed in GLib 2.64: before threads on Windows were not
    /// inheriting the thread priority but were spawned with the default priority.
    /// Starting with GLib 2.64 the behaviour is now consistent between Windows and
    /// POSIX and all threads inherit their parent thread's priority.
    extern fn g_thread_new(p_name: ?[*:0]const u8, p_func: glib.ThreadFunc, p_data: ?*anyopaque) *glib.Thread;
    pub const new = g_thread_new;

    /// This function is the same as `glib.Thread.new` except that
    /// it allows for the possibility of failure.
    ///
    /// If a thread can not be created (due to resource limits),
    /// `error` is set and `NULL` is returned.
    extern fn g_thread_try_new(p_name: ?[*:0]const u8, p_func: glib.ThreadFunc, p_data: ?*anyopaque, p_error: ?*?*glib.Error) ?*glib.Thread;
    pub const tryNew = g_thread_try_new;

    /// Gets the name of the thread.
    ///
    /// This function is intended for debugging purposes.
    extern fn g_thread_get_name(p_thread: *Thread) [*:0]const u8;
    pub const getName = g_thread_get_name;

    /// Waits until `thread` finishes, i.e. the function `func`, as
    /// given to `glib.Thread.new`, returns or `glib.threadExit` is called.
    /// If `thread` has already terminated, then `glib.Thread.join`
    /// returns immediately.
    ///
    /// Any thread can wait for any other thread by calling `glib.Thread.join`,
    /// not just its 'creator'. Calling `glib.Thread.join` from multiple threads
    /// for the same `thread` leads to undefined behaviour.
    ///
    /// The value returned by `func` or given to `glib.threadExit` is
    /// returned by this function.
    ///
    /// `glib.Thread.join` consumes the reference to the passed-in `thread`.
    /// This will usually cause the `glib.Thread` struct and associated resources
    /// to be freed. Use `glib.Thread.ref` to obtain an extra reference if you
    /// want to keep the GThread alive beyond the `glib.Thread.join` call.
    extern fn g_thread_join(p_thread: *Thread) ?*anyopaque;
    pub const join = g_thread_join;

    /// Increase the reference count on `thread`.
    extern fn g_thread_ref(p_thread: *Thread) *glib.Thread;
    pub const ref = g_thread_ref;

    /// This function does nothing.
    extern fn g_thread_set_priority(p_thread: *Thread, p_priority: glib.ThreadPriority) void;
    pub const setPriority = g_thread_set_priority;

    /// Decrease the reference count on `thread`, possibly freeing all
    /// resources associated with it.
    ///
    /// Note that each thread holds a reference to its `glib.Thread` while
    /// it is running, so it is safe to drop your own reference to it
    /// if you don't need it anymore.
    extern fn g_thread_unref(p_thread: *Thread) void;
    pub const unref = g_thread_unref;

    extern fn g_thread_get_type() usize;
    pub const getGObjectType = g_thread_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// This function table is no longer used by `glib.threadInit`
/// to initialize the thread system.
pub const ThreadFunctions = extern struct {
    /// virtual function pointer for `glib.mutexNew`
    f_mutex_new: ?*const fn () callconv(.c) *glib.Mutex,
    /// virtual function pointer for `glib.Mutex.lock`
    f_mutex_lock: ?*const fn (p_mutex: *glib.Mutex) callconv(.c) void,
    /// virtual function pointer for `glib.Mutex.trylock`
    f_mutex_trylock: ?*const fn (p_mutex: *glib.Mutex) callconv(.c) c_int,
    /// virtual function pointer for `glib.Mutex.unlock`
    f_mutex_unlock: ?*const fn (p_mutex: *glib.Mutex) callconv(.c) void,
    /// virtual function pointer for `glib.Mutex.free`
    f_mutex_free: ?*const fn (p_mutex: *glib.Mutex) callconv(.c) void,
    /// virtual function pointer for `glib.condNew`
    f_cond_new: ?*const fn () callconv(.c) *glib.Cond,
    /// virtual function pointer for `glib.Cond.signal`
    f_cond_signal: ?*const fn (p_cond: *glib.Cond) callconv(.c) void,
    /// virtual function pointer for `glib.Cond.broadcast`
    f_cond_broadcast: ?*const fn (p_cond: *glib.Cond) callconv(.c) void,
    /// virtual function pointer for `glib.Cond.wait`
    f_cond_wait: ?*const fn (p_cond: *glib.Cond, p_mutex: *glib.Mutex) callconv(.c) void,
    /// virtual function pointer for `glib.Cond.timedWait`
    f_cond_timed_wait: ?*const fn (p_cond: *glib.Cond, p_mutex: *glib.Mutex, p_end_time: *glib.TimeVal) callconv(.c) c_int,
    /// virtual function pointer for `glib.Cond.free`
    f_cond_free: ?*const fn (p_cond: *glib.Cond) callconv(.c) void,
    /// virtual function pointer for `glib.privateNew`
    f_private_new: ?*const fn (p_destructor: glib.DestroyNotify) callconv(.c) *glib.Private,
    /// virtual function pointer for `glib.Private.get`
    f_private_get: ?*const fn (p_private_key: *glib.Private) callconv(.c) *anyopaque,
    /// virtual function pointer for `glib.Private.set`
    f_private_set: ?*const fn (p_private_key: *glib.Private, p_data: *anyopaque) callconv(.c) void,
    /// virtual function pointer for `glib.threadCreate`
    f_thread_create: ?*const fn (p_func: glib.ThreadFunc, p_data: ?*anyopaque, p_stack_size: c_ulong, p_joinable: c_int, p_bound: c_int, p_priority: glib.ThreadPriority, p_thread: *anyopaque, p_error: ?*?*glib.Error) callconv(.c) void,
    /// virtual function pointer for `glib.threadYield`
    f_thread_yield: ?*const fn () callconv(.c) void,
    /// virtual function pointer for `glib.Thread.join`
    f_thread_join: ?*const fn (p_thread: *anyopaque) callconv(.c) void,
    /// virtual function pointer for `glib.threadExit`
    f_thread_exit: ?*const fn () callconv(.c) void,
    /// virtual function pointer for
    ///                       `glib.Thread.setPriority`
    f_thread_set_priority: ?*const fn (p_thread: *anyopaque, p_priority: glib.ThreadPriority) callconv(.c) void,
    /// virtual function pointer for `glib.threadSelf`
    f_thread_self: ?*const fn (p_thread: *anyopaque) callconv(.c) void,
    /// used internally by recursive mutex locks and by some
    ///                assertion checks
    f_thread_equal: ?*const fn (p_thread1: *anyopaque, p_thread2: *anyopaque) callconv(.c) c_int,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `GThreadPool` struct represents a thread pool.
///
/// A thread pool is useful when you wish to asynchronously fork out the execution of work
/// and continue working in your own thread. If that will happen often, the overhead of starting
/// and destroying a thread each time might be too high. In such cases reusing already started
/// threads seems like a good idea. And it indeed is, but implementing this can be tedious
/// and error-prone.
///
/// Therefore GLib provides thread pools for your convenience. An added advantage is, that the
/// threads can be shared between the different subsystems of your program, when they are using GLib.
///
/// To create a new thread pool, you use `glib.ThreadPool.new`.
/// It is destroyed by `glib.ThreadPool.free`.
///
/// If you want to execute a certain task within a thread pool, use `glib.ThreadPool.push`.
///
/// To get the current number of running threads you call `glib.ThreadPool.getNumThreads`.
/// To get the number of still unprocessed tasks you call `glib.ThreadPool.unprocessed`.
/// To control the maximum number of threads for a thread pool, you use
/// `glib.ThreadPool.getMaxThreads`. and `glib.ThreadPool.setMaxThreads`.
///
/// Finally you can control the number of unused threads, that are kept alive by GLib for future use.
/// The current number can be fetched with `glib.ThreadPool.getNumUnusedThreads`.
/// The maximum number can be controlled by `glib.ThreadPool.getMaxUnusedThreads` and
/// `glib.ThreadPool.setMaxUnusedThreads`. All currently unused threads
/// can be stopped by calling `glib.ThreadPool.stopUnusedThreads`.
pub const ThreadPool = extern struct {
    /// the function to execute in the threads of this pool
    f_func: ?glib.Func,
    /// the user data for the threads of this pool
    f_user_data: ?*anyopaque,
    /// are all threads exclusive to this pool
    f_exclusive: c_int,

    /// This function will return the maximum `interval` that a
    /// thread will wait in the thread pool for new tasks before
    /// being stopped.
    ///
    /// If this function returns 0, threads waiting in the thread
    /// pool for new work are not stopped.
    extern fn g_thread_pool_get_max_idle_time() c_uint;
    pub const getMaxIdleTime = g_thread_pool_get_max_idle_time;

    /// Returns the maximal allowed number of unused threads.
    extern fn g_thread_pool_get_max_unused_threads() c_int;
    pub const getMaxUnusedThreads = g_thread_pool_get_max_unused_threads;

    /// Returns the number of currently unused threads.
    extern fn g_thread_pool_get_num_unused_threads() c_uint;
    pub const getNumUnusedThreads = g_thread_pool_get_num_unused_threads;

    /// This function creates a new thread pool.
    ///
    /// Whenever you call `glib.ThreadPool.push`, either a new thread is
    /// created or an unused one is reused. At most `max_threads` threads
    /// are running concurrently for this thread pool. `max_threads` = -1
    /// allows unlimited threads to be created for this thread pool. The
    /// newly created or reused thread now executes the function `func`
    /// with the two arguments. The first one is the parameter to
    /// `glib.ThreadPool.push` and the second one is `user_data`.
    ///
    /// Pass `glib.getNumProcessors` to `max_threads` to create as many threads as
    /// there are logical processors on the system. This will not pin each thread to
    /// a specific processor.
    ///
    /// The parameter `exclusive` determines whether the thread pool owns
    /// all threads exclusive or shares them with other thread pools.
    /// If `exclusive` is `TRUE`, `max_threads` threads are started
    /// immediately and they will run exclusively for this thread pool
    /// until it is destroyed by `glib.ThreadPool.free`. If `exclusive` is
    /// `FALSE`, threads are created when needed and shared between all
    /// non-exclusive thread pools. This implies that `max_threads` may
    /// not be -1 for exclusive thread pools. Besides, exclusive thread
    /// pools are not affected by `glib.threadPoolSetMaxIdleTime`
    /// since their threads are never considered idle and returned to the
    /// global pool.
    ///
    /// Note that the threads used by exclusive thread pools will all inherit the
    /// scheduler settings of the current thread while the threads used by
    /// non-exclusive thread pools will inherit the scheduler settings from the
    /// first thread that created such a thread pool.
    ///
    /// At least one thread will be spawned when this function is called, either to
    /// create the `max_threads` exclusive threads, or to preserve the scheduler
    /// settings of the current thread for future spawns.
    ///
    /// `error` can be `NULL` to ignore errors, or non-`NULL` to report
    /// errors. An error can only occur when `exclusive` is set to `TRUE`
    /// and not all `max_threads` threads could be created.
    /// See `glib.ThreadError` for possible errors that may occur.
    /// Note, even in case of error a valid `glib.ThreadPool` is returned.
    extern fn g_thread_pool_new(p_func: glib.Func, p_user_data: ?*anyopaque, p_max_threads: c_int, p_exclusive: c_int, p_error: ?*?*glib.Error) ?*glib.ThreadPool;
    pub const new = g_thread_pool_new;

    /// This function creates a new thread pool similar to `glib.ThreadPool.new`
    /// but allowing `item_free_func` to be specified to free the data passed
    /// to `glib.ThreadPool.push` in the case that the `glib.ThreadPool` is stopped
    /// and freed before all tasks have been executed.
    ///
    /// `item_free_func` will *not* be called on items successfully passed to `func`.
    /// `func` is responsible for freeing the items passed to it.
    extern fn g_thread_pool_new_full(p_func: glib.Func, p_user_data: ?*anyopaque, p_item_free_func: ?glib.DestroyNotify, p_max_threads: c_int, p_exclusive: c_int, p_error: ?*?*glib.Error) ?*glib.ThreadPool;
    pub const newFull = g_thread_pool_new_full;

    /// This function will set the maximum `interval` that a thread
    /// waiting in the pool for new tasks can be idle for before
    /// being stopped. This function is similar to calling
    /// `glib.threadPoolStopUnusedThreads` on a regular timeout,
    /// except this is done on a per thread basis.
    ///
    /// By setting `interval` to 0, idle threads will not be stopped.
    ///
    /// The default value is 15000 (15 seconds).
    extern fn g_thread_pool_set_max_idle_time(p_interval: c_uint) void;
    pub const setMaxIdleTime = g_thread_pool_set_max_idle_time;

    /// Sets the maximal number of unused threads to `max_threads`.
    /// If `max_threads` is -1, no limit is imposed on the number
    /// of unused threads.
    ///
    /// The default value is 8 since GLib 2.84. Previously the default value was 2.
    extern fn g_thread_pool_set_max_unused_threads(p_max_threads: c_int) void;
    pub const setMaxUnusedThreads = g_thread_pool_set_max_unused_threads;

    /// Stops all currently unused threads. This does not change the
    /// maximal number of unused threads. This function can be used to
    /// regularly stop all unused threads e.g. from `glib.timeoutAdd`.
    extern fn g_thread_pool_stop_unused_threads() void;
    pub const stopUnusedThreads = g_thread_pool_stop_unused_threads;

    /// Frees all resources allocated for `pool`.
    ///
    /// If `immediate` is `TRUE`, no new task is processed for `pool`.
    /// Otherwise `pool` is not freed before the last task is processed.
    /// Note however, that no thread of this pool is interrupted while
    /// processing a task. Instead at least all still running threads
    /// can finish their tasks before the `pool` is freed.
    ///
    /// If `wait_` is `TRUE`, this function does not return before all
    /// tasks to be processed (dependent on `immediate`, whether all
    /// or only the currently running) are ready.
    /// Otherwise this function returns immediately.
    ///
    /// After calling this function `pool` must not be used anymore.
    extern fn g_thread_pool_free(p_pool: *ThreadPool, p_immediate: c_int, p_wait_: c_int) void;
    pub const free = g_thread_pool_free;

    /// Returns the maximal number of threads for `pool`.
    extern fn g_thread_pool_get_max_threads(p_pool: *ThreadPool) c_int;
    pub const getMaxThreads = g_thread_pool_get_max_threads;

    /// Returns the number of threads currently running in `pool`.
    extern fn g_thread_pool_get_num_threads(p_pool: *ThreadPool) c_uint;
    pub const getNumThreads = g_thread_pool_get_num_threads;

    /// Moves the item to the front of the queue of unprocessed
    /// items, so that it will be processed next.
    extern fn g_thread_pool_move_to_front(p_pool: *ThreadPool, p_data: ?*anyopaque) c_int;
    pub const moveToFront = g_thread_pool_move_to_front;

    /// Inserts `data` into the list of tasks to be executed by `pool`.
    ///
    /// When the number of currently running threads is lower than the
    /// maximal allowed number of threads, a new thread is started (or
    /// reused) with the properties given to `glib.ThreadPool.new`.
    /// Otherwise, `data` stays in the queue until a thread in this pool
    /// finishes its previous task and processes `data`.
    ///
    /// `error` can be `NULL` to ignore errors, or non-`NULL` to report
    /// errors. An error can only occur when a new thread couldn't be
    /// created. In that case `data` is simply appended to the queue of
    /// work to do.
    ///
    /// Before version 2.32, this function did not return a success status.
    extern fn g_thread_pool_push(p_pool: *ThreadPool, p_data: ?*anyopaque, p_error: ?*?*glib.Error) c_int;
    pub const push = g_thread_pool_push;

    /// Sets the maximal allowed number of threads for `pool`.
    /// A value of -1 means that the maximal number of threads
    /// is unlimited. If `pool` is an exclusive thread pool, setting
    /// the maximal number of threads to -1 is not allowed.
    ///
    /// Setting `max_threads` to 0 means stopping all work for `pool`.
    /// It is effectively frozen until `max_threads` is set to a non-zero
    /// value again.
    ///
    /// A thread is never terminated while calling `func`, as supplied by
    /// `glib.ThreadPool.new`. Instead the maximal number of threads only
    /// has effect for the allocation of new threads in `glib.ThreadPool.push`.
    /// A new thread is allocated, whenever the number of currently
    /// running threads in `pool` is smaller than the maximal number.
    ///
    /// `error` can be `NULL` to ignore errors, or non-`NULL` to report
    /// errors. An error can only occur when a new thread couldn't be
    /// created.
    ///
    /// Before version 2.32, this function did not return a success status.
    extern fn g_thread_pool_set_max_threads(p_pool: *ThreadPool, p_max_threads: c_int, p_error: ?*?*glib.Error) c_int;
    pub const setMaxThreads = g_thread_pool_set_max_threads;

    /// Sets the function used to sort the list of tasks. This allows the
    /// tasks to be processed by a priority determined by `func`, and not
    /// just in the order in which they were added to the pool.
    ///
    /// Note, if the maximum number of threads is more than 1, the order
    /// that threads are executed cannot be guaranteed 100%. Threads are
    /// scheduled by the operating system and are executed at random. It
    /// cannot be assumed that threads are executed in the order they are
    /// created.
    extern fn g_thread_pool_set_sort_function(p_pool: *ThreadPool, p_func: glib.CompareDataFunc, p_user_data: ?*anyopaque) void;
    pub const setSortFunction = g_thread_pool_set_sort_function;

    /// Returns the number of tasks still unprocessed in `pool`.
    extern fn g_thread_pool_unprocessed(p_pool: *ThreadPool) c_uint;
    pub const unprocessed = g_thread_pool_unprocessed;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Represents a precise time, with seconds and microseconds.
///
/// Similar to the struct timeval returned by the ``gettimeofday``
/// UNIX system call.
///
/// GLib is attempting to unify around the use of 64-bit integers to
/// represent microsecond-precision time. As such, this type will be
/// removed from a future version of GLib. A consequence of using `glong` for
/// `tv_sec` is that on 32-bit systems `GTimeVal` is subject to the year 2038
/// problem.
pub const TimeVal = extern struct {
    /// seconds
    f_tv_sec: c_long,
    /// microseconds
    f_tv_usec: c_long,

    /// Converts a string containing an ISO 8601 encoded date and time
    /// to a `glib.TimeVal` and puts it into `time_`.
    ///
    /// `iso_date` must include year, month, day, hours, minutes, and
    /// seconds. It can optionally include fractions of a second and a time
    /// zone indicator. (In the absence of any time zone indication, the
    /// timestamp is assumed to be in local time.)
    ///
    /// Any leading or trailing space in `iso_date` is ignored.
    ///
    /// This function was deprecated, along with `glib.TimeVal` itself, in GLib 2.62.
    /// Equivalent functionality is available using code like:
    /// ```
    /// GDateTime *dt = g_date_time_new_from_iso8601 (iso8601_string, NULL);
    /// gint64 time_val = g_date_time_to_unix (dt);
    /// g_date_time_unref (dt);
    /// ```
    extern fn g_time_val_from_iso8601(p_iso_date: [*:0]const u8, p_time_: *glib.TimeVal) c_int;
    pub const fromIso8601 = g_time_val_from_iso8601;

    /// Adds the given number of microseconds to `time_`. `microseconds` can
    /// also be negative to decrease the value of `time_`.
    extern fn g_time_val_add(p_time_: *TimeVal, p_microseconds: c_long) void;
    pub const add = g_time_val_add;

    /// Converts `time_` into an RFC 3339 encoded string, relative to the
    /// Coordinated Universal Time (UTC). This is one of the many formats
    /// allowed by ISO 8601.
    ///
    /// ISO 8601 allows a large number of date/time formats, with or without
    /// punctuation and optional elements. The format returned by this function
    /// is a complete date and time, with optional punctuation included, the
    /// UTC time zone represented as "Z", and the `tv_usec` part included if
    /// and only if it is nonzero, i.e. either
    /// "YYYY-MM-DDTHH:MM:SSZ" or "YYYY-MM-DDTHH:MM:SS.fffffZ".
    ///
    /// This corresponds to the Internet date/time format defined by
    /// [RFC 3339](https://www.ietf.org/rfc/rfc3339.txt),
    /// and to either of the two most-precise formats defined by
    /// the W3C Note
    /// [Date and Time Formats](http://www.w3.org/TR/NOTE-datetime-19980827).
    /// Both of these documents are profiles of ISO 8601.
    ///
    /// Use `glib.DateTime.format` or `glib.strdupPrintf` if a different
    /// variation of ISO 8601 format is required.
    ///
    /// If `time_` represents a date which is too large to fit into a `struct tm`,
    /// `NULL` will be returned. This is platform dependent. Note also that since
    /// `GTimeVal` stores the number of seconds as a `glong`, on 32-bit systems it
    /// is subject to the year 2038 problem. Accordingly, since GLib 2.62, this
    /// function has been deprecated. Equivalent functionality is available using:
    /// ```
    /// GDateTime *dt = g_date_time_new_from_unix_utc (time_val);
    /// iso8601_string = g_date_time_format_iso8601 (dt);
    /// g_date_time_unref (dt);
    /// ```
    ///
    /// The return value of `glib.TimeVal.toIso8601` has been nullable since GLib
    /// 2.54; before then, GLib would crash under the same conditions.
    extern fn g_time_val_to_iso8601(p_time_: *TimeVal) ?[*:0]u8;
    pub const toIso8601 = g_time_val_to_iso8601;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `GTimeZone` represents a time zone, at no particular point in time.
///
/// The `GTimeZone` struct is refcounted and immutable.
///
/// Each time zone has an identifier (for example, ‘Europe/London’) which is
/// platform dependent. See `glib.TimeZone.new` for information on the
/// identifier formats. The identifier of a time zone can be retrieved using
/// `glib.TimeZone.getIdentifier`.
///
/// A time zone contains a number of intervals. Each interval has an abbreviation
/// to describe it (for example, ‘PDT’), an offset to UTC and a flag indicating
/// if the daylight savings time is in effect during that interval. A time zone
/// always has at least one interval — interval 0. Note that interval abbreviations
/// are not the same as time zone identifiers (apart from ‘UTC’), and cannot be
/// passed to `glib.TimeZone.new`.
///
/// Every UTC time is contained within exactly one interval, but a given
/// local time may be contained within zero, one or two intervals (due to
/// incontinuities associated with daylight savings time).
///
/// An interval may refer to a specific period of time (eg: the duration
/// of daylight savings time during 2010) or it may refer to many periods
/// of time that share the same properties (eg: all periods of daylight
/// savings time).  It is also possible (usually for political reasons)
/// that some properties (like the abbreviation) change between intervals
/// without other properties changing.
pub const TimeZone = opaque {
    /// A version of `glib.TimeZone.newIdentifier` which returns the UTC time zone
    /// if `identifier` could not be parsed or loaded.
    ///
    /// If you need to check whether `identifier` was loaded successfully, use
    /// `glib.TimeZone.newIdentifier`.
    extern fn g_time_zone_new(p_identifier: ?[*:0]const u8) *glib.TimeZone;
    pub const new = g_time_zone_new;

    /// Creates a `glib.TimeZone` corresponding to `identifier`. If `identifier` cannot be
    /// parsed or loaded, `NULL` is returned.
    ///
    /// `identifier` can either be an RFC3339/ISO 8601 time offset or
    /// something that would pass as a valid value for the `TZ` environment
    /// variable (including `NULL`).
    ///
    /// In Windows, `identifier` can also be the unlocalized name of a time
    /// zone for standard time, for example "Pacific Standard Time".
    ///
    /// Valid RFC3339 time offsets are `"Z"` (for UTC) or
    /// `"±hh:mm"`.  ISO 8601 additionally specifies
    /// `"±hhmm"` and `"±hh"`.  Offsets are
    /// time values to be added to Coordinated Universal Time (UTC) to get
    /// the local time.
    ///
    /// In UNIX, the `TZ` environment variable typically corresponds
    /// to the name of a file in the zoneinfo database, an absolute path to a file
    /// somewhere else, or a string in
    /// "std offset [dst [offset],start[/time],end[/time]]" (POSIX) format.
    /// There  are  no spaces in the specification. The name of standard
    /// and daylight savings time zone must be three or more alphabetic
    /// characters. Offsets are time values to be added to local time to
    /// get Coordinated Universal Time (UTC) and should be
    /// `"[±]hh[[:]mm[:ss]]"`.  Dates are either
    /// `"Jn"` (Julian day with n between 1 and 365, leap
    /// years not counted), `"n"` (zero-based Julian day
    /// with n between 0 and 365) or `"Mm.w.d"` (day d
    /// (0 <= d <= 6) of week w (1 <= w <= 5) of month m (1 <= m <= 12), day
    /// 0 is a Sunday).  Times are in local wall clock time, the default is
    /// 02:00:00.
    ///
    /// In Windows, the "tzn[+|–]hh[:mm[:ss]][dzn]" format is used, but also
    /// accepts POSIX format.  The Windows format uses US rules for all time
    /// zones; daylight savings time is 60 minutes behind the standard time
    /// with date and time of change taken from Pacific Standard Time.
    /// Offsets are time values to be added to the local time to get
    /// Coordinated Universal Time (UTC).
    ///
    /// `glib.TimeZone.newLocal` calls this function with the value of the
    /// `TZ` environment variable. This function itself is independent of
    /// the value of `TZ`, but if `identifier` is `NULL` then `/etc/localtime`
    /// will be consulted to discover the correct time zone on UNIX and the
    /// registry will be consulted or `GetTimeZoneInformation` will be used
    /// to get the local time zone on Windows.
    ///
    /// If intervals are not available, only time zone rules from `TZ`
    /// environment variable or other means, then they will be computed
    /// from year 1900 to 2037.  If the maximum year for the rules is
    /// available and it is greater than 2037, then it will followed
    /// instead.
    ///
    /// See
    /// [RFC3339 §5.6](http://tools.ietf.org/html/rfc3339`section`-5.6)
    /// for a precise definition of valid RFC3339 time offsets
    /// (the `time-offset` expansion) and ISO 8601 for the
    /// full list of valid time offsets.  See
    /// [The GNU C Library manual](http://www.gnu.org/s/libc/manual/html_node/TZ-Variable.html)
    /// for an explanation of the possible
    /// values of the `TZ` environment variable. See
    /// [Microsoft Time Zone Index Values](http://msdn.microsoft.com/en-us/library/ms912391`@"28v"`=winembedded.11`@"29"`.aspx)
    /// for the list of time zones on Windows.
    ///
    /// You should release the return value by calling `glib.TimeZone.unref`
    /// when you are done with it.
    extern fn g_time_zone_new_identifier(p_identifier: ?[*:0]const u8) ?*glib.TimeZone;
    pub const newIdentifier = g_time_zone_new_identifier;

    /// Creates a `glib.TimeZone` corresponding to local time.  The local time
    /// zone may change between invocations to this function; for example,
    /// if the system administrator changes it.
    ///
    /// This is equivalent to calling `glib.TimeZone.new` with the value of
    /// the `TZ` environment variable (including the possibility of `NULL`).
    ///
    /// You should release the return value by calling `glib.TimeZone.unref`
    /// when you are done with it.
    extern fn g_time_zone_new_local() *glib.TimeZone;
    pub const newLocal = g_time_zone_new_local;

    /// Creates a `glib.TimeZone` corresponding to the given constant offset from UTC,
    /// in seconds.
    ///
    /// This is equivalent to calling `glib.TimeZone.new` with a string in the form
    /// `[+|-]hh[:mm[:ss]]`.
    ///
    /// It is possible for this function to fail if `seconds` is too big (greater than
    /// 24 hours), in which case this function will return the UTC timezone for
    /// backwards compatibility. To detect failures like this, use
    /// `glib.TimeZone.newIdentifier` directly.
    extern fn g_time_zone_new_offset(p_seconds: i32) *glib.TimeZone;
    pub const newOffset = g_time_zone_new_offset;

    /// Creates a `glib.TimeZone` corresponding to UTC.
    ///
    /// This is equivalent to calling `glib.TimeZone.new` with a value like
    /// "Z", "UTC", "+00", etc.
    ///
    /// You should release the return value by calling `glib.TimeZone.unref`
    /// when you are done with it.
    extern fn g_time_zone_new_utc() *glib.TimeZone;
    pub const newUtc = g_time_zone_new_utc;

    /// Finds an interval within `tz` that corresponds to the given `time_`,
    /// possibly adjusting `time_` if required to fit into an interval.
    /// The meaning of `time_` depends on `type`.
    ///
    /// This function is similar to `glib.TimeZone.findInterval`, with the
    /// difference that it always succeeds (by making the adjustments
    /// described below).
    ///
    /// In any of the cases where `glib.TimeZone.findInterval` succeeds then
    /// this function returns the same value, without modifying `time_`.
    ///
    /// This function may, however, modify `time_` in order to deal with
    /// non-existent times.  If the non-existent local `time_` of 02:30 were
    /// requested on March 14th 2010 in Toronto then this function would
    /// adjust `time_` to be 03:00 and return the interval containing the
    /// adjusted time.
    extern fn g_time_zone_adjust_time(p_tz: *TimeZone, p_type: glib.TimeType, p_time_: *i64) c_int;
    pub const adjustTime = g_time_zone_adjust_time;

    /// Finds an interval within `tz` that corresponds to the given `time_`.
    /// The meaning of `time_` depends on `type`.
    ///
    /// If `type` is `G_TIME_TYPE_UNIVERSAL` then this function will always
    /// succeed (since universal time is monotonic and continuous).
    ///
    /// Otherwise `time_` is treated as local time.  The distinction between
    /// `G_TIME_TYPE_STANDARD` and `G_TIME_TYPE_DAYLIGHT` is ignored except in
    /// the case that the given `time_` is ambiguous.  In Toronto, for example,
    /// 01:30 on November 7th 2010 occurred twice (once inside of daylight
    /// savings time and the next, an hour later, outside of daylight savings
    /// time).  In this case, the different value of `type` would result in a
    /// different interval being returned.
    ///
    /// It is still possible for this function to fail.  In Toronto, for
    /// example, 02:00 on March 14th 2010 does not exist (due to the leap
    /// forward to begin daylight savings time).  -1 is returned in that
    /// case.
    extern fn g_time_zone_find_interval(p_tz: *TimeZone, p_type: glib.TimeType, p_time_: i64) c_int;
    pub const findInterval = g_time_zone_find_interval;

    /// Determines the time zone abbreviation to be used during a particular
    /// `interval` of time in the time zone `tz`.
    ///
    /// For example, in Toronto this is currently "EST" during the winter
    /// months and "EDT" during the summer months when daylight savings time
    /// is in effect.
    extern fn g_time_zone_get_abbreviation(p_tz: *TimeZone, p_interval: c_int) [*:0]const u8;
    pub const getAbbreviation = g_time_zone_get_abbreviation;

    /// Get the identifier of this `glib.TimeZone`, as passed to `glib.TimeZone.new`.
    /// If the identifier passed at construction time was not recognised, `UTC` will
    /// be returned. If it was `NULL`, the identifier of the local timezone at
    /// construction time will be returned.
    ///
    /// The identifier will be returned in the same format as provided at
    /// construction time: if provided as a time offset, that will be returned by
    /// this function.
    extern fn g_time_zone_get_identifier(p_tz: *TimeZone) [*:0]const u8;
    pub const getIdentifier = g_time_zone_get_identifier;

    /// Determines the offset to UTC in effect during a particular `interval`
    /// of time in the time zone `tz`.
    ///
    /// The offset is the number of seconds that you add to UTC time to
    /// arrive at local time for `tz` (ie: negative numbers for time zones
    /// west of GMT, positive numbers for east).
    extern fn g_time_zone_get_offset(p_tz: *TimeZone, p_interval: c_int) i32;
    pub const getOffset = g_time_zone_get_offset;

    /// Determines if daylight savings time is in effect during a particular
    /// `interval` of time in the time zone `tz`.
    extern fn g_time_zone_is_dst(p_tz: *TimeZone, p_interval: c_int) c_int;
    pub const isDst = g_time_zone_is_dst;

    /// Increases the reference count on `tz`.
    extern fn g_time_zone_ref(p_tz: *TimeZone) *glib.TimeZone;
    pub const ref = g_time_zone_ref;

    /// Decreases the reference count on `tz`.
    extern fn g_time_zone_unref(p_tz: *TimeZone) void;
    pub const unref = g_time_zone_unref;

    extern fn g_time_zone_get_type() usize;
    pub const getGObjectType = g_time_zone_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GTimer` records a start time, and counts microseconds elapsed since
/// that time.
///
/// This is done somewhat differently on different platforms, and can be
/// tricky to get exactly right, so `GTimer` provides a portable/convenient interface.
pub const Timer = opaque {
    /// Creates a new timer, and starts timing (i.e. `glib.Timer.start` is
    /// implicitly called for you).
    extern fn g_timer_new() *glib.Timer;
    pub const new = g_timer_new;

    /// Resumes a timer that has previously been stopped with
    /// `glib.Timer.stop`. `glib.Timer.stop` must be called before using this
    /// function.
    extern fn g_timer_continue(p_timer: *Timer) void;
    pub const @"continue" = g_timer_continue;

    /// Destroys a timer, freeing associated resources.
    extern fn g_timer_destroy(p_timer: *Timer) void;
    pub const destroy = g_timer_destroy;

    /// If `timer` has been started but not stopped, obtains the time since
    /// the timer was started. If `timer` has been stopped, obtains the
    /// elapsed time between the time it was started and the time it was
    /// stopped. The return value is the number of seconds elapsed,
    /// including any fractional part. The `microseconds` out parameter is
    /// essentially useless.
    extern fn g_timer_elapsed(p_timer: *Timer, p_microseconds: *c_ulong) f64;
    pub const elapsed = g_timer_elapsed;

    /// Exposes whether the timer is currently active.
    extern fn g_timer_is_active(p_timer: *Timer) c_int;
    pub const isActive = g_timer_is_active;

    /// This function is useless; it's fine to call `glib.Timer.start` on an
    /// already-started timer to reset the start time, so `glib.Timer.reset`
    /// serves no purpose.
    extern fn g_timer_reset(p_timer: *Timer) void;
    pub const reset = g_timer_reset;

    /// Marks a start time, so that future calls to `glib.Timer.elapsed` will
    /// report the time since `glib.Timer.start` was called. `glib.Timer.new`
    /// automatically marks the start time, so no need to call
    /// `glib.Timer.start` immediately after creating the timer.
    extern fn g_timer_start(p_timer: *Timer) void;
    pub const start = g_timer_start;

    /// Marks an end time, so calls to `glib.Timer.elapsed` will return the
    /// difference between this end time and the start time.
    extern fn g_timer_stop(p_timer: *Timer) void;
    pub const stop = g_timer_stop;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A `GTrashStack` is an efficient way to keep a stack of unused allocated
/// memory chunks. Each memory chunk is required to be large enough to hold
/// a `gpointer`. This allows the stack to be maintained without any space
/// overhead, since the stack pointers can be stored inside the memory chunks.
///
/// There is no function to create a `GTrashStack`. A `NULL` `GTrashStack*`
/// is a perfectly valid empty stack.
///
/// Each piece of memory that is pushed onto the stack is cast to a
/// `GTrashStack*`.
///
/// There is no longer any good reason to use `GTrashStack`.  If you have
/// extra pieces of memory, ``free`` them and allocate them again later.
pub const TrashStack = extern struct {
    /// pointer to the previous element of the stack,
    ///     gets stored in the first `sizeof (gpointer)`
    ///     bytes of the element
    f_next: ?*glib.TrashStack,

    /// Returns the height of a `glib.TrashStack`.
    ///
    /// Note that execution of this function is of O(N) complexity
    /// where N denotes the number of items on the stack.
    extern fn g_trash_stack_height(p_stack_p: **glib.TrashStack) c_uint;
    pub const height = g_trash_stack_height;

    /// Returns the element at the top of a `glib.TrashStack`
    /// which may be `NULL`.
    extern fn g_trash_stack_peek(p_stack_p: **glib.TrashStack) ?*anyopaque;
    pub const peek = g_trash_stack_peek;

    /// Pops a piece of memory off a `glib.TrashStack`.
    extern fn g_trash_stack_pop(p_stack_p: **glib.TrashStack) ?*anyopaque;
    pub const pop = g_trash_stack_pop;

    /// Pushes a piece of memory onto a `glib.TrashStack`.
    extern fn g_trash_stack_push(p_stack_p: **glib.TrashStack, p_data_p: *anyopaque) void;
    pub const push = g_trash_stack_push;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The GTree struct is an opaque data structure representing a
/// [balanced binary tree](data-structures.html`binary`-trees). It should be
/// accessed only by using the following functions.
pub const Tree = opaque {
    /// Creates a new `glib.Tree`.
    extern fn g_tree_new(p_key_compare_func: glib.CompareFunc) *glib.Tree;
    pub const new = g_tree_new;

    /// Creates a new `glib.Tree` like `glib.Tree.new` and allows to specify functions
    /// to free the memory allocated for the key and value that get called when
    /// removing the entry from the `glib.Tree`.
    extern fn g_tree_new_full(p_key_compare_func: glib.CompareDataFunc, p_key_compare_data: ?*anyopaque, p_key_destroy_func: glib.DestroyNotify, p_value_destroy_func: ?glib.DestroyNotify) *glib.Tree;
    pub const newFull = g_tree_new_full;

    /// Creates a new `glib.Tree` with a comparison function that accepts user data.
    /// See `glib.Tree.new` for more details.
    extern fn g_tree_new_with_data(p_key_compare_func: glib.CompareDataFunc, p_key_compare_data: ?*anyopaque) *glib.Tree;
    pub const newWithData = g_tree_new_with_data;

    /// Removes all keys and values from the `glib.Tree` and decreases its
    /// reference count by one. If keys and/or values are dynamically
    /// allocated, you should either free them first or create the `glib.Tree`
    /// using `glib.Tree.newFull`. In the latter case the destroy functions
    /// you supplied will be called on all keys and values before destroying
    /// the `glib.Tree`.
    extern fn g_tree_destroy(p_tree: *Tree) void;
    pub const destroy = g_tree_destroy;

    /// Calls the given function for each of the key/value pairs in the `glib.Tree`.
    /// The function is passed the key and value of each pair, and the given
    /// `data` parameter. The tree is traversed in sorted order.
    ///
    /// The tree may not be modified while iterating over it (you can't
    /// add/remove items). To remove all items matching a predicate, you need
    /// to add each item to a list in your `glib.TraverseFunc` as you walk over
    /// the tree, then walk the list and remove each item.
    extern fn g_tree_foreach(p_tree: *Tree, p_func: glib.TraverseFunc, p_user_data: ?*anyopaque) void;
    pub const foreach = g_tree_foreach;

    /// Calls the given function for each of the nodes in the `glib.Tree`.
    /// The function is passed the pointer to the particular node, and the given
    /// `data` parameter. The tree traversal happens in-order.
    ///
    /// The tree may not be modified while iterating over it (you can't
    /// add/remove items). To remove all items matching a predicate, you need
    /// to add each item to a list in your `glib.TraverseFunc` as you walk over
    /// the tree, then walk the list and remove each item.
    extern fn g_tree_foreach_node(p_tree: *Tree, p_func: glib.TraverseNodeFunc, p_user_data: ?*anyopaque) void;
    pub const foreachNode = g_tree_foreach_node;

    /// Gets the height of a `glib.Tree`.
    ///
    /// If the `glib.Tree` contains no nodes, the height is 0.
    /// If the `glib.Tree` contains only one root node the height is 1.
    /// If the root node has children the height is 2, etc.
    extern fn g_tree_height(p_tree: *Tree) c_int;
    pub const height = g_tree_height;

    /// Inserts a key/value pair into a `glib.Tree`.
    ///
    /// Inserts a new key and value into a `glib.Tree` as `glib.Tree.insertNode` does,
    /// only this function does not return the inserted or set node.
    extern fn g_tree_insert(p_tree: *Tree, p_key: ?*anyopaque, p_value: ?*anyopaque) void;
    pub const insert = g_tree_insert;

    /// Inserts a key/value pair into a `glib.Tree`.
    ///
    /// If the given key already exists in the `glib.Tree` its corresponding value
    /// is set to the new value. If you supplied a `value_destroy_func` when
    /// creating the `glib.Tree`, the old value is freed using that function. If
    /// you supplied a `key_destroy_func` when creating the `glib.Tree`, the passed
    /// key is freed using that function.
    ///
    /// The tree is automatically 'balanced' as new key/value pairs are added,
    /// so that the distance from the root to every leaf is as small as possible.
    /// The cost of maintaining a balanced tree while inserting new key/value
    /// result in a O(n log(n)) operation where most of the other operations
    /// are O(log(n)).
    extern fn g_tree_insert_node(p_tree: *Tree, p_key: ?*anyopaque, p_value: ?*anyopaque) ?*glib.TreeNode;
    pub const insertNode = g_tree_insert_node;

    /// Gets the value corresponding to the given key. Since a `glib.Tree` is
    /// automatically balanced as key/value pairs are added, key lookup
    /// is O(log n) (where n is the number of key/value pairs in the tree).
    extern fn g_tree_lookup(p_tree: *Tree, p_key: ?*const anyopaque) ?*anyopaque;
    pub const lookup = g_tree_lookup;

    /// Looks up a key in the `glib.Tree`, returning the original key and the
    /// associated value. This is useful if you need to free the memory
    /// allocated for the original key, for example before calling
    /// `glib.Tree.remove`.
    extern fn g_tree_lookup_extended(p_tree: *Tree, p_lookup_key: ?*const anyopaque, p_orig_key: ?*anyopaque, p_value: ?*anyopaque) c_int;
    pub const lookupExtended = g_tree_lookup_extended;

    /// Gets the tree node corresponding to the given key. Since a `glib.Tree` is
    /// automatically balanced as key/value pairs are added, key lookup
    /// is O(log n) (where n is the number of key/value pairs in the tree).
    extern fn g_tree_lookup_node(p_tree: *Tree, p_key: ?*const anyopaque) ?*glib.TreeNode;
    pub const lookupNode = g_tree_lookup_node;

    /// Gets the lower bound node corresponding to the given key,
    /// or `NULL` if the tree is empty or all the nodes in the tree
    /// have keys that are strictly lower than the searched key.
    ///
    /// The lower bound is the first node that has its key greater
    /// than or equal to the searched key.
    extern fn g_tree_lower_bound(p_tree: *Tree, p_key: ?*const anyopaque) ?*glib.TreeNode;
    pub const lowerBound = g_tree_lower_bound;

    /// Gets the number of nodes in a `glib.Tree`.
    extern fn g_tree_nnodes(p_tree: *Tree) c_int;
    pub const nnodes = g_tree_nnodes;

    /// Returns the first in-order node of the tree, or `NULL`
    /// for an empty tree.
    extern fn g_tree_node_first(p_tree: *Tree) ?*glib.TreeNode;
    pub const nodeFirst = g_tree_node_first;

    /// Returns the last in-order node of the tree, or `NULL`
    /// for an empty tree.
    extern fn g_tree_node_last(p_tree: *Tree) ?*glib.TreeNode;
    pub const nodeLast = g_tree_node_last;

    /// Increments the reference count of `tree` by one.
    ///
    /// It is safe to call this function from any thread.
    extern fn g_tree_ref(p_tree: *Tree) *glib.Tree;
    pub const ref = g_tree_ref;

    /// Removes a key/value pair from a `glib.Tree`.
    ///
    /// If the `glib.Tree` was created using `glib.Tree.newFull`, the key and value
    /// are freed using the supplied destroy functions, otherwise you have to
    /// make sure that any dynamically allocated values are freed yourself.
    /// If the key does not exist in the `glib.Tree`, the function does nothing.
    ///
    /// The cost of maintaining a balanced tree while removing a key/value
    /// result in a O(n log(n)) operation where most of the other operations
    /// are O(log(n)).
    extern fn g_tree_remove(p_tree: *Tree, p_key: ?*const anyopaque) c_int;
    pub const remove = g_tree_remove;

    /// Removes all nodes from a `glib.Tree` and destroys their keys and values,
    /// then resets the `glib.Tree`’s root to `NULL`.
    extern fn g_tree_remove_all(p_tree: *Tree) void;
    pub const removeAll = g_tree_remove_all;

    /// Inserts a new key and value into a `glib.Tree` as `glib.Tree.replaceNode` does,
    /// only this function does not return the inserted or set node.
    extern fn g_tree_replace(p_tree: *Tree, p_key: ?*anyopaque, p_value: ?*anyopaque) void;
    pub const replace = g_tree_replace;

    /// Inserts a new key and value into a `glib.Tree` similar to `glib.Tree.insertNode`.
    /// The difference is that if the key already exists in the `glib.Tree`, it gets
    /// replaced by the new key. If you supplied a `value_destroy_func` when
    /// creating the `glib.Tree`, the old value is freed using that function. If you
    /// supplied a `key_destroy_func` when creating the `glib.Tree`, the old key is
    /// freed using that function.
    ///
    /// The tree is automatically 'balanced' as new key/value pairs are added,
    /// so that the distance from the root to every leaf is as small as possible.
    extern fn g_tree_replace_node(p_tree: *Tree, p_key: ?*anyopaque, p_value: ?*anyopaque) ?*glib.TreeNode;
    pub const replaceNode = g_tree_replace_node;

    /// Searches a `glib.Tree` using `search_func`.
    ///
    /// The `search_func` is called with a pointer to the key of a key/value
    /// pair in the tree, and the passed in `user_data`. If `search_func` returns
    /// 0 for a key/value pair, then the corresponding value is returned as
    /// the result of `glib.Tree.search`. If `search_func` returns -1, searching
    /// will proceed among the key/value pairs that have a smaller key; if
    /// `search_func` returns 1, searching will proceed among the key/value
    /// pairs that have a larger key.
    extern fn g_tree_search(p_tree: *Tree, p_search_func: glib.CompareFunc, p_user_data: ?*const anyopaque) ?*anyopaque;
    pub const search = g_tree_search;

    /// Searches a `glib.Tree` using `search_func`.
    ///
    /// The `search_func` is called with a pointer to the key of a key/value
    /// pair in the tree, and the passed in `user_data`. If `search_func` returns
    /// 0 for a key/value pair, then the corresponding node is returned as
    /// the result of `glib.Tree.search`. If `search_func` returns -1, searching
    /// will proceed among the key/value pairs that have a smaller key; if
    /// `search_func` returns 1, searching will proceed among the key/value
    /// pairs that have a larger key.
    extern fn g_tree_search_node(p_tree: *Tree, p_search_func: glib.CompareFunc, p_user_data: ?*const anyopaque) ?*glib.TreeNode;
    pub const searchNode = g_tree_search_node;

    /// Removes a key and its associated value from a `glib.Tree` without calling
    /// the key and value destroy functions.
    ///
    /// If the key does not exist in the `glib.Tree`, the function does nothing.
    extern fn g_tree_steal(p_tree: *Tree, p_key: ?*const anyopaque) c_int;
    pub const steal = g_tree_steal;

    /// Calls the given function for each node in the `glib.Tree`.
    extern fn g_tree_traverse(p_tree: *Tree, p_traverse_func: glib.TraverseFunc, p_traverse_type: glib.TraverseType, p_user_data: ?*anyopaque) void;
    pub const traverse = g_tree_traverse;

    /// Decrements the reference count of `tree` by one.
    /// If the reference count drops to 0, all keys and values will
    /// be destroyed (if destroy functions were specified) and all
    /// memory allocated by `tree` will be released.
    ///
    /// It is safe to call this function from any thread.
    extern fn g_tree_unref(p_tree: *Tree) void;
    pub const unref = g_tree_unref;

    /// Gets the upper bound node corresponding to the given key,
    /// or `NULL` if the tree is empty or all the nodes in the tree
    /// have keys that are lower than or equal to the searched key.
    ///
    /// The upper bound is the first node that has its key strictly greater
    /// than the searched key.
    extern fn g_tree_upper_bound(p_tree: *Tree, p_key: ?*const anyopaque) ?*glib.TreeNode;
    pub const upperBound = g_tree_upper_bound;

    extern fn g_tree_get_type() usize;
    pub const getGObjectType = g_tree_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// An opaque type which identifies a specific node in a `glib.Tree`.
pub const TreeNode = opaque {
    /// Gets the key stored at a particular tree node.
    extern fn g_tree_node_key(p_node: *TreeNode) ?*anyopaque;
    pub const key = g_tree_node_key;

    /// Returns the next in-order node of the tree, or `NULL`
    /// if the passed node was already the last one.
    extern fn g_tree_node_next(p_node: *TreeNode) ?*glib.TreeNode;
    pub const next = g_tree_node_next;

    /// Returns the previous in-order node of the tree, or `NULL`
    /// if the passed node was already the first one.
    extern fn g_tree_node_previous(p_node: *TreeNode) ?*glib.TreeNode;
    pub const previous = g_tree_node_previous;

    /// Gets the value stored at a particular tree node.
    extern fn g_tree_node_value(p_node: *TreeNode) ?*anyopaque;
    pub const value = g_tree_node_value;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.Tuples` struct is used to return records (or tuples) from the
/// `glib.Relation` by `glib.Relation.select`. It only contains one public
/// member - the number of records that matched. To access the matched
/// records, you must use `glib.Tuples.index`.
pub const Tuples = extern struct {
    /// the number of records that matched.
    f_len: c_uint,

    /// Frees the records which were returned by `glib.Relation.select`. This
    /// should always be called after `glib.Relation.select` when you are
    /// finished with the records. The records are not removed from the
    /// `glib.Relation`.
    extern fn g_tuples_destroy(p_tuples: *Tuples) void;
    pub const destroy = g_tuples_destroy;

    /// Gets a field from the records returned by `glib.Relation.select`. It
    /// returns the given field of the record at the given index. The
    /// returned value should not be changed.
    extern fn g_tuples_index(p_tuples: *Tuples, p_index_: c_int, p_field: c_int) ?*anyopaque;
    pub const index = g_tuples_index;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A Unix pipe. The advantage of this type over `int[2]` is that it can
/// be closed automatically when it goes out of scope, using `g_auto(GUnixPipe)`,
/// on compilers that support that feature.
pub const UnixPipe = extern struct {
    /// A pair of file descriptors, each negative if closed or not yet opened.
    ///  The file descriptor with index `G_UNIX_PIPE_END_READ` is readable.
    ///  The file descriptor with index `G_UNIX_PIPE_END_WRITE` is writable.
    f_fds: [2]c_int,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `GUri` type and related functions can be used to parse URIs into
/// their components, and build valid URIs from individual components.
///
/// Since `GUri` only represents absolute URIs, all `GUri`s will have a
/// URI scheme, so `glib.Uri.getScheme` will always return a non-`NULL`
/// answer. Likewise, by definition, all URIs have a path component, so
/// `glib.Uri.getPath` will always return a non-`NULL` string (which may
/// be empty).
///
/// If the URI string has an
/// [‘authority’ component](https://tools.ietf.org/html/rfc3986`section`-3) (that
/// is, if the scheme is followed by `://` rather than just `:`), then the
/// `GUri` will contain a hostname, and possibly a port and ‘userinfo’.
/// Additionally, depending on how the `GUri` was constructed/parsed (for example,
/// using the `G_URI_FLAGS_HAS_PASSWORD` and `G_URI_FLAGS_HAS_AUTH_PARAMS` flags),
/// the userinfo may be split out into a username, password, and
/// additional authorization-related parameters.
///
/// Normally, the components of a `GUri` will have all `%`-encoded
/// characters decoded. However, if you construct/parse a `GUri` with
/// `G_URI_FLAGS_ENCODED`, then the `%`-encoding will be preserved instead in
/// the userinfo, path, and query fields (and in the host field if also
/// created with `G_URI_FLAGS_NON_DNS`). In particular, this is necessary if
/// the URI may contain binary data or non-UTF-8 text, or if decoding
/// the components might change the interpretation of the URI.
///
/// For example, with the encoded flag:
///
/// ```c
/// g_autoptr(GUri) uri = g_uri_parse ("http://host/path?query=http`@"3A"``@"2F"``@"2Fhost"``@"2Fpath"``@"3Fparam"``@"3Dvalue"`", G_URI_FLAGS_ENCODED, &err);
/// g_assert_cmpstr (g_uri_get_query (uri), ==, "query=http`@"3A"``@"2F"``@"2Fhost"``@"2Fpath"``@"3Fparam"``@"3Dvalue"`");
/// ```
///
/// While the default `%`-decoding behaviour would give:
///
/// ```c
/// g_autoptr(GUri) uri = g_uri_parse ("http://host/path?query=http`@"3A"``@"2F"``@"2Fhost"``@"2Fpath"``@"3Fparam"``@"3Dvalue"`", G_URI_FLAGS_NONE, &err);
/// g_assert_cmpstr (g_uri_get_query (uri), ==, "query=http://host/path?param=value");
/// ```
///
/// During decoding, if an invalid UTF-8 string is encountered, parsing will fail
/// with an error indicating the bad string location:
///
/// ```c
/// g_autoptr(GUri) uri = g_uri_parse ("http://host/path?query=http`@"3A"``@"2F"``@"2Fhost"``@"2Fpath"``@"3Fbad"``@"3D"``@"00alue"`", G_URI_FLAGS_NONE, &err);
/// g_assert_error (err, G_URI_ERROR, G_URI_ERROR_BAD_QUERY);
/// ```
///
/// You should pass `G_URI_FLAGS_ENCODED` or `G_URI_FLAGS_ENCODED_QUERY` if you
/// need to handle that case manually. In particular, if the query string
/// contains `=` characters that are `%`-encoded, you should let
/// `glib.Uri.parseParams` do the decoding once of the query.
///
/// `GUri` is immutable once constructed, and can safely be accessed from
/// multiple threads. Its reference counting is atomic.
///
/// Note that the scope of `GUri` is to help manipulate URIs in various applications,
/// following [RFC 3986](https://tools.ietf.org/html/rfc3986). In particular,
/// it doesn't intend to cover web browser needs, and doesn’t implement the
/// [WHATWG URL](https://url.spec.whatwg.org/) standard. No APIs are provided to
/// help prevent
/// [homograph attacks](https://en.wikipedia.org/wiki/IDN_homograph_attack), so
/// `GUri` is not suitable for formatting URIs for display to the user for making
/// security-sensitive decisions.
///
/// ## Relative and absolute URIs
///
/// As defined in [RFC 3986](https://tools.ietf.org/html/rfc3986`section`-4), the
/// hierarchical nature of URIs means that they can either be ‘relative
/// references’ (sometimes referred to as ‘relative URIs’) or ‘URIs’ (for
/// clarity, ‘URIs’ are referred to in this documentation as
/// ‘absolute URIs’ — although
/// [in contrast to RFC 3986](https://tools.ietf.org/html/rfc3986`section`-4.3),
/// fragment identifiers are always allowed).
///
/// Relative references have one or more components of the URI missing. In
/// particular, they have no scheme. Any other component, such as hostname,
/// query, etc. may be missing, apart from a path, which has to be specified (but
/// may be empty). The path may be relative, starting with `./` rather than `/`.
///
/// For example, a valid relative reference is `./path?query`,
/// `/?query`fragment`` or `//example.com`.
///
/// Absolute URIs have a scheme specified. Any other components of the URI which
/// are missing are specified as explicitly unset in the URI, rather than being
/// resolved relative to a base URI using `glib.Uri.parseRelative`.
///
/// For example, a valid absolute URI is `file:///home/bob` or
/// `https://search.com?query=string`.
///
/// A `GUri` instance is always an absolute URI. A string may be an absolute URI
/// or a relative reference; see the documentation for individual functions as to
/// what forms they accept.
///
/// ## Parsing URIs
///
/// The most minimalist APIs for parsing URIs are `glib.Uri.split` and
/// `glib.Uri.splitWithUser`. These split a URI into its component
/// parts, and return the parts; the difference between the two is that
/// `glib.Uri.split` treats the ‘userinfo’ component of the URI as a
/// single element, while `glib.Uri.splitWithUser` can (depending on the
/// `glib.UriFlags` you pass) treat it as containing a username, password,
/// and authentication parameters. Alternatively, `glib.Uri.splitNetwork`
/// can be used when you are only interested in the components that are
/// needed to initiate a network connection to the service (scheme,
/// host, and port).
///
/// `glib.Uri.parse` is similar to `glib.Uri.split`, but instead of
/// returning individual strings, it returns a `GUri` structure (and it requires
/// that the URI be an absolute URI).
///
/// `glib.Uri.resolveRelative` and `glib.Uri.parseRelative` allow
/// you to resolve a relative URI relative to a base URI.
/// `glib.Uri.resolveRelative` takes two strings and returns a string,
/// and `glib.Uri.parseRelative` takes a `GUri` and a string and returns a
/// `GUri`.
///
/// All of the parsing functions take a `glib.UriFlags` argument describing
/// exactly how to parse the URI; see the documentation for that type
/// for more details on the specific flags that you can pass. If you
/// need to choose different flags based on the type of URI, you can
/// use `glib.Uri.peekScheme` on the URI string to check the scheme
/// first, and use that to decide what flags to parse it with.
///
/// For example, you might want to use `G_URI_PARAMS_WWW_FORM` when parsing the
/// params for a web URI, so compare the result of `glib.Uri.peekScheme`
/// against `http` and `https`.
///
/// ## Building URIs
///
/// `glib.Uri.join` and `glib.Uri.joinWithUser` can be used to construct
/// valid URI strings from a set of component strings. They are the
/// inverse of `glib.Uri.split` and `glib.Uri.splitWithUser`.
///
/// Similarly, `glib.Uri.build` and `glib.Uri.buildWithUser` can be
/// used to construct a `GUri` from a set of component strings.
///
/// As with the parsing functions, the building functions take a
/// `glib.UriFlags` argument. In particular, it is important to keep in mind
/// whether the URI components you are using are already `%`-encoded. If so,
/// you must pass the `G_URI_FLAGS_ENCODED` flag.
///
/// ## `file://` URIs
///
/// Note that Windows and Unix both define special rules for parsing
/// `file://` URIs (involving non-UTF-8 character sets on Unix, and the
/// interpretation of path separators on Windows). `GUri` does not
/// implement these rules. Use `glib.filenameFromUri` and
/// `glib.filenameToUri` if you want to properly convert between
/// `file://` URIs and local filenames.
///
/// ## URI Equality
///
/// Note that there is no `g_uri_equal ()` function, because comparing
/// URIs usefully requires scheme-specific knowledge that `GUri` does
/// not have. `GUri` can help with normalization if you use the various
/// encoded `glib.UriFlags` as well as `G_URI_FLAGS_SCHEME_NORMALIZE`
/// however it is not comprehensive.
/// For example, `data:,foo` and `data:;base64,Zm9v` resolve to the same
/// thing according to the `data:` URI specification which GLib does not
/// handle.
pub const Uri = opaque {
    /// Creates a new `glib.Uri` from the given components according to `flags`.
    ///
    /// See also `glib.uriBuildWithUser`, which allows specifying the
    /// components of the "userinfo" separately.
    extern fn g_uri_build(p_flags: glib.UriFlags, p_scheme: [*:0]const u8, p_userinfo: ?[*:0]const u8, p_host: ?[*:0]const u8, p_port: c_int, p_path: [*:0]const u8, p_query: ?[*:0]const u8, p_fragment: ?[*:0]const u8) *glib.Uri;
    pub const build = g_uri_build;

    /// Creates a new `glib.Uri` from the given components according to `flags`
    /// (`G_URI_FLAGS_HAS_PASSWORD` is added unconditionally). The `flags` must be
    /// coherent with the passed values, in particular use `%`-encoded values with
    /// `G_URI_FLAGS_ENCODED`.
    ///
    /// In contrast to `glib.uriBuild`, this allows specifying the components
    /// of the ‘userinfo’ field separately. Note that `user` must be non-`NULL`
    /// if either `password` or `auth_params` is non-`NULL`.
    extern fn g_uri_build_with_user(p_flags: glib.UriFlags, p_scheme: [*:0]const u8, p_user: ?[*:0]const u8, p_password: ?[*:0]const u8, p_auth_params: ?[*:0]const u8, p_host: ?[*:0]const u8, p_port: c_int, p_path: [*:0]const u8, p_query: ?[*:0]const u8, p_fragment: ?[*:0]const u8) *glib.Uri;
    pub const buildWithUser = g_uri_build_with_user;

    extern fn g_uri_error_quark() glib.Quark;
    pub const errorQuark = g_uri_error_quark;

    /// Escapes arbitrary data for use in a URI.
    ///
    /// Normally all characters that are not ‘unreserved’ (i.e. ASCII
    /// alphanumerical characters plus dash, dot, underscore and tilde) are
    /// escaped. But if you specify characters in `reserved_chars_allowed`
    /// they are not escaped. This is useful for the ‘reserved’ characters
    /// in the URI specification, since those are allowed unescaped in some
    /// portions of a URI.
    ///
    /// Though technically incorrect, this will also allow escaping nul
    /// bytes as `%``00`.
    extern fn g_uri_escape_bytes(p_unescaped: [*]const u8, p_length: usize, p_reserved_chars_allowed: ?[*:0]const u8) [*:0]u8;
    pub const escapeBytes = g_uri_escape_bytes;

    /// Escapes a string for use in a URI.
    ///
    /// Normally all characters that are not "unreserved" (i.e. ASCII
    /// alphanumerical characters plus dash, dot, underscore and tilde) are
    /// escaped. But if you specify characters in `reserved_chars_allowed`
    /// they are not escaped. This is useful for the "reserved" characters
    /// in the URI specification, since those are allowed unescaped in some
    /// portions of a URI.
    extern fn g_uri_escape_string(p_unescaped: [*:0]const u8, p_reserved_chars_allowed: ?[*:0]const u8, p_allow_utf8: c_int) [*:0]u8;
    pub const escapeString = g_uri_escape_string;

    /// Parses `uri_string` according to `flags`, to determine whether it is a valid
    /// [absolute URI](`relative`-and-absolute-uris), i.e. it does not need to be resolved
    /// relative to another URI using `glib.Uri.parseRelative`.
    ///
    /// If it’s not a valid URI, an error is returned explaining how it’s invalid.
    ///
    /// See `glib.uriSplit`, and the definition of `glib.UriFlags`, for more
    /// information on the effect of `flags`.
    extern fn g_uri_is_valid(p_uri_string: [*:0]const u8, p_flags: glib.UriFlags, p_error: ?*?*glib.Error) c_int;
    pub const isValid = g_uri_is_valid;

    /// Joins the given components together according to `flags` to create
    /// an absolute URI string. `path` may not be `NULL` (though it may be the empty
    /// string).
    ///
    /// When `host` is present, `path` must either be empty or begin with a slash (`/`)
    /// character. When `host` is not present, `path` cannot begin with two slash
    /// characters (`//`). See
    /// [RFC 3986, section 3](https://tools.ietf.org/html/rfc3986`section`-3).
    ///
    /// See also `glib.uriJoinWithUser`, which allows specifying the
    /// components of the ‘userinfo’ separately.
    ///
    /// `G_URI_FLAGS_HAS_PASSWORD` and `G_URI_FLAGS_HAS_AUTH_PARAMS` are ignored if set
    /// in `flags`.
    extern fn g_uri_join(p_flags: glib.UriFlags, p_scheme: ?[*:0]const u8, p_userinfo: ?[*:0]const u8, p_host: ?[*:0]const u8, p_port: c_int, p_path: [*:0]const u8, p_query: ?[*:0]const u8, p_fragment: ?[*:0]const u8) [*:0]u8;
    pub const join = g_uri_join;

    /// Joins the given components together according to `flags` to create
    /// an absolute URI string. `path` may not be `NULL` (though it may be the empty
    /// string).
    ///
    /// In contrast to `glib.uriJoin`, this allows specifying the components
    /// of the ‘userinfo’ separately. It otherwise behaves the same.
    ///
    /// `G_URI_FLAGS_HAS_PASSWORD` and `G_URI_FLAGS_HAS_AUTH_PARAMS` are ignored if set
    /// in `flags`.
    extern fn g_uri_join_with_user(p_flags: glib.UriFlags, p_scheme: ?[*:0]const u8, p_user: ?[*:0]const u8, p_password: ?[*:0]const u8, p_auth_params: ?[*:0]const u8, p_host: ?[*:0]const u8, p_port: c_int, p_path: [*:0]const u8, p_query: ?[*:0]const u8, p_fragment: ?[*:0]const u8) [*:0]u8;
    pub const joinWithUser = g_uri_join_with_user;

    /// Splits an URI list conforming to the text/uri-list
    /// mime type defined in RFC 2483 into individual URIs,
    /// discarding any comments. The URIs are not validated.
    extern fn g_uri_list_extract_uris(p_uri_list: [*:0]const u8) [*:null]?[*:0]u8;
    pub const listExtractUris = g_uri_list_extract_uris;

    /// Parses `uri_string` according to `flags`. If the result is not a
    /// valid [absolute URI](`relative`-and-absolute-uris), it will be discarded, and an
    /// error returned.
    extern fn g_uri_parse(p_uri_string: [*:0]const u8, p_flags: glib.UriFlags, p_error: ?*?*glib.Error) ?*glib.Uri;
    pub const parse = g_uri_parse;

    /// Many URI schemes include one or more attribute/value pairs as part of the URI
    /// value. This method can be used to parse them into a hash table. When an
    /// attribute has multiple occurrences, the last value is the final returned
    /// value. If you need to handle repeated attributes differently, use
    /// `glib.UriParamsIter`.
    ///
    /// The `params` string is assumed to still be `%`-encoded, but the returned
    /// values will be fully decoded. (Thus it is possible that the returned values
    /// may contain `=` or `separators`, if the value was encoded in the input.)
    /// Invalid `%`-encoding is treated as with the `G_URI_FLAGS_PARSE_RELAXED`
    /// rules for `glib.uriParse`. (However, if `params` is the path or query string
    /// from a `glib.Uri` that was parsed without `G_URI_FLAGS_PARSE_RELAXED` and
    /// `G_URI_FLAGS_ENCODED`, then you already know that it does not contain any
    /// invalid encoding.)
    ///
    /// `G_URI_PARAMS_WWW_FORM` is handled as documented for `glib.UriParamsIter.init`.
    ///
    /// If `G_URI_PARAMS_CASE_INSENSITIVE` is passed to `flags`, attributes will be
    /// compared case-insensitively, so a params string `attr=123&Attr=456` will only
    /// return a single attribute–value pair, `Attr=456`. Case will be preserved in
    /// the returned attributes.
    ///
    /// If `params` cannot be parsed (for example, it contains two `separators`
    /// characters in a row), then `error` is set and `NULL` is returned.
    extern fn g_uri_parse_params(p_params: [*:0]const u8, p_length: isize, p_separators: [*:0]const u8, p_flags: glib.UriParamsFlags, p_error: ?*?*glib.Error) ?*glib.HashTable;
    pub const parseParams = g_uri_parse_params;

    /// Gets the scheme portion of a URI string.
    /// [RFC 3986](https://tools.ietf.org/html/rfc3986`section`-3) decodes the scheme
    /// as:
    /// ```
    /// URI = scheme ":" hier-part [ "?" query ] [ "#" fragment ]
    /// ```
    /// Common schemes include `file`, `https`, `svn+ssh`, etc.
    extern fn g_uri_parse_scheme(p_uri: [*:0]const u8) ?[*:0]u8;
    pub const parseScheme = g_uri_parse_scheme;

    /// Gets the scheme portion of a URI string.
    /// [RFC 3986](https://tools.ietf.org/html/rfc3986`section`-3) decodes the scheme
    /// as:
    /// ```
    /// URI = scheme ":" hier-part [ "?" query ] [ "#" fragment ]
    /// ```
    /// Common schemes include `file`, `https`, `svn+ssh`, etc.
    ///
    /// Unlike `glib.uriParseScheme`, the returned scheme is normalized to
    /// all-lowercase and does not need to be freed.
    extern fn g_uri_peek_scheme(p_uri: [*:0]const u8) ?[*:0]const u8;
    pub const peekScheme = g_uri_peek_scheme;

    /// Parses `uri_ref` according to `flags` and, if it is a
    /// [relative URI](`relative`-and-absolute-uris), resolves it relative to
    /// `base_uri_string`. If the result is not a valid absolute URI, it will be
    /// discarded, and an error returned.
    ///
    /// (If `base_uri_string` is `NULL`, this just returns `uri_ref`, or
    /// `NULL` if `uri_ref` is invalid or not absolute.)
    extern fn g_uri_resolve_relative(p_base_uri_string: ?[*:0]const u8, p_uri_ref: [*:0]const u8, p_flags: glib.UriFlags, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const resolveRelative = g_uri_resolve_relative;

    /// Parses `uri_ref` (which can be an
    /// [absolute or relative URI](`relative`-and-absolute-uris)) according to `flags`, and
    /// returns the pieces. Any component that doesn't appear in `uri_ref` will be
    /// returned as `NULL` (but note that all URIs always have a path component,
    /// though it may be the empty string).
    ///
    /// If `flags` contains `G_URI_FLAGS_ENCODED`, then `%`-encoded characters in
    /// `uri_ref` will remain encoded in the output strings. (If not,
    /// then all such characters will be decoded.) Note that decoding will
    /// only work if the URI components are ASCII or UTF-8, so you will
    /// need to use `G_URI_FLAGS_ENCODED` if they are not.
    ///
    /// Note that the `G_URI_FLAGS_HAS_PASSWORD` and
    /// `G_URI_FLAGS_HAS_AUTH_PARAMS` `flags` are ignored by `glib.uriSplit`,
    /// since it always returns only the full userinfo; use
    /// `glib.uriSplitWithUser` if you want it split up.
    extern fn g_uri_split(p_uri_ref: [*:0]const u8, p_flags: glib.UriFlags, p_scheme: ?*[*:0]u8, p_userinfo: ?*[*:0]u8, p_host: ?*[*:0]u8, p_port: ?*c_int, p_path: ?*[*:0]u8, p_query: ?*[*:0]u8, p_fragment: ?*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const split = g_uri_split;

    /// Parses `uri_string` (which must be an [absolute URI](`relative`-and-absolute-uris))
    /// according to `flags`, and returns the pieces relevant to connecting to a host.
    /// See the documentation for `glib.uriSplit` for more details; this is
    /// mostly a wrapper around that function with simpler arguments.
    /// However, it will return an error if `uri_string` is a relative URI,
    /// or does not contain a hostname component.
    extern fn g_uri_split_network(p_uri_string: [*:0]const u8, p_flags: glib.UriFlags, p_scheme: ?*[*:0]u8, p_host: ?*[*:0]u8, p_port: ?*c_int, p_error: ?*?*glib.Error) c_int;
    pub const splitNetwork = g_uri_split_network;

    /// Parses `uri_ref` (which can be an
    /// [absolute or relative URI](`relative`-and-absolute-uris)) according to `flags`, and
    /// returns the pieces. Any component that doesn't appear in `uri_ref` will be
    /// returned as `NULL` (but note that all URIs always have a path component,
    /// though it may be the empty string).
    ///
    /// See `glib.uriSplit`, and the definition of `glib.UriFlags`, for more
    /// information on the effect of `flags`. Note that `password` will only
    /// be parsed out if `flags` contains `G_URI_FLAGS_HAS_PASSWORD`, and
    /// `auth_params` will only be parsed out if `flags` contains
    /// `G_URI_FLAGS_HAS_AUTH_PARAMS`.
    extern fn g_uri_split_with_user(p_uri_ref: [*:0]const u8, p_flags: glib.UriFlags, p_scheme: ?*[*:0]u8, p_user: ?*[*:0]u8, p_password: ?*[*:0]u8, p_auth_params: ?*[*:0]u8, p_host: ?*[*:0]u8, p_port: ?*c_int, p_path: ?*[*:0]u8, p_query: ?*[*:0]u8, p_fragment: ?*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const splitWithUser = g_uri_split_with_user;

    /// Unescapes a segment of an escaped string as binary data.
    ///
    /// Note that in contrast to `glib.uriUnescapeString`, this does allow
    /// nul bytes to appear in the output.
    ///
    /// If any of the characters in `illegal_characters` appears as an escaped
    /// character in `escaped_string`, then that is an error and `NULL` will be
    /// returned. This is useful if you want to avoid for instance having a slash
    /// being expanded in an escaped path element, which might confuse pathname
    /// handling.
    extern fn g_uri_unescape_bytes(p_escaped_string: [*:0]const u8, p_length: isize, p_illegal_characters: ?[*:0]const u8, p_error: ?*?*glib.Error) ?*glib.Bytes;
    pub const unescapeBytes = g_uri_unescape_bytes;

    /// Unescapes a segment of an escaped string.
    ///
    /// If any of the characters in `illegal_characters` or the NUL
    /// character appears as an escaped character in `escaped_string`, then
    /// that is an error and `NULL` will be returned. This is useful if you
    /// want to avoid for instance having a slash being expanded in an
    /// escaped path element, which might confuse pathname handling.
    ///
    /// Note: `NUL` byte is not accepted in the output, in contrast to
    /// `glib.uriUnescapeBytes`.
    extern fn g_uri_unescape_segment(p_escaped_string: ?[*:0]const u8, p_escaped_string_end: ?[*:0]const u8, p_illegal_characters: ?[*:0]const u8) ?[*:0]u8;
    pub const unescapeSegment = g_uri_unescape_segment;

    /// Unescapes a whole escaped string.
    ///
    /// If any of the characters in `illegal_characters` or the NUL
    /// character appears as an escaped character in `escaped_string`, then
    /// that is an error and `NULL` will be returned. This is useful if you
    /// want to avoid for instance having a slash being expanded in an
    /// escaped path element, which might confuse pathname handling.
    extern fn g_uri_unescape_string(p_escaped_string: [*:0]const u8, p_illegal_characters: ?[*:0]const u8) ?[*:0]u8;
    pub const unescapeString = g_uri_unescape_string;

    /// Gets `uri`'s authentication parameters, which may contain
    /// `%`-encoding, depending on the flags with which `uri` was created.
    /// (If `uri` was not created with `G_URI_FLAGS_HAS_AUTH_PARAMS` then this will
    /// be `NULL`.)
    ///
    /// Depending on the URI scheme, `glib.uriParseParams` may be useful for
    /// further parsing this information.
    extern fn g_uri_get_auth_params(p_uri: *Uri) ?[*:0]const u8;
    pub const getAuthParams = g_uri_get_auth_params;

    /// Gets `uri`'s flags set upon construction.
    extern fn g_uri_get_flags(p_uri: *Uri) glib.UriFlags;
    pub const getFlags = g_uri_get_flags;

    /// Gets `uri`'s fragment, which may contain `%`-encoding, depending on
    /// the flags with which `uri` was created.
    extern fn g_uri_get_fragment(p_uri: *Uri) ?[*:0]const u8;
    pub const getFragment = g_uri_get_fragment;

    /// Gets `uri`'s host. This will never have `%`-encoded characters,
    /// unless it is non-UTF-8 (which can only be the case if `uri` was
    /// created with `G_URI_FLAGS_NON_DNS`).
    ///
    /// If `uri` contained an IPv6 address literal, this value will be just
    /// that address, without the brackets around it that are necessary in
    /// the string form of the URI. Note that in this case there may also
    /// be a scope ID attached to the address. Eg, `fe80::1234%``em1` (or
    /// `fe80::1234%``25em1` if the string is still encoded).
    extern fn g_uri_get_host(p_uri: *Uri) ?[*:0]const u8;
    pub const getHost = g_uri_get_host;

    /// Gets `uri`'s password, which may contain `%`-encoding, depending on
    /// the flags with which `uri` was created. (If `uri` was not created
    /// with `G_URI_FLAGS_HAS_PASSWORD` then this will be `NULL`.)
    extern fn g_uri_get_password(p_uri: *Uri) ?[*:0]const u8;
    pub const getPassword = g_uri_get_password;

    /// Gets `uri`'s path, which may contain `%`-encoding, depending on the
    /// flags with which `uri` was created.
    extern fn g_uri_get_path(p_uri: *Uri) [*:0]const u8;
    pub const getPath = g_uri_get_path;

    /// Gets `uri`'s port.
    extern fn g_uri_get_port(p_uri: *Uri) c_int;
    pub const getPort = g_uri_get_port;

    /// Gets `uri`'s query, which may contain `%`-encoding, depending on the
    /// flags with which `uri` was created.
    ///
    /// For queries consisting of a series of `name=value` parameters,
    /// `glib.UriParamsIter` or `glib.uriParseParams` may be useful.
    extern fn g_uri_get_query(p_uri: *Uri) ?[*:0]const u8;
    pub const getQuery = g_uri_get_query;

    /// Gets `uri`'s scheme. Note that this will always be all-lowercase,
    /// regardless of the string or strings that `uri` was created from.
    extern fn g_uri_get_scheme(p_uri: *Uri) [*:0]const u8;
    pub const getScheme = g_uri_get_scheme;

    /// Gets the ‘username’ component of `uri`'s userinfo, which may contain
    /// `%`-encoding, depending on the flags with which `uri` was created.
    /// If `uri` was not created with `G_URI_FLAGS_HAS_PASSWORD` or
    /// `G_URI_FLAGS_HAS_AUTH_PARAMS`, this is the same as `glib.Uri.getUserinfo`.
    extern fn g_uri_get_user(p_uri: *Uri) ?[*:0]const u8;
    pub const getUser = g_uri_get_user;

    /// Gets `uri`'s userinfo, which may contain `%`-encoding, depending on
    /// the flags with which `uri` was created.
    extern fn g_uri_get_userinfo(p_uri: *Uri) ?[*:0]const u8;
    pub const getUserinfo = g_uri_get_userinfo;

    /// Parses `uri_ref` according to `flags` and, if it is a
    /// [relative URI](`relative`-and-absolute-uris), resolves it relative to `base_uri`.
    /// If the result is not a valid absolute URI, it will be discarded, and an error
    /// returned.
    extern fn g_uri_parse_relative(p_base_uri: ?*Uri, p_uri_ref: [*:0]const u8, p_flags: glib.UriFlags, p_error: ?*?*glib.Error) ?*glib.Uri;
    pub const parseRelative = g_uri_parse_relative;

    /// Increments the reference count of `uri` by one.
    extern fn g_uri_ref(p_uri: *Uri) *glib.Uri;
    pub const ref = g_uri_ref;

    /// Returns a string representing `uri`.
    ///
    /// This is not guaranteed to return a string which is identical to the
    /// string that `uri` was parsed from. However, if the source URI was
    /// syntactically correct (according to RFC 3986), and it was parsed
    /// with `G_URI_FLAGS_ENCODED`, then `glib.Uri.toString` is guaranteed to return
    /// a string which is at least semantically equivalent to the source
    /// URI (according to RFC 3986).
    ///
    /// If `uri` might contain sensitive details, such as authentication parameters,
    /// or private data in its query string, and the returned string is going to be
    /// logged, then consider using `glib.Uri.toStringPartial` to redact parts.
    extern fn g_uri_to_string(p_uri: *Uri) [*:0]u8;
    pub const toString = g_uri_to_string;

    /// Returns a string representing `uri`, subject to the options in
    /// `flags`. See `glib.Uri.toString` and `glib.UriHideFlags` for more details.
    extern fn g_uri_to_string_partial(p_uri: *Uri, p_flags: glib.UriHideFlags) [*:0]u8;
    pub const toStringPartial = g_uri_to_string_partial;

    /// Atomically decrements the reference count of `uri` by one.
    ///
    /// When the reference count reaches zero, the resources allocated by
    /// `uri` are freed
    extern fn g_uri_unref(p_uri: *Uri) void;
    pub const unref = g_uri_unref;

    extern fn g_uri_get_type() usize;
    pub const getGObjectType = g_uri_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Many URI schemes include one or more attribute/value pairs as part of the URI
/// value. For example `scheme://server/path?query=string&is=there` has two
/// attributes – `query=string` and `is=there` – in its query part.
///
/// A `glib.UriParamsIter` structure represents an iterator that can be used to
/// iterate over the attribute/value pairs of a URI query string. `glib.UriParamsIter`
/// structures are typically allocated on the stack and then initialized with
/// `glib.UriParamsIter.init`. See the documentation for `glib.UriParamsIter.init`
/// for a usage example.
pub const UriParamsIter = extern struct {
    f_dummy0: c_int,
    f_dummy1: ?*anyopaque,
    f_dummy2: ?*anyopaque,
    f_dummy3: [256]u8,

    /// Initializes an attribute/value pair iterator.
    ///
    /// The iterator keeps pointers to the `params` and `separators` arguments, those
    /// variables must thus outlive the iterator and not be modified during the
    /// iteration.
    ///
    /// If `G_URI_PARAMS_WWW_FORM` is passed in `flags`, `+` characters in the param
    /// string will be replaced with spaces in the output. For example, `foo=bar+baz`
    /// will give attribute `foo` with value `bar baz`. This is commonly used on the
    /// web (the `https` and `http` schemes only), but is deprecated in favour of
    /// the equivalent of encoding spaces as ``@"20"``.
    ///
    /// Unlike with `glib.uriParseParams`, `G_URI_PARAMS_CASE_INSENSITIVE` has no
    /// effect if passed to `flags` for `glib.UriParamsIter.init`. The caller is
    /// responsible for doing their own case-insensitive comparisons.
    ///
    /// ```
    /// GUriParamsIter iter;
    /// GError *error = NULL;
    /// gchar *unowned_attr, *unowned_value;
    ///
    /// g_uri_params_iter_init (&iter, "foo=bar&baz=bar&Foo=frob&baz=bar2", -1, "&", G_URI_PARAMS_NONE);
    /// while (g_uri_params_iter_next (&iter, &unowned_attr, &unowned_value, &error))
    ///   {
    ///     g_autofree gchar *attr = g_steal_pointer (&unowned_attr);
    ///     g_autofree gchar *value = g_steal_pointer (&unowned_value);
    ///     // do something with attr and value; this code will be called 4 times
    ///     // for the params string in this example: once with attr=foo and value=bar,
    ///     // then with baz/bar, then Foo/frob, then baz/bar2.
    ///   }
    /// if (error)
    ///   // handle parsing error
    /// ```
    extern fn g_uri_params_iter_init(p_iter: *UriParamsIter, p_params: [*:0]const u8, p_length: isize, p_separators: [*:0]const u8, p_flags: glib.UriParamsFlags) void;
    pub const init = g_uri_params_iter_init;

    /// Advances `iter` and retrieves the next attribute/value. `FALSE` is returned if
    /// an error has occurred (in which case `error` is set), or if the end of the
    /// iteration is reached (in which case `attribute` and `value` are set to `NULL`
    /// and the iterator becomes invalid). If `TRUE` is returned,
    /// `glib.UriParamsIter.next` may be called again to receive another
    /// attribute/value pair.
    ///
    /// Note that the same `attribute` may be returned multiple times, since URIs
    /// allow repeated attributes.
    extern fn g_uri_params_iter_next(p_iter: *UriParamsIter, p_attribute: ?*[*:0]u8, p_value: ?*[*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const next = g_uri_params_iter_next;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `GVariant` is a variant datatype; it can contain one or more values
/// along with information about the type of the values.
///
/// A `GVariant` may contain simple types, like an integer, or a boolean value;
/// or complex types, like an array of two strings, or a dictionary of key
/// value pairs. A `GVariant` is also immutable: once it’s been created neither
/// its type nor its content can be modified further.
///
/// `GVariant` is useful whenever data needs to be serialized, for example when
/// sending method parameters in D-Bus, or when saving settings using
/// [`GSettings`](../gio/class.Settings.html).
///
/// When creating a new `GVariant`, you pass the data you want to store in it
/// along with a string representing the type of data you wish to pass to it.
///
/// For instance, if you want to create a `GVariant` holding an integer value you
/// can use:
///
/// ```c
/// GVariant *v = g_variant_new ("u", 40);
/// ```
///
/// The string `u` in the first argument tells `GVariant` that the data passed to
/// the constructor (`40`) is going to be an unsigned integer.
///
/// More advanced examples of `GVariant` in use can be found in documentation for
/// [`GVariant` format strings](gvariant-format-strings.html`pointers`).
///
/// The range of possible values is determined by the type.
///
/// The type system used by `GVariant` is `glib.VariantType`.
///
/// `GVariant` instances always have a type and a value (which are given
/// at construction time).  The type and value of a `GVariant` instance
/// can never change other than by the `GVariant` itself being
/// destroyed.  A `GVariant` cannot contain a pointer.
///
/// `GVariant` is reference counted using `glib.Variant.ref` and
/// `glib.Variant.unref`.  `GVariant` also has floating reference counts —
/// see `glib.Variant.refSink`.
///
/// `GVariant` is completely threadsafe.  A `GVariant` instance can be
/// concurrently accessed in any way from any number of threads without
/// problems.
///
/// `GVariant` is heavily optimised for dealing with data in serialized
/// form.  It works particularly well with data located in memory-mapped
/// files.  It can perform nearly all deserialization operations in a
/// small constant time, usually touching only a single memory page.
/// Serialized `GVariant` data can also be sent over the network.
///
/// `GVariant` is largely compatible with D-Bus.  Almost all types of
/// `GVariant` instances can be sent over D-Bus.  See `glib.VariantType` for
/// exceptions.  (However, `GVariant`’s serialization format is not the same
/// as the serialization format of a D-Bus message body: use
/// [GDBusMessage](../gio/class.DBusMessage.html), in the GIO library, for those.)
///
/// For space-efficiency, the `GVariant` serialization format does not
/// automatically include the variant’s length, type or endianness,
/// which must either be implied from context (such as knowledge that a
/// particular file format always contains a little-endian
/// `G_VARIANT_TYPE_VARIANT` which occupies the whole length of the file)
/// or supplied out-of-band (for instance, a length, type and/or endianness
/// indicator could be placed at the beginning of a file, network message
/// or network stream).
///
/// A `GVariant`’s size is limited mainly by any lower level operating
/// system constraints, such as the number of bits in `gsize`.  For
/// example, it is reasonable to have a 2GB file mapped into memory
/// with `glib.MappedFile`, and call `glib.Variant.newFromData` on
/// it.
///
/// For convenience to C programmers, `GVariant` features powerful
/// varargs-based value construction and destruction.  This feature is
/// designed to be embedded in other libraries.
///
/// There is a Python-inspired text language for describing `GVariant`
/// values.  `GVariant` includes a printer for this language and a parser
/// with type inferencing.
///
/// ## Memory Use
///
/// `GVariant` tries to be quite efficient with respect to memory use.
/// This section gives a rough idea of how much memory is used by the
/// current implementation.  The information here is subject to change
/// in the future.
///
/// The memory allocated by `GVariant` can be grouped into 4 broad
/// purposes: memory for serialized data, memory for the type
/// information cache, buffer management memory and memory for the
/// `GVariant` structure itself.
///
/// ## Serialized Data Memory
///
/// This is the memory that is used for storing `GVariant` data in
/// serialized form.  This is what would be sent over the network or
/// what would end up on disk, not counting any indicator of the
/// endianness, or of the length or type of the top-level variant.
///
/// The amount of memory required to store a boolean is 1 byte. 16,
/// 32 and 64 bit integers and double precision floating point numbers
/// use their ‘natural’ size.  Strings (including object path and
/// signature strings) are stored with a nul terminator, and as such
/// use the length of the string plus 1 byte.
///
/// ‘Maybe’ types use no space at all to represent the null value and
/// use the same amount of space (sometimes plus one byte) as the
/// equivalent non-maybe-typed value to represent the non-null case.
///
/// Arrays use the amount of space required to store each of their
/// members, concatenated.  Additionally, if the items stored in an
/// array are not of a fixed-size (ie: strings, other arrays, etc)
/// then an additional framing offset is stored for each item.  The
/// size of this offset is either 1, 2 or 4 bytes depending on the
/// overall size of the container.  Additionally, extra padding bytes
/// are added as required for alignment of child values.
///
/// Tuples (including dictionary entries) use the amount of space
/// required to store each of their members, concatenated, plus one
/// framing offset (as per arrays) for each non-fixed-sized item in
/// the tuple, except for the last one.  Additionally, extra padding
/// bytes are added as required for alignment of child values.
///
/// Variants use the same amount of space as the item inside of the
/// variant, plus 1 byte, plus the length of the type string for the
/// item inside the variant.
///
/// As an example, consider a dictionary mapping strings to variants.
/// In the case that the dictionary is empty, 0 bytes are required for
/// the serialization.
///
/// If we add an item ‘width’ that maps to the int32 value of 500 then
/// we will use 4 bytes to store the int32 (so 6 for the variant
/// containing it) and 6 bytes for the string.  The variant must be
/// aligned to 8 after the 6 bytes of the string, so that’s 2 extra
/// bytes.  6 (string) + 2 (padding) + 6 (variant) is 14 bytes used
/// for the dictionary entry.  An additional 1 byte is added to the
/// array as a framing offset making a total of 15 bytes.
///
/// If we add another entry, ‘title’ that maps to a nullable string
/// that happens to have a value of null, then we use 0 bytes for the
/// null value (and 3 bytes for the variant to contain it along with
/// its type string) plus 6 bytes for the string.  Again, we need 2
/// padding bytes.  That makes a total of 6 + 2 + 3 = 11 bytes.
///
/// We now require extra padding between the two items in the array.
/// After the 14 bytes of the first item, that’s 2 bytes required.
/// We now require 2 framing offsets for an extra two
/// bytes. 14 + 2 + 11 + 2 = 29 bytes to encode the entire two-item
/// dictionary.
///
/// ## Type Information Cache
///
/// For each `GVariant` type that currently exists in the program a type
/// information structure is kept in the type information cache.  The
/// type information structure is required for rapid deserialization.
///
/// Continuing with the above example, if a `GVariant` exists with the
/// type `a{sv}` then a type information struct will exist for
/// `a{sv}`, `{sv}`, `s`, and `v`.  Multiple uses of the same type
/// will share the same type information.  Additionally, all
/// single-digit types are stored in read-only static memory and do
/// not contribute to the writable memory footprint of a program using
/// `GVariant`.
///
/// Aside from the type information structures stored in read-only
/// memory, there are two forms of type information.  One is used for
/// container types where there is a single element type: arrays and
/// maybe types.  The other is used for container types where there
/// are multiple element types: tuples and dictionary entries.
///
/// Array type info structures are `6 * sizeof (void *)`, plus the
/// memory required to store the type string itself.  This means that
/// on 32-bit systems, the cache entry for `a{sv}` would require 30
/// bytes of memory (plus allocation overhead).
///
/// Tuple type info structures are `6 * sizeof (void *)`, plus `4 *
/// sizeof (void *)` for each item in the tuple, plus the memory
/// required to store the type string itself.  A 2-item tuple, for
/// example, would have a type information structure that consumed
/// writable memory in the size of `14 * sizeof (void *)` (plus type
/// string)  This means that on 32-bit systems, the cache entry for
/// `{sv}` would require 61 bytes of memory (plus allocation overhead).
///
/// This means that in total, for our `a{sv}` example, 91 bytes of
/// type information would be allocated.
///
/// The type information cache, additionally, uses a `glib.HashTable` to
/// store and look up the cached items and stores a pointer to this
/// hash table in static storage.  The hash table is freed when there
/// are zero items in the type cache.
///
/// Although these sizes may seem large it is important to remember
/// that a program will probably only have a very small number of
/// different types of values in it and that only one type information
/// structure is required for many different values of the same type.
///
/// ## Buffer Management Memory
///
/// `GVariant` uses an internal buffer management structure to deal
/// with the various different possible sources of serialized data
/// that it uses.  The buffer is responsible for ensuring that the
/// correct call is made when the data is no longer in use by
/// `GVariant`.  This may involve a `glib.free` or
/// even `glib.MappedFile.unref`.
///
/// One buffer management structure is used for each chunk of
/// serialized data.  The size of the buffer management structure
/// is `4 * (void *)`.  On 32-bit systems, that’s 16 bytes.
///
/// ## GVariant structure
///
/// The size of a `GVariant` structure is `6 * (void *)`.  On 32-bit
/// systems, that’s 24 bytes.
///
/// `GVariant` structures only exist if they are explicitly created
/// with API calls.  For example, if a `GVariant` is constructed out of
/// serialized data for the example given above (with the dictionary)
/// then although there are 9 individual values that comprise the
/// entire dictionary (two keys, two values, two variants containing
/// the values, two dictionary entries, plus the dictionary itself),
/// only 1 `GVariant` instance exists — the one referring to the
/// dictionary.
///
/// If calls are made to start accessing the other values then
/// `GVariant` instances will exist for those values only for as long
/// as they are in use (ie: until you call `glib.Variant.unref`).  The
/// type information is shared.  The serialized data and the buffer
/// management structure for that serialized data is shared by the
/// child.
///
/// ## Summary
///
/// To put the entire example together, for our dictionary mapping
/// strings to variants (with two entries, as given above), we are
/// using 91 bytes of memory for type information, 29 bytes of memory
/// for the serialized data, 16 bytes for buffer management and 24
/// bytes for the `GVariant` instance, or a total of 160 bytes, plus
/// allocation overhead.  If we were to use `glib.Variant.getChildValue`
/// to access the two dictionary entries, we would use an additional 48
/// bytes.  If we were to have other dictionaries of the same type, we
/// would use more memory for the serialized data and buffer
/// management for those dictionaries, but the type information would
/// be shared.
pub const Variant = opaque {
    /// Determines if a given string is a valid D-Bus object path.  You
    /// should ensure that a string is a valid D-Bus object path before
    /// passing it to `glib.Variant.newObjectPath`.
    ///
    /// A valid object path starts with `/` followed by zero or more
    /// sequences of characters separated by `/` characters.  Each sequence
    /// must contain only the characters `[A-Z][a-z][0-9]_`.  No sequence
    /// (including the one following the final `/` character) may be empty.
    extern fn g_variant_is_object_path(p_string: [*:0]const u8) c_int;
    pub const isObjectPath = g_variant_is_object_path;

    /// Determines if a given string is a valid D-Bus type signature.  You
    /// should ensure that a string is a valid D-Bus type signature before
    /// passing it to `glib.Variant.newSignature`.
    ///
    /// D-Bus type signatures consist of zero or more definite `glib.VariantType`
    /// strings in sequence.
    extern fn g_variant_is_signature(p_string: [*:0]const u8) c_int;
    pub const isSignature = g_variant_is_signature;

    /// Parses a `glib.Variant` from a text representation.
    ///
    /// A single `glib.Variant` is parsed from the content of `text`.
    ///
    /// The format is described [here](gvariant-text-format.html).
    ///
    /// The memory at `limit` will never be accessed and the parser behaves as
    /// if the character at `limit` is the nul terminator.  This has the
    /// effect of bounding `text`.
    ///
    /// If `endptr` is non-`NULL` then `text` is permitted to contain data
    /// following the value that this function parses and `endptr` will be
    /// updated to point to the first character past the end of the text
    /// parsed by this function.  If `endptr` is `NULL` and there is extra data
    /// then an error is returned.
    ///
    /// If `type` is non-`NULL` then the value will be parsed to have that
    /// type.  This may result in additional parse errors (in the case that
    /// the parsed value doesn't fit the type) but may also result in fewer
    /// errors (in the case that the type would have been ambiguous, such as
    /// with empty arrays).
    ///
    /// In the event that the parsing is successful, the resulting `glib.Variant`
    /// is returned. It is never floating, and must be freed with
    /// `glib.Variant.unref`.
    ///
    /// In case of any error, `NULL` will be returned.  If `error` is non-`NULL`
    /// then it will be set to reflect the error that occurred.
    ///
    /// Officially, the language understood by the parser is “any string
    /// produced by `glib.Variant.print`”. This explicitly includes
    /// ``glib.Variant.print``’s annotated types like `int64 -1000`.
    ///
    /// There may be implementation specific restrictions on deeply nested values,
    /// which would result in a `G_VARIANT_PARSE_ERROR_RECURSION` error. `glib.Variant` is
    /// guaranteed to handle nesting up to at least 64 levels.
    extern fn g_variant_parse(p_type: ?*const glib.VariantType, p_text: [*:0]const u8, p_limit: ?[*:0]const u8, p_endptr: ?*[*:0]const u8, p_error: ?*?*glib.Error) ?*glib.Variant;
    pub const parse = g_variant_parse;

    /// Pretty-prints a message showing the context of a `glib.Variant` parse
    /// error within the string for which parsing was attempted.
    ///
    /// The resulting string is suitable for output to the console or other
    /// monospace media where newlines are treated in the usual way.
    ///
    /// The message will typically look something like one of the following:
    ///
    /// ```
    /// unterminated string constant:
    ///   (1, 2, 3, 'abc
    ///             ^^^^
    /// ```
    ///
    /// or
    ///
    /// ```
    /// unable to find a common type:
    ///   [1, 2, 3, 'str']
    ///    ^        ^^^^^
    /// ```
    ///
    /// The format of the message may change in a future version.
    ///
    /// `error` must have come from a failed attempt to `glib.variantParse` and
    /// `source_str` must be exactly the same string that caused the error.
    /// If `source_str` was not nul-terminated when you passed it to
    /// `glib.variantParse` then you must add nul termination before using this
    /// function.
    extern fn g_variant_parse_error_print_context(p_error: *glib.Error, p_source_str: [*:0]const u8) [*:0]u8;
    pub const parseErrorPrintContext = g_variant_parse_error_print_context;

    extern fn g_variant_parse_error_quark() glib.Quark;
    pub const parseErrorQuark = g_variant_parse_error_quark;

    /// Same as `g_variant_error_quark`.
    extern fn g_variant_parser_get_error_quark() glib.Quark;
    pub const parserGetErrorQuark = g_variant_parser_get_error_quark;

    /// Creates a new `glib.Variant` instance.
    ///
    /// Think of this function as an analogue to `glib.strdupPrintf`.
    ///
    /// The type of the created instance and the arguments that are expected
    /// by this function are determined by `format_string`. See the section on
    /// [GVariant format strings](gvariant-format-strings.html). Please note that
    /// the syntax of the format string is very likely to be extended in the
    /// future.
    ///
    /// The first character of the format string must not be '*' '?' '@' or
    /// 'r'; in essence, a new `glib.Variant` must always be constructed by this
    /// function (and not merely passed through it unmodified).
    ///
    /// Note that the arguments must be of the correct width for their types
    /// specified in `format_string`. This can be achieved by casting them. See
    /// the [GVariant varargs documentation](gvariant-format-strings.html`varargs`).
    ///
    /// ```
    /// MyFlags some_flags = FLAG_ONE | FLAG_TWO;
    /// const gchar *some_strings[] = { "a", "b", "c", NULL };
    /// GVariant *new_variant;
    ///
    /// new_variant = g_variant_new ("(t^as)",
    ///                              // This cast is required.
    ///                              (guint64) some_flags,
    ///                              some_strings);
    /// ```
    extern fn g_variant_new(p_format_string: [*:0]const u8, ...) *glib.Variant;
    pub const new = g_variant_new;

    /// Creates a new `glib.Variant` array from `children`.
    ///
    /// `child_type` must be non-`NULL` if `n_children` is zero.  Otherwise, the
    /// child type is determined by inspecting the first element of the
    /// `children` array.  If `child_type` is non-`NULL` then it must be a
    /// definite type.
    ///
    /// The items of the array are taken from the `children` array.  No entry
    /// in the `children` array may be `NULL`.
    ///
    /// All items in the array must have the same type, which must be the
    /// same as `child_type`, if given.
    ///
    /// If the `children` are floating references (see `glib.Variant.refSink`), the
    /// new instance takes ownership of them as if via `glib.Variant.refSink`.
    extern fn g_variant_new_array(p_child_type: ?*const glib.VariantType, p_children: ?[*]const *glib.Variant, p_n_children: usize) *glib.Variant;
    pub const newArray = g_variant_new_array;

    /// Creates a new boolean `glib.Variant` instance -- either `TRUE` or `FALSE`.
    extern fn g_variant_new_boolean(p_value: c_int) *glib.Variant;
    pub const newBoolean = g_variant_new_boolean;

    /// Creates a new byte `glib.Variant` instance.
    extern fn g_variant_new_byte(p_value: u8) *glib.Variant;
    pub const newByte = g_variant_new_byte;

    /// Creates an array-of-bytes `glib.Variant` with the contents of `string`.
    /// This function is just like `glib.Variant.newString` except that the
    /// string need not be valid UTF-8.
    ///
    /// The nul terminator character at the end of the string is stored in
    /// the array.
    extern fn g_variant_new_bytestring(p_string: [*]const u8) *glib.Variant;
    pub const newBytestring = g_variant_new_bytestring;

    /// Constructs an array of bytestring `glib.Variant` from the given array of
    /// strings.
    ///
    /// If `length` is -1 then `strv` is `NULL`-terminated.
    extern fn g_variant_new_bytestring_array(p_strv: [*]const ?[*:0]const u8, p_length: isize) *glib.Variant;
    pub const newBytestringArray = g_variant_new_bytestring_array;

    /// Creates a new dictionary entry `glib.Variant`. `key` and `value` must be
    /// non-`NULL`. `key` must be a value of a basic type (ie: not a container).
    ///
    /// If the `key` or `value` are floating references (see `glib.Variant.refSink`),
    /// the new instance takes ownership of them as if via `glib.Variant.refSink`.
    extern fn g_variant_new_dict_entry(p_key: *glib.Variant, p_value: *glib.Variant) *glib.Variant;
    pub const newDictEntry = g_variant_new_dict_entry;

    /// Creates a new double `glib.Variant` instance.
    extern fn g_variant_new_double(p_value: f64) *glib.Variant;
    pub const newDouble = g_variant_new_double;

    /// Constructs a new array `glib.Variant` instance, where the elements are
    /// of `element_type` type.
    ///
    /// `elements` must be an array with fixed-sized elements.  Numeric types are
    /// fixed-size as are tuples containing only other fixed-sized types.
    ///
    /// `element_size` must be the size of a single element in the array.
    /// For example, if calling this function for an array of 32-bit integers,
    /// you might say sizeof(gint32). This value isn't used except for the purpose
    /// of a double-check that the form of the serialized data matches the caller's
    /// expectation.
    ///
    /// `n_elements` must be the length of the `elements` array.
    extern fn g_variant_new_fixed_array(p_element_type: *const glib.VariantType, p_elements: ?*const anyopaque, p_n_elements: usize, p_element_size: usize) *glib.Variant;
    pub const newFixedArray = g_variant_new_fixed_array;

    /// Constructs a new serialized-mode `glib.Variant` instance.  This is the
    /// inner interface for creation of new serialized values that gets
    /// called from various functions in gvariant.c.
    ///
    /// A reference is taken on `bytes`.
    ///
    /// The data in `bytes` must be aligned appropriately for the `type` being loaded.
    /// Otherwise this function will internally create a copy of the memory (since
    /// GLib 2.60) or (in older versions) fail and exit the process.
    extern fn g_variant_new_from_bytes(p_type: *const glib.VariantType, p_bytes: *glib.Bytes, p_trusted: c_int) *glib.Variant;
    pub const newFromBytes = g_variant_new_from_bytes;

    /// Creates a new `glib.Variant` instance from serialized data.
    ///
    /// `type` is the type of `glib.Variant` instance that will be constructed.
    /// The interpretation of `data` depends on knowing the type.
    ///
    /// `data` is not modified by this function and must remain valid with an
    /// unchanging value until such a time as `notify` is called with
    /// `user_data`.  If the contents of `data` change before that time then
    /// the result is undefined.
    ///
    /// If `data` is trusted to be serialized data in normal form then
    /// `trusted` should be `TRUE`.  This applies to serialized data created
    /// within this process or read from a trusted location on the disk (such
    /// as a file installed in /usr/lib alongside your application).  You
    /// should set trusted to `FALSE` if `data` is read from the network, a
    /// file in the user's home directory, etc.
    ///
    /// If `data` was not stored in this machine's native endianness, any multi-byte
    /// numeric values in the returned variant will also be in non-native
    /// endianness. `glib.Variant.byteswap` can be used to recover the original values.
    ///
    /// `notify` will be called with `user_data` when `data` is no longer
    /// needed.  The exact time of this call is unspecified and might even be
    /// before this function returns.
    ///
    /// Note: `data` must be backed by memory that is aligned appropriately for the
    /// `type` being loaded. Otherwise this function will internally create a copy of
    /// the memory (since GLib 2.60) or (in older versions) fail and exit the
    /// process.
    extern fn g_variant_new_from_data(p_type: *const glib.VariantType, p_data: [*]const u8, p_size: usize, p_trusted: c_int, p_notify: glib.DestroyNotify, p_user_data: ?*anyopaque) *glib.Variant;
    pub const newFromData = g_variant_new_from_data;

    /// Creates a new handle `glib.Variant` instance.
    ///
    /// By convention, handles are indexes into an array of file descriptors
    /// that are sent alongside a D-Bus message.  If you're not interacting
    /// with D-Bus, you probably don't need them.
    extern fn g_variant_new_handle(p_value: i32) *glib.Variant;
    pub const newHandle = g_variant_new_handle;

    /// Creates a new int16 `glib.Variant` instance.
    extern fn g_variant_new_int16(p_value: i16) *glib.Variant;
    pub const newInt16 = g_variant_new_int16;

    /// Creates a new int32 `glib.Variant` instance.
    extern fn g_variant_new_int32(p_value: i32) *glib.Variant;
    pub const newInt32 = g_variant_new_int32;

    /// Creates a new int64 `glib.Variant` instance.
    extern fn g_variant_new_int64(p_value: i64) *glib.Variant;
    pub const newInt64 = g_variant_new_int64;

    /// Depending on if `child` is `NULL`, either wraps `child` inside of a
    /// maybe container or creates a Nothing instance for the given `type`.
    ///
    /// At least one of `child_type` and `child` must be non-`NULL`.
    /// If `child_type` is non-`NULL` then it must be a definite type.
    /// If they are both non-`NULL` then `child_type` must be the type
    /// of `child`.
    ///
    /// If `child` is a floating reference (see `glib.Variant.refSink`), the new
    /// instance takes ownership of `child`.
    extern fn g_variant_new_maybe(p_child_type: ?*const glib.VariantType, p_child: ?*glib.Variant) *glib.Variant;
    pub const newMaybe = g_variant_new_maybe;

    /// Creates a D-Bus object path `glib.Variant` with the contents of `object_path`.
    /// `object_path` must be a valid D-Bus object path.  Use
    /// `glib.variantIsObjectPath` if you're not sure.
    extern fn g_variant_new_object_path(p_object_path: [*:0]const u8) *glib.Variant;
    pub const newObjectPath = g_variant_new_object_path;

    /// Constructs an array of object paths `glib.Variant` from the given array of
    /// strings.
    ///
    /// Each string must be a valid `glib.Variant` object path; see
    /// `glib.variantIsObjectPath`.
    ///
    /// If `length` is -1 then `strv` is `NULL`-terminated.
    extern fn g_variant_new_objv(p_strv: [*]const ?[*:0]const u8, p_length: isize) *glib.Variant;
    pub const newObjv = g_variant_new_objv;

    /// Parses `format` and returns the result.
    ///
    /// `format` must be a text format `glib.Variant` with one extension: at any
    /// point that a value may appear in the text, a '%' character followed
    /// by a GVariant format string (as per `glib.Variant.new`) may appear.  In
    /// that case, the same arguments are collected from the argument list as
    /// `glib.Variant.new` would have collected.
    ///
    /// Note that the arguments must be of the correct width for their types
    /// specified in `format`. This can be achieved by casting them. See
    /// the [GVariant varargs documentation](gvariant-format-strings.html`varargs`).
    ///
    /// Consider this simple example:
    /// ```
    ///  g_variant_new_parsed ("[('one', 1), ('two', `i`), (`s`, 3)]", 2, "three");
    /// ```
    ///
    /// In the example, the variable argument parameters are collected and
    /// filled in as if they were part of the original string to produce the
    /// result of
    /// ```
    /// [('one', 1), ('two', 2), ('three', 3)]
    /// ```
    ///
    /// This function is intended only to be used with `format` as a string
    /// literal.  Any parse error is fatal to the calling process.  If you
    /// want to parse data from untrusted sources, use `glib.variantParse`.
    ///
    /// You may not use this function to return, unmodified, a single
    /// `glib.Variant` pointer from the argument list.  ie: `format` may not solely
    /// be anything along the lines of "%*", "%?", "\%r", or anything starting
    /// with "%@".
    extern fn g_variant_new_parsed(p_format: [*:0]const u8, ...) *glib.Variant;
    pub const newParsed = g_variant_new_parsed;

    /// Parses `format` and returns the result.
    ///
    /// This is the version of `glib.Variant.newParsed` intended to be used
    /// from libraries.
    ///
    /// The return value will be floating if it was a newly created GVariant
    /// instance.  In the case that `format` simply specified the collection
    /// of a `glib.Variant` pointer (eg: `format` was "%*") then the collected
    /// `glib.Variant` pointer will be returned unmodified, without adding any
    /// additional references.
    ///
    /// Note that the arguments in `app` must be of the correct width for their types
    /// specified in `format` when collected into the `va_list`. See
    /// the [GVariant varargs documentation](gvariant-format-strings.html`varargs`).
    ///
    /// In order to behave correctly in all cases it is necessary for the
    /// calling function to `glib.Variant.refSink` the return result before
    /// returning control to the user that originally provided the pointer.
    /// At this point, the caller will have their own full reference to the
    /// result.  This can also be done by adding the result to a container,
    /// or by passing it to another `glib.Variant.new` call.
    extern fn g_variant_new_parsed_va(p_format: [*:0]const u8, p_app: *std.builtin.VaList) *glib.Variant;
    pub const newParsedVa = g_variant_new_parsed_va;

    /// Creates a string-type GVariant using printf formatting.
    ///
    /// This is similar to calling `glib.strdupPrintf` and then
    /// `glib.Variant.newString` but it saves a temporary variable and an
    /// unnecessary copy.
    extern fn g_variant_new_printf(p_format_string: [*:0]const u8, ...) *glib.Variant;
    pub const newPrintf = g_variant_new_printf;

    /// Creates a D-Bus type signature `glib.Variant` with the contents of
    /// `string`.  `string` must be a valid D-Bus type signature.  Use
    /// `glib.variantIsSignature` if you're not sure.
    extern fn g_variant_new_signature(p_signature: [*:0]const u8) *glib.Variant;
    pub const newSignature = g_variant_new_signature;

    /// Creates a string `glib.Variant` with the contents of `string`.
    ///
    /// `string` must be valid UTF-8, and must not be `NULL`. To encode
    /// potentially-`NULL` strings, use `glib.Variant.new` with `ms` as the
    /// [format string](gvariant-format-strings.html`maybe`-types).
    extern fn g_variant_new_string(p_string: [*:0]const u8) *glib.Variant;
    pub const newString = g_variant_new_string;

    /// Constructs an array of strings `glib.Variant` from the given array of
    /// strings.
    ///
    /// If `length` is -1 then `strv` is `NULL`-terminated.
    extern fn g_variant_new_strv(p_strv: [*]const ?[*:0]const u8, p_length: isize) *glib.Variant;
    pub const newStrv = g_variant_new_strv;

    /// Creates a string `glib.Variant` with the contents of `string`.
    ///
    /// `string` must be valid UTF-8, and must not be `NULL`. To encode
    /// potentially-`NULL` strings, use this with `glib.Variant.newMaybe`.
    ///
    /// After this call, `string` belongs to the `glib.Variant` and may no longer be
    /// modified by the caller. The memory of `data` has to be dynamically
    /// allocated and will eventually be freed with `glib.free`.
    ///
    /// You must not modify or access `string` in any other way after passing
    /// it to this function.  It is even possible that `string` is immediately
    /// freed.
    extern fn g_variant_new_take_string(p_string: [*:0]u8) *glib.Variant;
    pub const newTakeString = g_variant_new_take_string;

    /// Creates a new tuple `glib.Variant` out of the items in `children`.  The
    /// type is determined from the types of `children`.  No entry in the
    /// `children` array may be `NULL`.
    ///
    /// If `n_children` is 0 then the unit tuple is constructed.
    ///
    /// If the `children` are floating references (see `glib.Variant.refSink`), the
    /// new instance takes ownership of them as if via `glib.Variant.refSink`.
    extern fn g_variant_new_tuple(p_children: [*]const *glib.Variant, p_n_children: usize) *glib.Variant;
    pub const newTuple = g_variant_new_tuple;

    /// Creates a new uint16 `glib.Variant` instance.
    extern fn g_variant_new_uint16(p_value: u16) *glib.Variant;
    pub const newUint16 = g_variant_new_uint16;

    /// Creates a new uint32 `glib.Variant` instance.
    extern fn g_variant_new_uint32(p_value: u32) *glib.Variant;
    pub const newUint32 = g_variant_new_uint32;

    /// Creates a new uint64 `glib.Variant` instance.
    extern fn g_variant_new_uint64(p_value: u64) *glib.Variant;
    pub const newUint64 = g_variant_new_uint64;

    /// This function is intended to be used by libraries based on
    /// `glib.Variant` that want to provide `glib.Variant.new`-like functionality
    /// to their users.
    ///
    /// The API is more general than `glib.Variant.new` to allow a wider range
    /// of possible uses.
    ///
    /// `format_string` must still point to a valid format string, but it only
    /// needs to be nul-terminated if `endptr` is `NULL`.  If `endptr` is
    /// non-`NULL` then it is updated to point to the first character past the
    /// end of the format string.
    ///
    /// `app` is a pointer to a `va_list`.  The arguments, according to
    /// `format_string`, are collected from this `va_list` and the list is left
    /// pointing to the argument following the last.
    ///
    /// Note that the arguments in `app` must be of the correct width for their
    /// types specified in `format_string` when collected into the `va_list`.
    /// See the [GVariant varargs documentation](gvariant-format-strings.html`varargs`).
    ///
    /// These two generalisations allow mixing of multiple calls to
    /// `glib.Variant.newVa` and `glib.Variant.getVa` within a single actual
    /// varargs call by the user.
    ///
    /// The return value will be floating if it was a newly created GVariant
    /// instance (for example, if the format string was "(ii)").  In the case
    /// that the format_string was '*', '?', 'r', or a format starting with
    /// '@' then the collected `glib.Variant` pointer will be returned unmodified,
    /// without adding any additional references.
    ///
    /// In order to behave correctly in all cases it is necessary for the
    /// calling function to `glib.Variant.refSink` the return result before
    /// returning control to the user that originally provided the pointer.
    /// At this point, the caller will have their own full reference to the
    /// result.  This can also be done by adding the result to a container,
    /// or by passing it to another `glib.Variant.new` call.
    extern fn g_variant_new_va(p_format_string: [*:0]const u8, p_endptr: ?*[*:0]const u8, p_app: *std.builtin.VaList) *glib.Variant;
    pub const newVa = g_variant_new_va;

    /// Boxes `value`.  The result is a `glib.Variant` instance representing a
    /// variant containing the original value.
    ///
    /// If `child` is a floating reference (see `glib.Variant.refSink`), the new
    /// instance takes ownership of `child`.
    extern fn g_variant_new_variant(p_value: *glib.Variant) *glib.Variant;
    pub const newVariant = g_variant_new_variant;

    /// Performs a byteswapping operation on the contents of `value`.  The
    /// result is that all multi-byte numeric data contained in `value` is
    /// byteswapped.  That includes 16, 32, and 64bit signed and unsigned
    /// integers as well as file handles and double precision floating point
    /// values.
    ///
    /// This function is an identity mapping on any value that does not
    /// contain multi-byte numeric data.  That include strings, booleans,
    /// bytes and containers containing only these things (recursively).
    ///
    /// While this function can safely handle untrusted, non-normal data, it is
    /// recommended to check whether the input is in normal form beforehand, using
    /// `glib.Variant.isNormalForm`, and to reject non-normal inputs if your
    /// application can be strict about what inputs it rejects.
    ///
    /// The returned value is always in normal form and is marked as trusted.
    /// A full, not floating, reference is returned.
    extern fn g_variant_byteswap(p_value: *Variant) *glib.Variant;
    pub const byteswap = g_variant_byteswap;

    /// Checks if calling `glib.Variant.get` with `format_string` on `value` would
    /// be valid from a type-compatibility standpoint.  `format_string` is
    /// assumed to be a valid format string (from a syntactic standpoint).
    ///
    /// If `copy_only` is `TRUE` then this function additionally checks that it
    /// would be safe to call `glib.Variant.unref` on `value` immediately after
    /// the call to `glib.Variant.get` without invalidating the result.  This is
    /// only possible if deep copies are made (ie: there are no pointers to
    /// the data inside of the soon-to-be-freed `glib.Variant` instance).  If this
    /// check fails then a `g_critical` is printed and `FALSE` is returned.
    ///
    /// This function is meant to be used by functions that wish to provide
    /// varargs accessors to `glib.Variant` values of uncertain values (eg:
    /// `glib.Variant.lookup` or `g_menu_model_get_item_attribute`).
    extern fn g_variant_check_format_string(p_value: *Variant, p_format_string: [*:0]const u8, p_copy_only: c_int) c_int;
    pub const checkFormatString = g_variant_check_format_string;

    /// Classifies `value` according to its top-level type.
    extern fn g_variant_classify(p_value: *Variant) glib.VariantClass;
    pub const classify = g_variant_classify;

    /// Compares `one` and `two`.
    ///
    /// The types of `one` and `two` are `gconstpointer` only to allow use of
    /// this function with `glib.Tree`, `glib.PtrArray`, etc.  They must each be a
    /// `glib.Variant`.
    ///
    /// Comparison is only defined for basic types (ie: booleans, numbers,
    /// strings).  For booleans, `FALSE` is less than `TRUE`.  Numbers are
    /// ordered in the usual way.  Strings are in ASCII lexographical order.
    ///
    /// It is a programmer error to attempt to compare container values or
    /// two values that have types that are not exactly equal.  For example,
    /// you cannot compare a 32-bit signed integer with a 32-bit unsigned
    /// integer.  Also note that this function is not particularly
    /// well-behaved when it comes to comparison of doubles; in particular,
    /// the handling of incomparable values (ie: NaN) is undefined.
    ///
    /// If you only require an equality comparison, `glib.Variant.equal` is more
    /// general.
    extern fn g_variant_compare(p_one: *const Variant, p_two: *const glib.Variant) c_int;
    pub const compare = g_variant_compare;

    /// Similar to `glib.Variant.getBytestring` except that instead of
    /// returning a constant string, the string is duplicated.
    ///
    /// The return value must be freed using `glib.free`.
    extern fn g_variant_dup_bytestring(p_value: *Variant, p_length: ?*usize) [*:0]u8;
    pub const dupBytestring = g_variant_dup_bytestring;

    /// Gets the contents of an array of array of bytes `glib.Variant`.  This call
    /// makes a deep copy; the return result should be released with
    /// `glib.strfreev`.
    ///
    /// If `length` is non-`NULL` then the number of elements in the result is
    /// stored there.  In any case, the resulting array will be
    /// `NULL`-terminated.
    ///
    /// For an empty array, `length` will be set to 0 and a pointer to a
    /// `NULL` pointer will be returned.
    extern fn g_variant_dup_bytestring_array(p_value: *Variant, p_length: ?*usize) [*][*:0]u8;
    pub const dupBytestringArray = g_variant_dup_bytestring_array;

    /// Gets the contents of an array of object paths `glib.Variant`.  This call
    /// makes a deep copy; the return result should be released with
    /// `glib.strfreev`.
    ///
    /// If `length` is non-`NULL` then the number of elements in the result
    /// is stored there.  In any case, the resulting array will be
    /// `NULL`-terminated.
    ///
    /// For an empty array, `length` will be set to 0 and a pointer to a
    /// `NULL` pointer will be returned.
    extern fn g_variant_dup_objv(p_value: *Variant, p_length: ?*usize) [*:null]?[*:0]u8;
    pub const dupObjv = g_variant_dup_objv;

    /// Similar to `glib.Variant.getString` except that instead of returning
    /// a constant string, the string is duplicated.
    ///
    /// The string will always be UTF-8 encoded.
    ///
    /// The return value must be freed using `glib.free`.
    extern fn g_variant_dup_string(p_value: *Variant, p_length: *usize) [*:0]u8;
    pub const dupString = g_variant_dup_string;

    /// Gets the contents of an array of strings `glib.Variant`.  This call
    /// makes a deep copy; the return result should be released with
    /// `glib.strfreev`.
    ///
    /// If `length` is non-`NULL` then the number of elements in the result
    /// is stored there.  In any case, the resulting array will be
    /// `NULL`-terminated.
    ///
    /// For an empty array, `length` will be set to 0 and a pointer to a
    /// `NULL` pointer will be returned.
    extern fn g_variant_dup_strv(p_value: *Variant, p_length: ?*usize) [*:null]?[*:0]u8;
    pub const dupStrv = g_variant_dup_strv;

    /// Checks if `one` and `two` have the same type and value.
    ///
    /// The types of `one` and `two` are `gconstpointer` only to allow use of
    /// this function with `glib.HashTable`.  They must each be a `glib.Variant`.
    extern fn g_variant_equal(p_one: *const Variant, p_two: *const glib.Variant) c_int;
    pub const equal = g_variant_equal;

    /// Deconstructs a `glib.Variant` instance.
    ///
    /// Think of this function as an analogue to `scanf`.
    ///
    /// The arguments that are expected by this function are entirely
    /// determined by `format_string`.  `format_string` also restricts the
    /// permissible types of `value`.  It is an error to give a value with
    /// an incompatible type.  See the section on
    /// [GVariant format strings](gvariant-format-strings.html).
    /// Please note that the syntax of the format string is very likely to be
    /// extended in the future.
    ///
    /// `format_string` determines the C types that are used for unpacking
    /// the values and also determines if the values are copied or borrowed,
    /// see the section on
    /// [`GVariant` format strings](gvariant-format-strings.html`pointers`).
    extern fn g_variant_get(p_value: *Variant, p_format_string: [*:0]const u8, ...) void;
    pub const get = g_variant_get;

    /// Returns the boolean value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than `G_VARIANT_TYPE_BOOLEAN`.
    extern fn g_variant_get_boolean(p_value: *Variant) c_int;
    pub const getBoolean = g_variant_get_boolean;

    /// Returns the byte value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than `G_VARIANT_TYPE_BYTE`.
    extern fn g_variant_get_byte(p_value: *Variant) u8;
    pub const getByte = g_variant_get_byte;

    /// Returns the string value of a `glib.Variant` instance with an
    /// array-of-bytes type.  The string has no particular encoding.
    ///
    /// If the array does not end with a nul terminator character, the empty
    /// string is returned.  For this reason, you can always trust that a
    /// non-`NULL` nul-terminated string will be returned by this function.
    ///
    /// If the array contains a nul terminator character somewhere other than
    /// the last byte then the returned string is the string, up to the first
    /// such nul character.
    ///
    /// `glib.Variant.getFixedArray` should be used instead if the array contains
    /// arbitrary data that could not be nul-terminated or could contain nul bytes.
    ///
    /// It is an error to call this function with a `value` that is not an
    /// array of bytes.
    ///
    /// The return value remains valid as long as `value` exists.
    extern fn g_variant_get_bytestring(p_value: *Variant) [*]const u8;
    pub const getBytestring = g_variant_get_bytestring;

    /// Gets the contents of an array of array of bytes `glib.Variant`.  This call
    /// makes a shallow copy; the return result should be released with
    /// `glib.free`, but the individual strings must not be modified.
    ///
    /// If `length` is non-`NULL` then the number of elements in the result is
    /// stored there.  In any case, the resulting array will be
    /// `NULL`-terminated.
    ///
    /// For an empty array, `length` will be set to 0 and a pointer to a
    /// `NULL` pointer will be returned.
    extern fn g_variant_get_bytestring_array(p_value: *Variant, p_length: ?*usize) [*][*:0]const u8;
    pub const getBytestringArray = g_variant_get_bytestring_array;

    /// Reads a child item out of a container `glib.Variant` instance and
    /// deconstructs it according to `format_string`.  This call is
    /// essentially a combination of `glib.Variant.getChildValue` and
    /// `glib.Variant.get`.
    ///
    /// `format_string` determines the C types that are used for unpacking
    /// the values and also determines if the values are copied or borrowed,
    /// see the section on
    /// [`GVariant` format strings](gvariant-format-strings.html`pointers`).
    extern fn g_variant_get_child(p_value: *Variant, p_index_: usize, p_format_string: [*:0]const u8, ...) void;
    pub const getChild = g_variant_get_child;

    /// Reads a child item out of a container `glib.Variant` instance.  This
    /// includes variants, maybes, arrays, tuples and dictionary
    /// entries.  It is an error to call this function on any other type of
    /// `glib.Variant`.
    ///
    /// It is an error if `index_` is greater than the number of child items
    /// in the container.  See `glib.Variant.nChildren`.
    ///
    /// The returned value is never floating.  You should free it with
    /// `glib.Variant.unref` when you're done with it.
    ///
    /// Note that values borrowed from the returned child are not guaranteed to
    /// still be valid after the child is freed even if you still hold a reference
    /// to `value`, if `value` has not been serialized at the time this function is
    /// called. To avoid this, you can serialize `value` by calling
    /// `glib.Variant.getData` and optionally ignoring the return value.
    ///
    /// There may be implementation specific restrictions on deeply nested values,
    /// which would result in the unit tuple being returned as the child value,
    /// instead of further nested children. `glib.Variant` is guaranteed to handle
    /// nesting up to at least 64 levels.
    ///
    /// This function is O(1).
    extern fn g_variant_get_child_value(p_value: *Variant, p_index_: usize) *glib.Variant;
    pub const getChildValue = g_variant_get_child_value;

    /// Returns a pointer to the serialized form of a `glib.Variant` instance.
    /// The returned data may not be in fully-normalised form if read from an
    /// untrusted source.  The returned data must not be freed; it remains
    /// valid for as long as `value` exists.
    ///
    /// If `value` is a fixed-sized value that was deserialized from a
    /// corrupted serialized container then `NULL` may be returned.  In this
    /// case, the proper thing to do is typically to use the appropriate
    /// number of nul bytes in place of `value`.  If `value` is not fixed-sized
    /// then `NULL` is never returned.
    ///
    /// In the case that `value` is already in serialized form, this function
    /// is O(1).  If the value is not already in serialized form,
    /// serialization occurs implicitly and is approximately O(n) in the size
    /// of the result.
    ///
    /// To deserialize the data returned by this function, in addition to the
    /// serialized data, you must know the type of the `glib.Variant`, and (if the
    /// machine might be different) the endianness of the machine that stored
    /// it. As a result, file formats or network messages that incorporate
    /// serialized `GVariants` must include this information either
    /// implicitly (for instance "the file always contains a
    /// `G_VARIANT_TYPE_VARIANT` and it is always in little-endian order") or
    /// explicitly (by storing the type and/or endianness in addition to the
    /// serialized data).
    extern fn g_variant_get_data(p_value: *Variant) ?*const anyopaque;
    pub const getData = g_variant_get_data;

    /// Returns a pointer to the serialized form of a `glib.Variant` instance.
    /// The semantics of this function are exactly the same as
    /// `glib.Variant.getData`, except that the returned `glib.Bytes` holds
    /// a reference to the variant data.
    extern fn g_variant_get_data_as_bytes(p_value: *Variant) *glib.Bytes;
    pub const getDataAsBytes = g_variant_get_data_as_bytes;

    /// Returns the double precision floating point value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than `G_VARIANT_TYPE_DOUBLE`.
    extern fn g_variant_get_double(p_value: *Variant) f64;
    pub const getDouble = g_variant_get_double;

    /// Provides access to the serialized data for an array of fixed-sized
    /// items.
    ///
    /// `value` must be an array with fixed-sized elements.  Numeric types are
    /// fixed-size, as are tuples containing only other fixed-sized types.
    ///
    /// `element_size` must be the size of a single element in the array,
    /// as given by the section on
    /// [serialized data memory](struct.Variant.html`serialized`-data-memory).
    ///
    /// In particular, arrays of these fixed-sized types can be interpreted
    /// as an array of the given C type, with `element_size` set to the size
    /// the appropriate type:
    /// - `G_VARIANT_TYPE_INT16` (etc.): `gint16` (etc.)
    /// - `G_VARIANT_TYPE_BOOLEAN`: `guchar` (not `gboolean`!)
    /// - `G_VARIANT_TYPE_BYTE`: `guint8`
    /// - `G_VARIANT_TYPE_HANDLE`: `guint32`
    /// - `G_VARIANT_TYPE_DOUBLE`: `gdouble`
    ///
    /// For example, if calling this function for an array of 32-bit integers,
    /// you might say `sizeof(gint32)`. This value isn't used except for the purpose
    /// of a double-check that the form of the serialized data matches the caller's
    /// expectation.
    ///
    /// `n_elements`, which must be non-`NULL`, is set equal to the number of
    /// items in the array.
    extern fn g_variant_get_fixed_array(p_value: *Variant, p_n_elements: *usize, p_element_size: usize) [*]const *anyopaque;
    pub const getFixedArray = g_variant_get_fixed_array;

    /// Returns the 32-bit signed integer value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type other
    /// than `G_VARIANT_TYPE_HANDLE`.
    ///
    /// By convention, handles are indexes into an array of file descriptors
    /// that are sent alongside a D-Bus message.  If you're not interacting
    /// with D-Bus, you probably don't need them.
    extern fn g_variant_get_handle(p_value: *Variant) i32;
    pub const getHandle = g_variant_get_handle;

    /// Returns the 16-bit signed integer value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than `G_VARIANT_TYPE_INT16`.
    extern fn g_variant_get_int16(p_value: *Variant) i16;
    pub const getInt16 = g_variant_get_int16;

    /// Returns the 32-bit signed integer value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than `G_VARIANT_TYPE_INT32`.
    extern fn g_variant_get_int32(p_value: *Variant) i32;
    pub const getInt32 = g_variant_get_int32;

    /// Returns the 64-bit signed integer value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than `G_VARIANT_TYPE_INT64`.
    extern fn g_variant_get_int64(p_value: *Variant) i64;
    pub const getInt64 = g_variant_get_int64;

    /// Given a maybe-typed `glib.Variant` instance, extract its value.  If the
    /// value is Nothing, then this function returns `NULL`.
    extern fn g_variant_get_maybe(p_value: *Variant) ?*glib.Variant;
    pub const getMaybe = g_variant_get_maybe;

    /// Gets a `glib.Variant` instance that has the same value as `value` and is
    /// trusted to be in normal form.
    ///
    /// If `value` is already trusted to be in normal form then a new
    /// reference to `value` is returned.
    ///
    /// If `value` is not already trusted, then it is scanned to check if it
    /// is in normal form.  If it is found to be in normal form then it is
    /// marked as trusted and a new reference to it is returned.
    ///
    /// If `value` is found not to be in normal form then a new trusted
    /// `glib.Variant` is created with the same value as `value`. The non-normal parts of
    /// `value` will be replaced with default values which are guaranteed to be in
    /// normal form.
    ///
    /// It makes sense to call this function if you've received `glib.Variant`
    /// data from untrusted sources and you want to ensure your serialized
    /// output is definitely in normal form.
    ///
    /// If `value` is already in normal form, a new reference will be returned
    /// (which will be floating if `value` is floating). If it is not in normal form,
    /// the newly created `glib.Variant` will be returned with a single non-floating
    /// reference. Typically, `glib.Variant.takeRef` should be called on the return
    /// value from this function to guarantee ownership of a single non-floating
    /// reference to it.
    extern fn g_variant_get_normal_form(p_value: *Variant) *glib.Variant;
    pub const getNormalForm = g_variant_get_normal_form;

    /// Gets the contents of an array of object paths `glib.Variant`.  This call
    /// makes a shallow copy; the return result should be released with
    /// `glib.free`, but the individual strings must not be modified.
    ///
    /// If `length` is non-`NULL` then the number of elements in the result
    /// is stored there.  In any case, the resulting array will be
    /// `NULL`-terminated.
    ///
    /// For an empty array, `length` will be set to 0 and a pointer to a
    /// `NULL` pointer will be returned.
    extern fn g_variant_get_objv(p_value: *Variant, p_length: ?*usize) [*:null]?[*:0]const u8;
    pub const getObjv = g_variant_get_objv;

    /// Determines the number of bytes that would be required to store `value`
    /// with `glib.Variant.store`.
    ///
    /// If `value` has a fixed-sized type then this function always returned
    /// that fixed size.
    ///
    /// In the case that `value` is already in serialized form or the size has
    /// already been calculated (ie: this function has been called before)
    /// then this function is O(1).  Otherwise, the size is calculated, an
    /// operation which is approximately O(n) in the number of values
    /// involved.
    extern fn g_variant_get_size(p_value: *Variant) usize;
    pub const getSize = g_variant_get_size;

    /// Returns the string value of a `glib.Variant` instance with a string
    /// type.  This includes the types `G_VARIANT_TYPE_STRING`,
    /// `G_VARIANT_TYPE_OBJECT_PATH` and `G_VARIANT_TYPE_SIGNATURE`.
    ///
    /// The string will always be UTF-8 encoded, will never be `NULL`, and will never
    /// contain nul bytes.
    ///
    /// If `length` is non-`NULL` then the length of the string (in bytes) is
    /// returned there.  For trusted values, this information is already
    /// known.  Untrusted values will be validated and, if valid, a `strlen` will be
    /// performed. If invalid, a default value will be returned — for
    /// `G_VARIANT_TYPE_OBJECT_PATH`, this is `"/"`, and for other types it is the
    /// empty string.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than those three.
    ///
    /// The return value remains valid as long as `value` exists.
    extern fn g_variant_get_string(p_value: *Variant, p_length: ?*usize) [*:0]const u8;
    pub const getString = g_variant_get_string;

    /// Gets the contents of an array of strings `glib.Variant`.  This call
    /// makes a shallow copy; the return result should be released with
    /// `glib.free`, but the individual strings must not be modified.
    ///
    /// If `length` is non-`NULL` then the number of elements in the result
    /// is stored there.  In any case, the resulting array will be
    /// `NULL`-terminated.
    ///
    /// For an empty array, `length` will be set to 0 and a pointer to a
    /// `NULL` pointer will be returned.
    extern fn g_variant_get_strv(p_value: *Variant, p_length: ?*usize) [*:null]?[*:0]const u8;
    pub const getStrv = g_variant_get_strv;

    /// Determines the type of `value`.
    ///
    /// The return value is valid for the lifetime of `value` and must not
    /// be freed.
    extern fn g_variant_get_type(p_value: *Variant) *const glib.VariantType;
    pub const getType = g_variant_get_type;

    /// Returns the type string of `value`.  Unlike the result of calling
    /// `glib.VariantType.peekString`, this string is nul-terminated.  This
    /// string belongs to `glib.Variant` and must not be freed.
    extern fn g_variant_get_type_string(p_value: *Variant) [*:0]const u8;
    pub const getTypeString = g_variant_get_type_string;

    /// Returns the 16-bit unsigned integer value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than `G_VARIANT_TYPE_UINT16`.
    extern fn g_variant_get_uint16(p_value: *Variant) u16;
    pub const getUint16 = g_variant_get_uint16;

    /// Returns the 32-bit unsigned integer value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than `G_VARIANT_TYPE_UINT32`.
    extern fn g_variant_get_uint32(p_value: *Variant) u32;
    pub const getUint32 = g_variant_get_uint32;

    /// Returns the 64-bit unsigned integer value of `value`.
    ///
    /// It is an error to call this function with a `value` of any type
    /// other than `G_VARIANT_TYPE_UINT64`.
    extern fn g_variant_get_uint64(p_value: *Variant) u64;
    pub const getUint64 = g_variant_get_uint64;

    /// This function is intended to be used by libraries based on `glib.Variant`
    /// that want to provide `glib.Variant.get`-like functionality to their
    /// users.
    ///
    /// The API is more general than `glib.Variant.get` to allow a wider range
    /// of possible uses.
    ///
    /// `format_string` must still point to a valid format string, but it only
    /// need to be nul-terminated if `endptr` is `NULL`.  If `endptr` is
    /// non-`NULL` then it is updated to point to the first character past the
    /// end of the format string.
    ///
    /// `app` is a pointer to a `va_list`.  The arguments, according to
    /// `format_string`, are collected from this `va_list` and the list is left
    /// pointing to the argument following the last.
    ///
    /// These two generalisations allow mixing of multiple calls to
    /// `glib.Variant.newVa` and `glib.Variant.getVa` within a single actual
    /// varargs call by the user.
    ///
    /// `format_string` determines the C types that are used for unpacking
    /// the values and also determines if the values are copied or borrowed,
    /// see the section on
    /// [`GVariant` format strings](gvariant-format-strings.html`pointers`).
    extern fn g_variant_get_va(p_value: *Variant, p_format_string: [*:0]const u8, p_endptr: ?*[*:0]const u8, p_app: *std.builtin.VaList) void;
    pub const getVa = g_variant_get_va;

    /// Unboxes `value`.  The result is the `glib.Variant` instance that was
    /// contained in `value`.
    extern fn g_variant_get_variant(p_value: *Variant) *glib.Variant;
    pub const getVariant = g_variant_get_variant;

    /// Generates a hash value for a `glib.Variant` instance.
    ///
    /// The output of this function is guaranteed to be the same for a given
    /// value only per-process.  It may change between different processor
    /// architectures or even different versions of GLib.  Do not use this
    /// function as a basis for building protocols or file formats.
    ///
    /// The type of `value` is `gconstpointer` only to allow use of this
    /// function with `glib.HashTable`.  `value` must be a `glib.Variant`.
    extern fn g_variant_hash(p_value: *const Variant) c_uint;
    pub const hash = g_variant_hash;

    /// Checks if `value` is a container.
    extern fn g_variant_is_container(p_value: *Variant) c_int;
    pub const isContainer = g_variant_is_container;

    /// Checks whether `value` has a floating reference count.
    ///
    /// This function should only ever be used to assert that a given variant
    /// is or is not floating, or for debug purposes. To acquire a reference
    /// to a variant that might be floating, always use `glib.Variant.refSink`
    /// or `glib.Variant.takeRef`.
    ///
    /// See `glib.Variant.refSink` for more information about floating reference
    /// counts.
    extern fn g_variant_is_floating(p_value: *Variant) c_int;
    pub const isFloating = g_variant_is_floating;

    /// Checks if `value` is in normal form.
    ///
    /// The main reason to do this is to detect if a given chunk of
    /// serialized data is in normal form: load the data into a `glib.Variant`
    /// using `glib.Variant.newFromData` and then use this function to
    /// check.
    ///
    /// If `value` is found to be in normal form then it will be marked as
    /// being trusted.  If the value was already marked as being trusted then
    /// this function will immediately return `TRUE`.
    ///
    /// There may be implementation specific restrictions on deeply nested values.
    /// GVariant is guaranteed to handle nesting up to at least 64 levels.
    extern fn g_variant_is_normal_form(p_value: *Variant) c_int;
    pub const isNormalForm = g_variant_is_normal_form;

    /// Checks if a value has a type matching the provided type.
    extern fn g_variant_is_of_type(p_value: *Variant, p_type: *const glib.VariantType) c_int;
    pub const isOfType = g_variant_is_of_type;

    /// Creates a heap-allocated `glib.VariantIter` for iterating over the items
    /// in `value`.
    ///
    /// Use `glib.VariantIter.free` to free the return value when you no longer
    /// need it.
    ///
    /// A reference is taken to `value` and will be released only when
    /// `glib.VariantIter.free` is called.
    extern fn g_variant_iter_new(p_value: *Variant) *glib.VariantIter;
    pub const iterNew = g_variant_iter_new;

    /// Looks up a value in a dictionary `glib.Variant`.
    ///
    /// This function is a wrapper around `glib.Variant.lookupValue` and
    /// `glib.Variant.get`.  In the case that `NULL` would have been returned,
    /// this function returns `FALSE`.  Otherwise, it unpacks the returned
    /// value and returns `TRUE`.
    ///
    /// `format_string` determines the C types that are used for unpacking
    /// the values and also determines if the values are copied or borrowed,
    /// see the section on
    /// [`GVariant` format strings](gvariant-format-strings.html`pointers`).
    ///
    /// This function is currently implemented with a linear scan.  If you
    /// plan to do many lookups then `glib.VariantDict` may be more efficient.
    extern fn g_variant_lookup(p_dictionary: *Variant, p_key: [*:0]const u8, p_format_string: [*:0]const u8, ...) c_int;
    pub const lookup = g_variant_lookup;

    /// Looks up a value in a dictionary `glib.Variant`.
    ///
    /// This function works with dictionaries of the type a{s*} (and equally
    /// well with type a{o*}, but we only further discuss the string case
    /// for sake of clarity).
    ///
    /// In the event that `dictionary` has the type a{sv}, the `expected_type`
    /// string specifies what type of value is expected to be inside of the
    /// variant. If the value inside the variant has a different type then
    /// `NULL` is returned. In the event that `dictionary` has a value type other
    /// than v then `expected_type` must directly match the value type and it is
    /// used to unpack the value directly or an error occurs.
    ///
    /// In either case, if `key` is not found in `dictionary`, `NULL` is returned.
    ///
    /// If the key is found and the value has the correct type, it is
    /// returned.  If `expected_type` was specified then any non-`NULL` return
    /// value will have this type.
    ///
    /// This function is currently implemented with a linear scan.  If you
    /// plan to do many lookups then `glib.VariantDict` may be more efficient.
    extern fn g_variant_lookup_value(p_dictionary: *Variant, p_key: [*:0]const u8, p_expected_type: ?*const glib.VariantType) *glib.Variant;
    pub const lookupValue = g_variant_lookup_value;

    /// Determines the number of children in a container `glib.Variant` instance.
    /// This includes variants, maybes, arrays, tuples and dictionary
    /// entries.  It is an error to call this function on any other type of
    /// `glib.Variant`.
    ///
    /// For variants, the return value is always 1.  For values with maybe
    /// types, it is always zero or one.  For arrays, it is the length of the
    /// array.  For tuples it is the number of tuple items (which depends
    /// only on the type).  For dictionary entries, it is always 2
    ///
    /// This function is O(1).
    extern fn g_variant_n_children(p_value: *Variant) usize;
    pub const nChildren = g_variant_n_children;

    /// Pretty-prints `value` in the format understood by `glib.variantParse`.
    ///
    /// The format is described [here](gvariant-text-format.html).
    ///
    /// If `type_annotate` is `TRUE`, then type information is included in
    /// the output.
    extern fn g_variant_print(p_value: *Variant, p_type_annotate: c_int) [*:0]u8;
    pub const print = g_variant_print;

    /// Behaves as `glib.Variant.print`, but operates on a `glib.String`.
    ///
    /// If `string` is non-`NULL` then it is appended to and returned.  Else,
    /// a new empty `glib.String` is allocated and it is returned.
    extern fn g_variant_print_string(p_value: *Variant, p_string: ?*glib.String, p_type_annotate: c_int) *glib.String;
    pub const printString = g_variant_print_string;

    /// Increases the reference count of `value`.
    extern fn g_variant_ref(p_value: *Variant) *glib.Variant;
    pub const ref = g_variant_ref;

    /// `glib.Variant` uses a floating reference count system.  All functions with
    /// names starting with `g_variant_new_` return floating
    /// references.
    ///
    /// Calling `glib.Variant.refSink` on a `glib.Variant` with a floating reference
    /// will convert the floating reference into a full reference.  Calling
    /// `glib.Variant.refSink` on a non-floating `glib.Variant` results in an
    /// additional normal reference being added.
    ///
    /// In other words, if the `value` is floating, then this call "assumes
    /// ownership" of the floating reference, converting it to a normal
    /// reference.  If the `value` is not floating, then this call adds a
    /// new normal reference increasing the reference count by one.
    ///
    /// All calls that result in a `glib.Variant` instance being inserted into a
    /// container will call `glib.Variant.refSink` on the instance.  This means
    /// that if the value was just created (and has only its floating
    /// reference) then the container will assume sole ownership of the value
    /// at that point and the caller will not need to unreference it.  This
    /// makes certain common styles of programming much easier while still
    /// maintaining normal refcounting semantics in situations where values
    /// are not floating.
    extern fn g_variant_ref_sink(p_value: *Variant) *glib.Variant;
    pub const refSink = g_variant_ref_sink;

    /// Stores the serialized form of `value` at `data`.  `data` should be
    /// large enough.  See `glib.Variant.getSize`.
    ///
    /// The stored data is in machine native byte order but may not be in
    /// fully-normalised form if read from an untrusted source.  See
    /// `glib.Variant.getNormalForm` for a solution.
    ///
    /// As with `glib.Variant.getData`, to be able to deserialize the
    /// serialized variant successfully, its type and (if the destination
    /// machine might be different) its endianness must also be available.
    ///
    /// This function is approximately O(n) in the size of `data`.
    extern fn g_variant_store(p_value: *Variant, p_data: *anyopaque) void;
    pub const store = g_variant_store;

    /// If `value` is floating, sink it.  Otherwise, do nothing.
    ///
    /// Typically you want to use `glib.Variant.refSink` in order to
    /// automatically do the correct thing with respect to floating or
    /// non-floating references, but there is one specific scenario where
    /// this function is helpful.
    ///
    /// The situation where this function is helpful is when creating an API
    /// that allows the user to provide a callback function that returns a
    /// `glib.Variant`.  We certainly want to allow the user the flexibility to
    /// return a non-floating reference from this callback (for the case
    /// where the value that is being returned already exists).
    ///
    /// At the same time, the style of the `glib.Variant` API makes it likely that
    /// for newly-created `glib.Variant` instances, the user can be saved some
    /// typing if they are allowed to return a `glib.Variant` with a floating
    /// reference.
    ///
    /// Using this function on the return value of the user's callback allows
    /// the user to do whichever is more convenient for them.  The caller
    /// will always receives exactly one full reference to the value: either
    /// the one that was returned in the first place, or a floating reference
    /// that has been converted to a full reference.
    ///
    /// This function has an odd interaction when combined with
    /// `glib.Variant.refSink` running at the same time in another thread on
    /// the same `glib.Variant` instance.  If `glib.Variant.refSink` runs first then
    /// the result will be that the floating reference is converted to a hard
    /// reference.  If `glib.Variant.takeRef` runs first then the result will
    /// be that the floating reference is converted to a hard reference and
    /// an additional reference on top of that one is added.  It is best to
    /// avoid this situation.
    extern fn g_variant_take_ref(p_value: *Variant) *glib.Variant;
    pub const takeRef = g_variant_take_ref;

    /// Decreases the reference count of `value`.  When its reference count
    /// drops to 0, the memory used by the variant is freed.
    extern fn g_variant_unref(p_value: *Variant) void;
    pub const unref = g_variant_unref;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A utility type for constructing container-type `glib.Variant` instances.
///
/// This is an opaque structure and may only be accessed using the
/// following functions.
///
/// `glib.VariantBuilder` is not threadsafe in any way.  Do not attempt to
/// access it from more than one thread.
pub const VariantBuilder = extern struct {
    anon0: extern union {
        anon0: extern struct {
            f_partial_magic: usize,
            f_type: ?*const glib.VariantType,
            f_y: [14]usize,
        },
        f_x: [16]usize,
    },

    /// Allocates and initialises a new `glib.VariantBuilder`.
    ///
    /// You should call `glib.VariantBuilder.unref` on the return value when it
    /// is no longer needed.  The memory will not be automatically freed by
    /// any other call.
    ///
    /// In most cases it is easier to place a `glib.VariantBuilder` directly on
    /// the stack of the calling function and initialise it with
    /// `glib.VariantBuilder.initStatic`.
    extern fn g_variant_builder_new(p_type: *const glib.VariantType) *glib.VariantBuilder;
    pub const new = g_variant_builder_new;

    /// Adds to a `glib.VariantBuilder`.
    ///
    /// This call is a convenience wrapper that is exactly equivalent to
    /// calling `glib.Variant.new` followed by `glib.VariantBuilder.addValue`.
    ///
    /// Note that the arguments must be of the correct width for their types
    /// specified in `format_string`. This can be achieved by casting them. See
    /// the [GVariant varargs documentation](gvariant-format-strings.html`varargs`).
    ///
    /// This function might be used as follows:
    ///
    /// ```
    /// GVariant *
    /// make_pointless_dictionary (void)
    /// {
    ///   GVariantBuilder builder;
    ///   int i;
    ///
    ///   g_variant_builder_init_static (&builder, G_VARIANT_TYPE_ARRAY);
    ///   for (i = 0; i < 16; i++)
    ///     {
    ///       gchar buf[3];
    ///
    ///       sprintf (buf, "`d`", i);
    ///       g_variant_builder_add (&builder, "{is}", i, buf);
    ///     }
    ///
    ///   return g_variant_builder_end (&builder);
    /// }
    /// ```
    extern fn g_variant_builder_add(p_builder: *VariantBuilder, p_format_string: [*:0]const u8, ...) void;
    pub const add = g_variant_builder_add;

    /// Adds to a `glib.VariantBuilder`.
    ///
    /// This call is a convenience wrapper that is exactly equivalent to
    /// calling `glib.Variant.newParsed` followed by
    /// `glib.VariantBuilder.addValue`.
    ///
    /// Note that the arguments must be of the correct width for their types
    /// specified in `format_string`. This can be achieved by casting them. See
    /// the [GVariant varargs documentation](gvariant-format-strings.html`varargs`).
    ///
    /// This function might be used as follows:
    ///
    /// ```
    /// GVariant *
    /// make_pointless_dictionary (void)
    /// {
    ///   GVariantBuilder builder;
    ///   int i;
    ///
    ///   g_variant_builder_init_static (&builder, G_VARIANT_TYPE_ARRAY);
    ///   g_variant_builder_add_parsed (&builder, "{'width', <`i`>}", 600);
    ///   g_variant_builder_add_parsed (&builder, "{'title', <`s`>}", "foo");
    ///   g_variant_builder_add_parsed (&builder, "{'transparency', <0.5>}");
    ///   return g_variant_builder_end (&builder);
    /// }
    /// ```
    extern fn g_variant_builder_add_parsed(p_builder: *VariantBuilder, p_format: [*:0]const u8, ...) void;
    pub const addParsed = g_variant_builder_add_parsed;

    /// Adds `value` to `builder`.
    ///
    /// It is an error to call this function in any way that would create an
    /// inconsistent value to be constructed.  Some examples of this are
    /// putting different types of items into an array, putting the wrong
    /// types or number of items in a tuple, putting more than one value into
    /// a variant, etc.
    ///
    /// If `value` is a floating reference (see `glib.Variant.refSink`),
    /// the `builder` instance takes ownership of `value`.
    extern fn g_variant_builder_add_value(p_builder: *VariantBuilder, p_value: *glib.Variant) void;
    pub const addValue = g_variant_builder_add_value;

    /// Releases all memory associated with a `glib.VariantBuilder` without
    /// freeing the `glib.VariantBuilder` structure itself.
    ///
    /// It typically only makes sense to do this on a stack-allocated
    /// `glib.VariantBuilder` if you want to abort building the value part-way
    /// through.  This function need not be called if you call
    /// `glib.VariantBuilder.end` and it also doesn't need to be called on
    /// builders allocated with `glib.VariantBuilder.new` (see
    /// `glib.VariantBuilder.unref` for that).
    ///
    /// This function leaves the `glib.VariantBuilder` structure set to all-zeros.
    /// It is valid to call this function on either an initialised
    /// `glib.VariantBuilder` or one that is set to all-zeros but it is not valid
    /// to call this function on uninitialised memory.
    extern fn g_variant_builder_clear(p_builder: *VariantBuilder) void;
    pub const clear = g_variant_builder_clear;

    /// Closes the subcontainer inside the given `builder` that was opened by
    /// the most recent call to `glib.VariantBuilder.open`.
    ///
    /// It is an error to call this function in any way that would create an
    /// inconsistent value to be constructed (ie: too few values added to the
    /// subcontainer).
    extern fn g_variant_builder_close(p_builder: *VariantBuilder) void;
    pub const close = g_variant_builder_close;

    /// Ends the builder process and returns the constructed value.
    ///
    /// It is not permissible to use `builder` in any way after this call
    /// except for reference counting operations (in the case of a
    /// heap-allocated `glib.VariantBuilder`) or by reinitialising it with
    /// `glib.VariantBuilder.init` (in the case of stack-allocated). This
    /// means that for the stack-allocated builders there is no need to
    /// call `glib.VariantBuilder.clear` after the call to
    /// `glib.VariantBuilder.end`.
    ///
    /// It is an error to call this function in any way that would create an
    /// inconsistent value to be constructed (ie: insufficient number of
    /// items added to a container with a specific number of children
    /// required).  It is also an error to call this function if the builder
    /// was created with an indefinite array or maybe type and no children
    /// have been added; in this case it is impossible to infer the type of
    /// the empty array.
    extern fn g_variant_builder_end(p_builder: *VariantBuilder) *glib.Variant;
    pub const end = g_variant_builder_end;

    /// Initialises a `glib.VariantBuilder` structure.
    ///
    /// `type` must be non-`NULL`.  It specifies the type of container to
    /// construct.  It can be an indefinite type such as
    /// `G_VARIANT_TYPE_ARRAY` or a definite type such as "as" or "(ii)".
    /// Maybe, array, tuple, dictionary entry and variant-typed values may be
    /// constructed.
    ///
    /// If using a static type such as one of the `G_VARIANT_TYPE_*` constants
    /// or a `G_VARIANT_TYPE ("(ii)")` macro, it is more performant to use
    /// `glib.VariantBuilder.initStatic` rather than `glib.VariantBuilder.init`.
    ///
    /// After the builder is initialised, values are added using
    /// `glib.VariantBuilder.addValue` or `glib.VariantBuilder.add`.
    ///
    /// After all the child values are added, `glib.VariantBuilder.end` frees
    /// the memory associated with the builder and returns the `glib.Variant` that
    /// was created.
    ///
    /// This function completely ignores the previous contents of `builder`.
    /// On one hand this means that it is valid to pass in completely
    /// uninitialised memory.  On the other hand, this means that if you are
    /// initialising over top of an existing `glib.VariantBuilder` you need to
    /// first call `glib.VariantBuilder.clear` in order to avoid leaking
    /// memory.
    ///
    /// You must not call `glib.VariantBuilder.ref` or
    /// `glib.VariantBuilder.unref` on a `glib.VariantBuilder` that was initialised
    /// with this function.  If you ever pass a reference to a
    /// `glib.VariantBuilder` outside of the control of your own code then you
    /// should assume that the person receiving that reference may try to use
    /// reference counting; you should use `glib.VariantBuilder.new` instead of
    /// this function.
    extern fn g_variant_builder_init(p_builder: *VariantBuilder, p_type: *const glib.VariantType) void;
    pub const init = g_variant_builder_init;

    /// Initialises a `glib.VariantBuilder` structure.
    ///
    /// This function works exactly like `glib.VariantBuilder.init` but does
    /// not make a copy of `type`. Therefore, `type` must remain valid for the
    /// lifetime of `builder`. This is always true of type constants like
    /// `G_VARIANT_TYPE_*` or `G_VARIANT_TYPE ("(ii)")`.
    extern fn g_variant_builder_init_static(p_builder: *VariantBuilder, p_type: *const glib.VariantType) void;
    pub const initStatic = g_variant_builder_init_static;

    /// Opens a subcontainer inside the given `builder`.  When done adding
    /// items to the subcontainer, `glib.VariantBuilder.close` must be called. `type`
    /// is the type of the container: so to build a tuple of several values, `type`
    /// must include the tuple itself.
    ///
    /// It is an error to call this function in any way that would cause an
    /// inconsistent value to be constructed (ie: adding too many values or
    /// a value of an incorrect type).
    ///
    /// Example of building a nested variant:
    /// ```
    /// GVariantBuilder builder;
    /// guint32 some_number = get_number ();
    /// g_autoptr (GHashTable) some_dict = get_dict ();
    /// GHashTableIter iter;
    /// const gchar *key;
    /// const GVariant *value;
    /// g_autoptr (GVariant) output = NULL;
    ///
    /// g_variant_builder_init (&builder, G_VARIANT_TYPE ("(ua{sv})"));
    /// g_variant_builder_add (&builder, "u", some_number);
    /// g_variant_builder_open (&builder, G_VARIANT_TYPE ("a{sv}"));
    ///
    /// g_hash_table_iter_init (&iter, some_dict);
    /// while (g_hash_table_iter_next (&iter, (gpointer *) &key, (gpointer *) &value))
    ///   {
    ///     g_variant_builder_open (&builder, G_VARIANT_TYPE ("{sv}"));
    ///     g_variant_builder_add (&builder, "s", key);
    ///     g_variant_builder_add (&builder, "v", value);
    ///     g_variant_builder_close (&builder);
    ///   }
    ///
    /// g_variant_builder_close (&builder);
    ///
    /// output = g_variant_builder_end (&builder);
    /// ```
    extern fn g_variant_builder_open(p_builder: *VariantBuilder, p_type: *const glib.VariantType) void;
    pub const open = g_variant_builder_open;

    /// Increases the reference count on `builder`.
    ///
    /// Don't call this on stack-allocated `glib.VariantBuilder` instances or bad
    /// things will happen.
    extern fn g_variant_builder_ref(p_builder: *VariantBuilder) *glib.VariantBuilder;
    pub const ref = g_variant_builder_ref;

    /// Decreases the reference count on `builder`.
    ///
    /// In the event that there are no more references, releases all memory
    /// associated with the `glib.VariantBuilder`.
    ///
    /// Don't call this on stack-allocated `glib.VariantBuilder` instances or bad
    /// things will happen.
    extern fn g_variant_builder_unref(p_builder: *VariantBuilder) void;
    pub const unref = g_variant_builder_unref;

    extern fn g_variant_builder_get_type() usize;
    pub const getGObjectType = g_variant_builder_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `glib.VariantDict` is a mutable interface to `glib.Variant` dictionaries.
///
/// It can be used for doing a sequence of dictionary lookups in an
/// efficient way on an existing `glib.Variant` dictionary or it can be used
/// to construct new dictionaries with a hashtable-like interface.  It
/// can also be used for taking existing dictionaries and modifying them
/// in order to create new ones.
///
/// `glib.VariantDict` can only be used with `G_VARIANT_TYPE_VARDICT`
/// dictionaries.
///
/// It is possible to use `glib.VariantDict` allocated on the stack or on the
/// heap.  When using a stack-allocated `glib.VariantDict`, you begin with a
/// call to `glib.VariantDict.init` and free the resources with a call to
/// `glib.VariantDict.clear`.
///
/// Heap-allocated `glib.VariantDict` follows normal refcounting rules: you
/// allocate it with `glib.VariantDict.new` and use `glib.VariantDict.ref`
/// and `glib.VariantDict.unref`.
///
/// `glib.VariantDict.end` is used to convert the `glib.VariantDict` back into a
/// dictionary-type `glib.Variant`.  When used with stack-allocated instances,
/// this also implicitly frees all associated memory, but for
/// heap-allocated instances, you must still call `glib.VariantDict.unref`
/// afterwards.
///
/// You will typically want to use a heap-allocated `glib.VariantDict` when
/// you expose it as part of an API.  For most other uses, the
/// stack-allocated form will be more convenient.
///
/// Consider the following two examples that do the same thing in each
/// style: take an existing dictionary and look up the "count" uint32
/// key, adding 1 to it if it is found, or returning an error if the
/// key is not found.  Each returns the new dictionary as a floating
/// `glib.Variant`.
///
/// ## Using a stack-allocated GVariantDict
///
/// ```
///   GVariant *
///   add_to_count (GVariant  *orig,
///                 GError   **error)
///   {
///     GVariantDict dict;
///     guint32 count;
///
///     g_variant_dict_init (&dict, orig);
///     if (!g_variant_dict_lookup (&dict, "count", "u", &count))
///       {
///         g_set_error (...);
///         g_variant_dict_clear (&dict);
///         return NULL;
///       }
///
///     g_variant_dict_insert (&dict, "count", "u", count + 1);
///
///     return g_variant_dict_end (&dict);
///   }
/// ```
///
/// ## Using heap-allocated GVariantDict
///
/// ```
///   GVariant *
///   add_to_count (GVariant  *orig,
///                 GError   **error)
///   {
///     GVariantDict *dict;
///     GVariant *result;
///     guint32 count;
///
///     dict = g_variant_dict_new (orig);
///
///     if (g_variant_dict_lookup (dict, "count", "u", &count))
///       {
///         g_variant_dict_insert (dict, "count", "u", count + 1);
///         result = g_variant_dict_end (dict);
///       }
///     else
///       {
///         g_set_error (...);
///         result = NULL;
///       }
///
///     g_variant_dict_unref (dict);
///
///     return result;
///   }
/// ```
pub const VariantDict = extern struct {
    anon0: extern union {
        anon0: extern struct {
            f_asv: ?*glib.Variant,
            f_partial_magic: usize,
            f_y: [14]usize,
        },
        f_x: [16]usize,
    },

    /// Allocates and initialises a new `glib.VariantDict`.
    ///
    /// You should call `glib.VariantDict.unref` on the return value when it
    /// is no longer needed.  The memory will not be automatically freed by
    /// any other call.
    ///
    /// In some cases it may be easier to place a `glib.VariantDict` directly on
    /// the stack of the calling function and initialise it with
    /// `glib.VariantDict.init`.  This is particularly useful when you are
    /// using `glib.VariantDict` to construct a `glib.Variant`.
    extern fn g_variant_dict_new(p_from_asv: ?*glib.Variant) *glib.VariantDict;
    pub const new = g_variant_dict_new;

    /// Releases all memory associated with a `glib.VariantDict` without freeing
    /// the `glib.VariantDict` structure itself.
    ///
    /// It typically only makes sense to do this on a stack-allocated
    /// `glib.VariantDict` if you want to abort building the value part-way
    /// through.  This function need not be called if you call
    /// `glib.VariantDict.end` and it also doesn't need to be called on dicts
    /// allocated with g_variant_dict_new (see `glib.VariantDict.unref` for
    /// that).
    ///
    /// It is valid to call this function on either an initialised
    /// `glib.VariantDict` or one that was previously cleared by an earlier call
    /// to `glib.VariantDict.clear` but it is not valid to call this function
    /// on uninitialised memory.
    extern fn g_variant_dict_clear(p_dict: *VariantDict) void;
    pub const clear = g_variant_dict_clear;

    /// Checks if `key` exists in `dict`.
    extern fn g_variant_dict_contains(p_dict: *VariantDict, p_key: [*:0]const u8) c_int;
    pub const contains = g_variant_dict_contains;

    /// Returns the current value of `dict` as a `glib.Variant` of type
    /// `G_VARIANT_TYPE_VARDICT`, clearing it in the process.
    ///
    /// It is not permissible to use `dict` in any way after this call except
    /// for reference counting operations (in the case of a heap-allocated
    /// `glib.VariantDict`) or by reinitialising it with `glib.VariantDict.init` (in
    /// the case of stack-allocated).
    extern fn g_variant_dict_end(p_dict: *VariantDict) *glib.Variant;
    pub const end = g_variant_dict_end;

    /// Initialises a `glib.VariantDict` structure.
    ///
    /// If `from_asv` is given, it is used to initialise the dictionary.
    ///
    /// This function completely ignores the previous contents of `dict`.  On
    /// one hand this means that it is valid to pass in completely
    /// uninitialised memory.  On the other hand, this means that if you are
    /// initialising over top of an existing `glib.VariantDict` you need to first
    /// call `glib.VariantDict.clear` in order to avoid leaking memory.
    ///
    /// You must not call `glib.VariantDict.ref` or `glib.VariantDict.unref` on a
    /// `glib.VariantDict` that was initialised with this function.  If you ever
    /// pass a reference to a `glib.VariantDict` outside of the control of your
    /// own code then you should assume that the person receiving that
    /// reference may try to use reference counting; you should use
    /// `glib.VariantDict.new` instead of this function.
    extern fn g_variant_dict_init(p_dict: *VariantDict, p_from_asv: ?*glib.Variant) void;
    pub const init = g_variant_dict_init;

    /// Inserts a value into a `glib.VariantDict`.
    ///
    /// This call is a convenience wrapper that is exactly equivalent to
    /// calling `glib.Variant.new` followed by `glib.VariantDict.insertValue`.
    extern fn g_variant_dict_insert(p_dict: *VariantDict, p_key: [*:0]const u8, p_format_string: [*:0]const u8, ...) void;
    pub const insert = g_variant_dict_insert;

    /// Inserts (or replaces) a key in a `glib.VariantDict`.
    ///
    /// `value` is consumed if it is floating.
    extern fn g_variant_dict_insert_value(p_dict: *VariantDict, p_key: [*:0]const u8, p_value: *glib.Variant) void;
    pub const insertValue = g_variant_dict_insert_value;

    /// Looks up a value in a `glib.VariantDict`.
    ///
    /// This function is a wrapper around `glib.VariantDict.lookupValue` and
    /// `glib.Variant.get`.  In the case that `NULL` would have been returned,
    /// this function returns `FALSE` and does not modify the values of the arguments
    /// passed in to @....  Otherwise, it unpacks the returned
    /// value and returns `TRUE`.
    ///
    /// `format_string` determines the C types that are used for unpacking the
    /// values and also determines if the values are copied or borrowed, see the
    /// section on [`GVariant` format strings](gvariant-format-strings.html`pointers`).
    extern fn g_variant_dict_lookup(p_dict: *VariantDict, p_key: [*:0]const u8, p_format_string: [*:0]const u8, ...) c_int;
    pub const lookup = g_variant_dict_lookup;

    /// Looks up a value in a `glib.VariantDict`.
    ///
    /// If `key` is not found in `dictionary`, `NULL` is returned.
    ///
    /// The `expected_type` string specifies what type of value is expected.
    /// If the value associated with `key` has a different type then `NULL` is
    /// returned.
    ///
    /// If the key is found and the value has the correct type, it is
    /// returned.  If `expected_type` was specified then any non-`NULL` return
    /// value will have this type.
    extern fn g_variant_dict_lookup_value(p_dict: *VariantDict, p_key: [*:0]const u8, p_expected_type: ?*const glib.VariantType) ?*glib.Variant;
    pub const lookupValue = g_variant_dict_lookup_value;

    /// Increases the reference count on `dict`.
    ///
    /// Don't call this on stack-allocated `glib.VariantDict` instances or bad
    /// things will happen.
    extern fn g_variant_dict_ref(p_dict: *VariantDict) *glib.VariantDict;
    pub const ref = g_variant_dict_ref;

    /// Removes a key and its associated value from a `glib.VariantDict`.
    extern fn g_variant_dict_remove(p_dict: *VariantDict, p_key: [*:0]const u8) c_int;
    pub const remove = g_variant_dict_remove;

    /// Decreases the reference count on `dict`.
    ///
    /// In the event that there are no more references, releases all memory
    /// associated with the `glib.VariantDict`.
    ///
    /// Don't call this on stack-allocated `glib.VariantDict` instances or bad
    /// things will happen.
    extern fn g_variant_dict_unref(p_dict: *VariantDict) void;
    pub const unref = g_variant_dict_unref;

    extern fn g_variant_dict_get_type() usize;
    pub const getGObjectType = g_variant_dict_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `glib.VariantIter` is an opaque data structure and can only be accessed
/// using the following functions.
pub const VariantIter = extern struct {
    f_x: [16]usize,

    /// Creates a new heap-allocated `glib.VariantIter` to iterate over the
    /// container that was being iterated over by `iter`.  Iteration begins on
    /// the new iterator from the current position of the old iterator but
    /// the two copies are independent past that point.
    ///
    /// Use `glib.VariantIter.free` to free the return value when you no longer
    /// need it.
    ///
    /// A reference is taken to the container that `iter` is iterating over
    /// and will be related only when `glib.VariantIter.free` is called.
    extern fn g_variant_iter_copy(p_iter: *VariantIter) *glib.VariantIter;
    pub const copy = g_variant_iter_copy;

    /// Frees a heap-allocated `glib.VariantIter`.  Only call this function on
    /// iterators that were returned by `glib.Variant.iterNew` or
    /// `glib.VariantIter.copy`.
    extern fn g_variant_iter_free(p_iter: *VariantIter) void;
    pub const free = g_variant_iter_free;

    /// Initialises (without allocating) a `glib.VariantIter`.  `iter` may be
    /// completely uninitialised prior to this call; its old value is
    /// ignored.
    ///
    /// The iterator remains valid for as long as `value` exists, and need not
    /// be freed in any way.
    extern fn g_variant_iter_init(p_iter: *VariantIter, p_value: *glib.Variant) usize;
    pub const init = g_variant_iter_init;

    /// Gets the next item in the container and unpacks it into the variable
    /// argument list according to `format_string`, returning `TRUE`.
    ///
    /// If no more items remain then `FALSE` is returned.
    ///
    /// On the first call to this function, the pointers appearing on the
    /// variable argument list are assumed to point at uninitialised memory.
    /// On the second and later calls, it is assumed that the same pointers
    /// will be given and that they will point to the memory as set by the
    /// previous call to this function.  This allows the previous values to
    /// be freed, as appropriate.
    ///
    /// This function is intended to be used with a while loop as
    /// demonstrated in the following example.  This function can only be
    /// used when iterating over an array.  It is only valid to call this
    /// function with a string constant for the format string and the same
    /// string constant must be used each time.  Mixing calls to this
    /// function and `glib.VariantIter.next` or `glib.VariantIter.nextValue` on
    /// the same iterator causes undefined behavior.
    ///
    /// If you break out of a such a while loop using `glib.VariantIter.loop` then
    /// you must free or unreference all the unpacked values as you would with
    /// `glib.Variant.get`. Failure to do so will cause a memory leak.
    ///
    /// Here is an example for memory management with `glib.VariantIter.loop`:
    /// ```
    ///   // Iterates a dictionary of type 'a{sv}'
    ///   void
    ///   iterate_dictionary (GVariant *dictionary)
    ///   {
    ///     GVariantIter iter;
    ///     GVariant *value;
    ///     gchar *key;
    ///
    ///     g_variant_iter_init (&iter, dictionary);
    ///     while (g_variant_iter_loop (&iter, "{sv}", &key, &value))
    ///       {
    ///         g_print ("Item '`s`' has type '`s`'\n", key,
    ///                  g_variant_get_type_string (value));
    ///
    ///         // no need to free 'key' and 'value' here
    ///         // unless breaking out of this loop
    ///       }
    ///   }
    /// ```
    ///
    /// For most cases you should use `glib.VariantIter.next`.
    ///
    /// This function is really only useful when unpacking into `glib.Variant` or
    /// `glib.VariantIter` in order to allow you to skip the call to
    /// `glib.Variant.unref` or `glib.VariantIter.free`.
    ///
    /// For example, if you are only looping over simple integer and string
    /// types, `glib.VariantIter.next` is definitely preferred.  For string
    /// types, use the '&' prefix to avoid allocating any memory at all (and
    /// thereby avoiding the need to free anything as well).
    ///
    /// `format_string` determines the C types that are used for unpacking
    /// the values and also determines if the values are copied or borrowed.
    ///
    /// See the section on
    /// [`GVariant` format strings](gvariant-format-strings.html`pointers`).
    extern fn g_variant_iter_loop(p_iter: *VariantIter, p_format_string: [*:0]const u8, ...) c_int;
    pub const loop = g_variant_iter_loop;

    /// Queries the number of child items in the container that we are
    /// iterating over.  This is the total number of items -- not the number
    /// of items remaining.
    ///
    /// This function might be useful for preallocation of arrays.
    extern fn g_variant_iter_n_children(p_iter: *VariantIter) usize;
    pub const nChildren = g_variant_iter_n_children;

    /// Gets the next item in the container and unpacks it into the variable
    /// argument list according to `format_string`, returning `TRUE`.
    ///
    /// If no more items remain then `FALSE` is returned.
    ///
    /// All of the pointers given on the variable arguments list of this
    /// function are assumed to point at uninitialised memory.  It is the
    /// responsibility of the caller to free all of the values returned by
    /// the unpacking process.
    ///
    /// Here is an example for memory management with `glib.VariantIter.next`:
    /// ```
    ///   // Iterates a dictionary of type 'a{sv}'
    ///   void
    ///   iterate_dictionary (GVariant *dictionary)
    ///   {
    ///     GVariantIter iter;
    ///     GVariant *value;
    ///     gchar *key;
    ///
    ///     g_variant_iter_init (&iter, dictionary);
    ///     while (g_variant_iter_next (&iter, "{sv}", &key, &value))
    ///       {
    ///         g_print ("Item '`s`' has type '`s`'\n", key,
    ///                  g_variant_get_type_string (value));
    ///
    ///         // must free data for ourselves
    ///         g_variant_unref (value);
    ///         g_free (key);
    ///       }
    ///   }
    /// ```
    ///
    /// For a solution that is likely to be more convenient to C programmers
    /// when dealing with loops, see `glib.VariantIter.loop`.
    ///
    /// `format_string` determines the C types that are used for unpacking
    /// the values and also determines if the values are copied or borrowed.
    ///
    /// See the section on
    /// [`GVariant` format strings](gvariant-format-strings.html`pointers`).
    extern fn g_variant_iter_next(p_iter: *VariantIter, p_format_string: [*:0]const u8, ...) c_int;
    pub const next = g_variant_iter_next;

    /// Gets the next item in the container.  If no more items remain then
    /// `NULL` is returned.
    ///
    /// Use `glib.Variant.unref` to drop your reference on the return value when
    /// you no longer need it.
    ///
    /// Here is an example for iterating with `glib.VariantIter.nextValue`:
    /// ```
    ///   // recursively iterate a container
    ///   void
    ///   iterate_container_recursive (GVariant *container)
    ///   {
    ///     GVariantIter iter;
    ///     GVariant *child;
    ///
    ///     g_variant_iter_init (&iter, container);
    ///     while ((child = g_variant_iter_next_value (&iter)))
    ///       {
    ///         g_print ("type '`s`'\n", g_variant_get_type_string (child));
    ///
    ///         if (g_variant_is_container (child))
    ///           iterate_container_recursive (child);
    ///
    ///         g_variant_unref (child);
    ///       }
    ///   }
    /// ```
    extern fn g_variant_iter_next_value(p_iter: *VariantIter) ?*glib.Variant;
    pub const nextValue = g_variant_iter_next_value;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A type in the `glib.Variant` type system.
///
/// `glib.Variant` types are represented as strings, but have a strict
/// syntax described below. All `glib.VariantType`s passed to GLib must be
/// valid, and they are typically expected to be static (i.e. not provided by
/// user input) as they determine how binary `glib.Variant` data is
/// interpreted.
///
/// To convert a static string to a `glib.VariantType` in C, use the
/// `glib.VARIANTTYPE` casting macro. When GLib is compiled with checks
/// enabled, it will validate the type. To check if an arbitrary string is a
/// valid `glib.VariantType`, use `glib.VariantType.stringIsValid`.
///
/// ## GVariant Type System
///
/// This section introduces the `glib.Variant` type system. It is based, in
/// large part, on the D-Bus type system, with two major changes and
/// some minor lifting of restrictions. The
/// [D-Bus specification](http://dbus.freedesktop.org/doc/dbus-specification.html),
/// therefore, provides a significant amount of
/// information that is useful when working with `glib.Variant`.
///
/// The first major change with respect to the D-Bus type system is the
/// introduction of maybe (or ‘nullable’) types.  Any type in `glib.Variant`
/// can be converted to a maybe type, in which case, `nothing` (or `null`)
/// becomes a valid value.  Maybe types have been added by introducing the
/// character `m` to type strings.
///
/// The second major change is that the `glib.Variant` type system supports
/// the concept of ‘indefinite types’ — types that are less specific than
/// the normal types found in D-Bus.  For example, it is possible to speak
/// of ‘an array of any type’ in `glib.Variant`, where the D-Bus type system
/// would require you to speak of ‘an array of integers’ or ‘an array of
/// strings’.  Indefinite types have been added by introducing the
/// characters `*`, `?` and `r` to type strings.
///
/// Finally, all arbitrary restrictions relating to the complexity of
/// types are lifted along with the restriction that dictionary entries
/// may only appear nested inside of arrays.
///
/// Just as in D-Bus, `glib.Variant` types are described with strings (‘type
/// strings’).  Subject to the differences mentioned above, these strings
/// are of the same form as those found in D-Bus.  Note, however: D-Bus
/// always works in terms of messages and therefore individual type
/// strings appear nowhere in its interface.  Instead, ‘signatures’
/// are a concatenation of the strings of the type of each argument in a
/// message.  `glib.Variant` deals with single values directly so
/// `glib.Variant` type strings always describe the type of exactly one
/// value.  This means that a D-Bus signature string is generally not a valid
/// `glib.Variant` type string — except in the case that it is the signature
/// of a message containing exactly one argument.
///
/// An indefinite type is similar in spirit to what may be called an
/// abstract type in other type systems.  No value can exist that has an
/// indefinite type as its type, but values can exist that have types
/// that are subtypes of indefinite types.  That is to say,
/// `glib.Variant.getType` will never return an indefinite type, but
/// calling `glib.Variant.isOfType` with an indefinite type may return
/// true.  For example, you cannot have a value that represents ‘an
/// array of no particular type’, but you can have an ‘array of integers’
/// which certainly matches the type of ‘an array of no particular type’,
/// since ‘array of integers’ is a subtype of ‘array of no particular
/// type’.
///
/// This is similar to how instances of abstract classes may not
/// directly exist in other type systems, but instances of their
/// non-abstract subtypes may.  For example, in GTK, no object that has
/// the type of [`GtkWidget`](https://docs.gtk.org/gtk4/class.Widget.html) can
/// exist (since `GtkWidget` is an abstract class), but a [`GtkWindow`](https://docs.gtk.org/gtk4/class.Window.html)
/// can certainly be instantiated, and you would say that a `GtkWindow` is a
/// `GtkWidget` (since `GtkWindow` is a subclass of `GtkWidget`).
///
/// Two types may not be compared by value; use `glib.VariantType.equal`
/// or `glib.VariantType.isSubtypeOf`  May be copied using
/// `glib.VariantType.copy` and freed using `glib.VariantType.free`.
///
/// ## GVariant Type Strings
///
/// A `glib.Variant` type string can be any of the following:
///
/// - any basic type string (listed below)
/// - `v`, `r` or `*`
/// - one of the characters `a` or `m`, followed by another type string
/// - the character `(`, followed by a concatenation of zero or more other
///   type strings, followed by the character `)`
/// - the character `{`, followed by a basic type string (see below),
///   followed by another type string, followed by the character `}`
///
/// A basic type string describes a basic type (as per
/// `glib.VariantType.isBasic`) and is always a single character in
/// length. The valid basic type strings are `b`, `y`, `n`, `q`, `i`, `u`, `x`,
/// `t`, `h`, `d`, `s`, `o`, `g` and `?`.
///
/// The above definition is recursive to arbitrary depth. `aaaaai` and
/// `(ui(nq((y)))s)` are both valid type strings, as is
/// `a(aa(ui)(qna{ya(yd)}))`. In order to not hit memory limits,
/// `glib.Variant` imposes a limit on recursion depth of 65 nested
/// containers. This is the limit in the D-Bus specification (64) plus one to
/// allow a [`GDBusMessage`](../gio/class.DBusMessage.html) to be nested in
/// a top-level tuple.
///
/// The meaning of each of the characters is as follows:
///
/// - `b`: the type string of `G_VARIANT_TYPE_BOOLEAN`; a boolean value.
/// - `y`: the type string of `G_VARIANT_TYPE_BYTE`; a byte.
/// - `n`: the type string of `G_VARIANT_TYPE_INT16`; a signed 16 bit integer.
/// - `q`: the type string of `G_VARIANT_TYPE_UINT16`; an unsigned 16 bit integer.
/// - `i`: the type string of `G_VARIANT_TYPE_INT32`; a signed 32 bit integer.
/// - `u`: the type string of `G_VARIANT_TYPE_UINT32`; an unsigned 32 bit integer.
/// - `x`: the type string of `G_VARIANT_TYPE_INT64`; a signed 64 bit integer.
/// - `t`: the type string of `G_VARIANT_TYPE_UINT64`; an unsigned 64 bit integer.
/// - `h`: the type string of `G_VARIANT_TYPE_HANDLE`; a signed 32 bit value
///   that, by convention, is used as an index into an array of file
///   descriptors that are sent alongside a D-Bus message.
/// - `d`: the type string of `G_VARIANT_TYPE_DOUBLE`; a double precision
///   floating point value.
/// - `s`: the type string of `G_VARIANT_TYPE_STRING`; a string.
/// - `o`: the type string of `G_VARIANT_TYPE_OBJECT_PATH`; a string in the form
///   of a D-Bus object path.
/// - `g`: the type string of `G_VARIANT_TYPE_SIGNATURE`; a string in the form of
///   a D-Bus type signature.
/// - `?`: the type string of `G_VARIANT_TYPE_BASIC`; an indefinite type that
///   is a supertype of any of the basic types.
/// - `v`: the type string of `G_VARIANT_TYPE_VARIANT`; a container type that
///   contain any other type of value.
/// - `a`: used as a prefix on another type string to mean an array of that
///   type; the type string `ai`, for example, is the type of an array of
///   signed 32-bit integers.
/// - `m`: used as a prefix on another type string to mean a ‘maybe’, or
///   ‘nullable’, version of that type; the type string `ms`, for example,
///   is the type of a value that maybe contains a string, or maybe contains
///   nothing.
/// - `()`: used to enclose zero or more other concatenated type strings to
///   create a tuple type; the type string `(is)`, for example, is the type of
///   a pair of an integer and a string.
/// - `r`: the type string of `G_VARIANT_TYPE_TUPLE`; an indefinite type that is
///   a supertype of any tuple type, regardless of the number of items.
/// - `{}`: used to enclose a basic type string concatenated with another type
///   string to create a dictionary entry type, which usually appears inside of
///   an array to form a dictionary; the type string `a{sd}`, for example, is
///   the type of a dictionary that maps strings to double precision floating
///   point values.
///
///   The first type (the basic type) is the key type and the second type is
///   the value type. The reason that the first type is restricted to being a
///   basic type is so that it can easily be hashed.
/// - `*`: the type string of `G_VARIANT_TYPE_ANY`; the indefinite type that is
///   a supertype of all types.  Note that, as with all type strings, this
///   character represents exactly one type. It cannot be used inside of tuples
///   to mean ‘any number of items’.
///
/// Any type string of a container that contains an indefinite type is,
/// itself, an indefinite type. For example, the type string `a*`
/// (corresponding to `G_VARIANT_TYPE_ARRAY`) is an indefinite type
/// that is a supertype of every array type. `(*s)` is a supertype
/// of all tuples that contain exactly two items where the second
/// item is a string.
///
/// `a{?*}` is an indefinite type that is a supertype of all arrays
/// containing dictionary entries where the key is any basic type and
/// the value is any type at all.  This is, by definition, a dictionary,
/// so this type string corresponds to `G_VARIANT_TYPE_DICTIONARY`. Note
/// that, due to the restriction that the key of a dictionary entry must
/// be a basic type, `{**}` is not a valid type string.
pub const VariantType = opaque {
    extern fn g_variant_type_checked_(p_type_string: [*:0]const u8) *const glib.VariantType;
    pub const checked = g_variant_type_checked_;

    extern fn g_variant_type_string_get_depth_(p_type_string: [*:0]const u8) usize;
    pub const stringGetDepth = g_variant_type_string_get_depth_;

    /// Checks if `type_string` is a valid
    /// [GVariant type string](./struct.VariantType.html`gvariant`-type-strings).
    ///
    /// This call is equivalent to calling `glib.VariantType.stringScan` and
    /// confirming that the following character is a nul terminator.
    extern fn g_variant_type_string_is_valid(p_type_string: [*:0]const u8) c_int;
    pub const stringIsValid = g_variant_type_string_is_valid;

    /// Scan for a single complete and valid GVariant type string in `string`.
    ///
    /// The memory pointed to by `limit` (or bytes beyond it) is never
    /// accessed.
    ///
    /// If a valid type string is found, `endptr` is updated to point to the
    /// first character past the end of the string that was found and `TRUE`
    /// is returned.
    ///
    /// If there is no valid type string starting at `string`, or if the type
    /// string does not end before `limit` then `FALSE` is returned.
    ///
    /// For the simple case of checking if a string is a valid type string,
    /// see `glib.VariantType.stringIsValid`.
    extern fn g_variant_type_string_scan(p_string: [*:0]const u8, p_limit: ?[*:0]const u8, p_endptr: ?*[*:0]const u8) c_int;
    pub const stringScan = g_variant_type_string_scan;

    /// Creates a new `glib.VariantType` corresponding to the type string given
    /// by `type_string`.
    ///
    /// It is appropriate to call `glib.VariantType.free` on the return value.
    ///
    /// It is a programmer error to call this function with an invalid type
    /// string.  Use `glib.VariantType.stringIsValid` if you are unsure.
    extern fn g_variant_type_new(p_type_string: [*:0]const u8) *glib.VariantType;
    pub const new = g_variant_type_new;

    /// Constructs the type corresponding to an array of elements of the
    /// type `type`.
    ///
    /// It is appropriate to call `glib.VariantType.first` on the return value.
    extern fn g_variant_type_new_array(p_element: *const glib.VariantType) *glib.VariantType;
    pub const newArray = g_variant_type_new_array;

    /// Constructs the type corresponding to a dictionary entry with a key
    /// of type `key` and a value of type `value`.
    ///
    /// It is appropriate to call `glib.VariantType.free` on the return value.
    extern fn g_variant_type_new_dict_entry(p_key: *const glib.VariantType, p_value: *const glib.VariantType) *glib.VariantType;
    pub const newDictEntry = g_variant_type_new_dict_entry;

    /// Constructs the type corresponding to a ‘maybe’ instance containing
    /// type `type` or `Nothing`.
    ///
    /// It is appropriate to call `glib.VariantType.free` on the return value.
    extern fn g_variant_type_new_maybe(p_element: *const glib.VariantType) *glib.VariantType;
    pub const newMaybe = g_variant_type_new_maybe;

    /// Constructs a new tuple type, from `items`.
    ///
    /// `length` is the number of items in `items`, or `-1` to indicate that
    /// `items` is `NULL`-terminated.
    ///
    /// It is appropriate to call `glib.VariantType.free` on the return value.
    extern fn g_variant_type_new_tuple(p_items: [*]const *const glib.VariantType, p_length: c_int) *glib.VariantType;
    pub const newTuple = g_variant_type_new_tuple;

    /// Makes a copy of a `glib.VariantType`.
    ///
    /// It is appropriate to call `glib.VariantType.free` on the return value.
    /// `type` may not be `NULL`.
    extern fn g_variant_type_copy(p_type: *const VariantType) *glib.VariantType;
    pub const copy = g_variant_type_copy;

    /// Returns a newly-allocated copy of the type string corresponding to `type`.
    ///
    /// The returned string is nul-terminated.  It is appropriate to call
    /// `glib.free` on the return value.
    extern fn g_variant_type_dup_string(p_type: *const VariantType) [*:0]u8;
    pub const dupString = g_variant_type_dup_string;

    /// Determines the element type of an array or ‘maybe’ type.
    ///
    /// This function may only be used with array or ‘maybe’ types.
    extern fn g_variant_type_element(p_type: *const VariantType) *const glib.VariantType;
    pub const element = g_variant_type_element;

    /// Compares `type1` and `type2` for equality.
    ///
    /// Only returns true if the types are exactly equal.  Even if one type
    /// is an indefinite type and the other is a subtype of it, false will
    /// be returned if they are not exactly equal.  If you want to check for
    /// subtypes, use `glib.VariantType.isSubtypeOf`.
    ///
    /// The argument types of `type1` and `type2` are only `gconstpointer` to
    /// allow use with `glib.HashTable` without function pointer casting.  For
    /// both arguments, a valid `glib.VariantType` must be provided.
    extern fn g_variant_type_equal(p_type1: *const VariantType, p_type2: *const glib.VariantType) c_int;
    pub const equal = g_variant_type_equal;

    /// Determines the first item type of a tuple or dictionary entry
    /// type.
    ///
    /// This function may only be used with tuple or dictionary entry types,
    /// but must not be used with the generic tuple type
    /// `G_VARIANT_TYPE_TUPLE`.
    ///
    /// In the case of a dictionary entry type, this returns the type of
    /// the key.
    ///
    /// `NULL` is returned in case of `type` being `G_VARIANT_TYPE_UNIT`.
    ///
    /// This call, together with `glib.VariantType.next` provides an iterator
    /// interface over tuple and dictionary entry types.
    extern fn g_variant_type_first(p_type: *const VariantType) ?*const glib.VariantType;
    pub const first = g_variant_type_first;

    /// Frees a `glib.VariantType` that was allocated with
    /// `glib.VariantType.copy`, `glib.VariantType.new` or one of the
    /// container type constructor functions.
    ///
    /// In the case that `type` is `NULL`, this function does nothing.
    ///
    /// Since 2.24
    extern fn g_variant_type_free(p_type: ?*VariantType) void;
    pub const free = g_variant_type_free;

    /// Returns the length of the type string corresponding to the given `type`.
    ///
    /// This function must be used to determine the valid extent of
    /// the memory region returned by `glib.VariantType.peekString`.
    extern fn g_variant_type_get_string_length(p_type: *const VariantType) usize;
    pub const getStringLength = g_variant_type_get_string_length;

    /// Hashes `type`.
    ///
    /// The argument type of `type` is only `gconstpointer` to allow use with
    /// `glib.HashTable` without function pointer casting.  A valid
    /// `glib.VariantType` must be provided.
    extern fn g_variant_type_hash(p_type: *const VariantType) c_uint;
    pub const hash = g_variant_type_hash;

    /// Determines if the given `type` is an array type.
    ///
    /// This is true if the type string for `type` starts with an `a`.
    ///
    /// This function returns true for any indefinite type for which every
    /// definite subtype is an array type — `G_VARIANT_TYPE_ARRAY`, for
    /// example.
    extern fn g_variant_type_is_array(p_type: *const VariantType) c_int;
    pub const isArray = g_variant_type_is_array;

    /// Determines if the given `type` is a basic type.
    ///
    /// Basic types are booleans, bytes, integers, doubles, strings, object
    /// paths and signatures.
    ///
    /// Only a basic type may be used as the key of a dictionary entry.
    ///
    /// This function returns `FALSE` for all indefinite types except
    /// `G_VARIANT_TYPE_BASIC`.
    extern fn g_variant_type_is_basic(p_type: *const VariantType) c_int;
    pub const isBasic = g_variant_type_is_basic;

    /// Determines if the given `type` is a container type.
    ///
    /// Container types are any array, maybe, tuple, or dictionary
    /// entry types plus the variant type.
    ///
    /// This function returns true for any indefinite type for which every
    /// definite subtype is a container — `G_VARIANT_TYPE_ARRAY`, for
    /// example.
    extern fn g_variant_type_is_container(p_type: *const VariantType) c_int;
    pub const isContainer = g_variant_type_is_container;

    /// Determines if the given `type` is definite (ie: not indefinite).
    ///
    /// A type is definite if its type string does not contain any indefinite
    /// type characters (`*`, `?`, or `r`).
    ///
    /// A `glib.Variant` instance may not have an indefinite type, so calling
    /// this function on the result of `glib.Variant.getType` will always
    /// result in true being returned.  Calling this function on an
    /// indefinite type like `G_VARIANT_TYPE_ARRAY`, however, will result in
    /// `FALSE` being returned.
    extern fn g_variant_type_is_definite(p_type: *const VariantType) c_int;
    pub const isDefinite = g_variant_type_is_definite;

    /// Determines if the given `type` is a dictionary entry type.
    ///
    /// This is true if the type string for `type` starts with a `{`.
    ///
    /// This function returns true for any indefinite type for which every
    /// definite subtype is a dictionary entry type —
    /// `G_VARIANT_TYPE_DICT_ENTRY`, for example.
    extern fn g_variant_type_is_dict_entry(p_type: *const VariantType) c_int;
    pub const isDictEntry = g_variant_type_is_dict_entry;

    /// Determines if the given `type` is a ‘maybe’ type.
    ///
    /// This is true if the type string for `type` starts with an `m`.
    ///
    /// This function returns true for any indefinite type for which every
    /// definite subtype is a ‘maybe’ type — `G_VARIANT_TYPE_MAYBE`, for
    /// example.
    extern fn g_variant_type_is_maybe(p_type: *const VariantType) c_int;
    pub const isMaybe = g_variant_type_is_maybe;

    /// Checks if `type` is a subtype of `supertype`.
    ///
    /// This function returns true if `type` is a subtype of `supertype`.  All
    /// types are considered to be subtypes of themselves.  Aside from that,
    /// only indefinite types can have subtypes.
    extern fn g_variant_type_is_subtype_of(p_type: *const VariantType, p_supertype: *const glib.VariantType) c_int;
    pub const isSubtypeOf = g_variant_type_is_subtype_of;

    /// Determines if the given `type` is a tuple type.
    ///
    /// This is true if the type string for `type` starts with a `(` or if `type` is
    /// `G_VARIANT_TYPE_TUPLE`.
    ///
    /// This function returns true for any indefinite type for which every
    /// definite subtype is a tuple type — `G_VARIANT_TYPE_TUPLE`, for
    /// example.
    extern fn g_variant_type_is_tuple(p_type: *const VariantType) c_int;
    pub const isTuple = g_variant_type_is_tuple;

    /// Determines if the given `type` is the variant type.
    extern fn g_variant_type_is_variant(p_type: *const VariantType) c_int;
    pub const isVariant = g_variant_type_is_variant;

    /// Determines the key type of a dictionary entry type.
    ///
    /// This function may only be used with a dictionary entry type.  Other
    /// than the additional restriction, this call is equivalent to
    /// `glib.VariantType.first`.
    extern fn g_variant_type_key(p_type: *const VariantType) *const glib.VariantType;
    pub const key = g_variant_type_key;

    /// Determines the number of items contained in a tuple or
    /// dictionary entry type.
    ///
    /// This function may only be used with tuple or dictionary entry types,
    /// but must not be used with the generic tuple type
    /// `G_VARIANT_TYPE_TUPLE`.
    ///
    /// In the case of a dictionary entry type, this function will always
    /// return `2`.
    extern fn g_variant_type_n_items(p_type: *const VariantType) usize;
    pub const nItems = g_variant_type_n_items;

    /// Determines the next item type of a tuple or dictionary entry
    /// type.
    ///
    /// `type` must be the result of a previous call to
    /// `glib.VariantType.first` or `glib.VariantType.next`.
    ///
    /// If called on the key type of a dictionary entry then this call
    /// returns the value type.  If called on the value type of a dictionary
    /// entry then this call returns `NULL`.
    ///
    /// For tuples, `NULL` is returned when `type` is the last item in the tuple.
    extern fn g_variant_type_next(p_type: *const VariantType) ?*const glib.VariantType;
    pub const next = g_variant_type_next;

    /// Returns the type string corresponding to the given `type`.
    ///
    /// The result is not nul-terminated; in order to determine its length you
    /// must call `glib.VariantType.getStringLength`.
    ///
    /// To get a nul-terminated string, see `glib.VariantType.dupString`.
    extern fn g_variant_type_peek_string(p_type: *const VariantType) [*:0]const u8;
    pub const peekString = g_variant_type_peek_string;

    /// Determines the value type of a dictionary entry type.
    ///
    /// This function may only be used with a dictionary entry type.
    extern fn g_variant_type_value(p_type: *const VariantType) *const glib.VariantType;
    pub const value = g_variant_type_value;

    extern fn g_variant_type_get_gtype() usize;
    pub const getGObjectType = g_variant_type_get_gtype;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.FloatIEEE754` and `glib.DoubleIEEE754` unions are used to access the sign,
/// mantissa and exponent of IEEE floats and doubles. These unions are defined
/// as appropriate for a given platform. IEEE floats and doubles are supported
/// (used for storage) by at least Intel, PPC and Sparc.
pub const DoubleIEEE754 = extern union {
    /// the double value
    f_v_double: f64,
    anon0: extern struct {
        bitfields0: packed struct(c_uint) {
            f_mantissa_low: u32,
        },
        bitfields1: packed struct(c_uint) {
            f_mantissa_high: u20,
            f_biased_exponent: u11,
            f_sign: u1,
        },
    },

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.FloatIEEE754` and `glib.DoubleIEEE754` unions are used to access the sign,
/// mantissa and exponent of IEEE floats and doubles. These unions are defined
/// as appropriate for a given platform. IEEE floats and doubles are supported
/// (used for storage) by at least Intel, PPC and Sparc.
pub const FloatIEEE754 = extern union {
    /// the double value
    f_v_float: f32,
    anon0: extern struct {
        bitfields0: packed struct(c_uint) {
            f_mantissa: u23,
            f_biased_exponent: u8,
            f_sign: u1,
        },
    },

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.Mutex` struct is an opaque data structure to represent a mutex
/// (mutual exclusion). It can be used to protect data against shared
/// access.
///
/// Take for example the following function:
/// ```
///   int
///   give_me_next_number (void)
///   {
///     static int current_number = 0;
///
///     // now do a very complicated calculation to calculate the new
///     // number, this might for example be a random number generator
///     current_number = calc_next_number (current_number);
///
///     return current_number;
///   }
/// ```
/// It is easy to see that this won't work in a multi-threaded
/// application. There current_number must be protected against shared
/// access. A `glib.Mutex` can be used as a solution to this problem:
/// ```
///   int
///   give_me_next_number (void)
///   {
///     static GMutex mutex;
///     static int current_number = 0;
///     int ret_val;
///
///     g_mutex_lock (&mutex);
///     ret_val = current_number = calc_next_number (current_number);
///     g_mutex_unlock (&mutex);
///
///     return ret_val;
///   }
/// ```
/// Notice that the `glib.Mutex` is not initialised to any particular value.
/// Its placement in static storage ensures that it will be initialised
/// to all-zeros, which is appropriate.
///
/// If a `glib.Mutex` is placed in other contexts (eg: embedded in a struct)
/// then it must be explicitly initialised using `glib.Mutex.init`.
///
/// A `glib.Mutex` should only be accessed via g_mutex_ functions.
pub const Mutex = extern union {
    f_p: ?*anyopaque,
    f_i: [2]c_uint,

    /// Allocates and initializes a new `glib.Mutex`.
    extern fn g_mutex_new() *glib.Mutex;
    pub const new = g_mutex_new;

    /// Frees the resources allocated to a mutex with `glib.Mutex.init`.
    ///
    /// This function should not be used with a `glib.Mutex` that has been
    /// statically allocated.
    ///
    /// Calling `glib.Mutex.clear` on a locked mutex leads to undefined
    /// behaviour.
    extern fn g_mutex_clear(p_mutex: *Mutex) void;
    pub const clear = g_mutex_clear;

    /// Destroys a `mutex` that has been created with `glib.mutexNew`.
    ///
    /// Calling `glib.Mutex.free` on a locked mutex may result
    /// in undefined behaviour.
    extern fn g_mutex_free(p_mutex: *Mutex) void;
    pub const free = g_mutex_free;

    /// Initializes a `glib.Mutex` so that it can be used.
    ///
    /// This function is useful to initialize a mutex that has been
    /// allocated on the stack, or as part of a larger structure.
    /// It is not necessary to initialize a mutex that has been
    /// statically allocated.
    ///
    /// ```
    ///   typedef struct {
    ///     GMutex m;
    ///     ...
    ///   } Blob;
    ///
    /// Blob *b;
    ///
    /// b = g_new (Blob, 1);
    /// g_mutex_init (&b->m);
    /// ```
    ///
    /// To undo the effect of `glib.Mutex.init` when a mutex is no longer
    /// needed, use `glib.Mutex.clear`.
    ///
    /// Calling `glib.Mutex.init` on an already initialized `glib.Mutex` leads
    /// to undefined behaviour.
    extern fn g_mutex_init(p_mutex: *Mutex) void;
    pub const init = g_mutex_init;

    /// Locks `mutex`. If `mutex` is already locked by another thread, the
    /// current thread will block until `mutex` is unlocked by the other
    /// thread.
    ///
    /// `glib.Mutex` is neither guaranteed to be recursive nor to be
    /// non-recursive.  As such, calling `glib.Mutex.lock` on a `glib.Mutex` that has
    /// already been locked by the same thread results in undefined behaviour
    /// (including but not limited to deadlocks).
    extern fn g_mutex_lock(p_mutex: *Mutex) void;
    pub const lock = g_mutex_lock;

    /// Tries to lock `mutex`. If `mutex` is already locked by another thread,
    /// it immediately returns `FALSE`. Otherwise it locks `mutex` and returns
    /// `TRUE`.
    ///
    /// `glib.Mutex` is neither guaranteed to be recursive nor to be
    /// non-recursive.  As such, calling `glib.Mutex.lock` on a `glib.Mutex` that has
    /// already been locked by the same thread results in undefined behaviour
    /// (including but not limited to deadlocks or arbitrary return values).
    extern fn g_mutex_trylock(p_mutex: *Mutex) c_int;
    pub const trylock = g_mutex_trylock;

    /// Unlocks `mutex`. If another thread is blocked in a `glib.Mutex.lock`
    /// call for `mutex`, it will become unblocked and can lock `mutex` itself.
    ///
    /// Calling `glib.Mutex.unlock` on a mutex that is not locked by the
    /// current thread leads to undefined behaviour.
    extern fn g_mutex_unlock(p_mutex: *Mutex) void;
    pub const unlock = g_mutex_unlock;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A union holding the value of the token.
pub const TokenValue = extern union {
    /// token symbol value
    f_v_symbol: ?*anyopaque,
    /// token identifier value
    f_v_identifier: ?[*:0]u8,
    /// token binary integer value
    f_v_binary: c_ulong,
    /// octal integer value
    f_v_octal: c_ulong,
    /// integer value
    f_v_int: c_ulong,
    /// 64-bit integer value
    f_v_int64: u64,
    /// floating point value
    f_v_float: f64,
    /// hex integer value
    f_v_hex: c_ulong,
    /// string value
    f_v_string: ?[*:0]u8,
    /// comment value
    f_v_comment: ?[*:0]u8,
    /// character value
    f_v_char: u8,
    /// error value
    f_v_error: c_uint,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by bookmark file parsing.
pub const BookmarkFileError = enum(c_int) {
    invalid_uri = 0,
    invalid_value = 1,
    app_not_registered = 2,
    uri_not_found = 3,
    read = 4,
    unknown_encoding = 5,
    write = 6,
    file_not_found = 7,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The hashing algorithm to be used by `glib.Checksum` when performing the
/// digest of some data.
///
/// Note that the `glib.ChecksumType` enumeration may be extended at a later
/// date to include new hashing algorithm types.
pub const ChecksumType = enum(c_int) {
    md5 = 0,
    sha1 = 1,
    sha256 = 2,
    sha512 = 3,
    sha384 = 4,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by character set conversion routines.
pub const ConvertError = enum(c_int) {
    no_conversion = 0,
    illegal_sequence = 1,
    failed = 2,
    partial_input = 3,
    bad_uri = 4,
    not_absolute_path = 5,
    no_memory = 6,
    embedded_nul = 7,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// This enumeration isn't used in the API, but may be useful if you need
/// to mark a number as a day, month, or year.
pub const DateDMY = enum(c_int) {
    day = 0,
    month = 1,
    year = 2,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Enumeration representing a month; values are `G_DATE_JANUARY`,
/// `G_DATE_FEBRUARY`, etc. `G_DATE_BAD_MONTH` is the invalid value.
pub const DateMonth = enum(c_int) {
    bad_month = 0,
    january = 1,
    february = 2,
    march = 3,
    april = 4,
    may = 5,
    june = 6,
    july = 7,
    august = 8,
    september = 9,
    october = 10,
    november = 11,
    december = 12,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Enumeration representing a day of the week; `G_DATE_MONDAY`,
/// `G_DATE_TUESDAY`, etc. `G_DATE_BAD_WEEKDAY` is an invalid weekday.
pub const DateWeekday = enum(c_int) {
    bad_weekday = 0,
    monday = 1,
    tuesday = 2,
    wednesday = 3,
    thursday = 4,
    friday = 5,
    saturday = 6,
    sunday = 7,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The possible errors, used in the `v_error` field
/// of `glib.TokenValue`, when the token is a `G_TOKEN_ERROR`.
pub const ErrorType = enum(c_int) {
    unknown = 0,
    unexp_eof = 1,
    unexp_eof_in_string = 2,
    unexp_eof_in_comment = 3,
    non_digit_in_const = 4,
    digit_radix = 5,
    float_radix = 6,
    float_malformed = 7,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Values corresponding to `errno` codes returned from file operations
/// on UNIX. Unlike `errno` codes, GFileError values are available on
/// all systems, even Windows. The exact meaning of each code depends
/// on what sort of file operation you were performing; the UNIX
/// documentation gives more details. The following error code descriptions
/// come from the GNU C Library manual, and are under the copyright
/// of that manual.
///
/// It's not very portable to make detailed assumptions about exactly
/// which errors will be returned from a given operation. Some errors
/// don't occur on some systems, etc., sometimes there are subtle
/// differences in when a system will report a given error, etc.
pub const FileError = enum(c_int) {
    exist = 0,
    isdir = 1,
    acces = 2,
    nametoolong = 3,
    noent = 4,
    notdir = 5,
    nxio = 6,
    nodev = 7,
    rofs = 8,
    txtbsy = 9,
    fault = 10,
    loop = 11,
    nospc = 12,
    nomem = 13,
    mfile = 14,
    nfile = 15,
    badf = 16,
    inval = 17,
    pipe = 18,
    again = 19,
    intr = 20,
    io = 21,
    perm = 22,
    nosys = 23,
    failed = 24,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by `glib.IOChannel` operations.
pub const IOChannelError = enum(c_int) {
    fbig = 0,
    inval = 1,
    io = 2,
    isdir = 3,
    nospc = 4,
    nxio = 5,
    overflow = 6,
    pipe = 7,
    failed = 8,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// `glib.IOError` is only used by the deprecated functions
/// `glib.IOChannel.read`, `glib.IOChannel.write`, and `glib.IOChannel.seek`.
pub const IOError = enum(c_int) {
    none = 0,
    again = 1,
    inval = 2,
    unknown = 3,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Statuses returned by most of the `glib.IOFuncs` functions.
pub const IOStatus = enum(c_int) {
    @"error" = 0,
    normal = 1,
    eof = 2,
    again = 3,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by key file parsing.
pub const KeyFileError = enum(c_int) {
    unknown_encoding = 0,
    parse = 1,
    not_found = 2,
    key_not_found = 3,
    group_not_found = 4,
    invalid_value = 5,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Return values from `GLogWriterFuncs` to indicate whether the given log entry
/// was successfully handled by the writer, or whether there was an error in
/// handling it (and hence a fallback writer should be used).
///
/// If a `glib.LogWriterFunc` ignores a log entry, it should return
/// `G_LOG_WRITER_HANDLED`.
pub const LogWriterOutput = enum(c_int) {
    handled = 1,
    unhandled = 0,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by markup parsing.
pub const MarkupError = enum(c_int) {
    bad_utf8 = 0,
    empty = 1,
    parse = 2,
    unknown_element = 3,
    unknown_attribute = 4,
    invalid_content = 5,
    missing_attribute = 6,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Defines how a Unicode string is transformed in a canonical
/// form, standardizing such issues as whether a character with
/// an accent is represented as a base character and combining
/// accent or as a single precomposed character. Unicode strings
/// should generally be normalized before comparing them.
pub const NormalizeMode = enum(c_int) {
    default = 0,
    default_compose = 1,
    all = 2,
    all_compose = 3,
    _,

    pub const nfd = NormalizeMode.default;
    pub const nfc = NormalizeMode.default_compose;
    pub const nfkd = NormalizeMode.all;
    pub const nfkc = NormalizeMode.all_compose;
    extern fn g_normalize_mode_get_type() usize;
    pub const getGObjectType = g_normalize_mode_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by functions converting a string to a number.
pub const NumberParserError = enum(c_int) {
    invalid = 0,
    out_of_bounds = 1,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The possible statuses of a one-time initialization function
/// controlled by a `glib.Once` struct.
pub const OnceStatus = enum(c_int) {
    notcalled = 0,
    progress = 1,
    ready = 2,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.OptionArg` enum values determine which type of extra argument the
/// options expect to find. If an option expects an extra argument, it can
/// be specified in several ways; with a short option: `-x arg`, with a long
/// option: `--name arg` or combined in a single argument: `--name=arg`.
pub const OptionArg = enum(c_int) {
    none = 0,
    string = 1,
    int = 2,
    callback = 3,
    filename = 4,
    string_array = 5,
    filename_array = 6,
    double = 7,
    int64 = 8,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by option parsing.
pub const OptionError = enum(c_int) {
    unknown_option = 0,
    bad_value = 1,
    failed = 2,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by regular expressions functions.
pub const RegexError = enum(c_int) {
    compile = 0,
    optimize = 1,
    replace = 2,
    match = 3,
    internal = 4,
    stray_backslash = 101,
    missing_control_char = 102,
    unrecognized_escape = 103,
    quantifiers_out_of_order = 104,
    quantifier_too_big = 105,
    unterminated_character_class = 106,
    invalid_escape_in_character_class = 107,
    range_out_of_order = 108,
    nothing_to_repeat = 109,
    unrecognized_character = 112,
    posix_named_class_outside_class = 113,
    unmatched_parenthesis = 114,
    inexistent_subpattern_reference = 115,
    unterminated_comment = 118,
    expression_too_large = 120,
    memory_error = 121,
    variable_length_lookbehind = 125,
    malformed_condition = 126,
    too_many_conditional_branches = 127,
    assertion_expected = 128,
    unknown_posix_class_name = 130,
    posix_collating_elements_not_supported = 131,
    hex_code_too_large = 134,
    invalid_condition = 135,
    single_byte_match_in_lookbehind = 136,
    infinite_loop = 140,
    missing_subpattern_name_terminator = 142,
    duplicate_subpattern_name = 143,
    malformed_property = 146,
    unknown_property = 147,
    subpattern_name_too_long = 148,
    too_many_subpatterns = 149,
    invalid_octal_value = 151,
    too_many_branches_in_define = 154,
    define_repetion = 155,
    inconsistent_newline_options = 156,
    missing_back_reference = 157,
    invalid_relative_reference = 158,
    backtracking_control_verb_argument_forbidden = 159,
    unknown_backtracking_control_verb = 160,
    number_too_big = 161,
    missing_subpattern_name = 162,
    missing_digit = 163,
    invalid_data_character = 164,
    extra_subpattern_name = 165,
    backtracking_control_verb_argument_required = 166,
    invalid_control_char = 168,
    missing_name = 169,
    not_supported_in_class = 171,
    too_many_forward_references = 172,
    name_too_long = 175,
    character_value_too_large = 176,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// An enumeration specifying the base position for a
/// `glib.IOChannel.seekPosition` operation.
pub const SeekType = enum(c_int) {
    cur = 0,
    set = 1,
    end = 2,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by shell functions.
pub const ShellError = enum(c_int) {
    bad_quoting = 0,
    empty_string = 1,
    failed = 2,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

pub const SliceConfig = enum(c_int) {
    always_malloc = 1,
    bypass_magazines = 2,
    working_set_msecs = 3,
    color_increment = 4,
    chunk_sizes = 5,
    contention_counter = 6,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by spawning processes.
pub const SpawnError = enum(c_int) {
    fork = 0,
    read = 1,
    chdir = 2,
    acces = 3,
    perm = 4,
    too_big = 5,
    noexec = 6,
    nametoolong = 7,
    noent = 8,
    nomem = 9,
    notdir = 10,
    loop = 11,
    txtbusy = 12,
    io = 13,
    nfile = 14,
    mfile = 15,
    inval = 16,
    isdir = 17,
    libbad = 18,
    failed = 19,
    _,

    pub const @"2big" = SpawnError.too_big;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The type of file to return the filename for, when used with
/// `glib.testBuildFilename`.
///
/// These two options correspond rather directly to the 'dist' and
/// 'built' terminology that automake uses and are explicitly used to
/// distinguish between the 'srcdir' and 'builddir' being separate. All
/// files in your project should either be dist (in the `EXTRA_DIST` or
/// `dist_schema_DATA` sense, in which case they will always be in the
/// srcdir) or built (in the `BUILT_SOURCES` sense, in which case they
/// will always be in the builddir).
///
/// Note: As a general rule of automake, files that are generated only as
/// part of the build-from-git process (but then are distributed with the
/// tarball) always go in srcdir (even if doing a srcdir != builddir
/// build from git) and are considered as distributed files.
///
/// The same principles apply for other build systems, such as meson.
pub const TestFileType = enum(c_int) {
    dist = 0,
    built = 1,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

pub const TestLogType = enum(c_int) {
    none = 0,
    @"error" = 1,
    start_binary = 2,
    list_case = 3,
    skip_case = 4,
    start_case = 5,
    stop_case = 6,
    min_result = 7,
    max_result = 8,
    message = 9,
    start_suite = 10,
    stop_suite = 11,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

pub const TestResult = enum(c_int) {
    success = 0,
    skipped = 1,
    failure = 2,
    incomplete = 3,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Possible errors of thread related functions.
pub const ThreadError = enum(c_int) {
    thread_error_again = 0,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Thread priorities.
pub const ThreadPriority = enum(c_int) {
    low = 0,
    normal = 1,
    high = 2,
    urgent = 3,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Disambiguates a given time in two ways.
///
/// First, specifies if the given time is in universal or local time.
///
/// Second, if the time is in local time, specifies if it is local
/// standard time or local daylight time.  This is important for the case
/// where the same local time occurs twice (during daylight savings time
/// transitions, for example).
pub const TimeType = enum(c_int) {
    standard = 0,
    daylight = 1,
    universal = 2,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The possible types of token returned from each
/// `glib.Scanner.getNextToken` call.
pub const TokenType = enum(c_int) {
    eof = 0,
    left_paren = 40,
    right_paren = 41,
    left_curly = 123,
    right_curly = 125,
    left_brace = 91,
    right_brace = 93,
    equal_sign = 61,
    comma = 44,
    none = 256,
    @"error" = 257,
    char = 258,
    binary = 259,
    octal = 260,
    int = 261,
    hex = 262,
    float = 263,
    string = 264,
    symbol = 265,
    identifier = 266,
    identifier_null = 267,
    comment_single = 268,
    comment_multi = 269,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Specifies the type of traversal performed by `glib.Tree.traverse`,
/// `glib.Node.traverse` and `glib.Node.find`.
///
/// The different orders are illustrated here:
///
/// - In order: A, B, C, D, E, F, G, H, I
///   <picture>
///     <source srcset="Sorted_binary_tree_inorder-dark.svg"
///      media="(prefers-color-scheme: dark)">
///     <img src="Sorted_binary_tree_inorder.svg"
///      alt="Sorted binary tree, in-order traversal">
///   </picture>
/// - Pre order: F, B, A, D, C, E, G, I, H
///   <picture>
///     <source srcset="Sorted_binary_tree_preorder-dark.svg"
///      media="(prefers-color-scheme: dark)">
///     <img src="Sorted_binary_tree_preorder.svg"
///      alt="Sorted binary tree, pre-order traversal">
///   </picture>
/// - Post order: A, C, E, D, B, H, I, G, F
///   <picture>
///     <source srcset="Sorted_binary_tree_postorder-dark.svg"
///      media="(prefers-color-scheme: dark)">
///     <img src="Sorted_binary_tree_postorder.svg"
///      alt="Sorted binary tree, post-order traversal">
///   </picture>
/// - Level order: F, B, G, A, D, I, C, E, H
///   <picture>
///     <source srcset="Sorted_binary_tree_breadth-first_traversal-dark.svg"
///      media="(prefers-color-scheme: dark)">
///     <img src="Sorted_binary_tree_breadth-first_traversal.svg"
///      alt="Sorted binary tree, breadth-first level order traversal">
///   </picture>
pub const TraverseType = enum(c_int) {
    in_order = 0,
    pre_order = 1,
    post_order = 2,
    level_order = 3,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// These are the possible line break classifications.
///
/// Since new Unicode versions may add new types here, applications should be ready
/// to handle unknown values. They may be regarded as `G_UNICODE_BREAK_UNKNOWN`.
///
/// See [Unicode Line Breaking Algorithm](https://www.unicode.org/reports/tr14/).
pub const UnicodeBreakType = enum(c_int) {
    mandatory = 0,
    carriage_return = 1,
    line_feed = 2,
    combining_mark = 3,
    surrogate = 4,
    zero_width_space = 5,
    inseparable = 6,
    non_breaking_glue = 7,
    contingent = 8,
    space = 9,
    after = 10,
    before = 11,
    before_and_after = 12,
    hyphen = 13,
    non_starter = 14,
    open_punctuation = 15,
    close_punctuation = 16,
    quotation = 17,
    exclamation = 18,
    ideographic = 19,
    numeric = 20,
    infix_separator = 21,
    symbol = 22,
    alphabetic = 23,
    prefix = 24,
    postfix = 25,
    complex_context = 26,
    ambiguous = 27,
    unknown = 28,
    next_line = 29,
    word_joiner = 30,
    hangul_l_jamo = 31,
    hangul_v_jamo = 32,
    hangul_t_jamo = 33,
    hangul_lv_syllable = 34,
    hangul_lvt_syllable = 35,
    close_paranthesis = 36,
    conditional_japanese_starter = 37,
    hebrew_letter = 38,
    regional_indicator = 39,
    emoji_base = 40,
    emoji_modifier = 41,
    zero_width_joiner = 42,
    aksara = 43,
    aksara_pre_base = 44,
    aksara_start = 45,
    virama_final = 46,
    virama = 47,
    _,

    pub const close_parenthesis = UnicodeBreakType.close_paranthesis;
    extern fn g_unicode_break_type_get_type() usize;
    pub const getGObjectType = g_unicode_break_type_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The `glib.UnicodeScript` enumeration identifies different writing
/// systems. The values correspond to the names as defined in the
/// Unicode standard. The enumeration has been added in GLib 2.14,
/// and is interchangeable with `PangoScript`.
///
/// Note that new types may be added in the future. Applications
/// should be ready to handle unknown values.
/// See [Unicode Standard Annex `@"24"`: Script names](http://www.unicode.org/reports/tr24/).
pub const UnicodeScript = enum(c_int) {
    invalid_code = -1,
    common = 0,
    inherited = 1,
    arabic = 2,
    armenian = 3,
    bengali = 4,
    bopomofo = 5,
    cherokee = 6,
    coptic = 7,
    cyrillic = 8,
    deseret = 9,
    devanagari = 10,
    ethiopic = 11,
    georgian = 12,
    gothic = 13,
    greek = 14,
    gujarati = 15,
    gurmukhi = 16,
    han = 17,
    hangul = 18,
    hebrew = 19,
    hiragana = 20,
    kannada = 21,
    katakana = 22,
    khmer = 23,
    lao = 24,
    latin = 25,
    malayalam = 26,
    mongolian = 27,
    myanmar = 28,
    ogham = 29,
    old_italic = 30,
    oriya = 31,
    runic = 32,
    sinhala = 33,
    syriac = 34,
    tamil = 35,
    telugu = 36,
    thaana = 37,
    thai = 38,
    tibetan = 39,
    canadian_aboriginal = 40,
    yi = 41,
    tagalog = 42,
    hanunoo = 43,
    buhid = 44,
    tagbanwa = 45,
    braille = 46,
    cypriot = 47,
    limbu = 48,
    osmanya = 49,
    shavian = 50,
    linear_b = 51,
    tai_le = 52,
    ugaritic = 53,
    new_tai_lue = 54,
    buginese = 55,
    glagolitic = 56,
    tifinagh = 57,
    syloti_nagri = 58,
    old_persian = 59,
    kharoshthi = 60,
    unknown = 61,
    balinese = 62,
    cuneiform = 63,
    phoenician = 64,
    phags_pa = 65,
    nko = 66,
    kayah_li = 67,
    lepcha = 68,
    rejang = 69,
    sundanese = 70,
    saurashtra = 71,
    cham = 72,
    ol_chiki = 73,
    vai = 74,
    carian = 75,
    lycian = 76,
    lydian = 77,
    avestan = 78,
    bamum = 79,
    egyptian_hieroglyphs = 80,
    imperial_aramaic = 81,
    inscriptional_pahlavi = 82,
    inscriptional_parthian = 83,
    javanese = 84,
    kaithi = 85,
    lisu = 86,
    meetei_mayek = 87,
    old_south_arabian = 88,
    old_turkic = 89,
    samaritan = 90,
    tai_tham = 91,
    tai_viet = 92,
    batak = 93,
    brahmi = 94,
    mandaic = 95,
    chakma = 96,
    meroitic_cursive = 97,
    meroitic_hieroglyphs = 98,
    miao = 99,
    sharada = 100,
    sora_sompeng = 101,
    takri = 102,
    bassa_vah = 103,
    caucasian_albanian = 104,
    duployan = 105,
    elbasan = 106,
    grantha = 107,
    khojki = 108,
    khudawadi = 109,
    linear_a = 110,
    mahajani = 111,
    manichaean = 112,
    mende_kikakui = 113,
    modi = 114,
    mro = 115,
    nabataean = 116,
    old_north_arabian = 117,
    old_permic = 118,
    pahawh_hmong = 119,
    palmyrene = 120,
    pau_cin_hau = 121,
    psalter_pahlavi = 122,
    siddham = 123,
    tirhuta = 124,
    warang_citi = 125,
    ahom = 126,
    anatolian_hieroglyphs = 127,
    hatran = 128,
    multani = 129,
    old_hungarian = 130,
    signwriting = 131,
    adlam = 132,
    bhaiksuki = 133,
    marchen = 134,
    newa = 135,
    osage = 136,
    tangut = 137,
    masaram_gondi = 138,
    nushu = 139,
    soyombo = 140,
    zanabazar_square = 141,
    dogra = 142,
    gunjala_gondi = 143,
    hanifi_rohingya = 144,
    makasar = 145,
    medefaidrin = 146,
    old_sogdian = 147,
    sogdian = 148,
    elymaic = 149,
    nandinagari = 150,
    nyiakeng_puachue_hmong = 151,
    wancho = 152,
    chorasmian = 153,
    dives_akuru = 154,
    khitan_small_script = 155,
    yezidi = 156,
    cypro_minoan = 157,
    old_uyghur = 158,
    tangsa = 159,
    toto = 160,
    vithkuqi = 161,
    math = 162,
    kawi = 163,
    nag_mundari = 164,
    todhri = 165,
    garay = 166,
    tulu_tigalari = 167,
    sunuwar = 168,
    gurung_khema = 169,
    kirat_rai = 170,
    ol_onal = 171,
    _,

    /// Looks up the Unicode script for `iso15924`.  ISO 15924 assigns four-letter
    /// codes to scripts.  For example, the code for Arabic is 'Arab'.
    /// This function accepts four letter codes encoded as a `guint32` in a
    /// big-endian fashion.  That is, the code expected for Arabic is
    /// 0x41726162 (0x41 is ASCII code for 'A', 0x72 is ASCII code for 'r', etc).
    ///
    /// See
    /// [Codes for the representation of names of scripts](http://unicode.org/iso15924/codelists.html)
    /// for details.
    extern fn g_unicode_script_from_iso15924(p_iso15924: u32) glib.UnicodeScript;
    pub const fromIso15924 = g_unicode_script_from_iso15924;

    /// Looks up the ISO 15924 code for `script`.  ISO 15924 assigns four-letter
    /// codes to scripts.  For example, the code for Arabic is 'Arab'.  The
    /// four letter codes are encoded as a `guint32` by this function in a
    /// big-endian fashion.  That is, the code returned for Arabic is
    /// 0x41726162 (0x41 is ASCII code for 'A', 0x72 is ASCII code for 'r', etc).
    ///
    /// See
    /// [Codes for the representation of names of scripts](http://unicode.org/iso15924/codelists.html)
    /// for details.
    extern fn g_unicode_script_to_iso15924(p_script: glib.UnicodeScript) u32;
    pub const toIso15924 = g_unicode_script_to_iso15924;

    extern fn g_unicode_script_get_type() usize;
    pub const getGObjectType = g_unicode_script_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// These are the possible character classifications from the
/// Unicode specification.
/// See [Unicode Character Database](http://www.unicode.org/reports/tr44/`General_Category_Values`).
pub const UnicodeType = enum(c_int) {
    control = 0,
    format = 1,
    unassigned = 2,
    private_use = 3,
    surrogate = 4,
    lowercase_letter = 5,
    modifier_letter = 6,
    other_letter = 7,
    titlecase_letter = 8,
    uppercase_letter = 9,
    spacing_mark = 10,
    enclosing_mark = 11,
    non_spacing_mark = 12,
    decimal_number = 13,
    letter_number = 14,
    other_number = 15,
    connect_punctuation = 16,
    dash_punctuation = 17,
    close_punctuation = 18,
    final_punctuation = 19,
    initial_punctuation = 20,
    other_punctuation = 21,
    open_punctuation = 22,
    currency_symbol = 23,
    modifier_symbol = 24,
    math_symbol = 25,
    other_symbol = 26,
    line_separator = 27,
    paragraph_separator = 28,
    space_separator = 29,
    _,

    extern fn g_unicode_type_get_type() usize;
    pub const getGObjectType = g_unicode_type_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Mnemonic constants for the ends of a Unix pipe.
pub const UnixPipeEnd = enum(c_int) {
    read = 0,
    write = 1,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by `glib.Uri` methods.
pub const UriError = enum(c_int) {
    failed = 0,
    bad_scheme = 1,
    bad_user = 2,
    bad_password = 3,
    bad_auth_params = 4,
    bad_host = 5,
    bad_port = 6,
    bad_path = 7,
    bad_query = 8,
    bad_fragment = 9,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// These are logical ids for special directories which are defined
/// depending on the platform used. You should use `glib.getUserSpecialDir`
/// to retrieve the full path associated to the logical id.
///
/// The `glib.UserDirectory` enumeration can be extended at later date. Not
/// every platform has a directory for every logical id in this
/// enumeration.
pub const UserDirectory = enum(c_int) {
    directory_desktop = 0,
    directory_documents = 1,
    directory_download = 2,
    directory_music = 3,
    directory_pictures = 4,
    directory_public_share = 5,
    directory_templates = 6,
    directory_videos = 7,
    n_directories = 8,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// The range of possible top-level types of `glib.Variant` instances.
pub const VariantClass = enum(c_int) {
    boolean = 98,
    byte = 121,
    int16 = 110,
    uint16 = 113,
    int32 = 105,
    uint32 = 117,
    int64 = 120,
    uint64 = 116,
    handle = 104,
    double = 100,
    string = 115,
    object_path = 111,
    signature = 103,
    variant = 118,
    maybe = 109,
    array = 97,
    tuple = 40,
    dict_entry = 123,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Error codes returned by parsing text-format GVariants.
pub const VariantParseError = enum(c_int) {
    failed = 0,
    basic_type_expected = 1,
    cannot_infer_type = 2,
    definite_type_expected = 3,
    input_not_at_end = 4,
    invalid_character = 5,
    invalid_format_string = 6,
    invalid_object_path = 7,
    invalid_signature = 8,
    invalid_type_string = 9,
    no_common_type = 10,
    number_out_of_range = 11,
    number_too_big = 12,
    type_error = 13,
    unexpected_token = 14,
    unknown_keyword = 15,
    unterminated_string_constant = 16,
    value_expected = 17,
    recursion = 18,
    _,

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

pub const AsciiType = packed struct(c_uint) {
    alnum: bool = false,
    alpha: bool = false,
    cntrl: bool = false,
    digit: bool = false,
    graph: bool = false,
    lower: bool = false,
    print: bool = false,
    punct: bool = false,
    space: bool = false,
    upper: bool = false,
    xdigit: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_alnum: AsciiType = @bitCast(@as(c_uint, 1));
    pub const flags_alpha: AsciiType = @bitCast(@as(c_uint, 2));
    pub const flags_cntrl: AsciiType = @bitCast(@as(c_uint, 4));
    pub const flags_digit: AsciiType = @bitCast(@as(c_uint, 8));
    pub const flags_graph: AsciiType = @bitCast(@as(c_uint, 16));
    pub const flags_lower: AsciiType = @bitCast(@as(c_uint, 32));
    pub const flags_print: AsciiType = @bitCast(@as(c_uint, 64));
    pub const flags_punct: AsciiType = @bitCast(@as(c_uint, 128));
    pub const flags_space: AsciiType = @bitCast(@as(c_uint, 256));
    pub const flags_upper: AsciiType = @bitCast(@as(c_uint, 512));
    pub const flags_xdigit: AsciiType = @bitCast(@as(c_uint, 1024));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags to pass to `glib.fileSetContentsFull` to affect its safety and
/// performance.
pub const FileSetContentsFlags = packed struct(c_uint) {
    consistent: bool = false,
    durable: bool = false,
    only_existing: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_none: FileSetContentsFlags = @bitCast(@as(c_uint, 0));
    pub const flags_consistent: FileSetContentsFlags = @bitCast(@as(c_uint, 1));
    pub const flags_durable: FileSetContentsFlags = @bitCast(@as(c_uint, 2));
    pub const flags_only_existing: FileSetContentsFlags = @bitCast(@as(c_uint, 4));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A test to perform on a file using `glib.fileTest`.
pub const FileTest = packed struct(c_uint) {
    is_regular: bool = false,
    is_symlink: bool = false,
    is_dir: bool = false,
    is_executable: bool = false,
    exists: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_is_regular: FileTest = @bitCast(@as(c_uint, 1));
    pub const flags_is_symlink: FileTest = @bitCast(@as(c_uint, 2));
    pub const flags_is_dir: FileTest = @bitCast(@as(c_uint, 4));
    pub const flags_is_executable: FileTest = @bitCast(@as(c_uint, 8));
    pub const flags_exists: FileTest = @bitCast(@as(c_uint, 16));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags to modify the format of the string returned by `glib.formatSizeFull`.
pub const FormatSizeFlags = packed struct(c_uint) {
    long_format: bool = false,
    iec_units: bool = false,
    bits: bool = false,
    only_value: bool = false,
    only_unit: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_default: FormatSizeFlags = @bitCast(@as(c_uint, 0));
    pub const flags_long_format: FormatSizeFlags = @bitCast(@as(c_uint, 1));
    pub const flags_iec_units: FormatSizeFlags = @bitCast(@as(c_uint, 2));
    pub const flags_bits: FormatSizeFlags = @bitCast(@as(c_uint, 4));
    pub const flags_only_value: FormatSizeFlags = @bitCast(@as(c_uint, 8));
    pub const flags_only_unit: FormatSizeFlags = @bitCast(@as(c_uint, 16));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags used internally in the `glib.Hook` implementation.
pub const HookFlagMask = packed struct(c_uint) {
    active: bool = false,
    in_call: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_active: HookFlagMask = @bitCast(@as(c_uint, 1));
    pub const flags_in_call: HookFlagMask = @bitCast(@as(c_uint, 2));
    pub const flags_mask: HookFlagMask = @bitCast(@as(c_uint, 15));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A bitwise combination representing a condition to watch for on an
/// event source.
pub const IOCondition = packed struct(c_uint) {
    in: bool = false,
    pri: bool = false,
    out: bool = false,
    err: bool = false,
    hup: bool = false,
    nval: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_in: IOCondition = @bitCast(@as(c_uint, 1));
    pub const flags_out: IOCondition = @bitCast(@as(c_uint, 4));
    pub const flags_pri: IOCondition = @bitCast(@as(c_uint, 2));
    pub const flags_err: IOCondition = @bitCast(@as(c_uint, 8));
    pub const flags_hup: IOCondition = @bitCast(@as(c_uint, 16));
    pub const flags_nval: IOCondition = @bitCast(@as(c_uint, 32));
    extern fn g_io_condition_get_type() usize;
    pub const getGObjectType = g_io_condition_get_type;

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Specifies properties of a `glib.IOChannel`. Some of the flags can only be
/// read with `glib.IOChannel.getFlags`, but not changed with
/// `glib.IOChannel.setFlags`.
pub const IOFlags = packed struct(c_uint) {
    append: bool = false,
    nonblock: bool = false,
    is_readable: bool = false,
    is_writable: bool = false,
    is_seekable: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_none: IOFlags = @bitCast(@as(c_uint, 0));
    pub const flags_append: IOFlags = @bitCast(@as(c_uint, 1));
    pub const flags_nonblock: IOFlags = @bitCast(@as(c_uint, 2));
    pub const flags_is_readable: IOFlags = @bitCast(@as(c_uint, 4));
    pub const flags_is_writable: IOFlags = @bitCast(@as(c_uint, 8));
    pub const flags_is_writeable: IOFlags = @bitCast(@as(c_uint, 8));
    pub const flags_is_seekable: IOFlags = @bitCast(@as(c_uint, 16));
    pub const flags_mask: IOFlags = @bitCast(@as(c_uint, 31));
    pub const flags_get_mask: IOFlags = @bitCast(@as(c_uint, 31));
    pub const flags_set_mask: IOFlags = @bitCast(@as(c_uint, 3));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags which influence the parsing.
pub const KeyFileFlags = packed struct(c_uint) {
    keep_comments: bool = false,
    keep_translations: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_none: KeyFileFlags = @bitCast(@as(c_uint, 0));
    pub const flags_keep_comments: KeyFileFlags = @bitCast(@as(c_uint, 1));
    pub const flags_keep_translations: KeyFileFlags = @bitCast(@as(c_uint, 2));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags specifying the level of log messages.
///
/// It is possible to change how GLib treats messages of the various
/// levels using `glib.logSetHandler` and `glib.logSetFatalMask`.
pub const LogLevelFlags = packed struct(c_uint) {
    flag_recursion: bool = false,
    flag_fatal: bool = false,
    level_error: bool = false,
    level_critical: bool = false,
    level_warning: bool = false,
    level_message: bool = false,
    level_info: bool = false,
    level_debug: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_flag_recursion: LogLevelFlags = @bitCast(@as(c_uint, 1));
    pub const flags_flag_fatal: LogLevelFlags = @bitCast(@as(c_uint, 2));
    pub const flags_level_error: LogLevelFlags = @bitCast(@as(c_uint, 4));
    pub const flags_level_critical: LogLevelFlags = @bitCast(@as(c_uint, 8));
    pub const flags_level_warning: LogLevelFlags = @bitCast(@as(c_uint, 16));
    pub const flags_level_message: LogLevelFlags = @bitCast(@as(c_uint, 32));
    pub const flags_level_info: LogLevelFlags = @bitCast(@as(c_uint, 64));
    pub const flags_level_debug: LogLevelFlags = @bitCast(@as(c_uint, 128));
    pub const flags_level_mask: LogLevelFlags = @bitCast(@as(c_int, -4));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags to pass to `glib.MainContext.newWithFlags` which affect the
/// behaviour of a `glib.MainContext`.
pub const MainContextFlags = packed struct(c_uint) {
    ownerless_polling: bool = false,
    _padding1: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_none: MainContextFlags = @bitCast(@as(c_uint, 0));
    pub const flags_ownerless_polling: MainContextFlags = @bitCast(@as(c_uint, 1));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A mixed enumerated type and flags field. You must specify one type
/// (string, strdup, boolean, tristate).  Additionally, you may  optionally
/// bitwise OR the type with the flag `G_MARKUP_COLLECT_OPTIONAL`.
///
/// It is likely that this enum will be extended in the future to
/// support other types.
pub const MarkupCollectType = packed struct(c_uint) {
    string: bool = false,
    strdup: bool = false,
    tristate: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    optional: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_invalid: MarkupCollectType = @bitCast(@as(c_uint, 0));
    pub const flags_string: MarkupCollectType = @bitCast(@as(c_uint, 1));
    pub const flags_strdup: MarkupCollectType = @bitCast(@as(c_uint, 2));
    pub const flags_boolean: MarkupCollectType = @bitCast(@as(c_uint, 3));
    pub const flags_tristate: MarkupCollectType = @bitCast(@as(c_uint, 4));
    pub const flags_optional: MarkupCollectType = @bitCast(@as(c_uint, 65536));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags that affect the behaviour of the parser.
pub const MarkupParseFlags = packed struct(c_uint) {
    do_not_use_this_unsupported_flag: bool = false,
    treat_cdata_as_text: bool = false,
    prefix_error_position: bool = false,
    ignore_qualified: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_default_flags: MarkupParseFlags = @bitCast(@as(c_uint, 0));
    pub const flags_do_not_use_this_unsupported_flag: MarkupParseFlags = @bitCast(@as(c_uint, 1));
    pub const flags_treat_cdata_as_text: MarkupParseFlags = @bitCast(@as(c_uint, 2));
    pub const flags_prefix_error_position: MarkupParseFlags = @bitCast(@as(c_uint, 4));
    pub const flags_ignore_qualified: MarkupParseFlags = @bitCast(@as(c_uint, 8));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags which modify individual options.
pub const OptionFlags = packed struct(c_uint) {
    hidden: bool = false,
    in_main: bool = false,
    reverse: bool = false,
    no_arg: bool = false,
    filename: bool = false,
    optional_arg: bool = false,
    @"noalias": bool = false,
    deprecated: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_none: OptionFlags = @bitCast(@as(c_uint, 0));
    pub const flags_hidden: OptionFlags = @bitCast(@as(c_uint, 1));
    pub const flags_in_main: OptionFlags = @bitCast(@as(c_uint, 2));
    pub const flags_reverse: OptionFlags = @bitCast(@as(c_uint, 4));
    pub const flags_no_arg: OptionFlags = @bitCast(@as(c_uint, 8));
    pub const flags_filename: OptionFlags = @bitCast(@as(c_uint, 16));
    pub const flags_optional_arg: OptionFlags = @bitCast(@as(c_uint, 32));
    pub const flags_noalias: OptionFlags = @bitCast(@as(c_uint, 64));
    pub const flags_deprecated: OptionFlags = @bitCast(@as(c_uint, 128));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags specifying compile-time options.
pub const RegexCompileFlags = packed struct(c_uint) {
    caseless: bool = false,
    multiline: bool = false,
    dotall: bool = false,
    extended: bool = false,
    anchored: bool = false,
    dollar_endonly: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    ungreedy: bool = false,
    _padding10: bool = false,
    raw: bool = false,
    no_auto_capture: bool = false,
    optimize: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    firstline: bool = false,
    dupnames: bool = false,
    newline_cr: bool = false,
    newline_lf: bool = false,
    _padding22: bool = false,
    bsr_anycrlf: bool = false,
    _padding24: bool = false,
    javascript_compat: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_default: RegexCompileFlags = @bitCast(@as(c_uint, 0));
    pub const flags_caseless: RegexCompileFlags = @bitCast(@as(c_uint, 1));
    pub const flags_multiline: RegexCompileFlags = @bitCast(@as(c_uint, 2));
    pub const flags_dotall: RegexCompileFlags = @bitCast(@as(c_uint, 4));
    pub const flags_extended: RegexCompileFlags = @bitCast(@as(c_uint, 8));
    pub const flags_anchored: RegexCompileFlags = @bitCast(@as(c_uint, 16));
    pub const flags_dollar_endonly: RegexCompileFlags = @bitCast(@as(c_uint, 32));
    pub const flags_ungreedy: RegexCompileFlags = @bitCast(@as(c_uint, 512));
    pub const flags_raw: RegexCompileFlags = @bitCast(@as(c_uint, 2048));
    pub const flags_no_auto_capture: RegexCompileFlags = @bitCast(@as(c_uint, 4096));
    pub const flags_optimize: RegexCompileFlags = @bitCast(@as(c_uint, 8192));
    pub const flags_firstline: RegexCompileFlags = @bitCast(@as(c_uint, 262144));
    pub const flags_dupnames: RegexCompileFlags = @bitCast(@as(c_uint, 524288));
    pub const flags_newline_cr: RegexCompileFlags = @bitCast(@as(c_uint, 1048576));
    pub const flags_newline_lf: RegexCompileFlags = @bitCast(@as(c_uint, 2097152));
    pub const flags_newline_crlf: RegexCompileFlags = @bitCast(@as(c_uint, 3145728));
    pub const flags_newline_anycrlf: RegexCompileFlags = @bitCast(@as(c_uint, 5242880));
    pub const flags_bsr_anycrlf: RegexCompileFlags = @bitCast(@as(c_uint, 8388608));
    pub const flags_javascript_compat: RegexCompileFlags = @bitCast(@as(c_uint, 33554432));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags specifying match-time options.
pub const RegexMatchFlags = packed struct(c_uint) {
    _padding0: bool = false,
    _padding1: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    anchored: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    notbol: bool = false,
    noteol: bool = false,
    _padding9: bool = false,
    notempty: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    partial: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    newline_cr: bool = false,
    newline_lf: bool = false,
    newline_any: bool = false,
    bsr_anycrlf: bool = false,
    bsr_any: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    partial_hard: bool = false,
    notempty_atstart: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_default: RegexMatchFlags = @bitCast(@as(c_uint, 0));
    pub const flags_anchored: RegexMatchFlags = @bitCast(@as(c_uint, 16));
    pub const flags_notbol: RegexMatchFlags = @bitCast(@as(c_uint, 128));
    pub const flags_noteol: RegexMatchFlags = @bitCast(@as(c_uint, 256));
    pub const flags_notempty: RegexMatchFlags = @bitCast(@as(c_uint, 1024));
    pub const flags_partial: RegexMatchFlags = @bitCast(@as(c_uint, 32768));
    pub const flags_newline_cr: RegexMatchFlags = @bitCast(@as(c_uint, 1048576));
    pub const flags_newline_lf: RegexMatchFlags = @bitCast(@as(c_uint, 2097152));
    pub const flags_newline_crlf: RegexMatchFlags = @bitCast(@as(c_uint, 3145728));
    pub const flags_newline_any: RegexMatchFlags = @bitCast(@as(c_uint, 4194304));
    pub const flags_newline_anycrlf: RegexMatchFlags = @bitCast(@as(c_uint, 5242880));
    pub const flags_bsr_anycrlf: RegexMatchFlags = @bitCast(@as(c_uint, 8388608));
    pub const flags_bsr_any: RegexMatchFlags = @bitCast(@as(c_uint, 16777216));
    pub const flags_partial_soft: RegexMatchFlags = @bitCast(@as(c_uint, 32768));
    pub const flags_partial_hard: RegexMatchFlags = @bitCast(@as(c_uint, 134217728));
    pub const flags_notempty_atstart: RegexMatchFlags = @bitCast(@as(c_uint, 268435456));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags passed to `glib.spawnSync`, `glib.spawnAsync` and `glib.spawnAsyncWithPipes`.
pub const SpawnFlags = packed struct(c_uint) {
    leave_descriptors_open: bool = false,
    do_not_reap_child: bool = false,
    search_path: bool = false,
    stdout_to_dev_null: bool = false,
    stderr_to_dev_null: bool = false,
    child_inherits_stdin: bool = false,
    file_and_argv_zero: bool = false,
    search_path_from_envp: bool = false,
    cloexec_pipes: bool = false,
    child_inherits_stdout: bool = false,
    child_inherits_stderr: bool = false,
    stdin_from_dev_null: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_default: SpawnFlags = @bitCast(@as(c_uint, 0));
    pub const flags_leave_descriptors_open: SpawnFlags = @bitCast(@as(c_uint, 1));
    pub const flags_do_not_reap_child: SpawnFlags = @bitCast(@as(c_uint, 2));
    pub const flags_search_path: SpawnFlags = @bitCast(@as(c_uint, 4));
    pub const flags_stdout_to_dev_null: SpawnFlags = @bitCast(@as(c_uint, 8));
    pub const flags_stderr_to_dev_null: SpawnFlags = @bitCast(@as(c_uint, 16));
    pub const flags_child_inherits_stdin: SpawnFlags = @bitCast(@as(c_uint, 32));
    pub const flags_file_and_argv_zero: SpawnFlags = @bitCast(@as(c_uint, 64));
    pub const flags_search_path_from_envp: SpawnFlags = @bitCast(@as(c_uint, 128));
    pub const flags_cloexec_pipes: SpawnFlags = @bitCast(@as(c_uint, 256));
    pub const flags_child_inherits_stdout: SpawnFlags = @bitCast(@as(c_uint, 512));
    pub const flags_child_inherits_stderr: SpawnFlags = @bitCast(@as(c_uint, 1024));
    pub const flags_stdin_from_dev_null: SpawnFlags = @bitCast(@as(c_uint, 2048));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags to pass to `glib.testTrapSubprocess` to control input and output.
///
/// Note that in contrast with `glib.testTrapFork`, the default
/// behavior of `glib.testTrapSubprocess` is to not show stdout
/// and stderr.
pub const TestSubprocessFlags = packed struct(c_uint) {
    inherit_stdin: bool = false,
    inherit_stdout: bool = false,
    inherit_stderr: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_default: TestSubprocessFlags = @bitCast(@as(c_uint, 0));
    pub const flags_inherit_stdin: TestSubprocessFlags = @bitCast(@as(c_uint, 1));
    pub const flags_inherit_stdout: TestSubprocessFlags = @bitCast(@as(c_uint, 2));
    pub const flags_inherit_stderr: TestSubprocessFlags = @bitCast(@as(c_uint, 4));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags to pass to `glib.testTrapFork` to control input and output.
///
/// Test traps are guards around forked tests. These flags determine what traps to set.
pub const TestTrapFlags = packed struct(c_uint) {
    _padding0: bool = false,
    _padding1: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    silence_stdout: bool = false,
    silence_stderr: bool = false,
    inherit_stdin: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_default: TestTrapFlags = @bitCast(@as(c_uint, 0));
    pub const flags_silence_stdout: TestTrapFlags = @bitCast(@as(c_uint, 128));
    pub const flags_silence_stderr: TestTrapFlags = @bitCast(@as(c_uint, 256));
    pub const flags_inherit_stdin: TestTrapFlags = @bitCast(@as(c_uint, 512));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Specifies which nodes are visited during several of the tree
/// functions, including `glib.Node.traverse` and `glib.Node.find`.
pub const TraverseFlags = packed struct(c_uint) {
    leaves: bool = false,
    non_leaves: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_leaves: TraverseFlags = @bitCast(@as(c_uint, 1));
    pub const flags_non_leaves: TraverseFlags = @bitCast(@as(c_uint, 2));
    pub const flags_all: TraverseFlags = @bitCast(@as(c_uint, 3));
    pub const flags_mask: TraverseFlags = @bitCast(@as(c_uint, 3));
    pub const flags_leafs: TraverseFlags = @bitCast(@as(c_uint, 1));
    pub const flags_non_leafs: TraverseFlags = @bitCast(@as(c_uint, 2));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags that describe a URI.
///
/// When parsing a URI, if you need to choose different flags based on
/// the type of URI, you can use `glib.uriPeekScheme` on the URI string
/// to check the scheme first, and use that to decide what flags to
/// parse it with.
pub const UriFlags = packed struct(c_uint) {
    parse_relaxed: bool = false,
    has_password: bool = false,
    has_auth_params: bool = false,
    encoded: bool = false,
    non_dns: bool = false,
    encoded_query: bool = false,
    encoded_path: bool = false,
    encoded_fragment: bool = false,
    scheme_normalize: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_none: UriFlags = @bitCast(@as(c_uint, 0));
    pub const flags_parse_relaxed: UriFlags = @bitCast(@as(c_uint, 1));
    pub const flags_has_password: UriFlags = @bitCast(@as(c_uint, 2));
    pub const flags_has_auth_params: UriFlags = @bitCast(@as(c_uint, 4));
    pub const flags_encoded: UriFlags = @bitCast(@as(c_uint, 8));
    pub const flags_non_dns: UriFlags = @bitCast(@as(c_uint, 16));
    pub const flags_encoded_query: UriFlags = @bitCast(@as(c_uint, 32));
    pub const flags_encoded_path: UriFlags = @bitCast(@as(c_uint, 64));
    pub const flags_encoded_fragment: UriFlags = @bitCast(@as(c_uint, 128));
    pub const flags_scheme_normalize: UriFlags = @bitCast(@as(c_uint, 256));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags describing what parts of the URI to hide in
/// `glib.Uri.toStringPartial`. Note that `G_URI_HIDE_PASSWORD` and
/// `G_URI_HIDE_AUTH_PARAMS` will only work if the `glib.Uri` was parsed with
/// the corresponding flags.
pub const UriHideFlags = packed struct(c_uint) {
    userinfo: bool = false,
    password: bool = false,
    auth_params: bool = false,
    query: bool = false,
    fragment: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_none: UriHideFlags = @bitCast(@as(c_uint, 0));
    pub const flags_userinfo: UriHideFlags = @bitCast(@as(c_uint, 1));
    pub const flags_password: UriHideFlags = @bitCast(@as(c_uint, 2));
    pub const flags_auth_params: UriHideFlags = @bitCast(@as(c_uint, 4));
    pub const flags_query: UriHideFlags = @bitCast(@as(c_uint, 8));
    pub const flags_fragment: UriHideFlags = @bitCast(@as(c_uint, 16));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// Flags modifying the way parameters are handled by `glib.uriParseParams` and
/// `glib.UriParamsIter`.
pub const UriParamsFlags = packed struct(c_uint) {
    case_insensitive: bool = false,
    www_form: bool = false,
    parse_relaxed: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    pub const flags_none: UriParamsFlags = @bitCast(@as(c_uint, 0));
    pub const flags_case_insensitive: UriParamsFlags = @bitCast(@as(c_uint, 1));
    pub const flags_www_form: UriParamsFlags = @bitCast(@as(c_uint, 2));
    pub const flags_parse_relaxed: UriParamsFlags = @bitCast(@as(c_uint, 4));

    test {
        @setEvalBranchQuota(100_000);
        std.testing.refAllDecls(@This());
    }
};

/// A wrapper for the POSIX `access` function. This function is used to
/// test a pathname for one or several of read, write or execute
/// permissions, or just existence.
///
/// On Windows, the file protection mechanism is not at all POSIX-like,
/// and the underlying function in the C library only checks the
/// FAT-style READONLY attribute, and does not look at the ACL of a
/// file at all. This function is this in practise almost useless on
/// Windows. Software that needs to handle file permissions on Windows
/// more exactly should use the Win32 API.
///
/// See your C library manual for more details about `access`.
extern fn g_access(p_filename: [*:0]const u8, p_mode: c_int) c_int;
pub const access = g_access;

/// This function is similar to `glib.malloc`, allocating (`n_blocks` * `n_block_bytes`)
/// bytes, but care is taken to align the allocated memory to with the given
/// alignment value. Additionally, it will detect possible overflow during
/// multiplication.
///
/// If the allocation fails (because the system is out of memory),
/// the program is terminated.
///
/// Aligned memory allocations returned by this function can only be
/// freed using `glib.alignedFreeSized` or `glib.alignedFree`.
extern fn g_aligned_alloc(p_n_blocks: usize, p_n_block_bytes: usize, p_alignment: usize) ?*anyopaque;
pub const alignedAlloc = g_aligned_alloc;

/// This function is similar to `glib.alignedAlloc`, but it will
/// also clear the allocated memory before returning it.
extern fn g_aligned_alloc0(p_n_blocks: usize, p_n_block_bytes: usize, p_alignment: usize) ?*anyopaque;
pub const alignedAlloc0 = g_aligned_alloc0;

/// Frees the memory allocated by `glib.alignedAlloc`.
extern fn g_aligned_free(p_mem: ?*anyopaque) void;
pub const alignedFree = g_aligned_free;

/// Frees the memory pointed to by `mem`, assuming it is has the given `size` and
/// `alignment`.
///
/// If `mem` is `NULL` this is a no-op (and `size` is ignored).
///
/// It is an error if `size` doesn’t match the size, or `alignment` doesn’t match
/// the alignment, passed when `mem` was allocated. `size` and `alignment` are
/// passed to this function to allow optimizations in the allocator. If you
/// don’t know either of them, use `glib.alignedFree` instead.
extern fn g_aligned_free_sized(p_mem: ?*anyopaque, p_alignment: usize, p_size: usize) void;
pub const alignedFreeSized = g_aligned_free_sized;

/// Determines the numeric value of a character as a decimal digit. If the
/// character is not a decimal digit according to `glib.asciiIsdigit`,
/// `-1` is returned.
///
/// Differs from `glib.unicharDigitValue` because it takes a char, so
/// there's no worry about sign extension if characters are signed.
extern fn g_ascii_digit_value(p_c: u8) c_int;
pub const asciiDigitValue = g_ascii_digit_value;

/// Converts a `gdouble` to a string, using the '.' as
/// decimal point.
///
/// This function generates enough precision that converting
/// the string back using `glib.asciiStrtod` gives the same machine-number
/// (on machines with IEEE compatible 64bit doubles). It is
/// guaranteed that the size of the resulting string will never
/// be larger than `glib.ASCII_DTOSTR_BUF_SIZE` bytes, including the terminating
/// nul character, which is always added.
extern fn g_ascii_dtostr(p_buffer: [*:0]u8, p_buf_len: c_int, p_d: f64) [*:0]u8;
pub const asciiDtostr = g_ascii_dtostr;

/// Converts a `gdouble` to a string, using the '.' as
/// decimal point. To format the number you pass in
/// a ``printf``-style format string. Allowed conversion
/// specifiers are 'e', 'E', 'f', 'F', 'g' and 'G'.
///
/// The `format` must just be a single format specifier
/// starting with `%`, expecting a `gdouble` argument.
///
/// The returned buffer is guaranteed to be nul-terminated.
///
/// If you just want to want to serialize the value into a
/// string, use `glib.asciiDtostr`.
extern fn g_ascii_formatd(p_buffer: [*:0]u8, p_buf_len: c_int, p_format: [*:0]const u8, p_d: f64) [*:0]u8;
pub const asciiFormatd = g_ascii_formatd;

/// Compare two strings, ignoring the case of ASCII characters.
///
/// Unlike the BSD ``strcasecmp`` function, this only recognizes standard
/// ASCII letters and ignores the locale, treating all non-ASCII
/// bytes as if they are not letters.
///
/// This function should be used only on strings that are known to be
/// in encodings where the bytes corresponding to ASCII letters always
/// represent themselves. This includes UTF-8 and the ISO-8859-*
/// charsets, but not for instance double-byte encodings like the
/// Windows Codepage 932, where the trailing bytes of double-byte
/// characters include all ASCII letters. If you compare two CP932
/// strings using this function, you will get false matches.
///
/// Both `s1` and `s2` must be non-`NULL`.
extern fn g_ascii_strcasecmp(p_s1: [*:0]const u8, p_s2: [*:0]const u8) c_int;
pub const asciiStrcasecmp = g_ascii_strcasecmp;

/// Converts all upper case ASCII letters to lower case ASCII letters, with
/// semantics that exactly match `glib.asciiTolower`.
extern fn g_ascii_strdown(p_str: [*:0]const u8, p_len: isize) [*:0]u8;
pub const asciiStrdown = g_ascii_strdown;

/// A convenience function for converting a string to a signed number.
///
/// This function assumes that `str` contains only a number of the given
/// `base` that is within inclusive bounds limited by `min` and `max`. If
/// this is true, then the converted number is stored in `out_num`. An
/// empty string is not a valid input. A string with leading or
/// trailing whitespace is also an invalid input.
///
/// `base` can be between 2 and 36 inclusive. Hexadecimal numbers must
/// not be prefixed with "0x" or "0X". Such a problem does not exist
/// for octal numbers, since they were usually prefixed with a zero
/// which does not change the value of the parsed number.
///
/// Parsing failures result in an error with the `G_NUMBER_PARSER_ERROR`
/// domain. If the input is invalid, the error code will be
/// `glib.@"NumberParserError.INVALID"`. If the parsed number is out of
/// bounds - `glib.@"NumberParserError.OUT_OF_BOUNDS"`.
///
/// See `glib.asciiStrtoll` if you have more complex needs such as
/// parsing a string which starts with a number, but then has other
/// characters.
extern fn g_ascii_string_to_signed(p_str: [*:0]const u8, p_base: c_uint, p_min: i64, p_max: i64, p_out_num: ?*i64, p_error: ?*?*glib.Error) c_int;
pub const asciiStringToSigned = g_ascii_string_to_signed;

/// A convenience function for converting a string to an unsigned number.
///
/// This function assumes that `str` contains only a number of the given
/// `base` that is within inclusive bounds limited by `min` and `max`. If
/// this is true, then the converted number is stored in `out_num`. An
/// empty string is not a valid input. A string with leading or
/// trailing whitespace is also an invalid input. A string with a leading sign
/// (`-` or `+`) is not a valid input for the unsigned parser.
///
/// `base` can be between 2 and 36 inclusive. Hexadecimal numbers must
/// not be prefixed with "0x" or "0X". Such a problem does not exist
/// for octal numbers, since they were usually prefixed with a zero
/// which does not change the value of the parsed number.
///
/// Parsing failures result in an error with the `G_NUMBER_PARSER_ERROR`
/// domain. If the input is invalid, the error code will be
/// `glib.@"NumberParserError.INVALID"`. If the parsed number is out of
/// bounds - `glib.@"NumberParserError.OUT_OF_BOUNDS"`.
///
/// See `glib.asciiStrtoull` if you have more complex needs such as
/// parsing a string which starts with a number, but then has other
/// characters.
extern fn g_ascii_string_to_unsigned(p_str: [*:0]const u8, p_base: c_uint, p_min: u64, p_max: u64, p_out_num: ?*u64, p_error: ?*?*glib.Error) c_int;
pub const asciiStringToUnsigned = g_ascii_string_to_unsigned;

/// Compare `s1` and `s2`, ignoring the case of ASCII characters and any
/// characters after the first `n` in each string. If either string is
/// less than `n` bytes long, comparison will stop at the first nul byte
/// encountered.
///
/// Unlike the BSD ``strncasecmp`` function, this only recognizes standard
/// ASCII letters and ignores the locale, treating all non-ASCII
/// characters as if they are not letters.
///
/// The same warning as in `glib.asciiStrcasecmp` applies: Use this
/// function only on strings known to be in encodings where bytes
/// corresponding to ASCII letters always represent themselves.
extern fn g_ascii_strncasecmp(p_s1: [*:0]const u8, p_s2: [*:0]const u8, p_n: usize) c_int;
pub const asciiStrncasecmp = g_ascii_strncasecmp;

/// Converts a string to a floating point value.
///
/// This function behaves like the standard ``strtod`` function
/// does in the C locale. It does this without actually changing
/// the current locale, since that would not be thread-safe.
/// A limitation of the implementation is that this function
/// will still accept localized versions of infinities and NANs.
///
/// This function is typically used when reading configuration
/// files or other non-user input that should be locale independent.
/// To handle input from the user you should normally use the
/// locale-sensitive system ``strtod`` function.
///
/// To convert from a gdouble to a string in a locale-insensitive
/// way, use `glib.asciiDtostr`.
///
/// If the correct value would cause overflow, plus or minus `HUGE_VAL`
/// is returned (according to the sign of the value), and `ERANGE` is
/// stored in `errno`. If the correct value would cause underflow,
/// zero is returned and `ERANGE` is stored in `errno`.
///
/// This function resets `errno` before calling ``strtod`` so that
/// you can reliably detect overflow and underflow.
extern fn g_ascii_strtod(p_nptr: [*:0]const u8, p_endptr: ?*[*:0]u8) f64;
pub const asciiStrtod = g_ascii_strtod;

/// Converts a string to a `gint64` value.
///
/// This function behaves like the standard ``strtoll`` function
/// does in the C locale. It does this without actually
/// changing the current locale, since that would not be
/// thread-safe.
///
/// This function is typically used when reading configuration
/// files or other non-user input that should be locale independent.
/// To handle input from the user you should normally use the
/// locale-sensitive system ``strtoll`` function.
///
/// If the correct value would cause overflow, `glib.MAXINT64` or
/// `glib.MININT64` is returned, and `ERANGE` is stored in `errno`.
/// If the base is outside the valid range, zero is returned, and
/// `EINVAL` is stored in `errno`. If the
/// string conversion fails, zero is returned, and `endptr` returns `nptr`
/// (if `endptr` is non-`NULL`).
extern fn g_ascii_strtoll(p_nptr: [*:0]const u8, p_endptr: ?*[*:0]u8, p_base: c_uint) i64;
pub const asciiStrtoll = g_ascii_strtoll;

/// Converts a string to a `guint64` value.
///
/// This function behaves like the standard ``strtoull`` function
/// does in the C locale. It does this without actually
/// changing the current locale, since that would not be
/// thread-safe.
///
/// Note that input with a leading minus sign (`-`) is accepted, and will return
/// the negation of the parsed number, unless that would overflow a `guint64`.
/// Critically, this means you cannot assume that a short fixed length input will
/// result in a low return value, as the input could have a leading `-`.
///
/// This function is typically used when reading configuration
/// files or other non-user input that should be locale independent.
/// To handle input from the user you should normally use the
/// locale-sensitive system ``strtoull`` function.
///
/// If the correct value would cause overflow, `glib.MAXUINT64`
/// is returned, and `ERANGE` is stored in `errno`.
/// If the base is outside the valid range, zero is returned, and
/// `EINVAL` is stored in `errno`.
/// If the string conversion fails, zero is returned, and `endptr` returns
/// `nptr` (if `endptr` is non-`NULL`).
extern fn g_ascii_strtoull(p_nptr: [*:0]const u8, p_endptr: ?*[*:0]u8, p_base: c_uint) u64;
pub const asciiStrtoull = g_ascii_strtoull;

/// Converts all lower case ASCII letters to upper case ASCII letters, with
/// semantics that exactly match `glib.asciiToupper`.
extern fn g_ascii_strup(p_str: [*:0]const u8, p_len: isize) [*:0]u8;
pub const asciiStrup = g_ascii_strup;

/// Convert a character to ASCII lower case. If the character is not an
/// ASCII upper case letter, it is returned unchanged.
///
/// Unlike the standard C library ``tolower`` function, this only
/// recognizes standard ASCII letters and ignores the locale, returning
/// all non-ASCII characters unchanged, even if they are lower case
/// letters in a particular character set. Also unlike the standard
/// library function, this takes and returns a char, not an int, so
/// don't call it on `EOF` but no need to worry about casting to `guchar`
/// before passing a possibly non-ASCII character in.
extern fn g_ascii_tolower(p_c: u8) u8;
pub const asciiTolower = g_ascii_tolower;

/// Convert a character to ASCII upper case. If the character is not an
/// ASCII lower case letter, it is returned unchanged.
///
/// Unlike the standard C library ``toupper`` function, this only
/// recognizes standard ASCII letters and ignores the locale, returning
/// all non-ASCII characters unchanged, even if they are upper case
/// letters in a particular character set. Also unlike the standard
/// library function, this takes and returns a char, not an int, so
/// don't call it on `EOF` but no need to worry about casting to `guchar`
/// before passing a possibly non-ASCII character in.
extern fn g_ascii_toupper(p_c: u8) u8;
pub const asciiToupper = g_ascii_toupper;

/// Determines the numeric value of a character as a hexadecimal digit. If the
/// character is not a hex digit according to `glib.asciiIsxdigit`,
/// `-1` is returned.
///
/// Differs from `glib.unicharXdigitValue` because it takes a char, so
/// there's no worry about sign extension if characters are signed.
///
/// Differs from `glib.unicharXdigitValue` because it takes a char, so
/// there's no worry about sign extension if characters are signed.
extern fn g_ascii_xdigit_value(p_c: u8) c_int;
pub const asciiXdigitValue = g_ascii_xdigit_value;

extern fn g_assert_warning(p_log_domain: [*:0]const u8, p_file: [*:0]const u8, p_line: c_int, p_pretty_function: [*:0]const u8, p_expression: [*:0]const u8) void;
pub const assertWarning = g_assert_warning;

extern fn g_assertion_message(p_domain: [*:0]const u8, p_file: [*:0]const u8, p_line: c_int, p_func: [*:0]const u8, p_message: [*:0]const u8) void;
pub const assertionMessage = g_assertion_message;

extern fn g_assertion_message_cmpint(p_domain: [*:0]const u8, p_file: [*:0]const u8, p_line: c_int, p_func: [*:0]const u8, p_expr: [*:0]const u8, p_arg1: u64, p_cmp: [*:0]const u8, p_arg2: u64, p_numtype: u8) void;
pub const assertionMessageCmpint = g_assertion_message_cmpint;

extern fn g_assertion_message_cmpnum(p_domain: [*:0]const u8, p_file: [*:0]const u8, p_line: c_int, p_func: [*:0]const u8, p_expr: [*:0]const u8, p_arg1: c_longdouble, p_cmp: [*:0]const u8, p_arg2: c_longdouble, p_numtype: u8) void;
pub const assertionMessageCmpnum = g_assertion_message_cmpnum;

extern fn g_assertion_message_cmpstr(p_domain: [*:0]const u8, p_file: [*:0]const u8, p_line: c_int, p_func: [*:0]const u8, p_expr: [*:0]const u8, p_arg1: [*:0]const u8, p_cmp: [*:0]const u8, p_arg2: [*:0]const u8) void;
pub const assertionMessageCmpstr = g_assertion_message_cmpstr;

extern fn g_assertion_message_cmpstrv(p_domain: [*:0]const u8, p_file: [*:0]const u8, p_line: c_int, p_func: [*:0]const u8, p_expr: [*:0]const u8, p_arg1: *const [*:0]const u8, p_arg2: *const [*:0]const u8, p_first_wrong_idx: usize) void;
pub const assertionMessageCmpstrv = g_assertion_message_cmpstrv;

extern fn g_assertion_message_error(p_domain: [*:0]const u8, p_file: [*:0]const u8, p_line: c_int, p_func: [*:0]const u8, p_expr: [*:0]const u8, p_error: *const glib.Error, p_error_domain: glib.Quark, p_error_code: c_int) void;
pub const assertionMessageError = g_assertion_message_error;

/// Internal function used to print messages from the public
/// `g_assert` and `g_assert_not_reached` macros.
extern fn g_assertion_message_expr(p_domain: ?[*:0]const u8, p_file: [*:0]const u8, p_line: c_int, p_func: [*:0]const u8, p_expr: ?[*:0]const u8) void;
pub const assertionMessageExpr = g_assertion_message_expr;

/// Specifies a function to be called at normal program termination.
///
/// Since GLib 2.8.2, on Windows `glib.atexit` actually is a preprocessor
/// macro that maps to a call to the `atexit` function in the C
/// library. This means that in case the code that calls `glib.atexit`,
/// i.e. `atexit`, is in a DLL, the function will be called when the
/// DLL is detached from the program. This typically makes more sense
/// than that the function is called when the GLib DLL is detached,
/// which happened earlier when `glib.atexit` was a function in the GLib
/// DLL.
///
/// The behaviour of `atexit` in the context of dynamically loaded
/// modules is not formally specified and varies wildly.
///
/// On POSIX systems, calling `glib.atexit` (or `atexit`) in a dynamically
/// loaded module which is unloaded before the program terminates might
/// well cause a crash at program exit.
///
/// Some POSIX systems implement `atexit` like Windows, and have each
/// dynamically loaded module maintain an own atexit chain that is
/// called when the module is unloaded.
///
/// On other POSIX systems, before a dynamically loaded module is
/// unloaded, the registered atexit functions (if any) residing in that
/// module are called, regardless where the code that registered them
/// resided. This is presumably the most robust approach.
///
/// As can be seen from the above, for portability it's best to avoid
/// calling `glib.atexit` (or `atexit`) except in the main executable of a
/// program.
extern fn g_atexit(p_func: glib.VoidFunc) void;
pub const atexit = g_atexit;

/// Atomically adds `val` to the value of `atomic`.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic += val; return tmp; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// Before version 2.30, this function did not return a value
/// (but `glib.atomicIntExchangeAndAdd` did, and had the same meaning).
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_int_add(p_atomic: *c_int, p_val: c_int) c_int;
pub const atomicIntAdd = g_atomic_int_add;

/// Performs an atomic bitwise 'and' of the value of `atomic` and `val`,
/// storing the result back in `atomic`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic &= val; return tmp; }`.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_int_and(p_atomic: *c_uint, p_val: c_uint) c_uint;
pub const atomicIntAnd = g_atomic_int_and;

/// Compares `atomic` to `oldval` and, if equal, sets it to `newval`.
/// If `atomic` was not equal to `oldval` then no change occurs.
///
/// This compare and exchange is done atomically.
///
/// Think of this operation as an atomic version of
/// `{ if (*atomic == oldval) { *atomic = newval; return TRUE; } else return FALSE; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_int_compare_and_exchange(p_atomic: *c_int, p_oldval: c_int, p_newval: c_int) c_int;
pub const atomicIntCompareAndExchange = g_atomic_int_compare_and_exchange;

/// Compares `atomic` to `oldval` and, if equal, sets it to `newval`.
/// If `atomic` was not equal to `oldval` then no change occurs.
/// In any case the value of `atomic` before this operation is stored in `preval`.
///
/// This compare and exchange is done atomically.
///
/// Think of this operation as an atomic version of
/// `{ *preval = *atomic; if (*atomic == oldval) { *atomic = newval; return TRUE; } else return FALSE; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// See also `glib.atomicIntCompareAndExchange`
extern fn g_atomic_int_compare_and_exchange_full(p_atomic: *c_int, p_oldval: c_int, p_newval: c_int, p_preval: *c_int) c_int;
pub const atomicIntCompareAndExchangeFull = g_atomic_int_compare_and_exchange_full;

/// Decrements the value of `atomic` by 1.
///
/// Think of this operation as an atomic version of
/// `{ *atomic -= 1; return (*atomic == 0); }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_int_dec_and_test(p_atomic: *c_int) c_int;
pub const atomicIntDecAndTest = g_atomic_int_dec_and_test;

/// Sets the `atomic` to `newval` and returns the old value from `atomic`.
///
/// This exchange is done atomically.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic = val; return tmp; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
extern fn g_atomic_int_exchange(p_atomic: *c_int, p_newval: c_int) c_int;
pub const atomicIntExchange = g_atomic_int_exchange;

/// This function existed before `glib.atomicIntAdd` returned the prior
/// value of the integer (which it now does).  It is retained only for
/// compatibility reasons.  Don't use this function in new code.
extern fn g_atomic_int_exchange_and_add(p_atomic: *c_int, p_val: c_int) c_int;
pub const atomicIntExchangeAndAdd = g_atomic_int_exchange_and_add;

/// Gets the current value of `atomic`.
///
/// This call acts as a full compiler and hardware
/// memory barrier (before the get).
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_int_get(p_atomic: *c_int) c_int;
pub const atomicIntGet = g_atomic_int_get;

/// Increments the value of `atomic` by 1.
///
/// Think of this operation as an atomic version of `{ *atomic += 1; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_int_inc(p_atomic: *c_int) void;
pub const atomicIntInc = g_atomic_int_inc;

/// Performs an atomic bitwise 'or' of the value of `atomic` and `val`,
/// storing the result back in `atomic`.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic |= val; return tmp; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_int_or(p_atomic: *c_uint, p_val: c_uint) c_uint;
pub const atomicIntOr = g_atomic_int_or;

/// Sets the value of `atomic` to `newval`.
///
/// This call acts as a full compiler and hardware
/// memory barrier (after the set).
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_int_set(p_atomic: *c_int, p_newval: c_int) void;
pub const atomicIntSet = g_atomic_int_set;

/// Performs an atomic bitwise 'xor' of the value of `atomic` and `val`,
/// storing the result back in `atomic`.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic ^= val; return tmp; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_int_xor(p_atomic: *c_uint, p_val: c_uint) c_uint;
pub const atomicIntXor = g_atomic_int_xor;

/// Atomically adds `val` to the value of `atomic`.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic += val; return tmp; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
///
/// In GLib 2.80, the return type was changed from `gssize` to `gintptr` to add
/// support for platforms with 128-bit pointers. This should not affect existing
/// code.
extern fn g_atomic_pointer_add(p_atomic: *anyopaque, p_val: isize) isize;
pub const atomicPointerAdd = g_atomic_pointer_add;

/// Performs an atomic bitwise 'and' of the value of `atomic` and `val`,
/// storing the result back in `atomic`.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic &= val; return tmp; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
///
/// In GLib 2.80, the return type was changed from `gsize` to `guintptr` to add
/// support for platforms with 128-bit pointers. This should not affect existing
/// code.
extern fn g_atomic_pointer_and(p_atomic: *anyopaque, p_val: usize) usize;
pub const atomicPointerAnd = g_atomic_pointer_and;

/// Compares `atomic` to `oldval` and, if equal, sets it to `newval`.
/// If `atomic` was not equal to `oldval` then no change occurs.
///
/// This compare and exchange is done atomically.
///
/// Think of this operation as an atomic version of
/// `{ if (*atomic == oldval) { *atomic = newval; return TRUE; } else return FALSE; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_pointer_compare_and_exchange(p_atomic: *anyopaque, p_oldval: ?*anyopaque, p_newval: ?*anyopaque) c_int;
pub const atomicPointerCompareAndExchange = g_atomic_pointer_compare_and_exchange;

/// Compares `atomic` to `oldval` and, if equal, sets it to `newval`.
/// If `atomic` was not equal to `oldval` then no change occurs.
/// In any case the value of `atomic` before this operation is stored in `preval`.
///
/// This compare and exchange is done atomically.
///
/// Think of this operation as an atomic version of
/// `{ *preval = *atomic; if (*atomic == oldval) { *atomic = newval; return TRUE; } else return FALSE; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// See also `glib.atomicPointerCompareAndExchange`
extern fn g_atomic_pointer_compare_and_exchange_full(p_atomic: *anyopaque, p_oldval: ?*anyopaque, p_newval: ?*anyopaque, p_preval: *anyopaque) c_int;
pub const atomicPointerCompareAndExchangeFull = g_atomic_pointer_compare_and_exchange_full;

/// Sets the `atomic` to `newval` and returns the old value from `atomic`.
///
/// This exchange is done atomically.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic = val; return tmp; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
extern fn g_atomic_pointer_exchange(p_atomic: ?*anyopaque, p_newval: ?*anyopaque) ?*anyopaque;
pub const atomicPointerExchange = g_atomic_pointer_exchange;

/// Gets the current value of `atomic`.
///
/// This call acts as a full compiler and hardware
/// memory barrier (before the get).
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_pointer_get(p_atomic: *anyopaque) ?*anyopaque;
pub const atomicPointerGet = g_atomic_pointer_get;

/// Performs an atomic bitwise 'or' of the value of `atomic` and `val`,
/// storing the result back in `atomic`.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic |= val; return tmp; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
///
/// In GLib 2.80, the return type was changed from `gsize` to `guintptr` to add
/// support for platforms with 128-bit pointers. This should not affect existing
/// code.
extern fn g_atomic_pointer_or(p_atomic: *anyopaque, p_val: usize) usize;
pub const atomicPointerOr = g_atomic_pointer_or;

/// Sets the value of `atomic` to `newval`.
///
/// This call acts as a full compiler and hardware
/// memory barrier (after the set).
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
extern fn g_atomic_pointer_set(p_atomic: *anyopaque, p_newval: ?*anyopaque) void;
pub const atomicPointerSet = g_atomic_pointer_set;

/// Performs an atomic bitwise 'xor' of the value of `atomic` and `val`,
/// storing the result back in `atomic`.
///
/// Think of this operation as an atomic version of
/// `{ tmp = *atomic; *atomic ^= val; return tmp; }`.
///
/// This call acts as a full compiler and hardware memory barrier.
///
/// While `atomic` has a `volatile` qualifier, this is a historical artifact and
/// the pointer passed to it should not be `volatile`.
///
/// In GLib 2.80, the return type was changed from `gsize` to `guintptr` to add
/// support for platforms with 128-bit pointers. This should not affect existing
/// code.
extern fn g_atomic_pointer_xor(p_atomic: *anyopaque, p_val: usize) usize;
pub const atomicPointerXor = g_atomic_pointer_xor;

/// Atomically acquires a reference on the data pointed by `mem_block`.
extern fn g_atomic_rc_box_acquire(p_mem_block: *anyopaque) *anyopaque;
pub const atomicRcBoxAcquire = g_atomic_rc_box_acquire;

/// Allocates `block_size` bytes of memory, and adds atomic
/// reference counting semantics to it.
///
/// The data will be freed when its reference count drops to
/// zero.
///
/// The allocated data is guaranteed to be suitably aligned for any
/// built-in type.
extern fn g_atomic_rc_box_alloc(p_block_size: usize) *anyopaque;
pub const atomicRcBoxAlloc = g_atomic_rc_box_alloc;

/// Allocates `block_size` bytes of memory, and adds atomic
/// reference counting semantics to it.
///
/// The contents of the returned data is set to zero.
///
/// The data will be freed when its reference count drops to
/// zero.
///
/// The allocated data is guaranteed to be suitably aligned for any
/// built-in type.
extern fn g_atomic_rc_box_alloc0(p_block_size: usize) *anyopaque;
pub const atomicRcBoxAlloc0 = g_atomic_rc_box_alloc0;

/// Allocates a new block of data with atomic reference counting
/// semantics, and copies `block_size` bytes of `mem_block`
/// into it.
extern fn g_atomic_rc_box_dup(p_block_size: usize, p_mem_block: *const anyopaque) *anyopaque;
pub const atomicRcBoxDup = g_atomic_rc_box_dup;

/// Retrieves the size of the reference counted data pointed by `mem_block`.
extern fn g_atomic_rc_box_get_size(p_mem_block: *anyopaque) usize;
pub const atomicRcBoxGetSize = g_atomic_rc_box_get_size;

/// Atomically releases a reference on the data pointed by `mem_block`.
///
/// If the reference was the last one, it will free the
/// resources allocated for `mem_block`.
extern fn g_atomic_rc_box_release(p_mem_block: *anyopaque) void;
pub const atomicRcBoxRelease = g_atomic_rc_box_release;

/// Atomically releases a reference on the data pointed by `mem_block`.
///
/// If the reference was the last one, it will call `clear_func`
/// to clear the contents of `mem_block`, and then will free the
/// resources allocated for `mem_block`.
///
/// Note that implementing weak references via `clear_func` is not thread-safe:
/// clearing a pointer to the memory from the callback can race with another
/// thread trying to access it as `mem_block` already has a reference count of 0
/// when the callback is called and will be freed.
extern fn g_atomic_rc_box_release_full(p_mem_block: *anyopaque, p_clear_func: glib.DestroyNotify) void;
pub const atomicRcBoxReleaseFull = g_atomic_rc_box_release_full;

/// Atomically compares the current value of `arc` with `val`.
extern fn g_atomic_ref_count_compare(p_arc: *c_int, p_val: c_int) c_int;
pub const atomicRefCountCompare = g_atomic_ref_count_compare;

/// Atomically decreases the reference count.
///
/// If `TRUE` is returned, the reference count reached 0. After this point, `arc`
/// is an undefined state and must be reinitialized with
/// `glib.atomicRefCountInit` to be used again.
extern fn g_atomic_ref_count_dec(p_arc: *c_int) c_int;
pub const atomicRefCountDec = g_atomic_ref_count_dec;

/// Atomically increases the reference count.
extern fn g_atomic_ref_count_inc(p_arc: *c_int) void;
pub const atomicRefCountInc = g_atomic_ref_count_inc;

/// Initializes a reference count variable to 1.
extern fn g_atomic_ref_count_init(p_arc: *c_int) void;
pub const atomicRefCountInit = g_atomic_ref_count_init;

/// Decode a sequence of Base-64 encoded text into binary data.  Note
/// that the returned binary data is not necessarily zero-terminated,
/// so it should not be used as a character string.
extern fn g_base64_decode(p_text: [*:0]const u8, p_out_len: *usize) [*]u8;
pub const base64Decode = g_base64_decode;

/// Decode a sequence of Base-64 encoded text into binary data
/// by overwriting the input data.
extern fn g_base64_decode_inplace(p_text: *[*]u8, p_out_len: *usize) *u8;
pub const base64DecodeInplace = g_base64_decode_inplace;

/// Incrementally decode a sequence of binary data from its Base-64 stringified
/// representation. By calling this function multiple times you can convert
/// data in chunks to avoid having to have the full encoded data in memory.
///
/// The output buffer must be large enough to fit all the data that will
/// be written to it. Since base64 encodes 3 bytes in 4 chars you need
/// at least: (`len` / 4) * 3 + 3 bytes (+ 3 may be needed in case of non-zero
/// state).
extern fn g_base64_decode_step(p_in: [*]const u8, p_len: usize, p_out: *[*]u8, p_state: *c_int, p_save: *c_uint) usize;
pub const base64DecodeStep = g_base64_decode_step;

/// Encode a sequence of binary data into its Base-64 stringified
/// representation.
extern fn g_base64_encode(p_data: ?[*]const u8, p_len: usize) [*:0]u8;
pub const base64Encode = g_base64_encode;

/// Flush the status from a sequence of calls to `glib.base64EncodeStep`.
///
/// The output buffer must be large enough to fit all the data that will
/// be written to it. It will need up to 4 bytes, or up to 5 bytes if
/// line-breaking is enabled.
///
/// The `out` array will not be automatically nul-terminated.
extern fn g_base64_encode_close(p_break_lines: c_int, p_out: *[*]u8, p_state: *c_int, p_save: *c_int) usize;
pub const base64EncodeClose = g_base64_encode_close;

/// Incrementally encode a sequence of binary data into its Base-64 stringified
/// representation. By calling this function multiple times you can convert
/// data in chunks to avoid having to have the full encoded data in memory.
///
/// When all of the data has been converted you must call
/// `glib.base64EncodeClose` to flush the saved state.
///
/// The output buffer must be large enough to fit all the data that will
/// be written to it. Due to the way base64 encodes you will need
/// at least: (`len` / 3 + 1) * 4 + 4 bytes (+ 4 may be needed in case of
/// non-zero state). If you enable line-breaking you will need at least:
/// ((`len` / 3 + 1) * 4 + 4) / 76 + 1 bytes of extra space.
///
/// `break_lines` is typically used when putting base64-encoded data in emails.
/// It breaks the lines at 76 columns instead of putting all of the text on
/// the same line. This avoids problems with long lines in the email system.
/// Note however that it breaks the lines with `LF` characters, not
/// `CR LF` sequences, so the result cannot be passed directly to SMTP
/// or certain other protocols.
extern fn g_base64_encode_step(p_in: [*]const u8, p_len: usize, p_break_lines: c_int, p_out: *[*]u8, p_state: *c_int, p_save: *c_int) usize;
pub const base64EncodeStep = g_base64_encode_step;

/// Gets the name of the file without any leading directory
/// components. It returns a pointer into the given file name
/// string.
extern fn g_basename(p_file_name: [*:0]const u8) [*:0]const u8;
pub const basename = g_basename;

/// Sets the indicated `lock_bit` in `address`.  If the bit is already
/// set, this call will block until `glib.bitUnlock` unsets the
/// corresponding bit.
///
/// Attempting to lock on two different bits within the same integer is
/// not supported and will very probably cause deadlocks.
///
/// The value of the bit that is set is (1u << `bit`).  If `bit` is not
/// between 0 and 31 then the result is undefined.
///
/// This function accesses `address` atomically.  All other accesses to
/// `address` must be atomic in order for this function to work
/// reliably. While `address` has a `volatile` qualifier, this is a historical
/// artifact and the argument passed to it should not be `volatile`.
extern fn g_bit_lock(p_address: *c_int, p_lock_bit: c_int) void;
pub const bitLock = g_bit_lock;

/// Find the position of the first bit set in `mask`, searching
/// from (but not including) `nth_bit` upwards. Bits are numbered
/// from 0 (least significant) to sizeof(`gulong`) * 8 - 1 (31 or 63,
/// usually). To start searching from the 0th bit, set `nth_bit` to -1.
extern fn g_bit_nth_lsf(p_mask: c_ulong, p_nth_bit: c_int) c_int;
pub const bitNthLsf = g_bit_nth_lsf;

/// Find the position of the first bit set in `mask`, searching
/// from (but not including) `nth_bit` downwards. Bits are numbered
/// from 0 (least significant) to sizeof(`gulong`) * 8 - 1 (31 or 63,
/// usually). To start searching from the last bit, set `nth_bit` to
/// -1 or GLIB_SIZEOF_LONG * 8.
extern fn g_bit_nth_msf(p_mask: c_ulong, p_nth_bit: c_int) c_int;
pub const bitNthMsf = g_bit_nth_msf;

/// Gets the number of bits used to hold `number`,
/// e.g. if `number` is 4, 3 bits are needed.
extern fn g_bit_storage(p_number: c_ulong) c_uint;
pub const bitStorage = g_bit_storage;

/// Sets the indicated `lock_bit` in `address`, returning `TRUE` if
/// successful.  If the bit is already set, returns `FALSE` immediately.
///
/// Attempting to lock on two different bits within the same integer is
/// not supported.
///
/// The value of the bit that is set is (1u << `bit`).  If `bit` is not
/// between 0 and 31 then the result is undefined.
///
/// This function accesses `address` atomically.  All other accesses to
/// `address` must be atomic in order for this function to work
/// reliably. While `address` has a `volatile` qualifier, this is a historical
/// artifact and the argument passed to it should not be `volatile`.
extern fn g_bit_trylock(p_address: *c_int, p_lock_bit: c_int) c_int;
pub const bitTrylock = g_bit_trylock;

/// Clears the indicated `lock_bit` in `address`.  If another thread is
/// currently blocked in `glib.bitLock` on this same bit then it will be
/// woken up.
///
/// This function accesses `address` atomically.  All other accesses to
/// `address` must be atomic in order for this function to work
/// reliably. While `address` has a `volatile` qualifier, this is a historical
/// artifact and the argument passed to it should not be `volatile`.
extern fn g_bit_unlock(p_address: *c_int, p_lock_bit: c_int) void;
pub const bitUnlock = g_bit_unlock;

extern fn g_blow_chunks() void;
pub const blowChunks = g_blow_chunks;

/// Creates a filename from a series of elements using the correct
/// separator for the current platform.
///
/// On Unix, this function behaves identically to `g_build_path
/// (G_DIR_SEPARATOR_S, first_element, ....)`.
///
/// On Windows, it takes into account that either the backslash
/// (`\` or slash (`/`) can be used as separator in filenames, but
/// otherwise behaves as on UNIX. When file pathname separators need
/// to be inserted, the one that last previously occurred in the
/// parameters (reading from left to right) is used.
///
/// No attempt is made to force the resulting filename to be an absolute
/// path. If the first element is a relative path, the result will
/// be a relative path.
///
/// If you are building a path programmatically you may want to use
/// `glib.PathBuf` instead.
extern fn g_build_filename(p_first_element: [*:0]const u8, ...) [*:0]u8;
pub const buildFilename = g_build_filename;

/// Creates a filename from a list of elements using the correct
/// separator for the current platform.
///
/// Behaves exactly like `glib.buildFilename`, but takes the path elements
/// as a va_list.
///
/// This function is mainly meant for implementing other variadic arguments
/// functions.
extern fn g_build_filename_valist(p_first_element: [*:0]const u8, p_args: *std.builtin.VaList) [*:0]u8;
pub const buildFilenameValist = g_build_filename_valist;

/// Creates a filename from a vector of elements using the correct
/// separator for the current platform.
///
/// This function behaves exactly like `glib.buildFilename`, but takes the path
/// elements as a string array, instead of varargs. This function is mainly
/// meant for language bindings.
///
/// If you are building a path programmatically you may want to use
/// `glib.PathBuf` instead.
extern fn g_build_filenamev(p_args: [*:null]?[*:0]u8) [*:0]u8;
pub const buildFilenamev = g_build_filenamev;

/// Creates a path from a series of elements using `separator` as the
/// separator between elements.
///
/// At the boundary between two elements, any trailing occurrences of
/// separator in the first element, or leading occurrences of separator
/// in the second element are removed and exactly one copy of the
/// separator is inserted.
///
/// Empty elements are ignored.
///
/// The number of leading copies of the separator on the result is
/// the same as the number of leading copies of the separator on
/// the first non-empty element.
///
/// The number of trailing copies of the separator on the result is
/// the same as the number of trailing copies of the separator on
/// the last non-empty element. (Determination of the number of
/// trailing copies is done without stripping leading copies, so
/// if the separator is `ABA`, then `ABABA` has 1 trailing copy.)
///
/// However, if there is only a single non-empty element, and there
/// are no characters in that element not part of the leading or
/// trailing separators, then the result is exactly the original value
/// of that element.
///
/// Other than for determination of the number of leading and trailing
/// copies of the separator, elements consisting only of copies
/// of the separator are ignored.
extern fn g_build_path(p_separator: [*:0]const u8, p_first_element: [*:0]const u8, ...) [*:0]u8;
pub const buildPath = g_build_path;

/// Behaves exactly like `glib.buildPath`, but takes the path elements
/// as a string array, instead of variadic arguments.
///
/// This function is mainly meant for language bindings.
extern fn g_build_pathv(p_separator: [*:0]const u8, p_args: [*:null]?[*:0]u8) [*:0]u8;
pub const buildPathv = g_build_pathv;

/// Gets the canonical file name from `filename`. All triple slashes are turned into
/// single slashes, and all `..` and `.`s resolved against `relative_to`.
///
/// Symlinks are not followed, and the returned path is guaranteed to be absolute.
///
/// If `filename` is an absolute path, `relative_to` is ignored. Otherwise,
/// `relative_to` will be prepended to `filename` to make it absolute. `relative_to`
/// must be an absolute path, or `NULL`. If `relative_to` is `NULL`, it'll fallback
/// to `glib.getCurrentDir`.
///
/// This function never fails, and will canonicalize file paths even if they don't
/// exist.
///
/// No file system I/O is done.
extern fn g_canonicalize_filename(p_filename: [*:0]const u8, p_relative_to: ?[*:0]const u8) [*:0]u8;
pub const canonicalizeFilename = g_canonicalize_filename;

/// A wrapper for the POSIX `chdir` function. The function changes the
/// current directory of the process to `path`.
///
/// See your C library manual for more details about `chdir`.
extern fn g_chdir(p_path: [*:0]const u8) c_int;
pub const chdir = g_chdir;

/// Checks that the GLib library in use is compatible with the
/// given version.
///
/// Generally you would pass in the constants `GLIB_MAJOR_VERSION`,
/// `GLIB_MINOR_VERSION`, `GLIB_MICRO_VERSION` as the three arguments
/// to this function; that produces a check that the library in use
/// is compatible with the version of GLib the application or module
/// was compiled against.
///
/// Compatibility is defined by two things: first the version
/// of the running library is newer than the version
/// ``required_major`.required_minor.`required_micro``. Second
/// the running library must be binary compatible with the
/// version ``required_major`.`required_minor`.`required_micro``
/// (same major version.)
extern fn glib_check_version(p_required_major: c_uint, p_required_minor: c_uint, p_required_micro: c_uint) ?[*:0]const u8;
pub const checkVersion = glib_check_version;

/// Sets a function to be called when the child indicated by `pid`
/// exits, at a default priority, `glib.PRIORITY_DEFAULT`.
///
/// If you obtain `pid` from `glib.spawnAsync` or
/// `glib.spawnAsyncWithPipes` you will need to pass
/// `G_SPAWN_DO_NOT_REAP_CHILD` as flag to the spawn function for the child
/// watching to work.
///
/// Note that on platforms where `glib.Pid` must be explicitly closed
/// (see `glib.spawnClosePid`) `pid` must not be closed while the
/// source is still active. Typically, you will want to call
/// `glib.spawnClosePid` in the callback function for the source.
///
/// GLib supports only a single callback per process id.
/// On POSIX platforms, the same restrictions mentioned for
/// `glib.childWatchSourceNew` apply to this function.
///
/// This internally creates a main loop source using
/// `glib.childWatchSourceNew` and attaches it to the main loop context
/// using `glib.Source.attach`. You can do these steps manually if you
/// need greater control.
extern fn g_child_watch_add(p_pid: glib.Pid, p_function: glib.ChildWatchFunc, p_data: ?*anyopaque) c_uint;
pub const childWatchAdd = g_child_watch_add;

/// Sets a function to be called when the child indicated by `pid`
/// exits, at the priority `priority`.
///
/// If you obtain `pid` from `glib.spawnAsync` or
/// `glib.spawnAsyncWithPipes` you will need to pass
/// `G_SPAWN_DO_NOT_REAP_CHILD` as flag to the spawn function for the child
/// watching to work.
///
/// In many programs, you will want to call `glib.spawnCheckWaitStatus`
/// in the callback to determine whether or not the child exited
/// successfully.
///
/// Also, note that on platforms where `glib.Pid` must be explicitly closed
/// (see `glib.spawnClosePid`) `pid` must not be closed while the source
/// is still active.  Typically, you should invoke `glib.spawnClosePid`
/// in the callback function for the source.
///
/// GLib supports only a single callback per process id.
/// On POSIX platforms, the same restrictions mentioned for
/// `glib.childWatchSourceNew` apply to this function.
///
/// This internally creates a main loop source using
/// `glib.childWatchSourceNew` and attaches it to the main loop context
/// using `glib.Source.attach`. You can do these steps manually if you
/// need greater control.
extern fn g_child_watch_add_full(p_priority: c_int, p_pid: glib.Pid, p_function: glib.ChildWatchFunc, p_data: ?*anyopaque, p_notify: ?glib.DestroyNotify) c_uint;
pub const childWatchAddFull = g_child_watch_add_full;

/// Creates a new child_watch source.
///
/// The source will not initially be associated with any
/// `glib.MainContext` and must be added to one with
/// `glib.Source.attach` before it will be executed.
///
/// Note that child watch sources can only be used in conjunction with
/// `g_spawn...` when the `G_SPAWN_DO_NOT_REAP_CHILD` flag is used.
///
/// Note that on platforms where `glib.Pid` must be explicitly closed
/// (see `glib.spawnClosePid`) `pid` must not be closed while the
/// source is still active. Typically, you will want to call
/// `glib.spawnClosePid` in the callback function for the source.
///
/// On POSIX platforms, the following restrictions apply to this API
/// due to limitations in POSIX process interfaces:
///
/// * `pid` must be a child of this process
/// * `pid` must be positive
/// * the application must not call `waitpid` with a non-positive
///   first argument, for instance in another thread
/// * the application must not wait for `pid` to exit by any other
///   mechanism, including `waitpid(pid, ...)` or a second child-watch
///   source for the same `pid`
/// * the application must not ignore `SIGCHLD`
/// * Before 2.78, the application could not send a signal (``kill``) to the
///   watched `pid` in a race free manner. Since 2.78, you can do that while the
///   associated `glib.MainContext` is acquired.
/// * Before 2.78, even after destroying the `glib.Source`, you could not
///   be sure that `pid` wasn't already reaped. Hence, it was also not
///   safe to ``kill`` or ``waitpid`` on the process ID after the child watch
///   source was gone. Destroying the source before it fired made it
///   impossible to reliably reap the process.
///
/// If any of those conditions are not met, this and related APIs will
/// not work correctly. This can often be diagnosed via a GLib warning
/// stating that `ECHILD` was received by `waitpid`.
///
/// Calling `waitpid` for specific processes other than `pid` remains a
/// valid thing to do.
extern fn g_child_watch_source_new(p_pid: glib.Pid) *glib.Source;
pub const childWatchSourceNew = g_child_watch_source_new;

/// A wrapper for the POSIX `chmod` function. The `chmod` function is
/// used to set the permissions of a file system object.
///
/// On Windows the file protection mechanism is not at all POSIX-like,
/// and the underlying `chmod` function in the C library just sets or
/// clears the FAT-style READONLY attribute. It does not touch any
/// ACL. Software that needs to manage file permissions on Windows
/// exactly should use the Win32 API.
///
/// See your C library manual for more details about `chmod`.
extern fn g_chmod(p_filename: [*:0]const u8, p_mode: c_int) c_int;
pub const chmod = g_chmod;

/// If `err` or `*err` is `NULL`, does nothing. Otherwise,
/// calls `glib.Error.free` on `*err` and sets `*err` to `NULL`.
extern fn g_clear_error(p_error: ?*?*glib.Error) void;
pub const clearError = g_clear_error;

/// Clears a numeric handler, such as a `glib.Source` ID.
///
/// `tag_ptr` must be a valid pointer to the variable holding the handler.
///
/// If the ID is zero then this function does nothing.
/// Otherwise, `clear_func` is called with the ID as a parameter, and the tag is
/// set to zero.
///
/// A macro is also included that allows this function to be used without
/// pointer casts.
extern fn g_clear_handle_id(p_tag_ptr: *c_uint, p_clear_func: glib.ClearHandleFunc) void;
pub const clearHandleId = g_clear_handle_id;

/// Clears a pointer to a `glib.List`, freeing it and, optionally, freeing its elements using `destroy`.
///
/// `list_ptr` must be a valid pointer. If `list_ptr` points to a null `glib.List`, this does nothing.
extern fn g_clear_list(p_list_ptr: **glib.List, p_destroy: ?glib.DestroyNotify) void;
pub const clearList = g_clear_list;

/// Clears a reference to a variable.
///
/// `pp` must not be `NULL`.
///
/// If the reference is `NULL` then this function does nothing.
/// Otherwise, the variable is destroyed using `destroy` and the
/// pointer is set to `NULL`.
///
/// A macro is also included that allows this function to be used without
/// pointer casts. This will mask any warnings about incompatible function types
/// or calling conventions, so you must ensure that your `destroy` function is
/// compatible with being called as `glib.DestroyNotify` using the
/// standard calling convention for the platform that GLib was compiled for;
/// otherwise the program will experience undefined behaviour.
///
/// Examples of this kind of undefined behaviour include using many Windows Win32
/// APIs, as well as many if not all OpenGL and Vulkan calls on 32-bit Windows,
/// which typically use the `__stdcall` calling convention rather than the
/// `__cdecl` calling convention.
///
/// The affected functions can be used by wrapping them in a
/// `glib.DestroyNotify` that is declared with the standard calling
/// convention:
///
/// ```c
/// // Wrapper needed to avoid mismatched calling conventions on Windows
/// static void
/// destroy_sync (void *sync)
/// {
///   glDeleteSync (sync);
/// }
///
/// // …
///
/// g_clear_pointer (&sync, destroy_sync);
/// ```
extern fn g_clear_pointer(p_pp: *anyopaque, p_destroy: glib.DestroyNotify) void;
pub const clearPointer = g_clear_pointer;

/// Clears a pointer to a `glib.SList`, freeing it and, optionally, freeing its elements using `destroy`.
///
/// `slist_ptr` must be a valid pointer. If `slist_ptr` points to a null `glib.SList`, this does nothing.
extern fn g_clear_slist(p_slist_ptr: **glib.SList, p_destroy: ?glib.DestroyNotify) void;
pub const clearSlist = g_clear_slist;

/// This wraps the `close` call. In case of error, `errno` will be
/// preserved, but the error will also be stored as a `glib.Error` in `error`.
/// In case of success, `errno` is undefined.
///
/// Besides using `glib.Error`, there is another major reason to prefer this
/// function over the call provided by the system; on Unix, it will
/// attempt to correctly handle `EINTR`, which has platform-specific
/// semantics.
///
/// It is a bug to call this function with an invalid file descriptor.
///
/// On POSIX platforms since GLib 2.76, this function is async-signal safe
/// if (and only if) `error` is `NULL` and `fd` is a valid open file descriptor.
/// This makes it safe to call from a signal handler or a `glib.SpawnChildSetupFunc`
/// under those conditions.
/// See [`signal(7)`](man:signal(7)) and
/// [`signal-safety(7)`](man:signal-safety(7)) for more details.
extern fn g_close(p_fd: c_int, p_error: ?*?*glib.Error) c_int;
pub const close = g_close;

/// Close every file descriptor equal to or greater than `lowfd`.
///
/// Typically `lowfd` will be 3, to leave standard input, standard output
/// and standard error open.
///
/// This is the same as Linux `close_range (lowfd, ~0U, 0)`,
/// but portable to other OSs and to older versions of Linux.
/// Equivalently, it is the same as BSD `closefrom (lowfd)`, but portable,
/// and async-signal-safe on all OSs.
///
/// This function is async-signal safe, making it safe to call from a
/// signal handler or a `glib.SpawnChildSetupFunc`, as long as `lowfd` is
/// non-negative.
/// See [`signal(7)`](man:signal(7)) and
/// [`signal-safety(7)`](man:signal-safety(7)) for more details.
extern fn g_closefrom(p_lowfd: c_int) c_int;
pub const closefrom = g_closefrom;

/// Computes the checksum for a binary `data`. This is a
/// convenience wrapper for `glib.Checksum.new`, `glib.Checksum.getString`
/// and `glib.Checksum.free`.
///
/// The hexadecimal string returned will be in lower case.
extern fn g_compute_checksum_for_bytes(p_checksum_type: glib.ChecksumType, p_data: *glib.Bytes) ?[*:0]u8;
pub const computeChecksumForBytes = g_compute_checksum_for_bytes;

/// Computes the checksum for a binary `data` of `length`. This is a
/// convenience wrapper for `glib.Checksum.new`, `glib.Checksum.getString`
/// and `glib.Checksum.free`.
///
/// The hexadecimal string returned will be in lower case.
extern fn g_compute_checksum_for_data(p_checksum_type: glib.ChecksumType, p_data: [*]const u8, p_length: usize) ?[*:0]u8;
pub const computeChecksumForData = g_compute_checksum_for_data;

/// Computes the checksum of a string.
///
/// The hexadecimal string returned will be in lower case.
extern fn g_compute_checksum_for_string(p_checksum_type: glib.ChecksumType, p_str: [*:0]const u8, p_length: isize) ?[*:0]u8;
pub const computeChecksumForString = g_compute_checksum_for_string;

/// Computes the HMAC for a binary `data`. This is a
/// convenience wrapper for `glib.Hmac.new`, `glib.Hmac.getString`
/// and `glib.Hmac.unref`.
///
/// The hexadecimal string returned will be in lower case.
extern fn g_compute_hmac_for_bytes(p_digest_type: glib.ChecksumType, p_key: *glib.Bytes, p_data: *glib.Bytes) [*:0]u8;
pub const computeHmacForBytes = g_compute_hmac_for_bytes;

/// Computes the HMAC for a binary `data` of `length`. This is a
/// convenience wrapper for `glib.Hmac.new`, `glib.Hmac.getString`
/// and `glib.Hmac.unref`.
///
/// The hexadecimal string returned will be in lower case.
extern fn g_compute_hmac_for_data(p_digest_type: glib.ChecksumType, p_key: [*]const u8, p_key_len: usize, p_data: [*]const u8, p_length: usize) [*:0]u8;
pub const computeHmacForData = g_compute_hmac_for_data;

/// Computes the HMAC for a string.
///
/// The hexadecimal string returned will be in lower case.
extern fn g_compute_hmac_for_string(p_digest_type: glib.ChecksumType, p_key: [*]const u8, p_key_len: usize, p_str: [*:0]const u8, p_length: isize) [*:0]u8;
pub const computeHmacForString = g_compute_hmac_for_string;

/// Converts a string from one character set to another.
///
/// Note that you should use `glib.iconv` for streaming conversions.
/// Despite the fact that `bytes_read` can return information about partial
/// characters, the g_convert_... functions are not generally suitable
/// for streaming. If the underlying converter maintains internal state,
/// then this won't be preserved across successive calls to `glib.convert`,
/// `glib.convertWithIconv` or `glib.convertWithFallback`. (An example of
/// this is the GNU C converter for CP1255 which does not emit a base
/// character until it knows that the next character is not a mark that
/// could combine with the base character.)
///
/// Using extensions such as "//TRANSLIT" may not work (or may not work
/// well) on many platforms.  Consider using `glib.strToAscii` instead.
extern fn g_convert(p_str: [*]const u8, p_len: isize, p_to_codeset: [*:0]const u8, p_from_codeset: [*:0]const u8, p_bytes_read: ?*usize, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) ?[*]u8;
pub const convert = g_convert;

extern fn g_convert_error_quark() glib.Quark;
pub const convertErrorQuark = g_convert_error_quark;

/// Converts a string from one character set to another, possibly
/// including fallback sequences for characters not representable
/// in the output. Note that it is not guaranteed that the specification
/// for the fallback sequences in `fallback` will be honored. Some
/// systems may do an approximate conversion from `from_codeset`
/// to `to_codeset` in their `iconv` functions,
/// in which case GLib will simply return that approximate conversion.
///
/// Note that you should use `glib.iconv` for streaming conversions.
/// Despite the fact that `bytes_read` can return information about partial
/// characters, the g_convert_... functions are not generally suitable
/// for streaming. If the underlying converter maintains internal state,
/// then this won't be preserved across successive calls to `glib.convert`,
/// `glib.convertWithIconv` or `glib.convertWithFallback`. (An example of
/// this is the GNU C converter for CP1255 which does not emit a base
/// character until it knows that the next character is not a mark that
/// could combine with the base character.)
extern fn g_convert_with_fallback(p_str: [*]const u8, p_len: isize, p_to_codeset: [*:0]const u8, p_from_codeset: [*:0]const u8, p_fallback: [*:0]const u8, p_bytes_read: ?*usize, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) ?[*]u8;
pub const convertWithFallback = g_convert_with_fallback;

/// Converts a string from one character set to another.
///
/// Note that you should use `glib.iconv` for streaming conversions.
/// Despite the fact that `bytes_read` can return information about partial
/// characters, the g_convert_... functions are not generally suitable
/// for streaming. If the underlying converter maintains internal state,
/// then this won't be preserved across successive calls to `glib.convert`,
/// `glib.convertWithIconv` or `glib.convertWithFallback`. (An example of
/// this is the GNU C converter for CP1255 which does not emit a base
/// character until it knows that the next character is not a mark that
/// could combine with the base character.)
///
/// Characters which are valid in the input character set, but which have no
/// representation in the output character set will result in a
/// `G_CONVERT_ERROR_ILLEGAL_SEQUENCE` error. This is in contrast to the `iconv`
/// specification, which leaves this behaviour implementation defined. Note that
/// this is the same error code as is returned for an invalid byte sequence in
/// the input character set. To get defined behaviour for conversion of
/// unrepresentable characters, use `glib.convertWithFallback`.
extern fn g_convert_with_iconv(p_str: [*]const u8, p_len: isize, p_converter: glib.IConv, p_bytes_read: ?*usize, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) ?[*]u8;
pub const convertWithIconv = g_convert_with_iconv;

/// A wrapper for the POSIX `creat` function. The `creat` function is
/// used to convert a pathname into a file descriptor, creating a file
/// if necessary.
///
/// On POSIX systems file descriptors are implemented by the operating
/// system. On Windows, it's the C library that implements `creat` and
/// file descriptors. The actual Windows API for opening files is
/// different, see MSDN documentation for `CreateFile`. The Win32 API
/// uses file handles, which are more randomish integers, not small
/// integers like file descriptors.
///
/// Because file descriptors are specific to the C library on Windows,
/// the file descriptor returned by this function makes sense only to
/// functions in the same C library. Thus if the GLib-using code uses a
/// different C library than GLib does, the file descriptor returned by
/// this function cannot be passed to C library functions like `write`
/// or `read`.
///
/// See your C library manual for more details about `creat`.
extern fn g_creat(p_filename: [*:0]const u8, p_mode: c_int) c_int;
pub const creat = g_creat;

/// Frees all the data elements of the datalist.
/// The data elements' destroy functions are called
/// if they have been set.
extern fn g_datalist_clear(p_datalist: **glib.Data) void;
pub const datalistClear = g_datalist_clear;

/// Calls the given function for each data element of the datalist. The
/// function is called with each data element's `glib.Quark` id and data,
/// together with the given `user_data` parameter. Note that this
/// function is NOT thread-safe. So unless `datalist` can be protected
/// from any modifications during invocation of this function, it should
/// not be called.
///
/// `func` can make changes to `datalist`, but the iteration will not
/// reflect changes made during the `glib.datalistForeach` call, other
/// than skipping over elements that are removed.
extern fn g_datalist_foreach(p_datalist: **glib.Data, p_func: glib.DataForeachFunc, p_user_data: ?*anyopaque) void;
pub const datalistForeach = g_datalist_foreach;

/// Gets a data element, using its string identifier. This is slower than
/// `glib.datalistIdGetData` because it compares strings.
extern fn g_datalist_get_data(p_datalist: **glib.Data, p_key: [*:0]const u8) ?*anyopaque;
pub const datalistGetData = g_datalist_get_data;

/// Gets flags values packed in together with the datalist.
/// See `glib.datalistSetFlags`.
extern fn g_datalist_get_flags(p_datalist: **glib.Data) c_uint;
pub const datalistGetFlags = g_datalist_get_flags;

/// This is a variant of `glib.datalistIdGetData` which
/// returns a 'duplicate' of the value. `dup_func` defines the
/// meaning of 'duplicate' in this context, it could e.g.
/// take a reference on a ref-counted object.
///
/// If the `key_id` is not set in the datalist then `dup_func`
/// will be called with a `NULL` argument.
///
/// Note that `dup_func` is called while the datalist is locked, so it
/// is not allowed to read or modify the datalist.
///
/// This function can be useful to avoid races when multiple
/// threads are using the same datalist and the same key.
extern fn g_datalist_id_dup_data(p_datalist: **glib.Data, p_key_id: glib.Quark, p_dup_func: ?glib.DuplicateFunc, p_user_data: ?*anyopaque) ?*anyopaque;
pub const datalistIdDupData = g_datalist_id_dup_data;

/// Retrieves the data element corresponding to `key_id`.
extern fn g_datalist_id_get_data(p_datalist: **glib.Data, p_key_id: glib.Quark) ?*anyopaque;
pub const datalistIdGetData = g_datalist_id_get_data;

/// Removes multiple keys from a datalist.
///
/// This is more efficient than calling `g_datalist_id_remove_data`
/// multiple times in a row.
///
/// Before 2.80, `n_keys` had to be not larger than 16.
/// Since 2.84, performance is improved for larger number of keys.
extern fn g_datalist_id_remove_multiple(p_datalist: **glib.Data, p_keys: [*]glib.Quark, p_n_keys: usize) void;
pub const datalistIdRemoveMultiple = g_datalist_id_remove_multiple;

/// Removes an element, without calling its destroy notification
/// function.
extern fn g_datalist_id_remove_no_notify(p_datalist: **glib.Data, p_key_id: glib.Quark) ?*anyopaque;
pub const datalistIdRemoveNoNotify = g_datalist_id_remove_no_notify;

/// Compares the member that is associated with `key_id` in
/// `datalist` to `oldval`, and if they are the same, replace
/// `oldval` with `newval`.
///
/// This is like a typical atomic compare-and-exchange
/// operation, for a member of `datalist`.
///
/// If the previous value was replaced then ownership of the
/// old value (`oldval`) is passed to the caller, including
/// the registered destroy notify for it (passed out in `old_destroy`).
/// Its up to the caller to free this as they wish, which may
/// or may not include using `old_destroy` as sometimes replacement
/// should not destroy the object in the normal way.
extern fn g_datalist_id_replace_data(p_datalist: **glib.Data, p_key_id: glib.Quark, p_oldval: ?*anyopaque, p_newval: ?*anyopaque, p_destroy: ?glib.DestroyNotify, p_old_destroy: ?*glib.DestroyNotify) c_int;
pub const datalistIdReplaceData = g_datalist_id_replace_data;

/// Sets the data corresponding to the given `glib.Quark` id, and the
/// function to be called when the element is removed from the datalist.
/// Any previous data with the same key is removed, and its destroy
/// function is called.
extern fn g_datalist_id_set_data_full(p_datalist: **glib.Data, p_key_id: glib.Quark, p_data: ?*anyopaque, p_destroy_func: ?glib.DestroyNotify) void;
pub const datalistIdSetDataFull = g_datalist_id_set_data_full;

/// Resets the datalist to `NULL`. It does not free any memory or call
/// any destroy functions.
extern fn g_datalist_init(p_datalist: **glib.Data) void;
pub const datalistInit = g_datalist_init;

/// Turns on flag values for a data list. This function is used
/// to keep a small number of boolean flags in an object with
/// a data list without using any additional space. It is
/// not generally useful except in circumstances where space
/// is very tight. (It is used in the base `GObject` type, for
/// example.)
extern fn g_datalist_set_flags(p_datalist: **glib.Data, p_flags: c_uint) void;
pub const datalistSetFlags = g_datalist_set_flags;

/// Turns off flag values for a data list. See `glib.datalistUnsetFlags`
extern fn g_datalist_unset_flags(p_datalist: **glib.Data, p_flags: c_uint) void;
pub const datalistUnsetFlags = g_datalist_unset_flags;

/// Destroys the dataset, freeing all memory allocated, and calling any
/// destroy functions set for data elements.
extern fn g_dataset_destroy(p_dataset_location: *const anyopaque) void;
pub const datasetDestroy = g_dataset_destroy;

/// Calls the given function for each data element which is associated
/// with the given location. Note that this function is NOT thread-safe.
/// So unless `dataset_location` can be protected from any modifications
/// during invocation of this function, it should not be called.
///
/// `func` can make changes to the dataset, but the iteration will not
/// reflect changes made during the `glib.datasetForeach` call, other
/// than skipping over elements that are removed.
extern fn g_dataset_foreach(p_dataset_location: *const anyopaque, p_func: glib.DataForeachFunc, p_user_data: ?*anyopaque) void;
pub const datasetForeach = g_dataset_foreach;

/// Gets the data element corresponding to a `glib.Quark`.
extern fn g_dataset_id_get_data(p_dataset_location: *const anyopaque, p_key_id: glib.Quark) ?*anyopaque;
pub const datasetIdGetData = g_dataset_id_get_data;

/// Removes an element, without calling its destroy notification
/// function.
extern fn g_dataset_id_remove_no_notify(p_dataset_location: *const anyopaque, p_key_id: glib.Quark) ?*anyopaque;
pub const datasetIdRemoveNoNotify = g_dataset_id_remove_no_notify;

/// Sets the data element associated with the given `glib.Quark` id, and also
/// the function to call when the data element is destroyed. Any
/// previous data with the same key is removed, and its destroy function
/// is called.
extern fn g_dataset_id_set_data_full(p_dataset_location: *const anyopaque, p_key_id: glib.Quark, p_data: ?*anyopaque, p_destroy_func: glib.DestroyNotify) void;
pub const datasetIdSetDataFull = g_dataset_id_set_data_full;

/// This is a variant of `glib.dgettext` that allows specifying a locale
/// category instead of always using `LC_MESSAGES`. See `glib.dgettext` for
/// more information about how this functions differs from calling
/// `dcgettext` directly.
extern fn g_dcgettext(p_domain: ?[*:0]const u8, p_msgid: [*:0]const u8, p_category: c_int) [*:0]const u8;
pub const dcgettext = g_dcgettext;

/// This function is a wrapper of `dgettext` which does not translate
/// the message if the default domain as set with `textdomain` has no
/// translations for the current locale.
///
/// The advantage of using this function over `dgettext` proper is that
/// libraries using this function (like GTK) will not use translations
/// if the application using the library does not have translations for
/// the current locale.  This results in a consistent English-only
/// interface instead of one having partial translations.  For this
/// feature to work, the call to `textdomain` and `setlocale` should
/// precede any `glib.dgettext` invocations.  For GTK, it means calling
/// `textdomain` before gtk_init or its variants.
///
/// This function disables translations if and only if upon its first
/// call all the following conditions hold:
///
/// - `domain` is not `NULL`
///
/// - `textdomain` has been called to set a default text domain
///
/// - there is no translations available for the default text domain
///   and the current locale
///
/// - current locale is not "C" or any English locales (those
///   starting with "en_")
///
/// Note that this behavior may not be desired for example if an application
/// has its untranslated messages in a language other than English. In those
/// cases the application should call `textdomain` after initializing GTK.
///
/// Applications should normally not use this function directly,
/// but use the `_` macro for translations.
extern fn g_dgettext(p_domain: ?[*:0]const u8, p_msgid: [*:0]const u8) [*:0]const u8;
pub const dgettext = g_dgettext;

/// Compares two `gpointer` arguments and returns `TRUE` if they are equal.
/// It can be passed to `glib.HashTable.new` as the `key_equal_func`
/// parameter, when using opaque pointers compared by pointer value as
/// keys in a `glib.HashTable`.
///
/// This equality function is also appropriate for keys that are integers
/// stored in pointers, such as `GINT_TO_POINTER (n)`.
extern fn g_direct_equal(p_v1: ?*const anyopaque, p_v2: ?*const anyopaque) c_int;
pub const directEqual = g_direct_equal;

/// Converts a gpointer to a hash value.
/// It can be passed to `glib.HashTable.new` as the `hash_func` parameter,
/// when using opaque pointers compared by pointer value as keys in a
/// `glib.HashTable`.
///
/// This hash function is also appropriate for keys that are integers
/// stored in pointers, such as `GINT_TO_POINTER (n)`.
extern fn g_direct_hash(p_v: ?*const anyopaque) c_uint;
pub const directHash = g_direct_hash;

/// This function is a wrapper of `dngettext` which does not translate
/// the message if the default domain as set with `textdomain` has no
/// translations for the current locale.
///
/// See `glib.dgettext` for details of how this differs from `dngettext`
/// proper.
extern fn g_dngettext(p_domain: ?[*:0]const u8, p_msgid: [*:0]const u8, p_msgid_plural: [*:0]const u8, p_n: c_ulong) [*:0]const u8;
pub const dngettext = g_dngettext;

/// Compares the two `gdouble` values being pointed to and returns
/// `TRUE` if they are equal.
/// It can be passed to `glib.HashTable.new` as the `key_equal_func`
/// parameter, when using non-`NULL` pointers to doubles as keys in a
/// `glib.HashTable`.
extern fn g_double_equal(p_v1: *const anyopaque, p_v2: *const anyopaque) c_int;
pub const doubleEqual = g_double_equal;

/// Converts a pointer to a `gdouble` to a hash value.
/// It can be passed to `glib.HashTable.new` as the `hash_func` parameter,
/// It can be passed to `glib.HashTable.new` as the `hash_func` parameter,
/// when using non-`NULL` pointers to doubles as keys in a `glib.HashTable`.
extern fn g_double_hash(p_v: *const anyopaque) c_uint;
pub const doubleHash = g_double_hash;

/// This function is a variant of `glib.dgettext` which supports
/// a disambiguating message context. GNU gettext uses the
/// '\004' character to separate the message context and
/// message id in `msgctxtid`.
/// If 0 is passed as `msgidoffset`, this function will fall back to
/// trying to use the deprecated convention of using "|" as a separation
/// character.
///
/// This uses `glib.dgettext` internally. See that functions for differences
/// with `dgettext` proper.
///
/// Applications should normally not use this function directly,
/// but use the `C_` macro for translations with context.
extern fn g_dpgettext(p_domain: ?[*:0]const u8, p_msgctxtid: [*:0]const u8, p_msgidoffset: usize) [*:0]const u8;
pub const dpgettext = g_dpgettext;

/// This function is a variant of `glib.dgettext` which supports
/// a disambiguating message context. GNU gettext uses the
/// '\004' character to separate the message context and
/// message id in `msgctxtid`.
///
/// This uses `glib.dgettext` internally. See that functions for differences
/// with `dgettext` proper.
///
/// This function differs from `C_` in that it is not a macro and
/// thus you may use non-string-literals as context and msgid arguments.
extern fn g_dpgettext2(p_domain: ?[*:0]const u8, p_context: [*:0]const u8, p_msgid: [*:0]const u8) [*:0]const u8;
pub const dpgettext2 = g_dpgettext2;

/// Returns the value of the environment variable `variable` in the
/// provided list `envp`.
extern fn g_environ_getenv(p_envp: ?[*:null]?[*:0]u8, p_variable: [*:0]const u8) ?[*:0]const u8;
pub const environGetenv = g_environ_getenv;

/// Sets the environment variable `variable` in the provided list
/// `envp` to `value`.
extern fn g_environ_setenv(p_envp: ?[*:null]?[*:0]u8, p_variable: [*:0]const u8, p_value: [*:0]const u8, p_overwrite: c_int) [*:null]?[*:0]u8;
pub const environSetenv = g_environ_setenv;

/// Removes the environment variable `variable` from the provided
/// environment `envp`.
extern fn g_environ_unsetenv(p_envp: ?[*:null]?[*:0]u8, p_variable: [*:0]const u8) [*:null]?[*:0]u8;
pub const environUnsetenv = g_environ_unsetenv;

/// Mark every file descriptor equal to or greater than `lowfd` to be closed
/// at the next ``execve`` or similar, as if via the `FD_CLOEXEC` flag.
///
/// Typically `lowfd` will be 3, to leave standard input, standard output
/// and standard error open after exec.
///
/// This is the same as Linux `close_range (lowfd, ~0U, CLOSE_RANGE_CLOEXEC)`,
/// but portable to other OSs and to older versions of Linux.
///
/// This function is async-signal safe, making it safe to call from a
/// signal handler or a `glib.SpawnChildSetupFunc`, as long as `lowfd` is
/// non-negative.
/// See [`signal(7)`](man:signal(7)) and
/// [`signal-safety(7)`](man:signal-safety(7)) for more details.
extern fn g_fdwalk_set_cloexec(p_lowfd: c_int) c_int;
pub const fdwalkSetCloexec = g_fdwalk_set_cloexec;

/// Gets a `glib.FileError` constant based on the passed-in `err_no`.
///
/// For example, if you pass in `EEXIST` this function returns
/// `G_FILE_ERROR_EXIST`. Unlike `errno` values, you can portably
/// assume that all `glib.FileError` values will exist.
///
/// Normally a `glib.FileError` value goes into a `glib.Error` returned
/// from a function that manipulates files. So you would use
/// `glib.fileErrorFromErrno` when constructing a `glib.Error`.
extern fn g_file_error_from_errno(p_err_no: c_int) glib.FileError;
pub const fileErrorFromErrno = g_file_error_from_errno;

extern fn g_file_error_quark() glib.Quark;
pub const fileErrorQuark = g_file_error_quark;

/// Reads an entire file into allocated memory, with good error
/// checking.
///
/// If the call was successful, it returns `TRUE` and sets `contents` to the file
/// contents and `length` to the length of the file contents in bytes. The string
/// stored in `contents` will be nul-terminated, so for text files you can pass
/// `NULL` for the `length` argument. If the call was not successful, it returns
/// `FALSE` and sets `error`. The error domain is `G_FILE_ERROR`. Possible error
/// codes are those in the `glib.FileError` enumeration. In the error case,
/// `contents` is set to `NULL` and `length` is set to zero.
extern fn g_file_get_contents(p_filename: [*:0]const u8, p_contents: *[*]u8, p_length: ?*usize, p_error: ?*?*glib.Error) c_int;
pub const fileGetContents = g_file_get_contents;

/// Opens a file for writing in the preferred directory for temporary
/// files (as returned by `glib.getTmpDir`).
///
/// `tmpl` should be a string in the GLib file name encoding containing
/// a sequence of six 'X' characters, as the parameter to `glib.mkstemp`.
/// However, unlike these functions, the template should only be a
/// basename, no directory components are allowed. If template is
/// `NULL`, a default template is used.
///
/// Note that in contrast to `glib.mkstemp` (and `mkstemp`) `tmpl` is not
/// modified, and might thus be a read-only literal string.
///
/// Upon success, and if `name_used` is non-`NULL`, the actual name used
/// is returned in `name_used`. This string should be freed with `glib.free`
/// when not needed any longer. The returned name is in the GLib file
/// name encoding.
extern fn g_file_open_tmp(p_tmpl: ?[*:0]const u8, p_name_used: *[*:0]u8, p_error: ?*?*glib.Error) c_int;
pub const fileOpenTmp = g_file_open_tmp;

/// Reads the contents of the symbolic link `filename` like the POSIX
/// ``readlink`` function.
///
/// The returned string is in the encoding used for filenames. Use
/// `glib.filenameToUtf8` to convert it to UTF-8.
///
/// The returned string may also be a relative path. Use `glib.buildFilename`
/// to convert it to an absolute path:
///
/// ```
/// g_autoptr(GError) local_error = NULL;
/// g_autofree gchar *link_target = g_file_read_link ("/etc/localtime", &local_error);
///
/// if (local_error != NULL)
///   g_error ("Error reading link: `s`", local_error->message);
///
/// if (!g_path_is_absolute (link_target))
///   {
///     g_autofree gchar *absolute_link_target = g_build_filename ("/etc", link_target, NULL);
///     g_free (link_target);
///     link_target = g_steal_pointer (&absolute_link_target);
///   }
/// ```
extern fn g_file_read_link(p_filename: [*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const fileReadLink = g_file_read_link;

/// Writes all of `contents` to a file named `filename`. This is a convenience
/// wrapper around calling `glib.fileSetContentsFull` with `flags` set to
/// `G_FILE_SET_CONTENTS_CONSISTENT | G_FILE_SET_CONTENTS_ONLY_EXISTING` and
/// `mode` set to `0666`.
extern fn g_file_set_contents(p_filename: [*:0]const u8, p_contents: [*]const u8, p_length: isize, p_error: ?*?*glib.Error) c_int;
pub const fileSetContents = g_file_set_contents;

/// Writes all of `contents` to a file named `filename`, with good error checking.
/// If a file called `filename` already exists it will be overwritten.
///
/// `flags` control the properties of the write operation: whether it’s atomic,
/// and what the tradeoff is between returning quickly or being resilient to
/// system crashes.
///
/// As this function performs file I/O, it is recommended to not call it anywhere
/// where blocking would cause problems, such as in the main loop of a graphical
/// application. In particular, if `flags` has any value other than
/// `G_FILE_SET_CONTENTS_NONE` then this function may call ``fsync``.
///
/// If `G_FILE_SET_CONTENTS_CONSISTENT` is set in `flags`, the operation is atomic
/// in the sense that it is first written to a temporary file which is then
/// renamed to the final name.
///
/// Notes:
///
/// - On UNIX, if `filename` already exists hard links to `filename` will break.
///   Also since the file is recreated, existing permissions, access control
///   lists, metadata etc. may be lost. If `filename` is a symbolic link,
///   the link itself will be replaced, not the linked file.
///
/// - On UNIX, if `filename` already exists and is non-empty, and if the system
///   supports it (via a journalling filesystem or equivalent), and if
///   `G_FILE_SET_CONTENTS_CONSISTENT` is set in `flags`, the ``fsync`` call (or
///   equivalent) will be used to ensure atomic replacement: `filename`
///   will contain either its old contents or `contents`, even in the face of
///   system power loss, the disk being unsafely removed, etc.
///
/// - On UNIX, if `filename` does not already exist or is empty, there is a
///   possibility that system power loss etc. after calling this function will
///   leave `filename` empty or full of NUL bytes, depending on the underlying
///   filesystem, unless `G_FILE_SET_CONTENTS_DURABLE` and
///   `G_FILE_SET_CONTENTS_CONSISTENT` are set in `flags`.
///
/// - On Windows renaming a file will not remove an existing file with the
///   new name, so on Windows there is a race condition between the existing
///   file being removed and the temporary file being renamed.
///
/// - On Windows there is no way to remove a file that is open to some
///   process, or mapped into memory. Thus, this function will fail if
///   `filename` already exists and is open.
///
/// If the call was successful, it returns `TRUE`. If the call was not successful,
/// it returns `FALSE` and sets `error`. The error domain is `G_FILE_ERROR`.
/// Possible error codes are those in the `glib.FileError` enumeration.
///
/// Note that the name for the temporary file is constructed by appending up
/// to 7 characters to `filename`.
///
/// If the file didn’t exist before and is created, it will be given the
/// permissions from `mode`. Otherwise, the permissions of the existing file may
/// be changed to `mode` depending on `flags`, or they may remain unchanged.
extern fn g_file_set_contents_full(p_filename: [*:0]const u8, p_contents: [*]const u8, p_length: isize, p_flags: glib.FileSetContentsFlags, p_mode: c_int, p_error: ?*?*glib.Error) c_int;
pub const fileSetContentsFull = g_file_set_contents_full;

/// Returns `TRUE` if any of the tests in the bitfield `test` are
/// `TRUE`. For example, `(G_FILE_TEST_EXISTS | G_FILE_TEST_IS_DIR)`
/// will return `TRUE` if the file exists; the check whether it's a
/// directory doesn't matter since the existence test is `TRUE`. With
/// the current set of available tests, there's no point passing in
/// more than one test at a time.
///
/// Apart from `G_FILE_TEST_IS_SYMLINK` all tests follow symbolic links,
/// so for a symbolic link to a regular file `glib.fileTest` will return
/// `TRUE` for both `G_FILE_TEST_IS_SYMLINK` and `G_FILE_TEST_IS_REGULAR`.
///
/// Note, that for a dangling symbolic link `glib.fileTest` will return
/// `TRUE` for `G_FILE_TEST_IS_SYMLINK` and `FALSE` for all other flags.
///
/// You should never use `glib.fileTest` to test whether it is safe
/// to perform an operation, because there is always the possibility
/// of the condition changing before you actually perform the operation,
/// see [TOCTOU](https://en.wikipedia.org/wiki/Time-of-check_to_time-of-use).
///
/// For example, you might think you could use `G_FILE_TEST_IS_SYMLINK`
/// to know whether it is safe to write to a file without being
/// tricked into writing into a different location. It doesn't work!
///
/// ```
///  // DON'T DO THIS
///  if (!g_file_test (filename, G_FILE_TEST_IS_SYMLINK))
///    {
///      fd = g_open (filename, O_WRONLY);
///      // write to fd
///    }
///
///  // DO THIS INSTEAD
///  fd = g_open (filename, O_WRONLY | O_NOFOLLOW | O_CLOEXEC);
///  if (fd == -1)
///    {
///      // check error
///      if (errno == ELOOP)
///        // file is a symlink and can be ignored
///      else
///        // handle errors as before
///    }
///  else
///    {
///      // write to fd
///    }
/// ```
///
/// Another thing to note is that `G_FILE_TEST_EXISTS` and
/// `G_FILE_TEST_IS_EXECUTABLE` are implemented using the `access`
/// system call. This usually doesn't matter, but if your program
/// is setuid or setgid it means that these tests will give you
/// the answer for the real user ID and group ID, rather than the
/// effective user ID and group ID.
///
/// On Windows, there are no symlinks, so testing for
/// `G_FILE_TEST_IS_SYMLINK` will always return `FALSE`. Testing for
/// `G_FILE_TEST_IS_EXECUTABLE` will just check that the file exists and
/// its name indicates that it is executable, checking for well-known
/// extensions and those listed in the `PATHEXT` environment variable.
extern fn g_file_test(p_filename: [*:0]const u8, p_test: glib.FileTest) c_int;
pub const fileTest = g_file_test;

/// Returns the display basename for the particular filename, guaranteed
/// to be valid UTF-8. The display name might not be identical to the filename,
/// for instance there might be problems converting it to UTF-8, and some files
/// can be translated in the display.
///
/// If GLib cannot make sense of the encoding of `filename`, as a last resort it
/// replaces unknown characters with U+FFFD, the Unicode replacement character.
/// You can search the result for the UTF-8 encoding of this character (which is
/// "\357\277\275" in octal notation) to find out if `filename` was in an invalid
/// encoding.
///
/// You must pass the whole absolute pathname to this functions so that
/// translation of well known locations can be done.
///
/// This function is preferred over `glib.filenameDisplayName` if you know the
/// whole path, as it allows translation.
extern fn g_filename_display_basename(p_filename: [*:0]const u8) [*:0]u8;
pub const filenameDisplayBasename = g_filename_display_basename;

/// Converts a filename into a valid UTF-8 string. The conversion is
/// not necessarily reversible, so you should keep the original around
/// and use the return value of this function only for display purposes.
/// Unlike `glib.filenameToUtf8`, the result is guaranteed to be non-`NULL`
/// even if the filename actually isn't in the GLib file name encoding.
///
/// If GLib cannot make sense of the encoding of `filename`, as a last resort it
/// replaces unknown characters with U+FFFD, the Unicode replacement character.
/// You can search the result for the UTF-8 encoding of this character (which is
/// "\357\277\275" in octal notation) to find out if `filename` was in an invalid
/// encoding.
///
/// If you know the whole pathname of the file you should use
/// `glib.filenameDisplayBasename`, since that allows location-based
/// translation of filenames.
extern fn g_filename_display_name(p_filename: [*:0]const u8) [*:0]u8;
pub const filenameDisplayName = g_filename_display_name;

/// Converts an escaped ASCII-encoded URI to a local filename in the
/// encoding used for filenames.
///
/// Since GLib 2.78, the query string and fragment can be present in the URI,
/// but are not part of the resulting filename.
/// We take inspiration from https://url.spec.whatwg.org/`file`-state,
/// but we don't support the entire standard.
extern fn g_filename_from_uri(p_uri: [*:0]const u8, p_hostname: ?*[*:0]u8, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const filenameFromUri = g_filename_from_uri;

/// Converts a string from UTF-8 to the encoding GLib uses for
/// filenames. Note that on Windows GLib uses UTF-8 for filenames;
/// on other platforms, this function indirectly depends on the
/// [current locale](running.html`locale`).
///
/// The input string shall not contain nul characters even if the `len`
/// argument is positive. A nul character found inside the string will result
/// in error `G_CONVERT_ERROR_ILLEGAL_SEQUENCE`. If the filename encoding is
/// not UTF-8 and the conversion output contains a nul character, the error
/// `G_CONVERT_ERROR_EMBEDDED_NUL` is set and the function returns `NULL`.
extern fn g_filename_from_utf8(p_utf8string: [*:0]const u8, p_len: isize, p_bytes_read: ?*usize, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const filenameFromUtf8 = g_filename_from_utf8;

/// Converts an absolute filename to an escaped ASCII-encoded URI, with the path
/// component following Section 3.3. of RFC 2396.
extern fn g_filename_to_uri(p_filename: [*:0]const u8, p_hostname: ?[*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const filenameToUri = g_filename_to_uri;

/// Converts a string which is in the encoding used by GLib for
/// filenames into a UTF-8 string. Note that on Windows GLib uses UTF-8
/// for filenames; on other platforms, this function indirectly depends on
/// the [current locale](running.html`locale`).
///
/// The input string shall not contain nul characters even if the `len`
/// argument is positive. A nul character found inside the string will result
/// in error `G_CONVERT_ERROR_ILLEGAL_SEQUENCE`.
/// If the source encoding is not UTF-8 and the conversion output contains a
/// nul character, the error `G_CONVERT_ERROR_EMBEDDED_NUL` is set and the
/// function returns `NULL`. Use `glib.convert` to produce output that
/// may contain embedded nul characters.
extern fn g_filename_to_utf8(p_opsysstring: [*:0]const u8, p_len: isize, p_bytes_read: ?*usize, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const filenameToUtf8 = g_filename_to_utf8;

/// Locates the first executable named `program` in the user's path, in the
/// same way that `execvp` would locate it. Returns an allocated string
/// with the absolute path name, or `NULL` if the program is not found in
/// the path. If `program` is already an absolute path, returns a copy of
/// `program` if `program` exists and is executable, and `NULL` otherwise.
///
/// On Windows, if `program` does not have a file type suffix, tries
/// with the suffixes .exe, .cmd, .bat and .com, and the suffixes in
/// the `PATHEXT` environment variable.
///
/// On Windows, it looks for the file in the same way as `CreateProcess`
/// would. This means first in the directory where the executing
/// program was loaded from, then in the current directory, then in the
/// Windows 32-bit system directory, then in the Windows directory, and
/// finally in the directories in the `PATH` environment variable. If
/// the program is found, the return value contains the full name
/// including the type suffix.
extern fn g_find_program_in_path(p_program: [*:0]const u8) ?[*:0]u8;
pub const findProgramInPath = g_find_program_in_path;

/// A wrapper for the stdio ``fopen`` function. The ``fopen`` function
/// opens a file and associates a new stream with it.
///
/// Because file descriptors are specific to the C library on Windows,
/// and a file descriptor is part of the `FILE` struct, the `FILE*` returned
/// by this function makes sense only to functions in the same C library.
/// Thus if the GLib-using code uses a different C library than GLib does,
/// the FILE* returned by this function cannot be passed to C library
/// functions like ``fprintf`` or ``fread``.
///
/// See your C library manual for more details about ``fopen``.
///
/// As ``close`` and ``fclose`` are part of the C library, this implies that it is
/// currently impossible to close a file if the application C library and the C library
/// used by GLib are different. Convenience functions like `glib.fileSetContentsFull`
/// avoid this problem.
extern fn g_fopen(p_filename: [*:0]const u8, p_mode: [*:0]const u8) ?*anyopaque;
pub const fopen = g_fopen;

/// Formats a size (for example the size of a file) into a human readable
/// string.  Sizes are rounded to the nearest size prefix (kB, MB, GB)
/// and are displayed rounded to the nearest tenth. E.g. the file size
/// 3292528 bytes will be converted into the string "3.2 MB". The returned string
/// is UTF-8, and may use a non-breaking space to separate the number and units,
/// to ensure they aren’t separated when line wrapped.
///
/// The prefix units base is 1000 (i.e. 1 kB is 1000 bytes).
///
/// This string should be freed with `glib.free` when not needed any longer.
///
/// See `glib.formatSizeFull` for more options about how the size might be
/// formatted.
extern fn g_format_size(p_size: u64) [*:0]u8;
pub const formatSize = g_format_size;

/// Formats a size (for example the size of a file) into a human
/// readable string. Sizes are rounded to the nearest size prefix
/// (KB, MB, GB) and are displayed rounded to the nearest tenth.
/// E.g. the file size 3292528 bytes will be converted into the
/// string "3.1 MB".
///
/// The prefix units base is 1024 (i.e. 1 KB is 1024 bytes).
///
/// This string should be freed with `glib.free` when not needed any longer.
extern fn g_format_size_for_display(p_size: i64) [*:0]u8;
pub const formatSizeForDisplay = g_format_size_for_display;

/// Formats a size.
///
/// This function is similar to `glib.formatSize` but allows for flags
/// that modify the output. See `glib.FormatSizeFlags`.
extern fn g_format_size_full(p_size: u64, p_flags: glib.FormatSizeFlags) [*:0]u8;
pub const formatSizeFull = g_format_size_full;

/// An implementation of the standard ``fprintf`` function which supports
/// positional parameters, as specified in the Single Unix Specification.
///
/// `glib/gprintf.h` must be explicitly included in order to use this function.
extern fn g_fprintf(p_file: *anyopaque, p_format: [*:0]const u8, ...) c_int;
pub const fprintf = g_fprintf;

/// Frees the memory pointed to by `mem`.
///
/// If you know the allocated size of `mem`, calling `glib.freeSized` may be faster,
/// depending on the libc implementation in use.
///
/// Starting from GLib 2.78, this may happen automatically in case a GCC
/// compatible compiler is used with some optimization level and the allocated
/// size is known at compile time (see [documentation of
/// ``__builtin_object_size``](https://gcc.gnu.org/onlinedocs/gcc/Object-Size-Checking.html)
/// to understand its caveats).
///
/// If `mem` is `NULL` it simply returns, so there is no need to check `mem`
/// against `NULL` before calling this function.
extern fn g_free(p_mem: ?*anyopaque) void;
pub const free = g_free;

/// Frees the memory pointed to by `mem`, assuming it is has the given `size`.
///
/// If `mem` is `NULL` this is a no-op (and `size` is ignored).
///
/// It is an error if `size` doesn’t match the size passed when `mem` was
/// allocated. `size` is passed to this function to allow optimizations in the
/// allocator. If you don’t know the allocation size, use `glib.free` instead.
///
/// In case a GCC compatible compiler is used, this function may be used
/// automatically via `glib.free` if the allocated size is known at compile time,
/// since GLib 2.78.
extern fn g_free_sized(p_mem: ?*anyopaque, p_size: usize) void;
pub const freeSized = g_free_sized;

/// A wrapper for the POSIX `freopen` function. The `freopen` function
/// opens a file and associates it with an existing stream.
///
/// See your C library manual for more details about `freopen`.
extern fn g_freopen(p_filename: [*:0]const u8, p_mode: [*:0]const u8, p_stream: ?*anyopaque) ?*anyopaque;
pub const freopen = g_freopen;

/// A wrapper for the POSIX ``fsync`` function. On Windows, ``_commit`` will be
/// used. On macOS, `fcntl(F_FULLFSYNC)` will be used.
/// The ``fsync`` function is used to synchronize a file's in-core
/// state with that of the disk.
///
/// This wrapper will handle retrying on `EINTR`.
///
/// See the C library manual for more details about `fsync`.
extern fn g_fsync(p_fd: c_int) c_int;
pub const fsync = g_fsync;

/// Gets a human-readable name for the application, as set by
/// `glib.setApplicationName`. This name should be localized if
/// possible, and is intended for display to the user.  Contrast with
/// `glib.getPrgname`, which gets a non-localized name. If
/// `glib.setApplicationName` has not been called, returns the result of
/// `glib.getPrgname` (which may be `NULL` if `glib.setPrgname` has also not
/// been called).
extern fn g_get_application_name() ?[*:0]const u8;
pub const getApplicationName = g_get_application_name;

/// Obtains the character set for the [current locale](running.html`locale`);
/// you might use this character set as an argument to `glib.convert`, to convert
/// from the current locale's encoding to some other encoding. (Frequently
/// `glib.localeToUtf8` and `glib.localeFromUtf8` are nice shortcuts, though.)
///
/// On Windows the character set returned by this function is the
/// so-called system default ANSI code-page. That is the character set
/// used by the "narrow" versions of C library and Win32 functions that
/// handle file names. It might be different from the character set
/// used by the C library's current locale.
///
/// On Linux, the character set is found by consulting `nl_langinfo` if
/// available. If not, the environment variables `LC_ALL`, `LC_CTYPE`, `LANG`
/// and `CHARSET` are queried in order. `nl_langinfo` returns the C locale if
/// no locale has been loaded by `setlocale`.
///
/// The return value is `TRUE` if the locale's encoding is UTF-8, in that
/// case you can perhaps avoid calling `glib.convert`.
///
/// The string returned in `charset` is not allocated, and should not be
/// freed.
extern fn g_get_charset(p_charset: ?*[*:0]const u8) c_int;
pub const getCharset = g_get_charset;

/// Gets the character set for the current locale.
extern fn g_get_codeset() [*:0]u8;
pub const getCodeset = g_get_codeset;

/// Obtains the character set used by the console attached to the process,
/// which is suitable for printing output to the terminal.
///
/// Usually this matches the result returned by `glib.getCharset`, but in
/// environments where the locale's character set does not match the encoding
/// of the console this function tries to guess a more suitable value instead.
///
/// On Windows the character set returned by this function is the
/// output code page used by the console associated with the calling process.
/// If the codepage can't be determined (for example because there is no
/// console attached) UTF-8 is assumed.
///
/// The return value is `TRUE` if the locale's encoding is UTF-8, in that
/// case you can perhaps avoid calling `glib.convert`.
///
/// The string returned in `charset` is not allocated, and should not be
/// freed.
extern fn g_get_console_charset(p_charset: ?*[*:0]const u8) c_int;
pub const getConsoleCharset = g_get_console_charset;

/// Gets the current directory.
///
/// The returned string should be freed when no longer needed.
/// The encoding of the returned string is system defined.
/// On Windows, it is always UTF-8.
///
/// Since GLib 2.40, this function will return the value of the "PWD"
/// environment variable if it is set and it happens to be the same as
/// the current directory.  This can make a difference in the case that
/// the current directory is the target of a symbolic link.
extern fn g_get_current_dir() [*:0]u8;
pub const getCurrentDir = g_get_current_dir;

/// Equivalent to the UNIX `gettimeofday` function, but portable.
///
/// You may find `glib.getRealTime` to be more convenient.
extern fn g_get_current_time(p_result: *glib.TimeVal) void;
pub const getCurrentTime = g_get_current_time;

/// Gets the list of environment variables for the current process.
///
/// The list is `NULL` terminated and each item in the list is of the
/// form 'NAME=VALUE'.
///
/// This is equivalent to direct access to the 'environ' global variable,
/// except portable.
///
/// The return value is freshly allocated and it should be freed with
/// `glib.strfreev` when it is no longer needed.
extern fn g_get_environ() [*:null]?[*:0]u8;
pub const getEnviron = g_get_environ;

/// Determines the preferred character sets used for filenames.
/// The first character set from the `charsets` is the filename encoding, the
/// subsequent character sets are used when trying to generate a displayable
/// representation of a filename, see `glib.filenameDisplayName`.
///
/// On Unix, the character sets are determined by consulting the
/// environment variables `G_FILENAME_ENCODING` and `G_BROKEN_FILENAMES`.
/// On Windows, the character set used in the GLib API is always UTF-8
/// and said environment variables have no effect.
///
/// `G_FILENAME_ENCODING` may be set to a comma-separated list of
/// character set names. The special token ``locale`` is taken to mean the
/// character set for the [current locale](running.html`locale`).
/// If `G_FILENAME_ENCODING` is not set, but `G_BROKEN_FILENAMES` is,
/// the character set of the current locale is taken as the filename
/// encoding. If neither environment variable  is set, UTF-8 is taken
/// as the filename encoding, but the character set of the current locale
/// is also put in the list of encodings.
///
/// The returned `charsets` belong to GLib and must not be freed.
///
/// Note that on Unix, regardless of the locale character set or
/// `G_FILENAME_ENCODING` value, the actual file names present
/// on a system might be in any random encoding or just gibberish.
extern fn g_get_filename_charsets(p_filename_charsets: *[*][*:0]const u8) c_int;
pub const getFilenameCharsets = g_get_filename_charsets;

/// Gets the current user's home directory.
///
/// As with most UNIX tools, this function will return the value of the
/// `HOME` environment variable if it is set to an existing absolute path
/// name, falling back to the `passwd` file in the case that it is unset.
///
/// If the path given in `HOME` is non-absolute, does not exist, or is
/// not a directory, the result is undefined.
///
/// Before version 2.36 this function would ignore the `HOME` environment
/// variable, taking the value from the `passwd` database instead. This was
/// changed to increase the compatibility of GLib with other programs (and
/// the XDG basedir specification) and to increase testability of programs
/// based on GLib (by making it easier to run them from test frameworks).
///
/// If your program has a strong requirement for either the new or the
/// old behaviour (and if you don't wish to increase your GLib
/// dependency to ensure that the new behaviour is in effect) then you
/// should either directly check the `HOME` environment variable yourself
/// or unset it before calling any functions in GLib.
extern fn g_get_home_dir() [*:0]const u8;
pub const getHomeDir = g_get_home_dir;

/// Return a name for the machine.
///
/// The returned name is not necessarily a fully-qualified domain name,
/// or even present in DNS or some other name service at all. It need
/// not even be unique on your local network or site, but usually it
/// is. Callers should not rely on the return value having any specific
/// properties like uniqueness for security purposes. Even if the name
/// of the machine is changed while an application is running, the
/// return value from this function does not change. The returned
/// string is owned by GLib and should not be modified or freed. If no
/// name can be determined, a default fixed string "localhost" is
/// returned.
///
/// The encoding of the returned string is UTF-8.
extern fn g_get_host_name() [*:0]const u8;
pub const getHostName = g_get_host_name;

/// Computes a list of applicable locale names, which can be used to
/// e.g. construct locale-dependent filenames or search paths. The returned
/// list is sorted from most desirable to least desirable and always contains
/// the default locale "C".
///
/// For example, if LANGUAGE=de:en_US, then the returned list is
/// "de", "en_US", "en", "C".
///
/// This function consults the environment variables `LANGUAGE`, `LC_ALL`,
/// `LC_MESSAGES` and `LANG` to find the list of locales specified by the
/// user.
extern fn g_get_language_names() [*]const [*:0]const u8;
pub const getLanguageNames = g_get_language_names;

/// Computes a list of applicable locale names with a locale category name,
/// which can be used to construct the fallback locale-dependent filenames
/// or search paths. The returned list is sorted from most desirable to
/// least desirable and always contains the default locale "C".
///
/// This function consults the environment variables `LANGUAGE`, `LC_ALL`,
/// `category_name`, and `LANG` to find the list of locales specified by the
/// user.
///
/// `glib.getLanguageNames` returns g_get_language_names_with_category("LC_MESSAGES").
extern fn g_get_language_names_with_category(p_category_name: [*:0]const u8) [*]const [*:0]const u8;
pub const getLanguageNamesWithCategory = g_get_language_names_with_category;

/// Returns a list of derived variants of `locale`, which can be used to
/// e.g. construct locale-dependent filenames or search paths. The returned
/// list is sorted from most desirable to least desirable.
/// This function handles territory, charset and extra locale modifiers. See
/// [`setlocale(3)`](man:setlocale) for information about locales and their format.
///
/// `locale` itself is guaranteed to be returned in the output.
///
/// For example, if `locale` is `fr_BE`, then the returned list
/// is `fr_BE`, `fr`. If `locale` is `en_GB.UTF-8`euro``, then the returned list
/// is `en_GB.UTF-8`euro``, `en_GB.UTF-8`, `en_GB`euro``, `en_GB`, `en.UTF-8`euro``,
/// `en.UTF-8`, `en`euro``, `en`.
///
/// If you need the list of variants for the current locale,
/// use `glib.getLanguageNames`.
extern fn g_get_locale_variants(p_locale: [*:0]const u8) [*:null]?[*:0]u8;
pub const getLocaleVariants = g_get_locale_variants;

/// Queries the system monotonic time.
///
/// The monotonic clock will always increase and doesn't suffer
/// discontinuities when the user (or NTP) changes the system time.  It
/// may or may not continue to tick during times where the machine is
/// suspended.
///
/// We try to use the clock that corresponds as closely as possible to
/// the passage of time as measured by system calls such as `poll` but it
/// may not always be possible to do this.
extern fn g_get_monotonic_time() i64;
pub const getMonotonicTime = g_get_monotonic_time;

/// Determine the approximate number of threads that the system will
/// schedule simultaneously for this process.  This is intended to be
/// used as a parameter to `glib.ThreadPool.new` for CPU bound tasks and
/// similar cases.
extern fn g_get_num_processors() c_uint;
pub const getNumProcessors = g_get_num_processors;

/// Get information about the operating system.
///
/// On Linux this comes from the `/etc/os-release` file. On other systems, it may
/// come from a variety of sources. You can either use the standard key names
/// like `G_OS_INFO_KEY_NAME` or pass any UTF-8 string key name. For example,
/// `/etc/os-release` provides a number of other less commonly used values that may
/// be useful. No key is guaranteed to be provided, so the caller should always
/// check if the result is `NULL`.
extern fn g_get_os_info(p_key_name: [*:0]const u8) ?[*:0]u8;
pub const getOsInfo = g_get_os_info;

/// Gets the name of the program. This name should not be localized,
/// in contrast to `glib.getApplicationName`.
///
/// If you are using `GApplication` the program name is set in
/// `g_application_run`. In case of GDK or GTK it is set in
/// `gdk_init`, which is called by `gtk_init` and the
/// `GtkApplication.signals.startup` handler. The program name is found by
/// taking the last component of `argv`[0].
extern fn g_get_prgname() ?[*:0]const u8;
pub const getPrgname = g_get_prgname;

/// Gets the real name of the user. This usually comes from the user's
/// entry in the `passwd` file. The encoding of the returned string is
/// system-defined. (On Windows, it is, however, always UTF-8.) If the
/// real user name cannot be determined, the string "Unknown" is
/// returned.
extern fn g_get_real_name() [*:0]const u8;
pub const getRealName = g_get_real_name;

/// Queries the system wall-clock time.
///
/// This call is functionally equivalent to `glib.getCurrentTime` except
/// that the return value is often more convenient than dealing with a
/// `glib.TimeVal`.
///
/// You should only use this call if you are actually interested in the real
/// wall-clock time. `glib.getMonotonicTime` is probably more useful for
/// measuring intervals.
extern fn g_get_real_time() i64;
pub const getRealTime = g_get_real_time;

/// Returns an ordered list of base directories in which to access
/// system-wide configuration information.
///
/// On UNIX platforms this is determined using the mechanisms described
/// in the
/// [XDG Base Directory Specification](http://www.freedesktop.org/Standards/basedir-spec).
/// In this case the list of directories retrieved will be `XDG_CONFIG_DIRS`.
///
/// On Windows it follows XDG Base Directory Specification if `XDG_CONFIG_DIRS` is defined.
/// If `XDG_CONFIG_DIRS` is undefined, the directory that contains application
/// data for all users is used instead. A typical path is
/// `C:\Documents and Settings\All Users\Application Data`.
/// This folder is used for application data
/// that is not user specific. For example, an application can store
/// a spell-check dictionary, a database of clip art, or a log file in the
/// FOLDERID_ProgramData folder. This information will not roam and is available
/// to anyone using the computer.
///
/// The return value is cached and modifying it at runtime is not supported, as
/// it’s not thread-safe to modify environment variables at runtime.
extern fn g_get_system_config_dirs() [*]const [*:0]const u8;
pub const getSystemConfigDirs = g_get_system_config_dirs;

/// Returns an ordered list of base directories in which to access
/// system-wide application data.
///
/// On UNIX platforms this is determined using the mechanisms described
/// in the
/// [XDG Base Directory Specification](http://www.freedesktop.org/Standards/basedir-spec)
/// In this case the list of directories retrieved will be `XDG_DATA_DIRS`.
///
/// On Windows it follows XDG Base Directory Specification if `XDG_DATA_DIRS` is defined.
/// If `XDG_DATA_DIRS` is undefined,
/// the first elements in the list are the Application Data
/// and Documents folders for All Users. (These can be determined only
/// on Windows 2000 or later and are not present in the list on other
/// Windows versions.) See documentation for FOLDERID_ProgramData and
/// FOLDERID_PublicDocuments.
///
/// Then follows the "share" subfolder in the installation folder for
/// the package containing the DLL that calls this function, if it can
/// be determined.
///
/// Finally the list contains the "share" subfolder in the installation
/// folder for GLib, and in the installation folder for the package the
/// application's .exe file belongs to.
///
/// The installation folders above are determined by looking up the
/// folder where the module (DLL or EXE) in question is located. If the
/// folder's name is "bin", its parent is used, otherwise the folder
/// itself.
///
/// Note that on Windows the returned list can vary depending on where
/// this function is called.
///
/// The return value is cached and modifying it at runtime is not supported, as
/// it’s not thread-safe to modify environment variables at runtime.
extern fn g_get_system_data_dirs() [*]const [*:0]const u8;
pub const getSystemDataDirs = g_get_system_data_dirs;

/// Gets the directory to use for temporary files.
///
/// On UNIX, this is taken from the `TMPDIR` environment variable.
/// If the variable is not set, `P_tmpdir` is
/// used, as defined by the system C library. Failing that, a
/// hard-coded default of "/tmp" is returned.
///
/// On Windows, the `TEMP` environment variable is used, with the
/// root directory of the Windows installation (eg: "C:\") used
/// as a default.
///
/// The encoding of the returned string is system-defined. On Windows,
/// it is always UTF-8. The return value is never `NULL` or the empty
/// string.
extern fn g_get_tmp_dir() [*:0]const u8;
pub const getTmpDir = g_get_tmp_dir;

/// Returns a base directory in which to store non-essential, cached
/// data specific to particular user.
///
/// On UNIX platforms this is determined using the mechanisms described
/// in the
/// [XDG Base Directory Specification](http://www.freedesktop.org/Standards/basedir-spec).
/// In this case the directory retrieved will be `XDG_CACHE_HOME`.
///
/// On Windows it follows XDG Base Directory Specification if `XDG_CACHE_HOME` is defined.
/// If `XDG_CACHE_HOME` is undefined, the directory that serves as a common
/// repository for temporary Internet files is used instead. A typical path is
/// `C:\Documents and Settings\username\Local Settings\Temporary Internet Files`.
/// See the [documentation for `FOLDERID_InternetCache`](https://docs.microsoft.com/en-us/windows/win32/shell/knownfolderid).
///
/// The return value is cached and modifying it at runtime is not supported, as
/// it’s not thread-safe to modify environment variables at runtime.
extern fn g_get_user_cache_dir() [*:0]const u8;
pub const getUserCacheDir = g_get_user_cache_dir;

/// Returns a base directory in which to store user-specific application
/// configuration information such as user preferences and settings.
///
/// On UNIX platforms this is determined using the mechanisms described
/// in the
/// [XDG Base Directory Specification](http://www.freedesktop.org/Standards/basedir-spec).
/// In this case the directory retrieved will be `XDG_CONFIG_HOME`.
///
/// On Windows it follows XDG Base Directory Specification if `XDG_CONFIG_HOME` is defined.
/// If `XDG_CONFIG_HOME` is undefined, the folder to use for local (as opposed
/// to roaming) application data is used instead. See the
/// [documentation for `FOLDERID_LocalAppData`](https://docs.microsoft.com/en-us/windows/win32/shell/knownfolderid).
/// Note that in this case on Windows it will be  the same
/// as what `glib.getUserDataDir` returns.
///
/// The return value is cached and modifying it at runtime is not supported, as
/// it’s not thread-safe to modify environment variables at runtime.
extern fn g_get_user_config_dir() [*:0]const u8;
pub const getUserConfigDir = g_get_user_config_dir;

/// Returns a base directory in which to access application data such
/// as icons that is customized for a particular user.
///
/// On UNIX platforms this is determined using the mechanisms described
/// in the
/// [XDG Base Directory Specification](http://www.freedesktop.org/Standards/basedir-spec).
/// In this case the directory retrieved will be `XDG_DATA_HOME`.
///
/// On Windows it follows XDG Base Directory Specification if `XDG_DATA_HOME`
/// is defined. If `XDG_DATA_HOME` is undefined, the folder to use for local (as
/// opposed to roaming) application data is used instead. See the
/// [documentation for `FOLDERID_LocalAppData`](https://docs.microsoft.com/en-us/windows/win32/shell/knownfolderid).
/// Note that in this case on Windows it will be the same
/// as what `glib.getUserConfigDir` returns.
///
/// The return value is cached and modifying it at runtime is not supported, as
/// it’s not thread-safe to modify environment variables at runtime.
extern fn g_get_user_data_dir() [*:0]const u8;
pub const getUserDataDir = g_get_user_data_dir;

/// Gets the user name of the current user. The encoding of the returned
/// string is system-defined. On UNIX, it might be the preferred file name
/// encoding, or something else, and there is no guarantee that it is even
/// consistent on a machine. On Windows, it is always UTF-8.
extern fn g_get_user_name() [*:0]const u8;
pub const getUserName = g_get_user_name;

/// Returns a directory that is unique to the current user on the local
/// system.
///
/// This is determined using the mechanisms described
/// in the
/// [XDG Base Directory Specification](http://www.freedesktop.org/Standards/basedir-spec).
/// This is the directory
/// specified in the `XDG_RUNTIME_DIR` environment variable.
/// In the case that this variable is not set, we return the value of
/// `glib.getUserCacheDir`, after verifying that it exists.
///
/// The return value is cached and modifying it at runtime is not supported, as
/// it’s not thread-safe to modify environment variables at runtime.
extern fn g_get_user_runtime_dir() [*:0]const u8;
pub const getUserRuntimeDir = g_get_user_runtime_dir;

/// Returns the full path of a special directory using its logical id.
///
/// On UNIX this is done using the XDG special user directories.
/// For compatibility with existing practise, `G_USER_DIRECTORY_DESKTOP`
/// falls back to `$HOME/Desktop` when XDG special user directories have
/// not been set up.
///
/// Depending on the platform, the user might be able to change the path
/// of the special directory without requiring the session to restart; GLib
/// will not reflect any change once the special directories are loaded.
extern fn g_get_user_special_dir(p_directory: glib.UserDirectory) ?[*:0]const u8;
pub const getUserSpecialDir = g_get_user_special_dir;

/// Returns a base directory in which to store state files specific to
/// particular user.
///
/// On UNIX platforms this is determined using the mechanisms described
/// in the
/// [XDG Base Directory Specification](http://www.freedesktop.org/Standards/basedir-spec).
/// In this case the directory retrieved will be `XDG_STATE_HOME`.
///
/// On Windows it follows XDG Base Directory Specification if `XDG_STATE_HOME` is defined.
/// If `XDG_STATE_HOME` is undefined, the folder to use for local (as opposed
/// to roaming) application data is used instead. See the
/// [documentation for `FOLDERID_LocalAppData`](https://docs.microsoft.com/en-us/windows/win32/shell/knownfolderid).
/// Note that in this case on Windows it will be the same
/// as what `glib.getUserDataDir` returns.
///
/// The return value is cached and modifying it at runtime is not supported, as
/// it’s not thread-safe to modify environment variables at runtime.
extern fn g_get_user_state_dir() [*:0]const u8;
pub const getUserStateDir = g_get_user_state_dir;

/// Returns the value of an environment variable.
///
/// On UNIX, the name and value are byte strings which might or might not
/// be in some consistent character set and encoding. On Windows, they are
/// in UTF-8.
/// On Windows, in case the environment variable's value contains
/// references to other environment variables, they are expanded.
extern fn g_getenv(p_variable: [*:0]const u8) ?[*:0]const u8;
pub const getenv = g_getenv;

/// Tests if `hostname` contains segments with an ASCII-compatible
/// encoding of an Internationalized Domain Name. If this returns
/// `TRUE`, you should decode the hostname with `glib.hostnameToUnicode`
/// before displaying it to the user.
///
/// Note that a hostname might contain a mix of encoded and unencoded
/// segments, and so it is possible for `glib.hostnameIsNonAscii` and
/// `glib.hostnameIsAsciiEncoded` to both return `TRUE` for a name.
extern fn g_hostname_is_ascii_encoded(p_hostname: [*:0]const u8) c_int;
pub const hostnameIsAsciiEncoded = g_hostname_is_ascii_encoded;

/// Tests if `hostname` is the string form of an IPv4 or IPv6 address.
/// (Eg, "192.168.0.1".)
///
/// Since 2.66, IPv6 addresses with a zone-id are accepted (RFC6874).
extern fn g_hostname_is_ip_address(p_hostname: [*:0]const u8) c_int;
pub const hostnameIsIpAddress = g_hostname_is_ip_address;

/// Tests if `hostname` contains Unicode characters. If this returns
/// `TRUE`, you need to encode the hostname with `glib.hostnameToAscii`
/// before using it in non-IDN-aware contexts.
///
/// Note that a hostname might contain a mix of encoded and unencoded
/// segments, and so it is possible for `glib.hostnameIsNonAscii` and
/// `glib.hostnameIsAsciiEncoded` to both return `TRUE` for a name.
extern fn g_hostname_is_non_ascii(p_hostname: [*:0]const u8) c_int;
pub const hostnameIsNonAscii = g_hostname_is_non_ascii;

/// Converts `hostname` to its canonical ASCII form; an ASCII-only
/// string containing no uppercase letters and not ending with a
/// trailing dot.
extern fn g_hostname_to_ascii(p_hostname: [*:0]const u8) ?[*:0]u8;
pub const hostnameToAscii = g_hostname_to_ascii;

/// Converts `hostname` to its canonical presentation form; a UTF-8
/// string in Unicode normalization form C, containing no uppercase
/// letters, no forbidden characters, and no ASCII-encoded segments,
/// and not ending with a trailing dot.
///
/// Of course if `hostname` is not an internationalized hostname, then
/// the canonical presentation form will be entirely ASCII.
extern fn g_hostname_to_unicode(p_hostname: [*:0]const u8) ?[*:0]u8;
pub const hostnameToUnicode = g_hostname_to_unicode;

/// Same as the standard UNIX routine `iconv`, but
/// may be implemented via libiconv on UNIX flavors that lack
/// a native implementation.
///
/// GLib provides `glib.convert` and `glib.localeToUtf8` which are likely
/// more convenient than the raw iconv wrappers.
///
/// Note that the behaviour of `iconv` for characters which are valid in the
/// input character set, but which have no representation in the output character
/// set, is implementation defined. This function may return success (with a
/// positive number of non-reversible conversions as replacement characters were
/// used), or it may return -1 and set an error such as `EILSEQ`, in such a
/// situation.
///
/// See [`iconv(3posix)`](man:iconv(3posix)) and [`iconv(3)`](man:iconv(3)) for more details about behavior when an
/// error occurs.
extern fn g_iconv(p_converter: glib.IConv, p_inbuf: *[*:0]u8, p_inbytes_left: *usize, p_outbuf: *[*:0]u8, p_outbytes_left: *usize) usize;
pub const iconv = g_iconv;

/// Adds a function to be called whenever there are no higher priority
/// events pending to the default main loop. The function is given the
/// default idle priority, `glib.PRIORITY_DEFAULT_IDLE`.  If the function
/// returns `FALSE` it is automatically removed from the list of event
/// sources and will not be called again.
///
/// See [mainloop memory management](main-loop.html`memory`-management-of-sources) for details
/// on how to handle the return value and memory management of `data`.
///
/// This internally creates a main loop source using `glib.idleSourceNew`
/// and attaches it to the global `glib.MainContext` using
/// `glib.Source.attach`, so the callback will be invoked in whichever
/// thread is running that main context. You can do these steps manually if you
/// need greater control or to use a custom main context.
extern fn g_idle_add(p_function: glib.SourceFunc, p_data: ?*anyopaque) c_uint;
pub const idleAdd = g_idle_add;

/// Adds a function to be called whenever there are no higher priority
/// events pending.
///
/// If the function returns `glib.SOURCE_REMOVE` or `FALSE` it is automatically
/// removed from the list of event sources and will not be called again.
///
/// See [mainloop memory management](main-loop.html`memory`-management-of-sources) for details
/// on how to handle the return value and memory management of `data`.
///
/// This internally creates a main loop source using `glib.idleSourceNew`
/// and attaches it to the global `glib.MainContext` using
/// `glib.Source.attach`, so the callback will be invoked in whichever
/// thread is running that main context. You can do these steps manually if you
/// need greater control or to use a custom main context.
extern fn g_idle_add_full(p_priority: c_int, p_function: glib.SourceFunc, p_data: ?*anyopaque, p_notify: ?glib.DestroyNotify) c_uint;
pub const idleAddFull = g_idle_add_full;

/// Adds a function to be called whenever there are no higher priority
/// events pending to the default main loop. The function is given the
/// default idle priority, `glib.PRIORITY_DEFAULT_IDLE`.
///
/// The function will only be called once and then the source will be
/// automatically removed from the main context.
///
/// This function otherwise behaves like `glib.idleAdd`.
extern fn g_idle_add_once(p_function: glib.SourceOnceFunc, p_data: ?*anyopaque) c_uint;
pub const idleAddOnce = g_idle_add_once;

/// Removes the idle function with the given data.
extern fn g_idle_remove_by_data(p_data: ?*anyopaque) c_int;
pub const idleRemoveByData = g_idle_remove_by_data;

/// Creates a new idle source.
///
/// The source will not initially be associated with any
/// `glib.MainContext` and must be added to one with
/// `glib.Source.attach` before it will be executed. Note that the
/// default priority for idle sources is `glib.PRIORITY_DEFAULT_IDLE`, as
/// compared to other sources which have a default priority of
/// `glib.PRIORITY_DEFAULT`.
extern fn g_idle_source_new() *glib.Source;
pub const idleSourceNew = g_idle_source_new;

/// Compares the two `gint64` values being pointed to and returns
/// `TRUE` if they are equal.
/// It can be passed to `glib.HashTable.new` as the `key_equal_func`
/// parameter, when using non-`NULL` pointers to 64-bit integers as keys in a
/// `glib.HashTable`.
extern fn g_int64_equal(p_v1: *const anyopaque, p_v2: *const anyopaque) c_int;
pub const int64Equal = g_int64_equal;

/// Converts a pointer to a `gint64` to a hash value.
///
/// It can be passed to `glib.HashTable.new` as the `hash_func` parameter,
/// when using non-`NULL` pointers to 64-bit integer values as keys in a
/// `glib.HashTable`.
extern fn g_int64_hash(p_v: *const anyopaque) c_uint;
pub const int64Hash = g_int64_hash;

/// Compares the two `gint` values being pointed to and returns
/// `TRUE` if they are equal.
/// It can be passed to `glib.HashTable.new` as the `key_equal_func`
/// parameter, when using non-`NULL` pointers to integers as keys in a
/// `glib.HashTable`.
///
/// Note that this function acts on pointers to `gint`, not on `gint`
/// directly: if your hash table's keys are of the form
/// `GINT_TO_POINTER (n)`, use `glib.directEqual` instead.
extern fn g_int_equal(p_v1: *const anyopaque, p_v2: *const anyopaque) c_int;
pub const intEqual = g_int_equal;

/// Converts a pointer to a `gint` to a hash value.
/// It can be passed to `glib.HashTable.new` as the `hash_func` parameter,
/// when using non-`NULL` pointers to integer values as keys in a `glib.HashTable`.
///
/// Note that this function acts on pointers to `gint`, not on `gint`
/// directly: if your hash table's keys are of the form
/// `GINT_TO_POINTER (n)`, use `glib.directHash` instead.
extern fn g_int_hash(p_v: *const anyopaque) c_uint;
pub const intHash = g_int_hash;

/// Returns a canonical representation for `string`. Interned strings
/// can be compared for equality by comparing the pointers, instead of
/// using `strcmp`. `glib.internStaticString` does not copy the string,
/// therefore `string` must not be freed or modified.
///
/// This function must not be used before library constructors have finished
/// running. In particular, this means it cannot be used to initialize global
/// variables in C++.
extern fn g_intern_static_string(p_string: ?[*:0]const u8) [*:0]const u8;
pub const internStaticString = g_intern_static_string;

/// Returns a canonical representation for `string`. Interned strings
/// can be compared for equality by comparing the pointers, instead of
/// using `strcmp`.
///
/// This function must not be used before library constructors have finished
/// running. In particular, this means it cannot be used to initialize global
/// variables in C++.
extern fn g_intern_string(p_string: ?[*:0]const u8) [*:0]const u8;
pub const internString = g_intern_string;

/// Adds the `glib.IOChannel` into the default main loop context
/// with the default priority.
extern fn g_io_add_watch(p_channel: *glib.IOChannel, p_condition: glib.IOCondition, p_func: glib.IOFunc, p_user_data: ?*anyopaque) c_uint;
pub const ioAddWatch = g_io_add_watch;

/// Adds the `glib.IOChannel` into the default main loop context
/// with the given priority.
///
/// This internally creates a main loop source using `glib.ioCreateWatch`
/// and attaches it to the main loop context with `glib.Source.attach`.
/// You can do these steps manually if you need greater control.
extern fn g_io_add_watch_full(p_channel: *glib.IOChannel, p_priority: c_int, p_condition: glib.IOCondition, p_func: glib.IOFunc, p_user_data: ?*anyopaque, p_notify: ?glib.DestroyNotify) c_uint;
pub const ioAddWatchFull = g_io_add_watch_full;

/// Creates a `glib.Source` that's dispatched when `condition` is met for the
/// given `channel`. For example, if condition is `G_IO_IN`, the source will
/// be dispatched when there's data available for reading.
///
/// The callback function invoked by the `glib.Source` should be added with
/// `glib.Source.setCallback`, but it has type `glib.IOFunc` (not `glib.SourceFunc`).
///
/// `glib.ioAddWatch` is a simpler interface to this same functionality, for
/// the case where you want to add the source to the default main loop context
/// at the default priority.
///
/// On Windows, polling a `glib.Source` created to watch a channel for a socket
/// puts the socket in non-blocking mode. This is a side-effect of the
/// implementation and unavoidable.
extern fn g_io_create_watch(p_channel: *glib.IOChannel, p_condition: glib.IOCondition) *glib.Source;
pub const ioCreateWatch = g_io_create_watch;

/// Gets the names of all variables set in the environment.
///
/// Programs that want to be portable to Windows should typically use
/// this function and `glib.getenv` instead of using the environ array
/// from the C library directly. On Windows, the strings in the environ
/// array are in system codepage encoding, while in most of the typical
/// use cases for environment variables in GLib-using programs you want
/// the UTF-8 encoding that this function and `glib.getenv` provide.
extern fn g_listenv() [*:null]?[*:0]u8;
pub const listenv = g_listenv;

/// Converts a string from UTF-8 to the encoding used for strings by
/// the C runtime (usually the same as that used by the operating
/// system) in the [current locale](running.html`locale`).
/// On Windows this means the system codepage.
///
/// The input string shall not contain nul characters even if the `len`
/// argument is positive. A nul character found inside the string will result
/// in error `G_CONVERT_ERROR_ILLEGAL_SEQUENCE`. Use `glib.convert` to convert
/// input that may contain embedded nul characters.
extern fn g_locale_from_utf8(p_utf8string: [*:0]const u8, p_len: isize, p_bytes_read: ?*usize, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) ?[*]u8;
pub const localeFromUtf8 = g_locale_from_utf8;

/// Converts a string which is in the encoding used for strings by
/// the C runtime (usually the same as that used by the operating
/// system) in the [current locale](running.html`locale`) into a UTF-8 string.
///
/// If the source encoding is not UTF-8 and the conversion output contains a
/// nul character, the error `G_CONVERT_ERROR_EMBEDDED_NUL` is set and the
/// function returns `NULL`.
/// If the source encoding is UTF-8, an embedded nul character is treated with
/// the `G_CONVERT_ERROR_ILLEGAL_SEQUENCE` error for backward compatibility with
/// earlier versions of this library. Use `glib.convert` to produce output that
/// may contain embedded nul characters.
extern fn g_locale_to_utf8(p_opsysstring: [*]const u8, p_len: isize, p_bytes_read: ?*usize, p_bytes_written: ?*usize, p_error: ?*?*glib.Error) ?[*:0]u8;
pub const localeToUtf8 = g_locale_to_utf8;

/// Logs an error or debugging message.
///
/// If the log level has been set as fatal, `glib.BREAKPOINT` is called
/// to terminate the program. See the documentation for `glib.BREAKPOINT` for
/// details of the debugging options this provides.
///
/// If `glib.logDefaultHandler` is used as the log handler function, a new-line
/// character will automatically be appended to @..., and need not be entered
/// manually.
///
/// If [structured logging is enabled](logging.html`using`-structured-logging) this will
/// output via the structured log writer function (see `glib.logSetWriterFunc`).
extern fn g_log(p_log_domain: ?[*:0]const u8, p_log_level: glib.LogLevelFlags, p_format: [*:0]const u8, ...) void;
pub const log = g_log;

/// The default log handler set up by GLib; `glib.logSetDefaultHandler`
/// allows to install an alternate default log handler.
///
/// This is used if no log handler has been set for the particular log
/// domain and log level combination. It outputs the message to `stderr`
/// or `stdout` and if the log level is fatal it calls `glib.BREAKPOINT`. It automatically
/// prints a new-line character after the message, so one does not need to be
/// manually included in `message`.
///
/// The behavior of this log handler can be influenced by a number of
/// environment variables:
///
///   - `G_MESSAGES_PREFIXED`: A `:`-separated list of log levels for which
///     messages should be prefixed by the program name and PID of the
///     application.
///   - `G_MESSAGES_DEBUG`: A space-separated list of log domains for
///     which debug and informational messages are printed. By default
///     these messages are not printed. If you need to set the allowed
///     domains at runtime, use `glib.logWriterDefaultSetDebugDomains`.
///   - `DEBUG_INVOCATION`: If set to `1`, this is equivalent to
///     `G_MESSAGES_DEBUG=all`. `DEBUG_INVOCATION` is a standard environment
///     variable set by systemd to prompt debug output. (Since: 2.84)
///
/// `stderr` is used for levels `glib.@"LogLevelFlags.LEVEL_ERROR"`,
/// `glib.@"LogLevelFlags.LEVEL_CRITICAL"`, `glib.@"LogLevelFlags.LEVEL_WARNING"` and
/// `glib.@"LogLevelFlags.LEVEL_MESSAGE"`. `stdout` is used for
/// the rest, unless `stderr` was requested by
/// `glib.logWriterDefaultSetUseStderr`.
///
/// This has no effect if structured logging is enabled; see
/// [Using Structured Logging](logging.html`using`-structured-logging).
extern fn g_log_default_handler(p_log_domain: ?[*:0]const u8, p_log_level: glib.LogLevelFlags, p_message: ?[*:0]const u8, p_unused_data: ?*anyopaque) void;
pub const logDefaultHandler = g_log_default_handler;

/// Return whether debug output from the GLib logging system is enabled.
///
/// Note that this should not be used to conditionalise calls to `glib.debug` or
/// other logging functions; it should only be used from `glib.LogWriterFunc`
/// implementations.
///
/// Note also that the value of this does not depend on `G_MESSAGES_DEBUG`, nor
/// `DEBUG_INVOCATION`, nor `glib.logWriterDefaultSetDebugDomains`; see
/// the docs for `glib.logSetDebugEnabled`.
extern fn g_log_get_debug_enabled() c_int;
pub const logGetDebugEnabled = g_log_get_debug_enabled;

/// Removes the log handler.
///
/// This has no effect if structured logging is enabled; see
/// [Using Structured Logging](logging.html`using`-structured-logging).
extern fn g_log_remove_handler(p_log_domain: [*:0]const u8, p_handler_id: c_uint) void;
pub const logRemoveHandler = g_log_remove_handler;

/// Sets the message levels which are always fatal, in any log domain.
///
/// When a message with any of these levels is logged the program terminates.
/// You can only set the levels defined by GLib to be fatal.
/// `glib.@"LogLevelFlags.LEVEL_ERROR"` is always fatal.
///
/// You can also make some message levels fatal at runtime by setting
/// the `G_DEBUG` environment variable (see
/// [Running GLib Applications](glib-running.html)).
///
/// Libraries should not call this function, as it affects all messages logged
/// by a process, including those from other libraries.
///
/// Structured log messages (using `glib.logStructured` and
/// `glib.logStructuredArray`) are fatal only if the default log writer is used;
/// otherwise it is up to the writer function to determine which log messages
/// are fatal. See [Using Structured Logging](logging.html`using`-structured-logging).
extern fn g_log_set_always_fatal(p_fatal_mask: glib.LogLevelFlags) glib.LogLevelFlags;
pub const logSetAlwaysFatal = g_log_set_always_fatal;

/// Enable or disable debug output from the GLib logging system for all domains.
///
/// This value interacts disjunctively with `G_MESSAGES_DEBUG`, `DEBUG_INVOCATION` and
/// `glib.logWriterDefaultSetDebugDomains` — if any of them would allow
/// a debug message to be outputted, it will be.
///
/// Note that this should not be used from within library code to enable debug
/// output — it is intended for external use.
extern fn g_log_set_debug_enabled(p_enabled: c_int) void;
pub const logSetDebugEnabled = g_log_set_debug_enabled;

/// Installs a default log handler which is used if no
/// log handler has been set for the particular log domain
/// and log level combination.
///
/// By default, GLib uses `glib.logDefaultHandler` as default log handler.
///
/// This has no effect if structured logging is enabled; see
/// [Using Structured Logging](logging.html`using`-structured-logging).
extern fn g_log_set_default_handler(p_log_func: glib.LogFunc, p_user_data: ?*anyopaque) glib.LogFunc;
pub const logSetDefaultHandler = g_log_set_default_handler;

/// Sets the log levels which are fatal in the given domain.
///
/// `glib.@"LogLevelFlags.LEVEL_ERROR"` is always fatal.
///
/// This has no effect on structured log messages (using `glib.logStructured` or
/// `glib.logStructuredArray`). To change the fatal behaviour for specific log
/// messages, programs must install a custom log writer function using
/// `glib.logSetWriterFunc`. See
/// [Using Structured Logging](logging.html`using`-structured-logging).
///
/// This function is mostly intended to be used with
/// `glib.@"LogLevelFlags.LEVEL_CRITICAL"`.  You should typically not set
/// `glib.@"LogLevelFlags.LEVEL_WARNING"`, `glib.@"LogLevelFlags.LEVEL_MESSAGE"`, `glib.@"LogLevelFlags.LEVEL_INFO"` or
/// `glib.@"LogLevelFlags.LEVEL_DEBUG"` as fatal except inside of test programs.
extern fn g_log_set_fatal_mask(p_log_domain: [*:0]const u8, p_fatal_mask: glib.LogLevelFlags) glib.LogLevelFlags;
pub const logSetFatalMask = g_log_set_fatal_mask;

/// Sets the log handler for a domain and a set of log levels.
///
/// To handle fatal and recursive messages the `log_levels` parameter
/// must be combined with the `glib.@"LogLevelFlags.FLAG_FATAL"` and `glib.@"LogLevelFlags.FLAG_RECURSION"`
/// bit flags.
///
/// Note that since the `glib.@"LogLevelFlags.LEVEL_ERROR"` log level is always fatal, if
/// you want to set a handler for this log level you must combine it with
/// `glib.@"LogLevelFlags.FLAG_FATAL"`.
///
/// This has no effect if structured logging is enabled; see
/// [Using Structured Logging](logging.html`using`-structured-logging).
///
/// The `log_domain` parameter can be set to `NULL` or an empty string to use the default
/// application domain.
///
/// Here is an example for adding a log handler for all warning messages
/// in the default domain:
///
/// ```c
/// g_log_set_handler (NULL, G_LOG_LEVEL_WARNING | G_LOG_FLAG_FATAL
///                    | G_LOG_FLAG_RECURSION, my_log_handler, NULL);
/// ```
///
/// This example adds a log handler for all critical messages from GTK:
///
/// ```c
/// g_log_set_handler ("Gtk", G_LOG_LEVEL_CRITICAL | G_LOG_FLAG_FATAL
///                    | G_LOG_FLAG_RECURSION, my_log_handler, NULL);
/// ```
///
/// This example adds a log handler for all messages from GLib:
///
/// ```c
/// g_log_set_handler ("GLib", G_LOG_LEVEL_MASK | G_LOG_FLAG_FATAL
///                    | G_LOG_FLAG_RECURSION, my_log_handler, NULL);
/// ```
extern fn g_log_set_handler(p_log_domain: ?[*:0]const u8, p_log_levels: glib.LogLevelFlags, p_log_func: glib.LogFunc, p_user_data: ?*anyopaque) c_uint;
pub const logSetHandler = g_log_set_handler;

/// Like `glib.logSetHandler`, but takes a destroy notify for the `user_data`.
///
/// This has no effect if structured logging is enabled; see
/// [Using Structured Logging](logging.html`using`-structured-logging).
///
/// The `log_domain` parameter can be set to `NULL` or an empty string to use the default
/// application domain.
extern fn g_log_set_handler_full(p_log_domain: ?[*:0]const u8, p_log_levels: glib.LogLevelFlags, p_log_func: glib.LogFunc, p_user_data: ?*anyopaque, p_destroy: ?glib.DestroyNotify) c_uint;
pub const logSetHandlerFull = g_log_set_handler_full;

/// Set a writer function which will be called to format and write out each log
/// message.
///
/// Each program should set a writer function, or the default writer
/// (`glib.logWriterDefault`) will be used.
///
/// Libraries **must not** call this function — only programs are allowed to
/// install a writer function, as there must be a single, central point where
/// log messages are formatted and outputted.
///
/// There can only be one writer function. It is an error to set more than one.
extern fn g_log_set_writer_func(p_func: ?glib.LogWriterFunc, p_user_data: ?*anyopaque, p_user_data_free: ?glib.DestroyNotify) void;
pub const logSetWriterFunc = g_log_set_writer_func;

/// Log a message with structured data.
///
/// The message will be passed through to the log writer set by the application
/// using `glib.logSetWriterFunc`. If the message is fatal (i.e. its log level
/// is `glib.@"LogLevelFlags.LEVEL_ERROR"`), the program will be aborted by calling
/// `glib.BREAKPOINT` at the end of this function. If the log writer returns
/// `glib.@"LogWriterOutput.UNHANDLED"` (failure), no other fallback writers will be tried.
/// See the documentation for `glib.LogWriterFunc` for information on chaining
/// writers.
///
/// The structured data is provided as key–value pairs, where keys are UTF-8
/// strings, and values are arbitrary pointers — typically pointing to UTF-8
/// strings, but that is not a requirement. To pass binary (non-nul-terminated)
/// structured data, use `glib.logStructuredArray`. The keys for structured data
/// should follow the [systemd journal
/// fields](https://www.freedesktop.org/software/systemd/man/systemd.journal-fields.html)
/// specification. It is suggested that custom keys are namespaced according to
/// the code which sets them. For example, custom keys from GLib all have a
/// `GLIB_` prefix.
///
/// Note that keys that expect UTF-8 strings (specifically `"MESSAGE"` and
/// `"GLIB_DOMAIN"`) must be passed as nul-terminated UTF-8 strings until GLib
/// version 2.74.1 because the default log handler did not consider the length of
/// the `GLogField`. Starting with GLib 2.74.1 this is fixed and
/// non-nul-terminated UTF-8 strings can be passed with their correct length,
/// with the exception of `"GLIB_DOMAIN"` which was only fixed with GLib 2.82.3.
///
/// The `log_domain` will be converted into a `GLIB_DOMAIN` field. `log_level` will
/// be converted into a
/// [`PRIORITY`](https://www.freedesktop.org/software/systemd/man/systemd.journal-fields.html`PRIORITY`=)
/// field. The format string will have its placeholders substituted for the provided
/// values and be converted into a
/// [`MESSAGE`](https://www.freedesktop.org/software/systemd/man/systemd.journal-fields.html`MESSAGE`=)
/// field.
///
/// Other fields you may commonly want to pass into this function:
///
///  * [`MESSAGE_ID`](https://www.freedesktop.org/software/systemd/man/systemd.journal-fields.html`MESSAGE_ID`=)
///  * [`CODE_FILE`](https://www.freedesktop.org/software/systemd/man/systemd.journal-fields.html`CODE_FILE`=)
///  * [`CODE_LINE`](https://www.freedesktop.org/software/systemd/man/systemd.journal-fields.html`CODE_LINE`=)
///  * [`CODE_FUNC`](https://www.freedesktop.org/software/systemd/man/systemd.journal-fields.html`CODE_FUNC`=)
///  * [`ERRNO`](https://www.freedesktop.org/software/systemd/man/systemd.journal-fields.html`ERRNO`=)
///
/// Note that `CODE_FILE`, `CODE_LINE` and `CODE_FUNC` are automatically set by
/// the logging macros, `glib.DEBUGHERE`, `glib.message`, `glib.warning`, `glib.critical`,
/// `glib.@"error"`, etc, if the symbol `G_LOG_USE_STRUCTURED` is defined before including
/// `glib.h`.
///
/// For example:
///
/// ```c
/// g_log_structured (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG,
///                   "MESSAGE_ID", "06d4df59e6c24647bfe69d2c27ef0b4e",
///                   "MY_APPLICATION_CUSTOM_FIELD", "some debug string",
///                   "MESSAGE", "This is a debug message about pointer `p` and integer `u`.",
///                   some_pointer, some_integer);
/// ```
///
/// Note that each `MESSAGE_ID` must be [uniquely and randomly
/// generated](https://www.freedesktop.org/software/systemd/man/systemd.journal-fields.html`MESSAGE_ID`=).
/// If adding a `MESSAGE_ID`, consider shipping a [message
/// catalog](https://www.freedesktop.org/wiki/Software/systemd/catalog/) with
/// your software.
///
/// To pass a user data pointer to the log writer function which is specific to
/// this logging call, you must use `glib.logStructuredArray` and pass the pointer
/// as a field with `GLogField.length` set to zero, otherwise it will be
/// interpreted as a string.
///
/// For example:
///
/// ```c
/// const GLogField fields[] = {
///   { "MESSAGE", "This is a debug message.", -1 },
///   { "MESSAGE_ID", "fcfb2e1e65c3494386b74878f1abf893", -1 },
///   { "MY_APPLICATION_CUSTOM_FIELD", "some debug string", -1 },
///   { "MY_APPLICATION_STATE", state_object, 0 },
/// };
/// g_log_structured_array (G_LOG_LEVEL_DEBUG, fields, G_N_ELEMENTS (fields));
/// ```
///
/// Note also that, even if no other structured fields are specified, there
/// must always be a `MESSAGE` key before the format string. The `MESSAGE`-format
/// pair has to be the last of the key-value pairs, and `MESSAGE` is the only
/// field for which ``printf``-style formatting is supported.
///
/// The default writer function for `stdout` and `stderr` will automatically
/// append a new-line character after the message, so you should not add one
/// manually to the format string.
extern fn g_log_structured(p_log_domain: [*:0]const u8, p_log_level: glib.LogLevelFlags, ...) void;
pub const logStructured = g_log_structured;

/// Log a message with structured data.
///
/// The message will be passed through to the log writer set by the application
/// using `glib.logSetWriterFunc`. If the
/// message is fatal (i.e. its log level is `glib.@"LogLevelFlags.LEVEL_ERROR"`), the program will
/// be aborted at the end of this function.
///
/// See `glib.logStructured` for more documentation.
///
/// This assumes that `log_level` is already present in `fields` (typically as the
/// `PRIORITY` field).
extern fn g_log_structured_array(p_log_level: glib.LogLevelFlags, p_fields: [*]const glib.LogField, p_n_fields: usize) void;
pub const logStructuredArray = g_log_structured_array;

extern fn g_log_structured_standard(p_log_domain: [*:0]const u8, p_log_level: glib.LogLevelFlags, p_file: [*:0]const u8, p_line: [*:0]const u8, p_func: [*:0]const u8, p_message_format: [*:0]const u8, ...) void;
pub const logStructuredStandard = g_log_structured_standard;

/// Log a message with structured data, accepting the data within a `glib.Variant`.
///
/// This version is especially useful for use in other languages, via introspection.
///
/// The only mandatory item in the `fields` dictionary is the `"MESSAGE"` which must
/// contain the text shown to the user.
///
/// The values in the `fields` dictionary are likely to be of type `G_VARIANT_TYPE_STRING`.
/// Array of bytes (`G_VARIANT_TYPE_BYTESTRING`) is also
/// supported. In this case the message is handled as binary and will be forwarded
/// to the log writer as such. The size of the array should not be higher than
/// `G_MAXSSIZE`. Otherwise it will be truncated to this size. For other types
/// `glib.Variant.print` will be used to convert the value into a string.
///
/// For more details on its usage and about the parameters, see `glib.logStructured`.
extern fn g_log_variant(p_log_domain: ?[*:0]const u8, p_log_level: glib.LogLevelFlags, p_fields: *glib.Variant) void;
pub const logVariant = g_log_variant;

/// Format a structured log message and output it to the default log destination
/// for the platform.
///
/// On Linux, this is typically the systemd journal, falling
/// back to `stdout` or `stderr` if running from the terminal or if output is
/// being redirected to a file.
///
/// Support for other platform-specific logging mechanisms may be added in
/// future. Distributors of GLib may modify this function to impose their own
/// (documented) platform-specific log writing policies.
///
/// This is suitable for use as a `glib.LogWriterFunc`, and is the default writer used
/// if no other is set using `glib.logSetWriterFunc`.
///
/// As with `glib.logDefaultHandler`, this function drops debug and informational
/// messages unless their log domain (or `all`) is listed in the space-separated
/// `G_MESSAGES_DEBUG` environment variable, or `DEBUG_INVOCATION=1` is set in
/// the environment, or set at runtime by `glib.logWriterDefaultSetDebugDomains`.
///
/// `glib.logWriterDefault` uses the mask set by `glib.logSetAlwaysFatal` to
/// determine which messages are fatal. When using a custom writer function instead it is
/// up to the writer function to determine which log messages are fatal.
extern fn g_log_writer_default(p_log_level: glib.LogLevelFlags, p_fields: [*]const glib.LogField, p_n_fields: usize, p_user_data: ?*anyopaque) glib.LogWriterOutput;
pub const logWriterDefault = g_log_writer_default;

/// Reset the list of domains to be logged, that might be initially set by the
/// `G_MESSAGES_DEBUG` or `DEBUG_INVOCATION` environment variables.
///
/// This function is thread-safe.
extern fn g_log_writer_default_set_debug_domains(p_domains: ?*const [*:0]const u8) void;
pub const logWriterDefaultSetDebugDomains = g_log_writer_default_set_debug_domains;

/// Configure whether the built-in log functions will output all log messages to
/// `stderr`.
///
/// The built-in log functions are `glib.logDefaultHandler` for the
/// old-style API, and both `glib.logWriterDefault` and
/// `glib.logWriterStandardStreams` for the structured API.
///
/// By default, log messages of levels `glib.@"LogLevelFlags.LEVEL_INFO"` and
/// `glib.@"LogLevelFlags.LEVEL_DEBUG"` are sent to `stdout`, and other log messages are
/// sent to `stderr`. This is problematic for applications that intend
/// to reserve `stdout` for structured output such as JSON or XML.
///
/// This function sets global state. It is not thread-aware, and should be
/// called at the very start of a program, before creating any other threads
/// or creating objects that could create worker threads of their own.
extern fn g_log_writer_default_set_use_stderr(p_use_stderr: c_int) void;
pub const logWriterDefaultSetUseStderr = g_log_writer_default_set_use_stderr;

/// Check whether `glib.logWriterDefault` and `glib.logDefaultHandler` would
/// ignore a message with the given domain and level.
///
/// As with `glib.logDefaultHandler`, this function drops debug and informational
/// messages unless their log domain (or `all`) is listed in the space-separated
/// `G_MESSAGES_DEBUG` environment variable, or `DEBUG_INVOCATION=1` is set in
/// the environment, or by `glib.logWriterDefaultSetDebugDomains`.
///
/// This can be used when implementing log writers with the same filtering
/// behaviour as the default, but a different destination or output format:
///
/// ```c
/// if (g_log_writer_default_would_drop (log_level, log_domain))
///   return G_LOG_WRITER_HANDLED;
/// ```
///
/// or to skip an expensive computation if it is only needed for a debugging
/// message, and `G_MESSAGES_DEBUG` and `DEBUG_INVOCATION` are not set:
///
/// ```c
/// if (!g_log_writer_default_would_drop (G_LOG_LEVEL_DEBUG, G_LOG_DOMAIN))
///   {
///     g_autofree gchar *result = expensive_computation (my_object);
///
///     g_debug ("my_object result: `s`", result);
///   }
/// ```
extern fn g_log_writer_default_would_drop(p_log_level: glib.LogLevelFlags, p_log_domain: ?[*:0]const u8) c_int;
pub const logWriterDefaultWouldDrop = g_log_writer_default_would_drop;

/// Format a structured log message as a string suitable for outputting to the
/// terminal (or elsewhere).
///
/// This will include the values of all fields it knows
/// how to interpret, which includes `MESSAGE` and `GLIB_DOMAIN` (see the
/// documentation for `glib.logStructured`). It does not include values from
/// unknown fields.
///
/// The returned string does **not** have a trailing new-line character. It is
/// encoded in the character set of the current locale, which is not necessarily
/// UTF-8.
extern fn g_log_writer_format_fields(p_log_level: glib.LogLevelFlags, p_fields: [*]const glib.LogField, p_n_fields: usize, p_use_color: c_int) [*:0]u8;
pub const logWriterFormatFields = g_log_writer_format_fields;

/// Check whether the given `output_fd` file descriptor is a connection to the
/// systemd journal, or something else (like a log file or `stdout` or
/// `stderr`).
///
/// Invalid file descriptors are accepted and return `FALSE`, which allows for
/// the following construct without needing any additional error handling:
/// ```c
/// is_journald = g_log_writer_is_journald (fileno (stderr));
/// ```
extern fn g_log_writer_is_journald(p_output_fd: c_int) c_int;
pub const logWriterIsJournald = g_log_writer_is_journald;

/// Format a structured log message and send it to the systemd journal as a set
/// of key–value pairs.
///
/// All fields are sent to the journal, but if a field has
/// length zero (indicating program-specific data) then only its key will be
/// sent.
///
/// This is suitable for use as a `glib.LogWriterFunc`.
///
/// If GLib has been compiled without systemd support, this function is still
/// defined, but will always return `glib.@"LogWriterOutput.UNHANDLED"`.
extern fn g_log_writer_journald(p_log_level: glib.LogLevelFlags, p_fields: [*]const glib.LogField, p_n_fields: usize, p_user_data: ?*anyopaque) glib.LogWriterOutput;
pub const logWriterJournald = g_log_writer_journald;

/// Format a structured log message and print it to either `stdout` or `stderr`,
/// depending on its log level.
///
/// `glib.@"LogLevelFlags.LEVEL_INFO"` and `glib.@"LogLevelFlags.LEVEL_DEBUG"` messages
/// are sent to `stdout`, or to `stderr` if requested by
/// `glib.logWriterDefaultSetUseStderr`;
/// all other log levels are sent to `stderr`. Only fields
/// which are understood by this function are included in the formatted string
/// which is printed.
///
/// If the output stream supports
/// [ANSI color escape sequences](https://en.wikipedia.org/wiki/ANSI_escape_code),
/// they will be used in the output.
///
/// A trailing new-line character is added to the log message when it is printed.
///
/// This is suitable for use as a `glib.LogWriterFunc`.
extern fn g_log_writer_standard_streams(p_log_level: glib.LogLevelFlags, p_fields: [*]const glib.LogField, p_n_fields: usize, p_user_data: ?*anyopaque) glib.LogWriterOutput;
pub const logWriterStandardStreams = g_log_writer_standard_streams;

/// Check whether the given `output_fd` file descriptor supports
/// [ANSI color escape sequences](https://en.wikipedia.org/wiki/ANSI_escape_code).
///
/// If so, they can safely be used when formatting log messages.
extern fn g_log_writer_supports_color(p_output_fd: c_int) c_int;
pub const logWriterSupportsColor = g_log_writer_supports_color;

/// Format a structured log message and send it to the syslog daemon. Only fields
/// which are understood by this function are included in the formatted string
/// which is printed.
///
/// Log facility will be defined via the SYSLOG_FACILITY field and accepts the following
/// values: "auth", "daemon", and "user". If SYSLOG_FACILITY is not specified, LOG_USER
/// facility will be used.
///
/// This is suitable for use as a `glib.LogWriterFunc`.
///
/// If syslog is not supported, this function is still defined, but will always
/// return `glib.@"LogWriterOutput.UNHANDLED"`.
extern fn g_log_writer_syslog(p_log_level: glib.LogLevelFlags, p_fields: [*]const glib.LogField, p_n_fields: usize, p_user_data: ?*anyopaque) glib.LogWriterOutput;
pub const logWriterSyslog = g_log_writer_syslog;

/// Logs an error or debugging message.
///
/// If the log level has been set as fatal, `glib.BREAKPOINT` is called
/// to terminate the program. See the documentation for `glib.BREAKPOINT` for
/// details of the debugging options this provides.
///
/// If `glib.logDefaultHandler` is used as the log handler function, a new-line
/// character will automatically be appended to @..., and need not be entered
/// manually.
///
/// If [structured logging is enabled](logging.html`using`-structured-logging) this will
/// output via the structured log writer function (see `glib.logSetWriterFunc`).
///
/// The `log_domain` parameter can be set to `NULL` or an empty string to use the default
/// application domain.
extern fn g_logv(p_log_domain: ?[*:0]const u8, p_log_level: glib.LogLevelFlags, p_format: [*:0]const u8, p_args: std.builtin.VaList) void;
pub const logv = g_logv;

/// A wrapper for the POSIX `lstat` function. The `lstat` function is
/// like `stat` except that in the case of symbolic links, it returns
/// information about the symbolic link itself and not the file that it
/// refers to. If the system does not support symbolic links `glib.lstat`
/// is identical to `glib.stat`.
///
/// See your C library manual for more details about `lstat`.
extern fn g_lstat(p_filename: [*:0]const u8, p_buf: *glib.StatBuf) c_int;
pub const lstat = g_lstat;

/// Returns the currently firing source for this thread.
extern fn g_main_current_source() ?*glib.Source;
pub const mainCurrentSource = g_main_current_source;

/// Returns the depth of the stack of calls to
/// `glib.MainContext.dispatch` on any `glib.MainContext` in the current thread.
/// That is, when called from the toplevel, it gives 0. When
/// called from within a callback from `glib.MainContext.iteration`
/// (or `glib.MainLoop.run`, etc.) it returns 1. When called from within
/// a callback to a recursive call to `glib.MainContext.iteration`,
/// it returns 2. And so forth.
///
/// This function is useful in a situation like the following:
/// Imagine an extremely simple "garbage collected" system.
///
/// ```
/// static GList *free_list;
///
/// gpointer
/// allocate_memory (gsize size)
/// {
///   gpointer result = g_malloc (size);
///   free_list = g_list_prepend (free_list, result);
///   return result;
/// }
///
/// void
/// free_allocated_memory (void)
/// {
///   GList *l;
///   for (l = free_list; l; l = l->next);
///     g_free (l->data);
///   g_list_free (free_list);
///   free_list = NULL;
///  }
///
/// [...]
///
/// while (TRUE);
///  {
///    g_main_context_iteration (NULL, TRUE);
///    `free_allocated_memory`;
///   }
/// ```
///
/// This works from an application, however, if you want to do the same
/// thing from a library, it gets more difficult, since you no longer
/// control the main loop. You might think you can simply use an idle
/// function to make the call to `free_allocated_memory`, but that
/// doesn't work, since the idle function could be called from a
/// recursive callback. This can be fixed by using `glib.mainDepth`
///
/// ```
/// gpointer
/// allocate_memory (gsize size)
/// {
///   FreeListBlock *block = g_new (FreeListBlock, 1);
///   block->mem = g_malloc (size);
///   block->depth = g_main_depth ();
///   free_list = g_list_prepend (free_list, block);
///   return block->mem;
/// }
///
/// void
/// free_allocated_memory (void)
/// {
///   GList *l;
///
///   int depth = g_main_depth ();
///   for (l = free_list; l; );
///     {
///       GList *next = l->next;
///       FreeListBlock *block = l->data;
///       if (block->depth > depth)
///         {
///           g_free (block->mem);
///           g_free (block);
///           free_list = g_list_delete_link (free_list, l);
///         }
///
///       l = next;
///     }
///   }
/// ```
///
/// There is a temptation to use `glib.mainDepth` to solve
/// problems with reentrancy. For instance, while waiting for data
/// to be received from the network in response to a menu item,
/// the menu item might be selected again. It might seem that
/// one could make the menu item's callback return immediately
/// and do nothing if `glib.mainDepth` returns a value greater than 1.
/// However, this should be avoided since the user then sees selecting
/// the menu item do nothing. Furthermore, you'll find yourself adding
/// these checks all over your code, since there are doubtless many,
/// many things that the user could do. Instead, you can use the
/// following techniques:
///
/// 1. Use `gtk_widget_set_sensitive` or modal dialogs to prevent
///    the user from interacting with elements while the main
///    loop is recursing.
///
/// 2. Avoid main loop recursion in situations where you can't handle
///    arbitrary  callbacks. Instead, structure your code so that you
///    simply return to the main loop and then get called again when
///    there is more work to do.
extern fn g_main_depth() c_int;
pub const mainDepth = g_main_depth;

/// Allocates `n_bytes` bytes of memory.
/// If `n_bytes` is 0 it returns `NULL`.
///
/// If the allocation fails (because the system is out of memory),
/// the program is terminated.
extern fn g_malloc(p_n_bytes: usize) ?*anyopaque;
pub const malloc = g_malloc;

/// Allocates `n_bytes` bytes of memory, initialized to 0's.
/// If `n_bytes` is 0 it returns `NULL`.
///
/// If the allocation fails (because the system is out of memory),
/// the program is terminated.
extern fn g_malloc0(p_n_bytes: usize) ?*anyopaque;
pub const malloc0 = g_malloc0;

/// This function is similar to `glib.malloc0`, allocating (`n_blocks` * `n_block_bytes`) bytes,
/// but care is taken to detect possible overflow during multiplication.
///
/// If the allocation fails (because the system is out of memory),
/// the program is terminated.
extern fn g_malloc0_n(p_n_blocks: usize, p_n_block_bytes: usize) ?*anyopaque;
pub const malloc0N = g_malloc0_n;

/// This function is similar to `glib.malloc`, allocating (`n_blocks` * `n_block_bytes`) bytes,
/// but care is taken to detect possible overflow during multiplication.
///
/// If the allocation fails (because the system is out of memory),
/// the program is terminated.
extern fn g_malloc_n(p_n_blocks: usize, p_n_block_bytes: usize) ?*anyopaque;
pub const mallocN = g_malloc_n;

/// Collects the attributes of the element from the data passed to the
/// `glib.MarkupParser` start_element function, dealing with common error
/// conditions and supporting boolean values.
///
/// This utility function is not required to write a parser but can save
/// a lot of typing.
///
/// The `element_name`, `attribute_names`, `attribute_values` and `error`
/// parameters passed to the start_element callback should be passed
/// unmodified to this function.
///
/// Following these arguments is a list of "supported" attributes to collect.
/// It is an error to specify multiple attributes with the same name. If any
/// attribute not in the list appears in the `attribute_names` array then an
/// unknown attribute error will result.
///
/// The `glib.MarkupCollectType` field allows specifying the type of collection
/// to perform and if a given attribute must appear or is optional.
///
/// The attribute name is simply the name of the attribute to collect.
///
/// The pointer should be of the appropriate type (see the descriptions
/// under `glib.MarkupCollectType`) and may be `NULL` in case a particular
/// attribute is to be allowed but ignored.
///
/// This function deals with issuing errors for missing attributes
/// (of type `G_MARKUP_ERROR_MISSING_ATTRIBUTE`), unknown attributes
/// (of type `G_MARKUP_ERROR_UNKNOWN_ATTRIBUTE`) and duplicate
/// attributes (of type `G_MARKUP_ERROR_INVALID_CONTENT`) as well
/// as parse errors for boolean-valued attributes (again of type
/// `G_MARKUP_ERROR_INVALID_CONTENT`). In all of these cases `FALSE`
/// will be returned and `error` will be set as appropriate.
extern fn g_markup_collect_attributes(p_element_name: [*:0]const u8, p_attribute_names: *[*:0]const u8, p_attribute_values: *[*:0]const u8, p_error: **glib.Error, p_first_type: glib.MarkupCollectType, p_first_attr: [*:0]const u8, ...) c_int;
pub const markupCollectAttributes = g_markup_collect_attributes;

extern fn g_markup_error_quark() glib.Quark;
pub const markupErrorQuark = g_markup_error_quark;

/// Escapes text so that the markup parser will parse it verbatim.
/// Less than, greater than, ampersand, etc. are replaced with the
/// corresponding entities. This function would typically be used
/// when writing out a file to be parsed with the markup parser.
///
/// Note that this function doesn't protect whitespace and line endings
/// from being processed according to the XML rules for normalization
/// of line endings and attribute values.
///
/// Note also that this function will produce character references in
/// the range of &`x1`; ... &`x1f`; for all control sequences
/// except for tabstop, newline and carriage return.  The character
/// references in this range are not valid XML 1.0, but they are
/// valid XML 1.1 and will be accepted by the GMarkup parser.
extern fn g_markup_escape_text(p_text: [*:0]const u8, p_length: isize) [*:0]u8;
pub const markupEscapeText = g_markup_escape_text;

/// Formats arguments according to `format`, escaping
/// all string and character arguments in the fashion
/// of `glib.markupEscapeText`. This is useful when you
/// want to insert literal strings into XML-style markup
/// output, without having to worry that the strings
/// might themselves contain markup.
///
/// ```
/// const char *store = "Fortnum & Mason";
/// const char *item = "Tea";
/// char *output;
///
/// output = g_markup_printf_escaped ("<purchase>"
///                                   "<store>`s`</store>"
///                                   "<item>`s`</item>"
///                                   "</purchase>",
///                                   store, item);
/// ```
extern fn g_markup_printf_escaped(p_format: [*:0]const u8, ...) [*:0]u8;
pub const markupPrintfEscaped = g_markup_printf_escaped;

/// Formats the data in `args` according to `format`, escaping
/// all string and character arguments in the fashion
/// of `glib.markupEscapeText`. See `glib.markupPrintfEscaped`.
extern fn g_markup_vprintf_escaped(p_format: [*:0]const u8, p_args: std.builtin.VaList) [*:0]u8;
pub const markupVprintfEscaped = g_markup_vprintf_escaped;

/// Checks whether the allocator used by `glib.malloc` is the system's
/// malloc implementation. If it returns `TRUE` memory allocated with
/// `malloc` can be used interchangeably with memory allocated using `glib.malloc`.
/// This function is useful for avoiding an extra copy of allocated memory returned
/// by a non-GLib-based API.
extern fn g_mem_is_system_malloc() c_int;
pub const memIsSystemMalloc = g_mem_is_system_malloc;

/// GLib used to support some tools for memory profiling, but this
/// no longer works. There are many other useful tools for memory
/// profiling these days which can be used instead.
extern fn g_mem_profile() void;
pub const memProfile = g_mem_profile;

/// This function used to let you override the memory allocation function.
/// However, its use was incompatible with the use of global constructors
/// in GLib and GIO, because those use the GLib allocators before main is
/// reached. Therefore this function is now deprecated and is just a stub.
extern fn g_mem_set_vtable(p_vtable: *glib.MemVTable) void;
pub const memSetVtable = g_mem_set_vtable;

/// Allocates `byte_size` bytes of memory, and copies `byte_size` bytes into it
/// from `mem`. If `mem` is `NULL` it returns `NULL`.
extern fn g_memdup(p_mem: ?*const anyopaque, p_byte_size: c_uint) ?*anyopaque;
pub const memdup = g_memdup;

/// Allocates `byte_size` bytes of memory, and copies `byte_size` bytes into it
/// from `mem`. If `mem` is `NULL` it returns `NULL`.
///
/// This replaces `glib.memdup`, which was prone to integer overflows when
/// converting the argument from a `gsize` to a `guint`.
extern fn g_memdup2(p_mem: ?*const anyopaque, p_byte_size: usize) ?*anyopaque;
pub const memdup2 = g_memdup2;

/// A wrapper for the POSIX `mkdir` function. The `mkdir` function
/// attempts to create a directory with the given name and permissions.
/// The mode argument is ignored on Windows.
///
/// See your C library manual for more details about `mkdir`.
extern fn g_mkdir(p_filename: [*:0]const u8, p_mode: c_int) c_int;
pub const mkdir = g_mkdir;

/// Create a directory if it doesn't already exist. Create intermediate
/// parent directories as needed, too.
extern fn g_mkdir_with_parents(p_pathname: [*:0]const u8, p_mode: c_int) c_int;
pub const mkdirWithParents = g_mkdir_with_parents;

/// Creates a temporary directory in the current directory.
///
/// See the [``mkdtemp``](man:mkdtemp(3)) documentation on most UNIX-like systems.
///
/// The parameter is a string that should follow the rules for
/// `mkdtemp` templates, i.e. contain the string "XXXXXX".
/// `glib.mkdtemp` is slightly more flexible than `mkdtemp` in that the
/// sequence does not have to occur at the very end of the template.
/// The X string will be modified to form the name of a directory that
/// didn't exist.
/// The string should be in the GLib file name encoding. Most importantly,
/// on Windows it should be in UTF-8.
///
/// If you are going to be creating a temporary directory inside the
/// directory returned by `glib.getTmpDir`, you might want to use
/// `glib.dirMakeTmp` instead.
extern fn g_mkdtemp(p_tmpl: [*:0]u8) ?[*:0]u8;
pub const mkdtemp = g_mkdtemp;

/// Creates a temporary directory in the current directory.
///
/// See the [``mkdtemp``](man:mkdtemp(3)) documentation on most UNIX-like systems.
///
/// The parameter is a string that should follow the rules for
/// `mkdtemp` templates, i.e. contain the string "XXXXXX".
/// `glib.mkdtempFull` is slightly more flexible than `mkdtemp